/*****************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: tms
 *
 * FILE: rules.h
 *
 * ABSTRACT:
 * There are 6 types of rules -- 2 for tms_nodes, 2 for cells and 2 for
 * maintaining consistency with the outside world.
 *
 * 1) Node_rule ---
 *    <name fn priority tms_node>
 *    Attached to a node.  
 *    Fires the first time the node comes IN (first only once).
 *    The "fn" is called with the tms_node.
 *
 * 2) Node_constraint_rule ---
 *    <name fn priority consequent_cell antecedent_nodes>
 *    Attached to one of the antecedent tms_nodes.
 *    Fires when that node comes IN and if the rest of the
 *    antecedents are IN. Otherwise, attaches itself to one of the OUT nodes,
 *    waiting until all antecedent nodes are IN at the same time.
 *    DOES NOT FIRE if the node it is currently attached to is itself derived
 *    by "name" (i.e. the informant of the derivation is the same as the
 *    "name"). NOTE: thus, "name" should be unique for each
 *    invocation of this rule (see function ...).
 *    The "fn" is called with the consequent cell and list of antecedent_nodes.
 *
 * 3) Cell_rule ---
 *    <name fn priority>
 *    Attached to a cell.
 *    Fires when a new tms_node is added to the cell (see file "cells").
 *    NOTE: the node does not have to be IN for the cell_rule to fire.
 *    Adds a node_rule to the tms_node added.
 *
 * 4) Constraint_rule ---
 *    <name fn priority consequent_cell antecedent_cells>
 *    Attached to each of the antecendent cells.
 *    Fires on the cross product of all antecendents (i.e. whenever a 
 *    tms_node is added to an antecedent cell, create the cross product of 
 *    that node and all other nodes of the other antecedents).
 *    Adds a node_constraint_rule to the antecedent tms_nodes. 
 * 
 * 5) In_rule (for maintaining consistency with the outside world) ---
 *    <name fn priority>
 *    Attached to a cell.  
 *    Fires whenever a node in that cell's contextual_value list comes IN, 
 *    unless the cell is a ConstrainedCell, in which case the node must 
 *    also be the most constrained. The "fn" is called with the node IN-ned.
 *
 * 6) Out_rule (for maintaining consistency with the outside world) ---
 *    <name fn priority>
 *    Attached to a cell.  
 *    Fires whenever a node in that cell's contextual_value list goes OUT
 *    (it is not fired if the node is initially OUT when the rule is added)
 *    Argument is the node OUT-ted.
 *
 * In addition, all rules can be parameterized, so that the same "fn" can be
 * used for different rule instances.  For example, a rule to apply a function
 * to the value of the arguments can be parameterized with different functions,
 * such as "max", "min", etc.
 *
 * For parameterized rules, the rule "fn" is called with the arguments as 
 * indicated above, followed by a pointer to a structure containing the 
 * parameters.
 *
 * REVISION HISTORY
 *
 * $Log: rules.h,v $
 * Revision 1.14  1996/01/30  15:04:57  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.13  1996/01/27  21:54:20  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.12  1995/12/17  20:22:10  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.11  1995/07/06  21:17:21  rich
 * Solaris and Linux changes.
 *
 * Revision 1.10  1995/05/31  19:36:44  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.9  1995/01/18  22:42:45  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.8  1994/11/07  04:26:50  rich
 * Committing Reid's changes needed to use the qlattice for xavier navigation.
 *
 * Revision 1.7  1994/05/17  23:17:38  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.6  1993/12/14  17:35:13  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/12/01  18:04:26  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.4  1993/11/21  20:19:32  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:16:55  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:12  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:48  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:15  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:25:42  fedor
 * Added Logging.
 *
 * 03-Jul-1991 Reid Simmons at School of Computer Science, CMU
 * Added a "used" slot to rule structures to facilitate memory management
 *
 * $Revision: 1.14 $
 * $Date: 1996/01/30 15:04:57 $
 * $Author: rich $
 *
 *****************************************************************************/

#ifndef INCrules
#define INCrules

typedef enum {
  NodeRule=0, NodeConstraintRule=1, CellRule=2, 
  ConstraintRule=3, InRule=4, OutRule=5 
#ifdef FORCE_32BIT_ENUM
    , dummyRuleClass = 0x7FFFFFFF
#endif
} RULE_CLASS_TYPE;


typedef struct { 
  CELL_PTR consequentCell;
  LIST_PTR antecedentList;
} ADDL_CONSTRAINT_TYPE, *ADDL_CONSTRAINT_PTR;

typedef void 
(* CELL_RULE_FN)(const char *, CELL_PTR,
		 LIST_PTR, const char *);
typedef void 
(* NODE_RULE_FN)(const char *, TMS_NODE_PTR, const char *);

typedef union { 
  CELL_RULE_FN cell_rule; 
  NODE_RULE_FN node_rule; 
} RULE_FN;

typedef struct {
  RULE_CLASS_TYPE rule_class;
  const char *name;
  RULE_FN fn;
  int32 priority;
  const char *parameters;
  /* 2-Mar-90: fedor may want to try a union here */
  TMS_NODE_PTR node;
  LIST_PTR innedNodes; 
  LIST_PTR outtedNodes;
  ADDL_CONSTRAINT_PTR addlConstraints;
  int32 used; /* Number of applications of this rule */
} RULE_TYPE, *RULE_PTR;


/* The physical size of the Tms_Priority_Queues array */
#define MAX_POSSIBLE_PRIORITY 20

#define PRIORITY_QUEUE(index) GET_S_GLOBAL(tmsPriorityQueuesGlobal)[index-1]

/**********************************************************************
 * Execute the 'body' without firing rules.
 **********************************************************************/
#define DELAY_RULES(body)	\
{ int32 tmpDelay;		\
    tmpDelay = GET_S_GLOBAL(isDelayedGlobal); \
    GET_S_GLOBAL(isDelayedGlobal) = TRUE;     \
    body;			\
    GET_S_GLOBAL(isDelayedGlobal) = tmpDelay;	\
    runRules(-1);		\
}

RULE_PTR ruleCreateNode(const char *name, NODE_RULE_FN function, int32 priority,
			TMS_NODE_PTR node, const void *parameters);
RULE_PTR ruleCreateNodeConstraint(const char *name, NODE_RULE_FN function, 
				  int32 priority, CELL_PTR consequentCell, 
				  LIST_PTR antecedentNodes,
				  const void *parameters);
RULE_PTR ruleCreateConstraint(const char *name, CELL_RULE_FN function,
			      int32 priority, CELL_PTR consequentCell, 
			      LIST_PTR antecedentCells,
			      const char *parameters);
RULE_PTR ruleCreateCell(const char *name, CELL_RULE_FN function,
			int32 priority, const void *parameters);
RULE_PTR ruleCreateIn(const char *name, NODE_RULE_FN function, 
		      int32 priority, const void *parameters);
RULE_PTR ruleCreateOut(const char *name, NODE_RULE_FN function, 
		       int32 priority, const void *parameters);
void rulesFree(LIST_PTR ruleList);
void putRuleOnCell(RULE_PTR rule, CELL_PTR cell);
void addRuleToCell(RULE_PTR rule, CELL_PTR cell);
void addRuleToNode(RULE_PTR rule, TMS_NODE_PTR node);
void fireCellRules(TMS_NODE_PTR node, CELL_PTR cell);
void addInRule(RULE_PTR rule, CELL_PTR cell);
void addOutRule(RULE_PTR rule, CELL_PTR cell);
void runInRules(TMS_NODE_PTR node, CELL_PTR cell);
void runOutRules(TMS_NODE_PTR node, CELL_PTR cell);
void rulesInit(int32 numRuleQueues);
int32 runRules(int32 highestPriority);
void removeOutRule(NODE_RULE_FN ruleProcedure, CELL_PTR cell);
void ruleFree(RULE_PTR *rule);

#endif /* INCrules */
