/*****************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: tms
 *
 * FILE: rules.c
 *
 * ABSTRACT:
 * There are 6 types of rules -- 2 for tms_nodes, 2 for cells and 2 
 * for maintaining consistency with the outside world.
 *
 * 1) Node_rule ---
 *    <name fn priority tms_node>
 *    Attached to a node.  
 *    Fires the first time the node comes IN (first only once).
 *    The "fn" is called with the name and the tms_node (the "rule-name"
 *    should be used as the "informant" to any justifications created
 *    by the rule function).
 *
 * 2) Node_constraint_rule ---
 *    <name fn priority consequent_cell antecedent_nodes>
 *    Attached to one of the antecedent tms_nodes.
 *    Fires when that node comes IN and if the rest of the
 *    antecedents are IN. Otherwise, attaches itself to one of the OUT nodes,
 *    waiting until all antecedent nodes are IN at the same time.
 *    DOES NOT FIRE if the node it is currently attached to is itself derived
 *    by "name" (i.e. the informant of the derivation is the same as the
 *    "name"). NOTE: thus, "name" should be unique for each
 *    invocation of this rule.
 *
 *    The "fn" is called with the name, the consequent cell, and the
 *    list of antecedent_nodes (the "rule-name" should be used as the
 *    "informant" to any justifications created by the rule function).
 *
 * 3) Cell_rule ---
 *    <name fn priority>
 *    Attached to a cell.
 *    Fires when a new tms_node is added to the cell (see file "cells").
 *    NOTE: the node does not have to be IN for the cell_rule to fire.
 *    Adds a node_rule to the tms_node added.
 *  
 * 4) Constraint_rule ---
 *    <name fn priority consequent_cell antecedent_cells>
 *    Attached to each of the antecendent cells.
 *    Fires on the cross product of all antecendents (i.e. whenever a
 *    tms_node is added to an antecedent cell, create the cross product
 *    of that node and all other nodes of the other antecedents).
 *    Adds a node_constraint_rule to the antecedent tms_nodes. 
 *  
 * 5) In_rule (for maintaining consistency with the outside world) ---
 *    <name fn priority>
 *    Attached to a cell.  
 *    Fires whenever a node in that cell's contextual_value list comes IN, 
 *    unless the cell is a ConstrainedCell, in which case the node must also 
 *    be the most constrained.
 *
 *    The "fn" is called with the name and the node IN-ned.
 * 
 * 6) Out_rule (for maintaining consistency with the outside world) ---
 *    <name fn priority>
 *    Attached to a cell.  
 *    Fires whenever a node in that cell's contextual_value list goes OUT
 *    (it is not fired if the node is initially OUT when the rule is added)
 *    The "fn" is called with the name and the node OUT-ted.
 *  
 *  In addition, all rules can be parameterized, so that the same "fn" can be
 *  used for different rule instances.  For example, a rule to apply a function
 *  to the value of the arguments can be parameterized with different
 *  functions, such as "max", "min", etc.
 *
 *  For parameterized rules, the rule "fn" is called with the arguments as 
 *  indicated above, followed by a pointer to a structure containing the 
 *  parameters.
 *
 *  Finally, the global variable "The_Rule_Informant" is set to the rule-name
 *  of the rule currently being run.  It can be used to create justifications.
 *  It is especially useful for constraint rules, as the rule will not fire if
 *  one of the antecedents is justified by the rule name (which should
 *  therefore be unique for each invocation of the rule).
 *
 * EXPORTS:
 *
 * User Interface Functions:
 *
 * rulesInit()
 * Initialize variables used by the rule functions.  
 * This must be called prior to the first use the rule system.
 *
 * runRules(highest_priority)
 * Run all the rules whose priority is no greater than "highest_priority".
 * The rule mechanism supports prioritized queues -- each queue is given 
 * a priority number.  The function runRules takes an argument which
 * indicates that only queues with that priority or lower should be run.
 * If the argument is 0, defaults to the maximum priority, that is, all the 
 * queues are run.
 *
 * Create_Node_Rule (name, function, priority, tms_node, parameters)
 *
 * Create_Node_Constraint_Rule (name, function, priority, consequent_cell, 
 *                              antecedent_nodes, parameters)
 *
 * Create_Cell_Rule (name, function, priority, parameters)
 *
 * Create_Constraint_Rule (name, function, priority, consequent_cell,
 *                         antecedent_cells, parameters)
 *
 * Create_In_Rule (name, function, priority, parameters)
 *
 * Create_Out_Rule (name, function, priority, parameters)
 *
 * Free_Rules (rule_list, cell)
 *
 * REVISION HISTORY
 *
 * $Log: rules.c,v $
 * Revision 1.15  1995/12/17  20:22:08  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.14  1995/12/15  01:23:23  rich
 * Moved Makefile to Makefile.generic to encourage people to use
 * GNUmakefile.
 * Fixed a memory leak when a module is closed and some other small fixes.
 *
 * Revision 1.13  1995/07/06  21:17:19  rich
 * Solaris and Linux changes.
 *
 * Revision 1.12  1995/06/14  03:22:21  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.11  1995/05/31  19:36:42  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.10  1995/01/18  22:42:39  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.9  1994/11/07  04:26:42  rich
 * Committing Reid's changes needed to use the qlattice for xavier navigation.
 *
 * Revision 1.8  1994/05/17  23:17:36  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.7  1994/04/28  16:17:16  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.6  1993/12/14  17:35:10  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/11/21  20:19:29  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/10/21  16:14:25  rich
 * Fixed compiler warnings.
 *
 * Revision 1.3  1993/08/27  07:16:50  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:19:09  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:47  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:32:12  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:25:38  fedor
 * Added Logging.
 *
 *  3-Jul-1991 Reid Simmons at School of Computer Science, CMU
 * Added a "used" slot to rule structures to facilitate memory management
 * Also, create the innedNodes and outtedNodes lists only for the relevant type
 *   of rule.
 *
 *  7-Nov-89  Christopher Fedor at School of Computer Science, CMU
 * Revised to Software Standards
 *
 * Ancient    Reid Simmons  at School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.15 $
 * $Date: 1995/12/17 20:22:08 $
 * $Author: rich $
 *
 *****************************************************************************/


#include "globalS.h"
#include "tcaMem.h"
#include "rules.h"


/*****************************************************************************
 *
 * FUNCTION:
 * RULE_PTR ruleCreate(ruleClass, ruleName, function, priority, parameters)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_CLASS_TYPE ruleClass;
 * const char *ruleName;
 * RULE_FN function; 
 * int32 priority;
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/

/* the version of the compiler on the sun4 machines is very old.
 * It won't allow you to cast a NODE_RULE_FN to a RULE_FN, so I have
 * put in this workaround until the new compiler is available.
 */

#if __GNUC__ != 2
static RULE_PTR ruleCreate(RULE_CLASS_TYPE class,
			   const char *name,
			   NODE_RULE_FN function,
			   int32 priority,
			   const void *parameters)
#else
     static RULE_PTR ruleCreate(RULE_CLASS_TYPE class,
				const char *name,
				RULE_FN function,
				int32 priority,
				const void *parameters)
#endif
{
  RULE_PTR rule;
  
  rule = NEW(RULE_TYPE);
  rule->rule_class = class;
  rule->name = String_Copy(name);
#if __GNUC__ != 2
  rule->fn.node_rule = function;
#else
  rule->fn = function;
#endif
  rule->priority = priority;
  rule->parameters = parameters;
  
  
  /* 2-Mar-90: fedor: more thought can make these a union and avoid
     initial list creation. */
  
  rule->node = NULL;
  rule->innedNodes = NULL;
  rule->outtedNodes = NULL;
  
  rule->addlConstraints = NULL;
  
  rule->used = 0;
  
  return rule;
}


/*****************************************************************************
 *
 * FUNCTION: RULE_PTR ruleCreateNode(name, function, priority,
 *                                   node, parameters)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * RULE_FN function; 
 * int32 priority; 
 * TMS_NODE_PTR node; 
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/

RULE_PTR ruleCreateNode(const char *name, NODE_RULE_FN function, int32 priority,
			TMS_NODE_PTR node, const void *parameters)
{ 
  RULE_PTR rule;
  
#if __GNUC__ != 2
  rule = ruleCreate(NodeRule, name, 
		    (NODE_RULE_FN) function, priority, parameters);
#else
  rule = ruleCreate(NodeRule, name, (RULE_FN) function, priority, parameters);
#endif
  rule->node = node;
  
  return rule;
}


/*****************************************************************************
 *
 * FUNCTION:
 * RULE_PTR ruleCreateNodeConstraint(name, function, priority, consequentCell, 
 * 				  antecedentNodes, parameters)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * NODE_RULE_FN function; 
 * int32 priority; 
 * CELL_PTR consequentCell; 
 * LIST_PTR antecedentNodes; 
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/

RULE_PTR ruleCreateNodeConstraint(const char *name, 
				  NODE_RULE_FN function, 
				  int32 priority,
				  CELL_PTR consequentCell,
				  LIST_PTR antecedentNodes, 
				  const void *parameters)
{ 
  RULE_PTR rule;
  
#if __GNUC__ != 2
  rule = ruleCreate(NodeConstraintRule, name,(NODE_RULE_FN) function, 
		    priority, parameters);
#else
  rule = ruleCreate(NodeConstraintRule, name, (RULE_FN) function, 
		    priority, parameters);
#endif
  rule->addlConstraints = NEW(ADDL_CONSTRAINT_TYPE);
  rule->addlConstraints->consequentCell = consequentCell;
  rule->addlConstraints->antecedentList = antecedentNodes;
  
  return rule;
}


/*****************************************************************************
 *
 * FUNCTION: RULE_PTR ruleCreateCell(name, function, priority, parameters)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * CELL_RULE_FN function; 
 * int32 priority; 
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/

RULE_PTR ruleCreateCell(const char *name, CELL_RULE_FN function,
			int32 priority, const void *parameters)
{ 
#if __GNUC__ != 2
  return ruleCreate(CellRule, name, (NODE_RULE_FN) function, 
		    priority, parameters);
#else
  return ruleCreate(CellRule, name, (RULE_FN) function, priority, parameters);
#endif
}


/*****************************************************************************
 *
 * FUNCTION:
 * RULE_PTR ruleCreateConstraint(name, function, priority, consequentCell, 
 *			      antecedentCells, parameters)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * CELL_RULE_FN function; 
 * int32 priority; 
 * CELL_PTR consequentCell; 
 * LIST_PTR antecedentCells; 
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/
RULE_PTR ruleCreateConstraint(const char *name, CELL_RULE_FN function,
			      int32 priority, CELL_PTR consequentCell, 
			      LIST_PTR antecedentCells, const char *parameters)
{
  RULE_PTR rule;
  
#if __GNUC__ != 2
  rule = ruleCreate(ConstraintRule, name, (NODE_RULE_FN) function,
		    priority, parameters);
#else
  rule = ruleCreate(ConstraintRule, name, (RULE_FN) function,
		    priority, parameters);
#endif
  rule->addlConstraints = NEW(ADDL_CONSTRAINT_TYPE);
  rule->addlConstraints->consequentCell = consequentCell;
  rule->addlConstraints->antecedentList = antecedentCells;
  
  return rule;
}

/*****************************************************************************
 *
 * FUNCTION: RULE_PTR ruleCreateIn(name, function, priority, parameters)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name;
 * NODE_RULE_FN function; 
 * int32 priority; 
 * char *parameters;
 *
 * OUTPUTS: RULE_PTR
 *
 *****************************************************************************/

RULE_PTR ruleCreateIn(const char *name, NODE_RULE_FN function, 
		      int32 priority, const void *parameters)
{ 
  RULE_PTR rule;
  
#if __GNUC__ != 2
  rule = ruleCreate(InRule, name, (NODE_RULE_FN) function, 
		    priority, parameters);
#else
  rule = ruleCreate(InRule, name, (RULE_FN) function, priority, parameters);
#endif
  rule->innedNodes = listCreate();
  
  return rule;
}


/*****************************************************************************
 *
 * FUNCTION:
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

RULE_PTR ruleCreateOut(const char *name, NODE_RULE_FN function, 
		       int32 priority, const void *parameters)
{ 
  RULE_PTR rule;
  
#if __GNUC__ != 2
  rule = ruleCreate(OutRule, name, (NODE_RULE_FN) function, 
		    priority, parameters);
#else
  rule = ruleCreate(OutRule, name, (RULE_FN) function, priority, parameters);
#endif
  rule->outtedNodes = listCreate();
  
  return rule;
}


/*****************************************************************************
 *
 * FUNCTION: void ruleFree(rule) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: RULE_PTR rule
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void ruleFree(RULE_PTR *rule)
{
  if ((*rule) == NULL) return;

  tcaFree((char *)(*rule)->name);
  switch ((*rule)->rule_class) {
  case NodeConstraintRule:
  case ConstraintRule: 
    listFree(&((*rule)->addlConstraints->antecedentList));
    tcaFree((char *)(*rule)->addlConstraints);
    break;
  default:
    break;
  };
  
  listFree(&((*rule)->innedNodes));
  listFree(&((*rule)->outtedNodes));
  
  tcaFree((char *)(*rule));
  *rule = NULL;
}


/*****************************************************************************
 *
 * FUNCTION: void rulesFree(rules) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * LIST_PTR rules; 
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void rulesFree(LIST_PTR ruleList)
{
  RULE_PTR rule;
  
  rule = (RULE_PTR)listFirst(ruleList);
  while (rule) {
    rule->used--;
    if (rule->used == 0) 
      ruleFree(&rule);
    rule = (RULE_PTR)listNext(ruleList);
  }
  
  listFree(&(ruleList));
}

/*****************************************************************************
 *
 * FUNCTION: void oneByN(value, subLists, crossProduct)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * char *value; 
 * LIST_PTR subLists; 
 * LIST_PTR crossProduct;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void oneByN(void *value, LIST_PTR subLists, LIST_PTR product)
{
  LIST_PTR list, subproduct;
  
  if (subLists) {
    list = (LIST_PTR)listFirst(subLists);
    while (list) {
      subproduct = listCopy(list);
      listInsertItemLast((char *)value, subproduct);
      listInsertItemFirst((char *)subproduct, product);
      list = (LIST_PTR)listNext(subLists);
    }
  } else {
    listInsertItemFirst((char *)listMake1(value), product);
  }
}


/*****************************************************************************
 *
 * FUNCTION: LIST_PTR createCrossProductOfContextualValues(cell, cellNode,
 *                                                         otherCells)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * TMS_NODE_PTR cellNode; 
 * LIST_PTR otherCells;
 *
 * OUTPUTS: LIST_PTR
 *
 *****************************************************************************/

static LIST_PTR createCrossProductOfContextualValues(CELL_PTR cell, 
						     TMS_NODE_PTR cellNode,
						     LIST_PTR otherCells)
{ 
  TMS_NODE_PTR node;
  CELL_PTR nextCell;
  LIST_PTR crossProduct, subCrossProduct;
  
  subCrossProduct = NULL;
  crossProduct = listCreate();
  
  nextCell = (CELL_PTR)listFirst(otherCells);
  while (nextCell) {
    if (cell == nextCell) 
      oneByN((char *)cellNode, subCrossProduct, crossProduct);
    else {
      node = (TMS_NODE_PTR)listFirst(nextCell->contextualValues);
      while (node) {
	oneByN((char *)node, subCrossProduct, crossProduct);
	node = (TMS_NODE_PTR)listNext(nextCell->contextualValues);
      }
    }
    if (subCrossProduct) listFree(&subCrossProduct);
    subCrossProduct = crossProduct;
    crossProduct = listCreate();
    nextCell = (CELL_PTR)listNext(otherCells);
  }
  
  crossProduct = subCrossProduct;
  return crossProduct;
  
  /* 24-Feb-90: fedor: memory problems ... */
  
}


/*****************************************************************************
 *
 * FUNCTION: int32 isMostConstrainedNode(node) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 isMostConstrainedNode(TMS_NODE_PTR node)
{ 
  CELL_PTR cell;
  
  cell = (CELL_PTR)node->cell;
  return (!cell ||
	  cell->type != ConstrainedCell ||
	  (cell->sameValueFN)(node->datum,
			      CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell)));
}


/*****************************************************************************
 *
 * FUNCTION: void callNodeRule(rule, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none
 *
 *****************************************************************************/

static void callNodeRule(RULE_PTR rule, TMS_NODE_PTR node)
{
  (rule->fn.node_rule)(rule->name, node, rule->parameters);
  /* A node rule is run just once -- so we can free it */
  if (rule->rule_class == NodeRule) 
    ruleFree(&rule);
}


/*****************************************************************************
 *
 * FUNCTION: void callConstraintRule(rule, antecedents)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * LIST_PTR antecedents;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void callConstraintRule(RULE_PTR rule, LIST_PTR antecedents)
{ 
  (rule->fn.cell_rule)(rule->name, rule->addlConstraints->consequentCell,
		       antecedents, rule->parameters);
  /* A node rule is run just once -- so we can free it */
  if (rule->rule_class == NodeConstraintRule) 
    ruleFree(&rule);
}


/*****************************************************************************
 *
 * FUNCTION: int32 isFireable(node)
 * 
 * DESCRIPTION:
 * Returns T if the node is IN and is the most constrained 
 * value, or is IN and not part of a constrained cell.
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 isFireable(TMS_NODE_PTR node)
{ 
  return (IS_IN(node) && isMostConstrainedNode(node));
}


/*****************************************************************************
 *
 * FUNCTION: void putRuleOnNode(rule, node) 
 * 
 * DESCRIPTION:
 * Load this rule into the rules slot of the node (to await inning or outing)
 *
 * INPUTS:
 * RULE_PTR rule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void putRuleOnNode(RULE_PTR rule, TMS_NODE_PTR node)
{ 
  rule->used++;
  listInsertItem((const char *)rule, node->rules);
}


/*****************************************************************************
 *
 * FUNCTION: void putRuleOnCell(rule, cell)
 * 
 * DESCRIPTION:
 * Load this rule onto the rules slot of the cell (to await firing)
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void putRuleOnCell(RULE_PTR rule, CELL_PTR cell)
{ 
  rule->used++;
  listInsertItemFirst((const char *)rule, cell->rules); 
}


/*****************************************************************************
 *
 * FUNCTION: void tryRunningRule(rule)
 * 
 * DESCRIPTION: Run a node_rule or a node_constraint_rule if it is IN.
 *
 * INPUTS: RULE_PTR rule;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void tryRunningRule(RULE_PTR rule)
{ 
  RULE_CLASS_TYPE ruleClass;
  LIST_PTR innedNodes, outtedNodes;
  TMS_NODE_PTR ruleNode, antecedent, innedNode, outtedNode;
  
  if (GET_S_GLOBAL(ruleTraceGlobal)) 
    (void)printf("%d", ++GET_S_GLOBAL(numRulesRunGlobal));
  
  ruleClass = rule->rule_class;
  if (ruleClass == NodeRule) {
    ruleNode = rule->node;
    rule->used--;
    
    if (IS_IN(ruleNode) && isMostConstrainedNode(ruleNode)) {
      if (GET_S_GLOBAL(ruleTraceGlobal)) 
	(void)printf("n");
      callNodeRule(rule, ruleNode);
    }
    else 
      putRuleOnNode(rule, ruleNode);
  }
  else if (ruleClass == NodeConstraintRule) {
    
    antecedent = 
      (TMS_NODE_PTR)listFirst(rule->addlConstraints->antecedentList);
    while (antecedent) {
      rule->used--;
      if (IS_OUT(antecedent) ||
	  sameReason(rule->name, antecedent->supportJustification) ||
	  !isMostConstrainedNode(antecedent)) {
	putRuleOnNode(rule, antecedent);
	return;
      }
      antecedent = 
	(TMS_NODE_PTR)listNext(rule->addlConstraints->antecedentList);
    }
    
    if (GET_S_GLOBAL(ruleTraceGlobal)) 
      (void)printf("c");
    callConstraintRule(rule, rule->addlConstraints->antecedentList);
  }
  else if (ruleClass == InRule) {
    innedNodes = rule->innedNodes;
    /* 2-Mar-90: fedor: does this rule have to a have an empty 
       Node list before executing? */
    rule->innedNodes = listCreate();
    
    innedNode = (TMS_NODE_PTR)listFirst(innedNodes);
    while (innedNode) {
      if (isFireable(innedNode)) {
	if (GET_S_GLOBAL(ruleTraceGlobal)) 
	  (void)printf("i");
	callNodeRule(rule, innedNode);
      }
      innedNode = (TMS_NODE_PTR)listNext(innedNodes);
    }
    listFree(&innedNodes);
  }
  else if (ruleClass == OutRule) {
    outtedNodes = rule->outtedNodes;
    /* 2-Mar-90: fedor: does this rule have to a have an empty 
       Node list before executing? */
    rule->outtedNodes = listCreate();
    
    outtedNode = (TMS_NODE_PTR)listFirst(outtedNodes);
    while (outtedNode) {
      if (IS_OUT(outtedNode)) {
	if (GET_S_GLOBAL(ruleTraceGlobal)) 
	  (void)printf("o");
	callNodeRule(rule, outtedNode);
      }
      outtedNode = (TMS_NODE_PTR)listNext(outtedNodes);
    }
    listFree(&outtedNodes);
  }
  else 
    tcaError("Trying to run an unknown rule type");
}


/*****************************************************************************
 *
 * FUNCTION: int32 fireImmediately(rule)
 * 
 * DESCRIPTION:
 *
 * INPUTS: RULE_PTR rule;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 fireImmediately(RULE_PTR rule)
{ 
  return (rule->priority < 0);
}


/*****************************************************************************
 *
 * FUNCTION: void enqueueRule(rule) 
 * 
 * DESCRIPTION:
 * Enqueue this rule on the appropriate priority 
 * queue, or try to run now if zero priority.
 *
 * INPUTS: RULE_PTR rule;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void enqueueRule(RULE_PTR rule)
{ 
  if (rule->priority == 0) {
    if (GET_S_GLOBAL(ruleTraceGlobal)) 
      (void)printf("!");
    tryRunningRule(rule);
  }
  else 
    Enqueue((void *)rule, PRIORITY_QUEUE(rule->priority));
}


/*****************************************************************************
 *
 * FUNCTION: void addRuleToNode(rule, node)
 * 
 * DESCRIPTION:
 * Put this rule on a rule_queue if it is ready to be fired, 
 * or onto the node's rules list if not ready to be fired
 *
 * INPUTS:
 * RULE_PTR rule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void addRuleToNode(RULE_PTR rule, TMS_NODE_PTR node)
{ 
  if (isFireable(node)) 
    enqueueRule(rule);
  else 
    putRuleOnNode(rule, node);
}


/*****************************************************************************
 *
 * FUNCTION: void instantiateCellRule(rule, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void instantiateCellRule(RULE_PTR rule, TMS_NODE_PTR node)
{
  if (fireImmediately(rule))
    callNodeRule(rule, node);
  else
    addRuleToNode(ruleCreateNode(rule->name, rule->fn.node_rule,
				 rule->priority, node, rule->parameters),
		  node);
}


/*****************************************************************************
 *
 * FUNCTION: void instantiateConstraintRule(rule, nodes)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * LIST_PTR nodes;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void instantiateConstraintRule(RULE_PTR rule, LIST_PTR nodeList)
{
  if (fireImmediately(rule))
    callConstraintRule(rule, nodeList);
  else 
    addRuleToNode(ruleCreateNodeConstraint(rule->name, rule->fn.node_rule,
					   rule->priority, 
					   rule->addlConstraints->consequentCell,
					   nodeList, rule->parameters), 
		  (TMS_NODE_PTR)listFirst(nodeList));
}


/*****************************************************************************
 *
 * FUNCTION: void fireCellRule(rule, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void fireCellRule(RULE_PTR rule, TMS_NODE_PTR node)
{ 
  instantiateCellRule(rule, node);
}


/*****************************************************************************
 *
 * FUNCTION: void printNodeList(list) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: LIST_PTR list;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

#if 0
/* No longer used */
static void printNodeList(LIST_PTR list)
{ 
  TMS_NODE_PTR node;
  
  (void)printf("Node List ");
  
  node = (TMS_NODE_PTR)listFirst(list);
  while (node) {
    sayNodeValue(node); 
    (void)printf(" ");
    node = (TMS_NODE_PTR)listNext(list);
  }
  
  (void)printf("\n");
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void fireConstraintRule(rule, antecedentCell, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR antecedentCell; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void fireConstraintRule(RULE_PTR rule, CELL_PTR antecedentCell,
			       TMS_NODE_PTR node)
{ 
  int32 every;
  CELL_PTR antecedent;
  LIST_PTR antecedentList, crossProduct, ruleArgument;
  
  antecedentList = rule->addlConstraints->antecedentList;
  
  if (listLength(antecedentList) > 1) {
    every = TRUE;
    
    antecedent = (CELL_PTR)listFirst(antecedentList);
    while (antecedent && every) {
      every = every && (listLength(antecedent->contextualValues));
      antecedent = (CELL_PTR)listNext(antecedentList);
    }
    
    if (every) {
      crossProduct = createCrossProductOfContextualValues(antecedentCell, 
							  node, antecedentList);
      
      ruleArgument = (LIST_PTR)listFirst(crossProduct);
      while (ruleArgument) {
	instantiateConstraintRule(rule, ruleArgument);
	ruleArgument = (LIST_PTR)listNext(crossProduct);
      }
      listFree(&crossProduct);
    }
  }
  else 
    /* Only one antecedent, and its the one passed in. */
    instantiateConstraintRule(rule, listMake1((const char *)node));
}


/*****************************************************************************
 *
 * FUNCTION: void fireRule(rule, cell, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR cell; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void fireRule(RULE_PTR rule, CELL_PTR cell, TMS_NODE_PTR node)
{
  if (rule->rule_class == CellRule)
    fireCellRule(rule, node);
  else if (rule->rule_class == ConstraintRule)
    fireConstraintRule(rule, cell, node);
  else 
    perror("Trying to fire unknown rule type");
}


/*****************************************************************************
 *
 * FUNCTION: void fireCellRules(node, cell)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * CELL_PTR cell; 
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void fireCellRules(TMS_NODE_PTR node, CELL_PTR cell)
{ 
  RULE_PTR rule;
  
  rule = (RULE_PTR)listFirst(cell->rules);
  while (rule) {
    fireRule(rule, cell, node);
    rule = (RULE_PTR)listNext(cell->rules);
  }
}


/*****************************************************************************
 *
 * FUNCTION: void fireInRule(inRule, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * RULE_PTR inRule; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void fireInRule(RULE_PTR inRule, TMS_NODE_PTR node)
{ 
  LIST_PTR innedNodeList;
  int32 needToEnqueue;
  
  innedNodeList = inRule->innedNodes;
  
  needToEnqueue = listLength(innedNodeList) == 0;
  
  if (!listMemberItem((char *)node, innedNodeList))
    listInsertItemFirst((char *)node, innedNodeList);
  
  if (needToEnqueue) enqueueRule(inRule);
}


/*****************************************************************************
 *
 * FUNCTION: void addRuleToCell(rule, cell)
 * 
 * DESCRIPTION:
 * Put this rule on the cell's rules slot, then 
 * fire it for each CV owned by the cell.
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
void addRuleToCell(RULE_PTR rule, CELL_PTR cell)
{
  TMS_NODE_PTR node;
  
  if ((rule->rule_class == CellRule) || (rule->rule_class == ConstraintRule)) {
    putRuleOnCell(rule, cell);
    
    node = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (node) {
      fireRule(rule, cell, node);
      node = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
  }
  else 
    perror("Adding wrong type rule to cell");
}

/*****************************************************************************
 *
 * FUNCTION: void addInRule(rule, cell)
 * 
 * DESCRIPTION:
 * Add rule to cell's IN_RULE slot, fires for all CVs in CELL (Most
 * constrained only) inRules fire on values which are IN, except if the
 * cell is a ConstrainedCell, in which case only the
 * current_value_supporter is fired.
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void addInRule(RULE_PTR rule, CELL_PTR cell)
{
  TMS_NODE_PTR node;
  
  if (rule->rule_class != InRule) 
    perror("Not an in_rule");
  else { 
    rule->used++;
    listInsertItemFirst((char *)rule, cell->inRules);
    
    node = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (node) {
      if (isFireable(node)) 
	fireInRule(rule, node);
      node = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void addOutRule(rule, cell)
 * 
 * DESCRIPTION:
 * outRules do not fire on values which are already 
 * OUT at the time the rule is added.
 *
 * INPUTS:
 * RULE_PTR rule; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void addOutRule(RULE_PTR rule, CELL_PTR cell)
{
  if (rule->rule_class != OutRule) 
    perror("Not an out_rule");
  else  {
    rule->used++;
    listInsertItemFirst((char *)rule, cell->outRules);
  }
}


/*****************************************************************************
 *
 * FUNCTION: void runInRules(node, cell)
 * 
 * DESCRIPTION:
 * Run IN_RULES for node (and zap rules slot); If CV, run parent cell's inRules
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static int32 iterateEnqueueRule(void *param, RULE_PTR nodeRule)
{
#ifdef applec
#pragma unused(param)
#endif
  enqueueRule(nodeRule);
  return TRUE;
}

static int32 iterateFireInRule(TMS_NODE_PTR node, RULE_PTR inRule)
{
  fireInRule(inRule, node);
  return TRUE;
}

void runInRules(TMS_NODE_PTR node, CELL_PTR cell)
{ 
  LIST_PTR rules;
  
  rules = node->rules;
  node->rules = listCreate();
  
  (void)listIterate((LIST_ITER_FN)iterateEnqueueRule, (char *)NULL, rules);
  
  if (cell)
    (void)listIterate((LIST_ITER_FN)iterateFireInRule, 
		      (char *)node, cell->inRules);
  
  listFree(&rules);
}


/*****************************************************************************
 *
 * FUNCTION: void runOutRules(node, cell)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void runOutRules(TMS_NODE_PTR node, CELL_PTR cell)
{ 
  RULE_PTR outRule;
  LIST_PTR outtedNodeList;
  int32 needToEnqueue;
  
  if (cell) {
    outRule = (RULE_PTR)listFirst(cell->outRules);
    while (outRule) {
      outtedNodeList = outRule->outtedNodes;
      
      needToEnqueue = listLength(outtedNodeList) == 0;
      
      if (!listMemberItem((char *)node, outtedNodeList))
	listInsertItemFirst((char *)node, outtedNodeList);
      
      if (needToEnqueue) enqueueRule(outRule);
      
      outRule = (RULE_PTR)listNext(cell->outRules);
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void rulesInit(numRuleQueues) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: int32 numRuleQueues;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void rulesInit(int32 numRuleQueues)
{ 
  int32 index;
  
  if (numRuleQueues > MAX_POSSIBLE_PRIORITY)
    perror("Can not have that many rule queues");
  else { 
    for (index=0; index < numRuleQueues; index++)
      GET_S_GLOBAL(tmsPriorityQueuesGlobal)[index] = Create_Queue();
    GET_S_GLOBAL(numRuleQueuesGlobal) = numRuleQueues;
    GET_S_GLOBAL(numRulesRunGlobal) = 0;
    /* set_up_cell_rules(); */
    /* set_up_contradiction_handlers(); */
  }
}


/*****************************************************************************
 *
 * FUNCTION: int32 runRules(highestPriority) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: int32 highestPriority;
 *
 * OUTPUTS: int32 (TRUE / FALSE)
 *
 *****************************************************************************/

int32 runRules(int32 highestPriority)
{ 
  int32 priority;
  RULE_PTR rule;
  int32 wasRun = FALSE;
  
  if (highestPriority < 0) 
    highestPriority = GET_S_GLOBAL(numRuleQueuesGlobal);
  
  if (!GET_S_GLOBAL(isDelayedGlobal)) {
    GET_S_GLOBAL(isDelayedGlobal) = TRUE;
  A: priority = 1;
  B: rule = (RULE_PTR)Dequeue(PRIORITY_QUEUE(priority));
    if (rule) {
      wasRun = TRUE;
      tryRunningRule(rule);
      goto A;
    }
    else 
      if ((priority == highestPriority) || 
	  (priority == GET_S_GLOBAL(numRuleQueuesGlobal))) {
	GET_S_GLOBAL(isDelayedGlobal) = FALSE;
	return wasRun;
      }
      else {
	priority++;
	goto B;
      }
  }
  else 
    return FALSE;
}


/*****************************************************************************
 *
 * FUNCTION:
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

/* Currently not used (RGS: 11/11/92)
   void addConstraintRule(name, function, priority, consequentCell, 
   antecedentCells, parameters)
   const char *name; 
   CELL_RULE_FN function; 
   int32 priority; 
   CELL_PTR consequentCell; 
   LIST_PTR antecedentCells; 
   const char *parameters;
   { 
   CELL_PTR antecedentCell;
   RULE_PTR constraintRule;
   
   constraintRule = ruleCreateConstraint(name, function, priority, 
   consequentCell, antecedentCells, 
   parameters);
   
   addRuleToCell(constraintRule, (CELL_PTR)listFirst(antecedentCells));
   
   antecedentCell = (CELL_PTR)listNext(antecedentCells);
   while (antecedentCell) {
   putRuleOnCell(constraintRule, antecedentCell);
   antecedentCell = (CELL_PTR)listNext(antecedentCells);
   }
   }
   */

/*****************************************************************************
 *
 * FUNCTION: int32 sameRuleP(proc, rule)
 * 
 * DESCRIPTION:
 * returns TRUE if the proc equals the rule function, returns FALSE otherwise.
 *
 * INPUTS:
 * RULE_FN proc;
 * RULE_PTR rule;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 sameRuleP(RULE_FN proc, RULE_PTR rule)
{
  return (proc.cell_rule == rule->fn.cell_rule);
}


/*****************************************************************************
 *
 * FUNCTION: void removeOutRule(ruleProcedure, cell)
 * 
 * DESCRIPTION:
 * Remove all the "out rules" on the Cell that have the
 * Rule_Procedure as their function.
 *
 * INPUTS:
 * NODE_RULE_FN ruleProcedure;
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void removeOutRule(NODE_RULE_FN ruleProcedure, CELL_PTR cell)
{
  listTestDeleteItemAll((LIST_ITER_FN)sameRuleP, (const char *)ruleProcedure, 
			cell->outRules);
}


/*****************************************************************************
 *
 * FUNCTION: void removeInRule(ruleProcedure, cell)
 * 
 * DESCRIPTION:
 * Remove all the "in rules" on the Cell that have the 
 * Rule_Procedure as their function.
 *
 * INPUTS:
 * NODE_RULE_FN ruleProcedure;
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92)
   void removeInRule(ruleProcedure, cell)
   NODE_RULE_FN ruleProcedure;
   CELL_PTR cell;
   {
   listTestDeleteItemAll((LIST_ITER_FN)sameRuleP, 
   (char *)ruleProcedure, cell->inRules);
   }
   */

/*****************************************************************************
 *
 * FUNCTION: void removeCellRule(ruleProcedure, cell)
 * 
 * DESCRIPTION:
 * Remove all the "rules" on the Cell that have the 
 * Rule_Procedure as their function.
 *
 * INPUTS:
 * CELL_RULE_FN ruleProcedure;
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92)
   void removeCellRule(ruleProcedure, cell)
   CELL_RULE_FN ruleProcedure;
   CELL_PTR cell;
   {
   listTestDeleteItemAll((LIST_ITER_FN) sameRuleP,
   (char *)ruleProcedure, cell->rules);
   }
   */

/*****************************************************************************
 *
 * FUNCTION: void removeNodeRule(ruleProcedure, node)
 * 
 * DESCRIPTION:
 * Remove all the "rules" on the Tms_Node that have the 
 * Rule_Procedure as their function.
 *
 * INPUTS:
 * NODE_RULE_FN ruleProcedure;
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92)
   void removeNodeRule(ruleProcedure, node)
   NODE_RULE_FN ruleProcedure;
   TMS_NODE_PTR node;
   {
   listTestDeleteItemAll((LIST_ITER_FN) sameRuleP,
   (char *)ruleProcedure, node->rules);
   }
   */
