/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: 
 *
 * FILE: 
 *
 * ABSTRACT:
 This file contains functions to generate and manipulate queues, stacks,
 and priority queues.  It also contains code for `remembering' lists
 of nodes.
 
 The abstractions defined are QUEUE, STACK, PRIORITY_QUEUE and MEMORY.
 Each abstraction `X' supports the functions CREATE_X, FREE_X, 
 CLEAR_X, and IS_X_EMPTY.
 
 QUEUE additionally supports ENQUEUE and DEQUEUE, and it supports
 HEAD_ENQUEUE for getting items in a LIFO manner.
 
 STACK additionally supports PUSH_STACK and POP_STACK.
 
 PRIORITY_QUEUE supports PRIORITY_ENQUEUE and PRIORITY_DEQUEUE.
 
 MEMORY supports REMEMBER and DO_MEMORY (which applys a function to each
 `remembered' item in a last in, first out manner).
 
 29-Jul-91 Reid Simmons   Rewrote entirely to use Chris' list package.
 26-Jul-91 Reid Simmons   Fixed "head_enqueue"
 9-Aug-89 Reid Simmons   Fixed "extend_queue".
 2-Aug-89 Reid Simmons   Changed "printf"s to use logging facility.
 26-Jun-89 Reid Simmons   Added code to make QUEUES, STACKS, and MEMORIES 
 dynamically expandable.  
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/src/queue.c,v $ 
 * $Revision: 1.9 $
 * $Date: 1996/06/25 20:51:18 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: queue.c,v $
 * Revision 1.9  1996/06/25  20:51:18  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.8  1995/12/17  20:21:58  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.7  1995/07/06  21:17:04  rich
 * Solaris and Linux changes.
 *
 * Revision 1.6  1995/06/14  03:22:05  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.5  1995/03/30  15:43:53  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.4  1995/01/18  22:42:14  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.3  1993/08/27  07:16:25  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.5  1993/08/23  17:40:00  rich
 * Fixed the type definitions for function pointers. Added const
 * declarations.  Removed definitions VOID_FN and INT_FN.
 *
 * Revision 1.4  1993/07/08  05:39:05  rich
 * Added function prototypes
 *
 * Revision 1.3  1993/06/22  14:00:13  rich
 * Added makefile.depend.  Dependencies automatically generated using gcc.
 * Fixed some warnings.
 * Updated the -D<arch> flags to correspond to those generated
 * automatically by the makefile.
 * Changed system includes to the proper format "stdio.h" -> <stdio.h>.
 * This was needed so that the automatic dependency generation can
 * distinguish between "local" and system headers.  The location of the
 * system headers changes from architecture to architecture and should not
 * be included in the dependency list.
 *
 * Revision 1.2  1993/05/27  22:19:57  rich
 * Added automatic logging.
 *
 *****************************************************************************/

#include "globalM.h"
#include "queue.h"

/*
   .______________.
   .              .
   .    QUEUES    .
   .______________.
   
   */

queue_ptr Create_Queue(void) 
{ 
  queue_ptr new_queue;
  
  new_queue = NEW(queue_type);
  new_queue->list = listCreate();
  return new_queue;
}

/* Currently not used (RGS: 11/11/92) */
void Free_Queue(queue_ptr *queue)
{
  if ((*queue) == NULL) return;

  listFree(&((*queue)->list));
  tcaFree((char *) (*queue));
  *queue = NULL;
}

/* This is actually quite efficient, but a simple listClear operation
   would be cleaner */
void Clear_Queue (queue_ptr queue)
{
  listFree(&(queue->list));
  queue->list = listCreate();
}

/* Currently not used (RGS: 11/11/92)
   int32 Is_Queue_Empty (queue) queue_ptr queue;
   { 
   return (listLength(queue->list) == 0);
   }
   */

void Enqueue (const void *data, queue_ptr queue)
{ 
  listInsertItemLast(data, queue->list);
}

const void *Dequeue (queue_ptr queue)
{ 
  return listPopItem(queue->list);
}

void Head_Enqueue (const void *data, queue_ptr queue)
{ 
  listInsertItemFirst(data, queue->list);
}


/*
   .______________.
   .              .
   .    STACKS    .
   .______________.
   
   */

stack_ptr Create_Stack(void)
{ stack_ptr new_stack;
  
  new_stack = NEW(stack_type);
  new_stack->list = listCreate();
  return new_stack;
}

/* Currently not used (RGS: 11/11/92) */
void Free_Stack (stack_ptr *stack)
{
  if ((*stack) == NULL) return;

  listFree(&((*stack)->list));
  tcaFree((char *) (*stack));
  *stack = NULL;
}

/* This is actually quite efficient, but a simple listClear operation
   would be cleaner */
/* Currently not used (RGS: 11/11/92)
   void Clear_Stack (stack)
   stack_ptr stack;
   {
   listFree(&(stack->list));
   stack->list = listCreate();
   }
   */

/* Currently not used (RGS: 11/11/92)
   int32 Is_Stack_Empty (stack)
   stack_ptr stack;
   { 
   return (listLength(stack->list) == 0);
   }
   */

void Push_Stack (const void *data, stack_ptr stack)
{ 
  listInsertItemFirst(data, stack->list);
}

const void *Pop_Stack(stack_ptr stack)
{ 
  return listPopItem(stack->list);
}

#if 0
/* Currently not used (RGS: 11/11/92) */

/*
   ._______________________.
   .                       .
   .    PRIORITY_QUEUES    .
   ._______________________.
   
   
   INEFFICIENT IMPLEMENTATION OF PRIORITY-QUEUES
   Has the property that if two values are EQUAL, then the newest entry
   gets inserted ahead of the older entries (it's really like a PRIORITY-STACK 
   in that way!!!)
   */

priority_queue_ptr Create_Priority_Queue(valueLesspFn)
     BOOLEAN (* valueLesspFn)(const char *, const char *);
{ priority_queue_ptr new_priority_queue;
  
  new_priority_queue = NEW(priority_queue_type);
  new_priority_queue->list = listCreate();
  new_priority_queue->valueLesspFn = valueLesspFn;
  return new_priority_queue;
}

void Free_Priority_Queue (priority_queue)
     priority_queue_ptr priority_queue;
{
  if (!priority_queue) return;

  listFree(&(priority_queue->list));
  tcaFree((char *) priority_queue);
}

/* This is actually quite efficient, but a simple listClear operation
   would be cleaner */
void Clear_Priority_Queue (priority_queue)
     priority_queue_ptr priority_queue;
{
  listFree(&(priority_queue->list));
  priority_queue->list = listCreate();
}

int32 Is_Priority_Queue_Empty (priority_queue)
     priority_queue_ptr priority_queue;
{ 
  return (listLength(priority_queue->list) == 0);
}

void Priority_Enqueue (data, priority_queue)
     char *data; 
     priority_queue_ptr priority_queue;
{
  listInsertItemFirst(data, priority_queue->list);
}

/* Inefficient, but makes use of already existing list package primitives */
char *Priority_Dequeue(priority_queue) 
     priority_queue_ptr priority_queue;
{ 
  char *bestItem, *item;
  
  bestItem = listFirst(priority_queue->list);
  item = listNext(priority_queue->list);
  while (item) {
    if ((priority_queue->valueLesspFn)(bestItem, item)) {
      bestItem = item;
    }
    item = listNext(priority_queue->list);
  }
  return bestItem;
}
#endif

/*
   .________________.
   .                .
   .    MEMORIES    .
   .________________.
   
   `Memory' is a way of keeping track of a list of things.
   */

memory_ptr Create_Memory(void)
{ memory_ptr new_memory;
  
  new_memory = NEW(memory_type);
  new_memory->list = listCreate();
  return new_memory;
}

/* Currently not used (RGS: 11/11/92) */
void Free_Memory(memory_ptr *memory)
{
  if ((*memory) == NULL) return;

  listFree(&((*memory)->list));
  tcaFree((char *) (*memory));
  *memory = NULL;
}

/* This is actually quite efficient, but a simple listClear operation
   would be cleaner */
void Clear_Memory (memory_ptr memory)
{
  listFree(&(memory->list));
  memory->list = listCreate();
}

/* Currently not used (RGS: 11/11/92)
   int32 Is_Memory_Empty (memory)
   memory_ptr memory;
   { 
   return (listLength(memory->list) == 0);
   }
   */

void Remember (const void *data, memory_ptr memory)
{ 
  listInsertItemFirst(data, memory->list);
}

/*
   Execute the function, iterating over the items in memory in
   a last in, first out manner.
   */

static int32 applyMemoryFunc (DO_MEMORY_FN memoryFn, const void *item)
{
  memoryFn(item);
  return 1;
}

void Do_Memory (memory_ptr memory, DO_MEMORY_FN memoryFn)
{ 
  (void)listIterateFromLast((LIST_ITER_FN)applyMemoryFunc, 
			    (char *)memoryFn, memory->list);
}
