/**************************************************************************
 * 
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: qlattice
 *
 * FILE: qlattice.h
 *
 * ABSTRACT:
 *
 * QUANTITY LATTICE STRUCTURES
 * The upper and lower bounds are contextual-value "cells" which define
 * the interval range of the quantity.  The upper and lower bounds default 
 * to the values "[minus_infinity...plus_infinity]."
 *
 * The first four slots are mainly for bookkeeping and 
 * doing searches in the lattice.
 *
 * The (real) interval value of a quantity is given by the next two slots, 
 * each of which is a contextual_value "cell".  
 *
 * The "relations" slot is a list of relations with other quantities 
 * (a relation is also a contextual_value "cell"
 *
 * The "arithmetic_fields" is used to hang on additional slots that are needed
 * by the arithmetic package (qarith).
 *
 * REVISION HISTORY 
 *
 * $Log: qlattice.h,v $
 * Revision 1.13  1996/01/30  15:04:50  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.12  1996/01/27  21:54:06  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.11  1995/12/17  20:21:57  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.10  1995/05/31  19:36:14  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.9  1995/01/18  22:42:11  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.8  1994/11/07  04:26:39  rich
 * Committing Reid's changes needed to use the qlattice for xavier navigation.
 *
 * Revision 1.7  1994/05/17  23:17:10  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.6  1993/12/14  17:34:50  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/12/01  18:04:16  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.4  1993/11/21  20:19:14  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:16:23  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:46  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:46  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:49  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:25:11  fedor
 * Added Logging.
 *
 * $Revision: 1.13 $
 * $Date: 1996/01/30 15:04:50 $
 * $Author: rich $
 *
 **************************************************************************/

#ifndef INCqlattice
#define INCqlattice


typedef enum {
  No_Relationship=0, Less_Or_Equal=1, Not_Equal=2, 
  Greater_Or_Equal=3, Greater=4, Equal=5, Less=6, 
  Inconsistent=7, Real_Valued=8
#ifdef FORCE_32BIT_ENUM
    , dummyQrelation = 0x7FFFFFFF
#endif
} QRELATION_TYPE;

typedef enum {
  Is_False=0, Is_True=1, Is_Unknown=2
#ifdef FORCE_32BIT_ENUM
    , dummyTriValued = 0x7FFFFFFF
#endif
} TRI_VALUED_TYPE;

typedef struct { 
  float value;
  int32 limit;
} QBOUND_TYPE, *QBOUND_PTR;

/* 21-Nov-89: fedor: this needs more investigation to find how it is
   used to avoid much mallocing of memory */

/* 27-Nov-89: fedor: replaced SJ_TYPE lists with two justification slots */
typedef struct { 
  QRELATION_TYPE qRel1, qRel2;
  CELL_PTR justRel1, justRel2;
} SJ_TYPE;

typedef struct _QUANTITY_TYPE { 
  const char *name;
  int32 qnumber;
  SJ_TYPE searchJust;
  QRELATION_TYPE mark;
  CELL_PTR lower_bound;
  CELL_PTR upper_bound;
  LIST_PTR relations;
  struct _arith_fields *arithmetic_fields;
} QUANTITY_TYPE, *QUANTITY_PTR;

typedef struct { 
  QUANTITY_PTR arg1;
  QUANTITY_PTR arg2;
} REL_DATA_TYPE, *REL_DATA_PTR;

typedef struct {
  char *name;
  QRELATION_TYPE relation;
} REL_TO_SYMB_TYPE;

#define MAXQRELATION 8

/* Run the propagation and demon queues. */
#define LRUNQ() (void)runRules(2)

/* 
   Used for indexing relations -- largest number of quantities that the 
   system can handle 
   */
#define HIGHEST_QUANTITY_NUMBER 45000
#define RELATION_HASH_TABLE_SIZE 8009

#define IS_PLUS_INFINITY(r) ((r) == GET_S_GLOBAL(PLUS_INFINITY))
#define IS_MINUS_INFINITY(r) ((r) == GET_S_GLOBAL(MINUS_INFINITY))

#define IS_REAL_VALUED(q) ((q)->mark == Real_Valued)

#define CLOSED TRUE /* The end of the interval is closed */
#define OPEN FALSE /* The end of the interval is open */
#define STRICT TRUE /* The relationship is strict (< or >) */
#define NOT_STRICT FALSE /* The relationship is not strict (<= or >=) */
#define LOWERBOUND TRUE /* The left end of an interval */
#define UPPERBOUND FALSE /* The right end of an interval */

#define IS_OPEN(limit) (limit == OPEN)
#define IS_CLOSED(limit) (limit == CLOSED)
#define IS_STRICT(s) (s == STRICT)
#define IS_NOT_STRICT(s) (s == NOT_STRICT)
#define ARE_BOTH_CLOSED(limit1, limit2) \
(IS_CLOSED(limit1) && IS_CLOSED(limit2))

#define THE_LOWER_QBOUND(lbound) \
((QBOUND_PTR)CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(lbound))
#define THE_UPPER_QBOUND(ubound) \
((QBOUND_PTR)CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(ubound))

#define QUANTITY_OF_BOUND(bound) ((QUANTITY_PTR)((bound)->datum))

#define QUANTITY_LOWER_QBOUND_VALUE(quantity) \
((quantityLowerQbound(quantity))->value)

#define QUANTITY_UPPER_QBOUND_VALUE(quantity) \
((quantityUpperQbound(quantity))->value)

#define SYMMETRIC_RELATIONSHIP(relationship) \
(Symmetric_Relationships[(int32)relationship])

#define INVERSE_RELATIONSHIP(relationship) \
(Inverse_Relationships[(int32)relationship])

#define TRANSITIVITY(relationship1, relationship2) \
Relationship_Transitivity\
[(int32)(relationship1)][(int32)(relationship2)]

#define REL_UNION(relationship1, relationship2) \
Relationship_Unions\
[(int32)(relationship1)][(int32)(relationship2)]

/*
 * A relationship is "under_constrained" if it is a disjunction of 
 * the relationships "<",">", or "=".
 */
#define IS_UNDER_CONSTRAINED(relationship) ((int32)relationship < (int32)Greater)

/*
 * Two relationships are contradictory if the union of their 
 * relationships does not make  sense 
 * (e.g. something cannot be both ">" and "<").
 */
#define IS_CONTRADICTORY(relationship1, relationship2) \
(Inconsistent == REL_UNION(relationship1, relationship2))

/*
 * Does the actual relationship already subsume the desired relationship
 * (e.g. it is the same relationship or is more restrictive, such as ">" 
 * being more restrictive than ">=").
 */
#define IS_ALREADY_TRUE(actual_relationship, desired_relationship) \
(!is_more_constrained(desired_relationship, actual_relationship))

/* 
 * Is the actual relationship not contradictory with the desired 
 * relationship. 
 */
#define IS_IMPLIED_RELATION(actual_relationship, desired_relationship) \
(!IS_CONTRADICTORY(actual_relationship, desired_relationship))

#define IS_NO_RELATIONSHIP(relationship) ((relationship) == No_Relationship)
#define IS_RELATIONSHIP(relationship) ((relationship) != No_Relationship)

#define IMPLIES(antecedent_node, consequent_node) \
tmsJustifyNode\
(consequent_node, \
 tmsCreateJustification1("Subsumption", \
			 listMake1((char *)antecedent_node)))

/*
 * Adds the "relationship" to the relation, with the given justification 
 * (see the file "tms.c" for a description of what a legal justification
 * is).
 */
#define UPDATE_RELATIONSHIP(new_relationship, relation, q1, justification) \
(tmsJustifyNode(relationshipSupporter1(new_relationship, relation, q1), \
		justification))

#define REFINES(relation, node1, node2, refined_relationship)\
(UPDATE_RELATIONSHIP(refined_relationship, relation, REL_ARG1(relation), \
		     tmsCreateJustification1("REFINEMENT", \
					     listMake2((char *)node1,\
						       (char *)node2))))

#define REL_CONTRADICTION(relationship_supporter1, relationship_supporter2)\
(tmsContradictory(listMake2((char *)relationship_supporter1,\
			    (char *)relationship_supporter2),\
		  "Contradictory_Relationships"))

#define APPROX_EQUAL(a, b) approx_equal1(a, b, GET_S_GLOBAL(Qlat_Epsilon))

#define IS_GREATER_REAL(r1, r2)  is_less_real(r2, r1)

#define SEARCH_RELATIONSHIP(q) (QRELATION_TYPE)((q)->mark)

/* Total number of (known) relations between "q" and some other 
 *	quantity.
 */
#define NUM_RELATIONS(q) listLength((q)->relations)

#define QUANTITY_OF_QBOUND(qbound_cv) \
(QUANTITY_OF_BOUND((CELL_PTR)(qbound_cv)->cell))

#define LESS_OR_EQUAL_RELP(relationship) \
(IS_CONTRADICTORY(Greater, relationship))

#define GREATER_OR_EQUAL_RELP(relationship) \
(IS_CONTRADICTORY(Less, relationship))

#define NEW_LIMIT(strictness, constraining_limit) \
((IS_CLOSED(constraining_limit) && IS_NOT_STRICT(strictness)) ? CLOSED \
 : OPEN)

/*
 * Is the current relationship between the quantities 
 * "quant1" and "quant2"
 * consistent with the proposed "relationship" between them.
 */
#define IS_CONSISTENT(relationship, quant1, quant2) \
(!(IS_ALREADY_TRUE(qrelationship1(quant1, quant2, No_Relationship), \
		   INVERSE_RELATIONSHIP(relationship))))

#define QUNASSUME(q1, relationship, q2) \
(QRetract(q1, relationship, q2, "Assumption", NULL))

#define QASSUME(q1, relationship, q2) \
(QAssert(q1, relationship, q2, "Assumption", NULL))

#define RELATION_DATA(relation) ((REL_DATA_PTR)((relation)->datum))

#define REL_ARG1(relation) (RELATION_DATA(relation)->arg1)
#define REL_ARG2(relation) (RELATION_DATA(relation)->arg2)

#define CV_RELP(relp_supporter) ((QRELATION_TYPE)((relp_supporter)->datum))

#define CV_QBOUND(qbound_supporter) \
((QBOUND_PTR)((qbound_supporter)->datum))

int32 isInfinite(float r);

QBOUND_PTR createQbound(float value, int32 limit);
CELL_PTR qval_lower(QUANTITY_PTR quantity);
CELL_PTR qval_upper(QUANTITY_PTR quantity);
QUANTITY_PTR createQuantity(const char *name);
QUANTITY_PTR createRealQuantity(float realValue);
QUANTITY_PTR createQuantity1(const char *name,
			     QBOUND_PTR lowerBound, QBOUND_PTR upperBound);

QBOUND_PTR quantityLowerQbound(QUANTITY_PTR quantity);
QBOUND_PTR quantityUpperQbound(QUANTITY_PTR quantity);

CELL_PTR get_relation(QUANTITY_PTR q1, QUANTITY_PTR q2);
CELL_PTR relationGetOrMake(QUANTITY_PTR q1, QUANTITY_PTR q2);
QRELATION_TYPE true_relationship(QRELATION_TYPE relationship,
				 CELL_PTR relation, QUANTITY_PTR q1);
QUANTITY_PTR relation_other_arg(CELL_PTR relation, QUANTITY_PTR q1);
QRELATION_TYPE end_relationship1(float num1, int32 limit1, 
				 float num2, int32 limit2,
				 int32 is_same_bound, int32 is_lowerbound);

QRELATION_TYPE qrelationship1(QUANTITY_PTR q1, QUANTITY_PTR q2,
			      QRELATION_TYPE searchingFor);
const char *QRelationship(QUANTITY_PTR q1, QUANTITY_PTR q2);
TRI_VALUED_TYPE QIs(QUANTITY_PTR q1, const char *relationship,
		    QUANTITY_PTR q2);
int32 QIs_True(QUANTITY_PTR q1, const char *relationship,
	     QUANTITY_PTR q2);
void QAssert(QUANTITY_PTR q1, const char *relationship,
	     QUANTITY_PTR q2, const char *informant);
void QRetract(QUANTITY_PTR q1, const char *relationship,
	      QUANTITY_PTR q2, const char *informant);

TMS_NODE_PTR QNode(QUANTITY_PTR q1, const char *relationship,
		   QUANTITY_PTR q2);
void FreeQuantity(QUANTITY_PTR *quantity);
void removeQuantityRules(QUANTITY_PTR quantity);

void Print_Quantity(QUANTITY_PTR quant, int32 endLine); 
void Print_Relationship(QUANTITY_PTR q1, QUANTITY_PTR q2, int32 end_line);
void qlatticeInit(void);

#endif /* INCqlattice */
