/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: qlattice
 *
 * FILE: qlattice.c
 *
 * ABSTRACT:
 *
 * This file contains the code needed for maintaining and querying the 
 * quantity lattice.
 * 
 * Terminology:
 *
 * QUANTITY 
 * The representation of a number.  
 * Can have "relations" to other quantities, and a real-valued
 * interval range (in which its true value falls).
 *
 * RELATION
 * The "arc" between two quantities, recording the "relationship" between
 * them and the justification of how it was derived.  A relation can have
 * more than justification if incremental pieces are added (e.g. if first
 * we are told the relationship is ">=", then we are told ">", both ">=" and
 * ">" will have justifiers).
 *
 * A "relation" is the "arc" between two quantities in the lattice
 * ("arg1" and "arg2").
 * A relation is implemented as a "constrained" contextual_value cell.
 * (the type "relation_ptr" is really the same as "CELL_PTR").
 * Each contextual_value associated with the cell encodes the truth of the 
 * relationship (arg1 relationship arg2)".  
 * That is, if the an IN contextual_value
 * has the value ">" it means that "arg > arg2".
 *
 * RELATIONSHIP
 * The relationship between two quantities.  The allowable relationships
 * are <, <=, >, >=, <> (not equal), = (equal), ?? (no relationship).
 * Note: for efficiency, relationships are stored internally as integers,
 * but the user never sees this implementation detail.
 *
 * INTERVAL
 * The real valued range of a quantity.  The actual value of the quantity lies
 * somewhere within the interval range.  Initially, all intervals range from
 * [-infinity..+infinity].  Parentheses indicate that the interval is open; 
 * brackets indicate that the interval is closed.
 *
 * BOUND 
 * The upper or lower end of an interval.  The "bound" is the structure that
 * represents the value, the "qbound" refers to the value itself.
 *
 * LIMIT 
 * Either open or closed, referring to an end.
 *
 * STRICTNESS 
 * Whether a relationship is strict (< or >) or not-strict (<= or >=).
 *
 *   ***   RELATIONSHIPS    ***
 *   >
 *   =
 *   <
 *   >=
 *   <=
 *   <>
 *   ??  (unknown relationship)
 * 
 *   ***   Special quantities   ***
 *
 *   Zero_Quant	    =  0.0
 *   One_Quant	    =  1.0
 *   Minus_One_Quant = -1.0
 * 
 *   ***   ARITHMETIC FUNCTIONS   ***
 *
 *   (QPLUS Q1 Q2)              q1 + q2
 *   (QDIFFERENCE Q1 Q2)        q1 - q2
 *   (QTIMES Q1 Q2)             q1 * q2
 *   (QQUOTIENT Q1 Q2)          q1 / q2
 *   (QMINUS Q)	         - q
 *   (QABS Q)			 |q|
 *   (QSIN Q) 
 *   (QCOS Q) 
 *   (QTAN Q) 
 *        where Q, Q1 and Q2 are quantities
 *
 *   example use:  (QLATTICE-INIT) -> :DONE
 *                 (SETQ A (CREATE-QUANTITY 'A)) -> #<QUANTITY A>
 *                 (SETQ FOUR (CREATE-REAL-QUANTITY 4)) -> #<QUANTITY 4>
 *                 (QUANTITY-VALUE A) -> "[-INFINITY ... +INFINITY]"
 *                 (QASSERT A '> FOUR) -> :DONE 
 *                 (QUANTITY-VALUE A) -> "(4 ... +INFINITY]"
 *                 (SETQ B (CREATE-QUANTITY 'B)) -> #<QUANTITY B>
 *                 (QASSERT B '= (QPLUS A FOUR)) -> :DONE
 *                 (QRELATIONSHIP A B) ->  < (because 4+A > 8, and A < 8)
 *
 ********************************
 *
 * This is the most general form of the numeric_propagation constraint,
 * but it is inefficient in that it makes lots of extra rules, most of which
 * will never be fired (it needs rules for each relation and each 
 * combination of upper and lower bounds).
 *
 * Instead, we implement a more space_efficient version of 
 * "numeric_constraint_propagation" (see my AAAI_86 article) but which 
 * might fire rules more than once on the same arguments (so it is less 
 * time effiecient)
 *
 * For each relation R between A and B, this constraint is put on the cells
 *  1) R, Upper_bound A, Upper_bound B
 *  2) R, Upper_bound A, Lower_bound B
 *  3) R, Lower_bound A, Upper_bound B
 *  4) R, Lower_bound A, Lower_bound B
 *
 * (def_constraint NUMERIC_CONSTRAINT_PROPAGATION ((relation bound1 bound2) 
 * :priority 1)
 * (bound1 (relation bound2) (numeric_propagation1 bound2_value relation_value))
 * (bound2 (relation bound1) (numeric_propagation1 bound1_value relation_value))
 * (relation (bound1 bound2) (numeric_qrelationship bound1_value bound2_value)))
 *
 * The propagation algorithm we implement is as follows :
 * 1) A rule is put on the upper and lower bounds of each quantity.
 *    This rule fires on the most_constrained value for the bound and looks at
 *    each relation of the quantity to see if
 *    a) it can constrain the upper (or lower) bounds of the other quantity.
 *    b) determine the relationship numerically between the two quantities.
 *
 * 2) A rule is put on each relation (between two quantities Q1 and Q2).
 *    Whenever a relationship becomes the most_constrained relationship the
 *    rule 
 *    is fired and it looks to see if it can constrain the interval bounds of
 *    Q1 and Q2.  
 *    NOTE, this rule is fired everytime the relationship becomes the most
 *    constrained, not just the first time -- this is where the 
 *    time_inefficiency comes in, but it saves us from consing up lots of 
 *    needless rules and in practice it is probably not that bad, since the 
 *    amount of context switching is bound to be fairly small (if lots of 
 *    context switching happens __ then perhaps the more rule_intensive
 *    method should be used instead)
 *
 * 3) The algorithm tries to conserve space by only adding a new value to the 
 *    lower bound if it is more constraining than the current bound. 
 *    Thus, whenever a bound value goes OUT a repropagation rule is fired to 
 *    see whether any existing relation can constrain the value more.
 *
 ********************************
 *
 * EXPORTS:
 * 
 * 
 * Part of QLATTICE system
 *
 * INITIALIZATION
 *
 * qlatticeInit()
 * Initialize the Quantity Lattice (and the TMS).  
 * Should be the first call made.
 *
 * CREATING AND MANIPULATING QUANTITIES
 * 
 * createQuantity(name)
 * Creates a named quantity, where the name is string.
 * The quantity value defaults to "[-infinity ... +infinity]".
 * 
 * createRealQuantity(real_value)
 * Creates a quantity and assigns it a point on the reals.
 * 
 * Is_Exact_Valued(quantity)
 * Checks if quantity is a point (rather than finite interval).
 * 
 * ADDING AND DELETING FACTS
 * 
 * QAssert(q1, relationship, q2, informant, justifiers)
 * Asserts that the "relationship" holds between the two quantities
 * (as long as all the justifiers, if any, which are tms-nodes, are IN).
 *
 * Relationship" can be one of those listed.
 * If "informant" is NULL, makes it a premise ; if "informant" is
 * "Assumption", makes it an assumption; else uses the given 
 * "informant" and "justifiers" to justify the assertion.
 * 
 * QRetract(q1, relationship, q2, informant, justifiers)
 * Retracts the assertion that the informant and justifiers support the 
 * relationship between the two quantities.  
 * Uses the same method for determining justifications as given above.
 *
 * Note, the relationship might still hold, if it was justified some other way.
 *
 * QUERY FUNCTIONS
 * 
 * QRelationship(q1, q2) 
 * Returns relationship between quantities q1 and q2.
 * 
 * QIs(q1, relationship, q2)
 * Tests whether or not the "relationship" holds between the two
 * quantities, returning Is_True, Is_False, or Is_Unknown.
 * 
 * QIs_True(q1, rel, q2)
 * Same as "QIs" except returns TRUE if true, FALSE if false or unknown.
 *
 ********************************
 * 
 * (QNEIGHBOR-RELATIONS QUANT &OPTIONAL INCLUDE-INFERRED?)
 * Returns a list of pairs "(RELATION . QUANTITY)", for all quantities 
 * related to QUANT. If "INCLUDE-INFERRED?" is nil, then only asserted
 * relations will be returned.  If non-nil, then both asserted and inferred 
 * relations will be returned.
 * 
 * (QSUPPORTS Q1 Q2)
 * If the relationship between the two quantities is known and is
 * a derived relationship, returns a list of tms-nodes which represent
 * the "qassertions" which underly the inference.  For example, if one
 * states "(qassert a '> b)" and "(qassert b '> c)" then "(qsupports a c)"
 * will return the tms-nodes representing "a>b" and "b>c".
 * 
 * (QJUSTIFIER Q1 Q2)
 * If the relationship between the two quantities is known and was
 * "qasserted" directly, returns the justification used (combination of
 * the "informant" and "justifiers" used when doing the "qassert").
 * 
 * (QWHY Q1 Q2 &OPTIONAL RELATIONSHIP)
 * Interactive tool for tracing dependencies.
 * Helps describe why the two quantities are related.  If relationship
 * is omitted, the most-constrained relationship is used.
 * See the documentation on the function "WHY" in file "contextual-value"
 * to learn how the 'why' function works.
 *
 * REVISION HISTORY:
 *
 * $Log: qlattice.c,v $
 * Revision 1.17  1996/01/27  21:54:02  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.16  1995/12/17  20:21:53  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.15  1995/07/06  21:17:00  rich
 * Solaris and Linux changes.
 *
 * Revision 1.14  1995/06/14  03:21:59  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.13  1995/01/18  22:42:05  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.12  1994/11/07  04:26:21  rich
 * Committing Reid's changes needed to use the qlattice for xavier navigation.
 *
 * Revision 1.11  1994/10/25  17:10:39  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.10  1994/04/28  16:16:56  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.9  1993/12/14  17:34:43  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.8  1993/12/01  18:04:11  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.7  1993/11/21  20:19:09  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.6  1993/10/21  16:14:14  rich
 * Fixed compiler warnings.
 *
 * Revision 1.5  1993/08/30  21:54:15  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.4  1993/08/27  07:16:18  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/06/13  23:28:23  rich
 * Made changes for lisp needed for vx works style global variables.
 * Fixed some random compiler warnings.
 * Moved test routines to test directory.
 *
 * Revision 1.2  1993/05/26  23:18:42  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:31  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:43  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:25:08  fedor
 * Added Logging.
 *
 * 15-Aug-91  Christopher Fedor at School of Computer Science, CMU
 * The routine numeric_qrelationship was missing return values for
 * some cases - added them.
 *
 *  3-Jul-91  Reid Simmons at School of Computer Science, CMU
 * The qlattice rules interacted badly with killing of task tree nodes.
 * A simple fix is make them zero-priority rules (run without queuing).
 * This means they will run even if "isDelayedGlobal" is TRUE.
 *
 * 24-Jun-91  Reid Simmons (reids) at CMU
 * Added code to free a quantity and its associated relations.
 *
 * 23-May-91  Christopher Fedor at School of Computer Science, CMU
 * Discovered much to my horror that find_ordering2 takes only
 * two parameters but was always called with three. Changed the calls to two.
 *
 *  9-Nov-89  Christopher Fedor at School of Computer Science, CMU
 * Revised to Software Standards
 *
 * Ancient    Reid Simmons at School of Computer Science, CMU
 * (c) Copyright 1986 Reid G. Simmons.  All rights reserved.
 * Revised and ported from Zetalisp to
 * Common LISP 1987 Colin L. Campbell, Morton Thiokol, Inc.
 * Updated by Reid Simmons, 1988
 *
 * $Revision: 1.17 $
 * $Date: 1996/01/27 21:54:02 $
 * $Author: rich $
 *
 *****************************************************************************/


#include "globalS.h"
#include "tcaMem.h"

/* Defined in tplConstr.c; Included alone because including tplConst.h
   needs to bring in too many additional files*/
char *qName(QUANTITY_PTR q);

/* Defined below; forward references */
static void constrain_relationships(CELL_PTR relation, TMS_NODE_PTR relSupport,
				    QRELATION_TYPE relationship);
static void cacheOrderingSearch(QUANTITY_PTR q1, QUANTITY_PTR q2,
				CELL_PTR relation);

static REL_TO_SYMB_TYPE rel_to_symb_array[] = {
  {"??", No_Relationship},
  {"<=", Less_Or_Equal},
  {"<>", Not_Equal},
  {">=", Greater_Or_Equal},
  {">", Greater},
  {"=", Equal},
  {"<", Less},
  {"INC", Inconsistent}};

/*    
 * The "symmetric_relation" between two quantities is the one to use if you 
 * reverse the order of the arguments (e.g. for "a < b", ">" is the symmetric 
 * relation, since "b > a" is the same expression).
 */
static QRELATION_TYPE Symmetric_Relationships[7]
= {No_Relationship, Greater_Or_Equal, Not_Equal, Less_Or_Equal, 
     Less, Equal, Greater};

/*
 * The "inverse_relation" between two quantities is the one to use if you 
 * negate the relationship (e.g. for "a < b", ">=" is the inverse relation,
 * since  "~(a >= b)" is the same expression).
 */
static QRELATION_TYPE Inverse_Relationships[7]
= {No_Relationship, Greater, Equal, Less, Less_Or_Equal, 
     Not_Equal, Greater_Or_Equal};

/*
 * The transitivity table determines the relationship between "a" and "b", 
 * given that "a rel1 c" and "c rel2 b".
 */
static QRELATION_TYPE Relationship_Transitivity[7][7]
= {{No_Relationship, No_Relationship, No_Relationship, No_Relationship, 
      No_Relationship, No_Relationship, No_Relationship},
     {No_Relationship, Less_Or_Equal, No_Relationship, No_Relationship, 
	No_Relationship, Less_Or_Equal, Less},
     {No_Relationship, No_Relationship, No_Relationship, No_Relationship, 
	No_Relationship, Not_Equal, No_Relationship},
     {No_Relationship, No_Relationship, No_Relationship, Greater_Or_Equal, 
	Greater, Greater_Or_Equal, No_Relationship},
     {No_Relationship, No_Relationship, No_Relationship, Greater, Greater, 
	Greater, No_Relationship},
     {No_Relationship, Less_Or_Equal, Not_Equal, Greater_Or_Equal, Greater, 
	Equal, Less},
     {No_Relationship, Less, No_Relationship, No_Relationship, No_Relationship,
	Less, Less}};


/*
 * The union table determines the strictest relationship between "a" and "b",
 * given that we know both that "a rel1 b" and "a rel2 b".
 */
static QRELATION_TYPE Relationship_Unions[7][7]
= {{No_Relationship, Less_Or_Equal, Not_Equal, Greater_Or_Equal, 
      Greater, Equal, Less},
     {Less_Or_Equal, Less_Or_Equal, Less, Equal, Inconsistent, Equal, Less},
     {Not_Equal, Less, Not_Equal, Greater, Greater, Inconsistent, Less},
     {Greater_Or_Equal, Equal, Greater, Greater_Or_Equal, Greater, 
	Equal, Inconsistent},
     {Greater, Inconsistent, Greater, Greater, Greater, Inconsistent, 
	Inconsistent},
     {Equal, Equal, Inconsistent, Equal, Inconsistent, Equal, Inconsistent},
     {Less, Less, Less, Inconsistent, Inconsistent, Inconsistent, Less}};


/*****************************************************************************
 *
 * FUNCTION: int32 isInfinite(r)
 *
 * DESCRIPTION:
 *
 * INPUTS: float r;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

int32 isInfinite(float r)
{ 
  return(IS_PLUS_INFINITY(r) || IS_MINUS_INFINITY(r));
}


/*****************************************************************************
 *
 * FUNCTION: int32 approx_equal1(a, b, epsilon)
 *
 *****************************************************************************/

static int32 approx_equal1(float a, float b, float epsilon)
{ 
  float max, diff;
  
  if (a == b) return TRUE;
  else {
    diff = a-b;
    if (diff < 0) diff = -diff;
    if (a < 0) a = -a;
    if (b < 0) b = -b;
    max = 1.0;
    if (a > max) max = a;
    if (b > max) max = b;
    
    return diff/max <= epsilon;
  }
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_less_real(r1, r2) 
 *
 * DESCRIPTION:
 * The "_real" functions are needed to handle the special values of 
 * PLUS_INFINITY and MINUS_INFINITY.
 *
 *****************************************************************************/

static int32 is_less_real(float r1, float r2)
{ 
  /* r1 < r2 ? */
  if (IS_PLUS_INFINITY(r1) || IS_MINUS_INFINITY(r2))
    return FALSE;
  else if (IS_PLUS_INFINITY(r2) || IS_MINUS_INFINITY(r1))
    /* at this point, know r1 != +inf, and r2 != -inf */
    return TRUE; 
  else 
    return (r1 < r2);
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_equal_real(r1, r2) 
 *
 *****************************************************************************/

static int32 is_equal_real(float r1, float r2)
{ 
  /* r1 ~= r2 ? */
  if (isInfinite(r1)) 
    return (r1 == r2);
  else if (isInfinite(r2)) 
    return FALSE;
  else 
    return APPROX_EQUAL(r1, r2);
}


/*****************************************************************************
 *
 * FUNCTION: 
 * QRELATION_TYPE end_relationship1(num1, limit1, num2, limit2,
 *                                  is_same_bound, is_lowerbound)
 *
 *****************************************************************************/

QRELATION_TYPE end_relationship1(float num1, int32 limit1, 
				 float num2, int32 limit2,
				 int32 is_same_bound, int32 is_lowerbound)
{
  if (is_equal_real(num1, num2)) {
    if (is_same_bound) {
      if (limit1 == limit2) 
	return ((IS_CLOSED(limit1) && !(isInfinite(num1))) 
		? Equal : No_Relationship);
      else if (IS_GREATER_REAL(num1, num2)) 
	return Greater;
      else if (IS_CLOSED(limit1) == is_lowerbound) 
	return Less;
      else 
	return Greater;
    }
    else if (!(ARE_BOTH_CLOSED(limit1, limit2))) 
      return Less;
    else if (!(isInfinite(num1))) 
      return Equal;
    else 
      return No_Relationship;
  }
  else if (is_less_real(num1, num2)) 
    return Less;
  else 
    return Greater;
}


/*****************************************************************************
 *
 * FUNCTION: 
 * QRELATION_TYPE end_relationship(qbound1, qbound2, is_same_bound, 
 is_lowerbound)
 *
 *
 *****************************************************************************/

static QRELATION_TYPE end_relationship(QBOUND_PTR qbound1, QBOUND_PTR qbound2,
				       int32 is_same_bound, int32 is_lowerbound)
{ 
  return end_relationship1(qbound1->value, qbound1->limit, 
			   qbound2->value, qbound2->limit,
			   is_same_bound, is_lowerbound);
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_lower_bound_more_constrained(lower_qbound1, lower_qbound2)
 *
 *****************************************************************************/

static int32 is_lower_bound_more_constrained(QBOUND_PTR lower_qbound1,
					   QBOUND_PTR lower_qbound2)
{ 
  return end_relationship(lower_qbound1, lower_qbound2, TRUE, TRUE)
    == Greater; 
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_upper_bound_more_constrained(upper_qbound1, upper_qbound2)
 *
 *****************************************************************************/

static int32 is_upper_bound_more_constrained(QBOUND_PTR upper_qbound1,
					   QBOUND_PTR upper_qbound2)
{ 
  return end_relationship(upper_qbound1, upper_qbound2, TRUE, FALSE) == Less; 
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_more_constrained(relationship1, relationship2)
 *
 * DESCRIPTION: Is relationship1 more restrictive than relationship2? 
 *
 * INPUTS: QRELATION_TYPE relationship1, relationship2;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 is_more_constrained(QRELATION_TYPE relationship1,
			       QRELATION_TYPE relationship2)
{ 
  return relationship2 != REL_UNION(relationship2, relationship1); 
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE symbToRel(symb)
 *
 * DESCRIPTION:
 * Translate from a symbolic relationship to an internal numeric one.
 *
 * INPUTS: char *symb;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static QRELATION_TYPE symbToRel(const char *symb)
{
  int32 i;
  
  if (!symb)
    return No_Relationship;
  
  for (i=0;i < MAXQRELATION;i++)
    if (STREQ(rel_to_symb_array[i].name, symb))
      return rel_to_symb_array[i].relation;
  
  return No_Relationship;
}

static const char *relToSymb(QRELATION_TYPE relation)
{
  int32 i;
  
  for(i=0;i < MAXQRELATION;i++)
    if (relation == rel_to_symb_array[i].relation)
      return rel_to_symb_array[i].name;
  
  return "??";
}

/*****************************************************************************
 *
 * FUNCTION: void print_qvalue(value) 
 *
 * DESCRIPTION: Prints out the value of a quantity in a readable form.
 *
 * INPUTS: float value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void print_qvalue(float value)
{
  if (value == GET_S_GLOBAL(MINUS_INFINITY)) 
    Log("-inf");
  else if (value == GET_S_GLOBAL(PLUS_INFINITY)) 
    Log("+inf");
  else {
    Log("%4f", value);
  }
}

/*****************************************************************************
 *
 * FUNCTION: void print_lower_qbound(qbound) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QBOUND_PTR qbound;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void print_lower_qbound(QBOUND_PTR qbound)
{ 
  Log((IS_OPEN(qbound->limit) ? "(" : "["));
  print_qvalue(qbound->value);
}

/*****************************************************************************
 *
 * FUNCTION: void print_upper_qbound(qbound) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QBOUND_PTR qbound;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void print_upper_qbound(QBOUND_PTR qbound)
{ 
  print_qvalue(qbound->value);
  Log((IS_OPEN(qbound->limit) ? ")" : "]"));
}

/*****************************************************************************
 *
 * FUNCTION: void say_relation(relation, value)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR relation; 
 * QRELATION_TYPE value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void say_relation(CELL_PTR relation, QRELATION_TYPE value)
{ 
  REL_DATA_PTR rel_data;
  
  rel_data = RELATION_DATA(relation);
  (void)printf("%s %s %s", 
	       qName(rel_data->arg1), relToSymb(value), qName(rel_data->arg2));
}

/*****************************************************************************
 *
 * FUNCTION: void say_lower_bound(bound, value)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR bound; 
 * QBOUND_PTR value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void say_lower_bound(CELL_PTR bound, QBOUND_PTR value)
{
  (void)printf("The lower bound of %s is ", QUANTITY_OF_BOUND(bound)->name);
  if (IS_OPEN(value->limit)) 
    (void)printf("greater than ");
  print_qvalue(value->value);
}

/*****************************************************************************
 *
 * FUNCTION: void say_upper_bound(bound, value)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR bound; 
 * QBOUND_PTR value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void say_upper_bound(CELL_PTR bound, QBOUND_PTR value)
{
  (void)printf("The upperbound of %s is ", QUANTITY_OF_BOUND(bound)->name);
  if (IS_OPEN(value->limit)) 
    (void)printf("less than ");
  print_qvalue(value->value);
}


/*****************************************************************************
 *
 * FUNCTION: QBOUND_PTR createQbound(value, limit)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * float value; 
 * int32 limit;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

QBOUND_PTR createQbound(float value, int32 limit)
{ 
  QBOUND_PTR qbound;
  
  qbound = NEW(QBOUND_TYPE);
  qbound->value = value;
  qbound->limit = limit;
  return qbound;
}


/*****************************************************************************
 *
 * FUNCTION: CELL_PTR create_lower_bound(quantity, initialLbound)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QUANTITY_PTR quantity; 
 * QBOUND_PTR initialLbound;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static CELL_PTR create_lower_bound(QUANTITY_PTR quantity,
				   QBOUND_PTR initialLbound)
{ 
  CELL_PTR bound;
  
  ++GET_S_GLOBAL(numBoundsGlobal);
  bound = cellCreateConstrained((const char *)quantity, 
				(SAME_VALUE_FN) listItemEq,
				(MOST_CONSTRAINED_FN) 
				is_lower_bound_more_constrained,
				(CELL_SAY_FN) say_lower_bound);
  quantity->lower_bound = bound;
  justify(bound, (const char *)initialLbound,
	  tmsCreateJustification("Initial_Value", (LIST_PTR)NULL));
  if (!IS_REAL_VALUED(quantity)) {
    putRuleOnCell(GET_S_GLOBAL(Lower_Bound_Propagation_Rule), bound);
    addOutRule(GET_S_GLOBAL(Repropagate_Lower_Bound_Rule), bound);
  }
  return bound;
}


/*****************************************************************************
 *
 * FUNCTION:
 *
 *****************************************************************************/

static CELL_PTR create_upper_bound(QUANTITY_PTR quantity, 
				   QBOUND_PTR initial_ubound)
{ 
  CELL_PTR bound;
  
  ++GET_S_GLOBAL(numBoundsGlobal);
  bound = cellCreateConstrained((const char *)quantity, 
				(SAME_VALUE_FN) listItemEq,
				(MOST_CONSTRAINED_FN)
				is_upper_bound_more_constrained,
				(CELL_SAY_FN) say_upper_bound);
  quantity->upper_bound = bound;
  justify(bound, (const char *)initial_ubound, 
	  tmsCreateJustification("Initial_Value", (LIST_PTR)NULL));
  
  if (!IS_REAL_VALUED(quantity)) {
    putRuleOnCell(GET_S_GLOBAL(Upper_Bound_Propagation_Rule), bound);
    addOutRule(GET_S_GLOBAL(Repropagate_Upper_Bound_Rule), bound);
  }
  
  return bound;
}


/*****************************************************************************
 *
 * FUNCTION: 
 * CELL_PTR qval_lower(quantity) 
 * CELL_PTR qval_upper(quantity) 
 *
 * DESIGN:
 * We use a `virtual' upper and lower `qval' bound.  As long as one just 
 * needs the value, the default lower or upper qbound is returned.  
 * When you need the actual qval, it ``turns into'' a contextual_value cell.
 
 *****************************************************************************/

CELL_PTR qval_lower(QUANTITY_PTR quantity)
{ 
  return ((quantity->lower_bound) ? quantity->lower_bound :
	  create_lower_bound(quantity, GET_S_GLOBAL(Default_Lower_Qbound)));
}

CELL_PTR qval_upper(QUANTITY_PTR quantity)
{ 
  return ((quantity->upper_bound) ? quantity->upper_bound :
	  create_upper_bound(quantity, GET_S_GLOBAL(Default_Upper_Qbound)));
}

/*****************************************************************************
 *
 * FUNCTION: QUANTITY_PTR createQuantity1(name, lowerBound, upperBound)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * QBOUND_PTR lowerBound, upperBound;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

QUANTITY_PTR createQuantity1(const char *name,
			     QBOUND_PTR lowerBound, QBOUND_PTR upperBound)
{ 
  QUANTITY_PTR quantity;
  
  quantity = NEW(QUANTITY_TYPE);
  quantity->name = name;
  quantity->mark = No_Relationship;
  quantity->qnumber = GET_S_GLOBAL(numQuantitiesGlobal)++;
  
  quantity->searchJust.justRel1 = NULL;
  quantity->searchJust.justRel2 = NULL;
  
  if (lowerBound) 
    (void)create_lower_bound(quantity, lowerBound);
  else 
    quantity->lower_bound = NULL;
  
  if (upperBound) 
    (void)create_upper_bound(quantity, upperBound);
  else 
    quantity->upper_bound = NULL;
  
  quantity->relations = listCreate();
  
  quantity->arithmetic_fields = NULL;
  
  if (GET_S_GLOBAL(numQuantitiesGlobal) > HIGHEST_QUANTITY_NUMBER)
    tcaError("createQuantity1 -- increase size of 'HIGHEST_QUANTITY_NUMBER'\n");
  
  LRUNQ();
  
  return quantity;
}


/*****************************************************************************
 *
 * FUNCTION: QUANTITY_PTR createQuantity(name)
 * 
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *
 * OUTPUTS: QUANTITY_PTR
 *
 *****************************************************************************/

QUANTITY_PTR createQuantity(const char *name)
{ 
  return createQuantity1(name, (QBOUND_PTR)NULL, (QBOUND_PTR)NULL);
}


/*****************************************************************************
 *
 * FUNCTION: QUANTITY_PTR createRealQuantity(realValue) 
 *
 * DESCRIPTION: Create a quantity and assign it a point on the reals.
 *
 * INPUTS: float realValue;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

QUANTITY_PTR createRealQuantity(float realValue)
{ 
  QBOUND_PTR realBound;
  QUANTITY_PTR realQuantity;
  char *realName;
  
  realBound = createQbound(realValue, CLOSED);
  
  (void)sprintf(GET_S_GLOBAL(name_buffer), "%4f", realValue);
  realName = (char *)tcaMalloc((unsigned)(1+
					  strlen(GET_S_GLOBAL(name_buffer))));
  (void)strcpy(realName, GET_S_GLOBAL(name_buffer));
  
  realQuantity = createQuantity1(realName, realBound, realBound);
  realQuantity->mark = Real_Valued;
  
  return realQuantity;
}


/*****************************************************************************
 *
 * FUNCTION: QBOUND_PTR quantityLowerQbound(quantity) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR quantity;
 *
 * OUTPUTS: QBOUND_PTR
 *
 *****************************************************************************/

QBOUND_PTR quantityLowerQbound(QUANTITY_PTR quantity)
{ 
  return ((quantity->lower_bound)
	  ? THE_LOWER_QBOUND(quantity->lower_bound)
	  : GET_S_GLOBAL(Default_Lower_Qbound));
}


/*****************************************************************************
 *
 * FUNCTION: QBOUND_PTR quantityUpperQbound(quantity) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR quantity;
 *
 * OUTPUTS: QBOUND_PTR
 *
 *****************************************************************************/

QBOUND_PTR quantityUpperQbound(QUANTITY_PTR quantity)
{
  return ((quantity->upper_bound)
	  ? THE_UPPER_QBOUND(quantity->upper_bound)
	  : GET_S_GLOBAL(Default_Upper_Qbound));
}


/*****************************************************************************
 *
 * FUNCTION: CELL_PTR createRelation(q1, q2)
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/

static CELL_PTR createRelation(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  CELL_PTR relation;
  REL_DATA_PTR relData;
  
  relData = NEW(REL_DATA_TYPE);
  relData->arg1 = q1;
  relData->arg2 = q2;
  
  relation = cellCreateConstrained((char *)relData, (SAME_VALUE_FN) listItemEq,
				   (MOST_CONSTRAINED_FN) is_more_constrained, 
				   (CELL_SAY_FN)say_relation);
  
  (void)hashTableInsert((char *)RELATION_DATA(relation), sizeof(REL_DATA_TYPE),
			(char *)relation, GET_S_GLOBAL(Relation_Hash_Table));
  ++GET_S_GLOBAL(numRelationsGlobal);
  
  listInsertItem((char *)relation, q1->relations);
  if (q1 == q2)
    tmsJustifyNode(getOrAddValueNode(relation, (char *)Equal),
		   tmsCreateJustification("Premise", (LIST_PTR)NULL));
  else
    listInsertItem((char *)relation, q2->relations);
  
  addInRule(GET_S_GLOBAL(Relation_Propagation_Rule), relation);
  
  return relation;
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE relationRelationship(relation) 
 *
 * DESCRIPTION:
 *
 * INPUTS: CELL_PTR relation;
 *
 * OUTPUTS: QRELATION_PTR
 *
 *****************************************************************************/

static QRELATION_TYPE relationRelationship(CELL_PTR relation)
{ 
  const char *value;
  
  value = CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(relation);
  return ((value) ? (QRELATION_TYPE)value : No_Relationship);
}


/*****************************************************************************
 *
 * FUNCTION: int32 rel_hash_key(relData) 
 *
 * DESCRIPTION:
 *
 * INPUTS: REL_DATA_PTR relData;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 rel_hash_key(REL_DATA_PTR relData)
{ 
  int32 q1_number, q2_number;
  
  q1_number = relData->arg1->qnumber;
  q2_number = relData->arg2->qnumber;
  
  return ((q1_number < q2_number)
	  ? (q1_number + q2_number * HIGHEST_QUANTITY_NUMBER)
	  : (q2_number + q1_number * HIGHEST_QUANTITY_NUMBER));
}


/*****************************************************************************
 *
 * FUNCTION: int32 same_rel_data(relData1, relData2)
 *
 * DESCRIPTION:
 *
 * INPUTS: REL_DATA_PTR relData1, relData2;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 same_rel_data(REL_DATA_PTR relData1, REL_DATA_PTR relData2)
{
  return (((relData1->arg1 == relData2->arg1) && 
	   (relData1->arg2 == relData2->arg2))
	  || ((relData1->arg1 == relData2->arg2) &&
	      (relData1->arg2 == relData2->arg1)));
}


/*****************************************************************************
 *
 * FUNCTION: CELL_PTR get_relation(q1, q2)
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: CELL_PTR
 *
 * DESIGN:
 * Relations are stored in an equal hash table, the key is of 
 * type "REL_DATA_TYPE" 
 *
 *****************************************************************************/

CELL_PTR get_relation(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  REL_DATA_TYPE rel_data;
  
  rel_data.arg1 = q1;
  rel_data.arg2 = q2;
  return (CELL_PTR)hashTableFind((char *)&rel_data,
				 GET_S_GLOBAL(Relation_Hash_Table));
}


/*****************************************************************************
 *
 * FUNCTION: CELL_PTR relationGetOrMake(q1, q2)
 *
 * DESCRIPTION:
 * Return the relation between q1 and q2, if it exists.  
 * If not, create such a relation.
 *
 * INPUTS: QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/

CELL_PTR relationGetOrMake(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  CELL_PTR rel;
  
  rel = get_relation(q1, q2);
  if (!rel)
    rel = createRelation(q1, q2);
  
  return rel;
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE relationship(relation, q1)
 *
 * DESCRIPTION:
 * Returns the relationship assuming that "q1" is the first argument 
 * of the "relation".
 *
 * INPUTS:
 * CELL_PTR relation; 
 * QUANTITY_PTR q1;
 *
 * OPUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

static QRELATION_TYPE relationship(CELL_PTR relation, QUANTITY_PTR q1)
{ 
  return ((q1 == REL_ARG1(relation)) ? relationRelationship(relation)
	  : SYMMETRIC_RELATIONSHIP(relationRelationship(relation)));
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE true_relationship(relationship, relation, q1)
 *
 * DESCRIPTION: 
 * If q1 is the second argument of the relation, the "true" relationship for 
 * that  relation is the actually the symmetric relationship.
 *
 * INPUTS:
 * QRELATION_TYPE relationship; 
 * CELL_PTR relation; 
 * QUANTITY_PTR q1;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

QRELATION_TYPE true_relationship(QRELATION_TYPE relationship,
				 CELL_PTR relation, QUANTITY_PTR q1)
{
  return ((q1 == REL_ARG1(relation))
	  ? relationship : SYMMETRIC_RELATIONSHIP(relationship));
}


/*****************************************************************************
 *
 * FUNCTION: QUANTITY_PTR relation_other_arg(relation, q1)
 *
 * DESCRIPTION: Returns the other argument in the relation. 
 *
 * INPUTS:
 * CELL_PTR relation; 
 * QUANTITY_PTR q1;
 *
 * OUTPUTS: QUANTITY_PTR
 *
 *****************************************************************************/

QUANTITY_PTR relation_other_arg(CELL_PTR relation, QUANTITY_PTR q1)
{ 
  REL_DATA_PTR relData;
  
  relData = RELATION_DATA(relation);
  return ((q1 == relData->arg1) ? relData->arg2 : relData->arg1);
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR relationshipSupporter1(relationship, relation, q1)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QRELATION_TYPE relationship; 
 * CELL_PTR relation; 
 * QUANTITY_PTR q1;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

static TMS_NODE_PTR relationshipSupporter1(QRELATION_TYPE relationship,
					   CELL_PTR relation, QUANTITY_PTR q1)
{ 
  QRELATION_TYPE true_rel;
  TMS_NODE_PTR    supporter;
  
  true_rel = true_relationship(relationship, relation, q1);
  supporter = getValueNode(relation, (char *)true_rel);
  if (!supporter) {
    supporter = addValueNode(relation, (char *)true_rel);
    constrain_relationships(relation, supporter, true_rel);
  }
  return supporter;
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR relationship_supporter(relationship, q1, q2)
 * 
 * DESCRIPTION: 
 * Add a tms_node representing a relationship between two quantities to 
 * the relation.
 *
 * INPUTS:
 * QRELATION_TYPE relationship; 
 * QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

static TMS_NODE_PTR relationship_supporter(QRELATION_TYPE relationship,
					   QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  return relationshipSupporter1(relationship, relationGetOrMake(q1, q2), q1);
}


/*****************************************************************************
 *
 * FUNCTION:
 * void constrain_relationship(relation, relSuppot, relationship, otherSupport)
 *
 * DESCRIPTION: 
 * Add constraints such as "A>=B and B>=A => A=B" or "A>B => A>=B". 
 *
 * INPUTS:
 * CELL_PTR relation; 
 * TMS_NODE_PTR relSupport; 
 * QRELATION_TYPE relationship;
 * TMS_NODE_PTR otherSupport; 
 * 
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void constrain_relationship(CELL_PTR relation, TMS_NODE_PTR relSupport,
				   QRELATION_TYPE relationship,
				   TMS_NODE_PTR otherSupport)
{ 
  QRELATION_TYPE otherRelationship, relationshipUnion;
  
  otherRelationship = CV_RELP(otherSupport);
  relationshipUnion = REL_UNION(relationship, otherRelationship);
  
  if (relationshipUnion == Inconsistent) {
    /* CONTRADICTORY */
    REL_CONTRADICTION(relSupport, otherSupport);
  }
  else if (relationshipUnion == relationship) {
    /* SUBSUMES (A r B => A r' B) */
    IMPLIES(relSupport, otherSupport);
  }
  else if (relationshipUnion == otherRelationship) {
    /* SUBSUMED_BY (A r' B => A r B) */
    IMPLIES(otherSupport, relSupport);
  }
  else if ((int32)otherRelationship < (int32)relationship) {
    /* CONSTRAINING RELATIONSHIPS (A r B and A r' B) => A r'' B */
    REFINES(relation, otherSupport, relSupport, relationshipUnion);
  }
  else 
    REFINES(relation, relSupport, otherSupport, relationshipUnion);
}


/*****************************************************************************
 *
 * FUNCTION:
 * void constrain_relationships(relation, relSupport, relationship)
 * 
 * DESCRIPTION: Add constraints such as "A>=B and B>=A => A=B" or
 *              "A>B => A>=B".
 *
 * INPUTS:
 * CELL_PTR relation; 
 * TMS_NODE_PTR relSupport; 
 * QRELATION_TYPE relationship;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void constrain_relationships(CELL_PTR relation, TMS_NODE_PTR relSupport,
				    QRELATION_TYPE relationship)
{
  LIST_PTR list;
  TMS_NODE_PTR otherSupport;
  
  list = relation->contextualValues;
  
  otherSupport = (TMS_NODE_PTR)listFirst(list);
  while (otherSupport) {
    if (otherSupport != relSupport)
      constrain_relationship(relation, relSupport, relationship,
			     otherSupport);
    otherSupport = (TMS_NODE_PTR)listNext(list);
  }
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR Get_Relationship_Supporter(q1, relationship, q2)
 *
 * DESRIPTION:
 * Returns the tms_node which represents the relationship between q1 and q2.
 * Fixed 11/87 Campbell (was rel_to_symb, should be symb_to_rel)
 *
 * INPUTS:
 * QUANTITY_PTR q1; 
 * const char *relationship; 
 * QUANTITY_PTR q2;
 *
 * OUTPUTS:
 *
 *****************************************************************************/

#if 0
/* No longer used */
static TMS_NODE_PTR Get_Relationship_Supporter
(QUANTITY_PTR q1, const char *relationship, QUANTITY_PTR q2)
{ 
  return relationship_supporter(symbToRel(relationship), q1, q2); 
}
#endif

/*****************************************************************************
 *
 * FUNCTION: int32 do_propagate_lbound(strictness, lower_qbound1, lower_qbound2)
 *
 *****************************************************************************/

static int32 do_propagate_lbound(int32 strictness, QBOUND_PTR lower_qbound1,
			       QBOUND_PTR lower_qbound2)
{ 
  float save_epsilon;
  QRELATION_TYPE relationship;
  
  save_epsilon = GET_S_GLOBAL(Qlat_Epsilon);
  GET_S_GLOBAL(Qlat_Epsilon) = GET_S_GLOBAL(Prop_Epsilon);
  relationship = end_relationship(lower_qbound1, lower_qbound2, TRUE, TRUE);
  GET_S_GLOBAL(Qlat_Epsilon) = save_epsilon;
  
  return ((relationship == Greater) ||
	  ((relationship == Equal) && IS_STRICT(strictness)));
}


/*****************************************************************************
 *
 * FUNCTION: int32 do_propagate_ubound(strictness, upper_qbound1, upper_qbound2)
 *
 *****************************************************************************/

static int32 do_propagate_ubound(int32 strictness, QBOUND_PTR upper_qbound1, 
			       QBOUND_PTR upper_qbound2)
{ 
  float save_epsilon;
  QRELATION_TYPE relationship;
  
  save_epsilon = GET_S_GLOBAL(Qlat_Epsilon);
  GET_S_GLOBAL(Qlat_Epsilon) = GET_S_GLOBAL(Prop_Epsilon);
  relationship = end_relationship(upper_qbound1, upper_qbound2, TRUE, FALSE);
  GET_S_GLOBAL(Qlat_Epsilon) = save_epsilon;
  
  return ((relationship == Less) ||
	  ((relationship == Equal) && IS_STRICT(strictness)));
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE upper_lower_relationship(upperQqbound, lowerQbound)
 *
 *****************************************************************************/

static QRELATION_TYPE upper_lower_relationship(QBOUND_PTR upperQbound,
					       QBOUND_PTR lowerQbound)
{ 
  QRELATION_TYPE relationship;
  
  relationship = end_relationship(upperQbound, lowerQbound, FALSE, FALSE);
  if (relationship == Less) 
    return Less;
  else if (relationship == Equal) 
    return Less_Or_Equal;
  else 
    return No_Relationship;
}


/*****************************************************************************
 *
 * FUNCTION: void unmark_search1(quantity) 
 *
 *****************************************************************************/

static void unmark_search1(QUANTITY_PTR quantity)
{ 
  quantity->mark = No_Relationship;
  
  /* 27-Nov-89: fedor: may be dropping memeory here! */
  quantity->searchJust.justRel1 = NULL;
  quantity->searchJust.justRel2 = NULL;
}

/*****************************************************************************
 *
 * FUNCTION: void unmark_search()
 *
 *****************************************************************************/

static void unmark_search(void)
{
  Do_Memory(GET_S_GLOBAL(Search_Memory), (DO_MEMORY_FN) unmark_search1);
  Clear_Memory(GET_S_GLOBAL(Search_Memory));
}


/*****************************************************************************
 *
 * FUNCTION: 
 * QRELATION_TYPE updateSearchPath(addedRel, node, oldRel, justifier)
 *
 * DESCRIPTION:
 * The "search_path" tells how we got to a node.  Sometimes there are more than
 * one path to the node, one being more restrictive than another.  If the more 
 * restrictive path is found later, the search_path must be updated to reflect 
 * this new search path (there will never be more than two different
 * relationships).
 * Returns the most restrictive relation which is a combination of "added_rel" 
 * and "old_rel".
 *
 *****************************************************************************/

static QRELATION_TYPE updateSearchPath(QRELATION_TYPE addedRel,
				       QUANTITY_PTR node,
				       QRELATION_TYPE oldRel,
				       CELL_PTR justifier)
{ 
  QRELATION_TYPE newRel;
  
  newRel = REL_UNION(oldRel, addedRel);
  
  if (newRel != oldRel) {
    node->mark = newRel;
    
    if (newRel != addedRel) {
      if (node->searchJust.justRel2)
	tcaError("updateSearchPath: Search Node has three justifications.");
      
      node->searchJust.qRel1 = oldRel;
      /* 27-Mar-90: fedor: node->searchJust.justRel1 = not updated! */
      
      node->searchJust.qRel2 = addedRel;
      node->searchJust.justRel2 = justifier;
    }
    else {
      node->searchJust.justRel1 = justifier;
      node->searchJust.justRel2 = NULL;
      
      /* 27-Mar-90: fedor: unclear about updates for qRels here */
      node->searchJust.qRel1 = addedRel;
      node->searchJust.qRel2 = No_Relationship;
    }
    
    Remember((char *)node, GET_S_GLOBAL(Search_Memory));
  }
  return newRel;
}


/*****************************************************************************
 *
 * FUNCTION: 
 * QRELATION_TYPE sufficient_relationship(desiredRelationship, relSoFar)
 *
 *****************************************************************************/

static QRELATION_TYPE 
sufficient_relationship(QRELATION_TYPE desiredRelationship,
			QRELATION_TYPE relSoFar)
{
  switch (desiredRelationship) {
  case Equal: return Equal;
    
  case Less: return ((relSoFar == Less) ? Less_Or_Equal : Less);
    
  case Greater: return ((relSoFar == Greater) ? Greater_Or_Equal : Greater);
    
  case Less_Or_Equal:
  case Greater_Or_Equal: return desiredRelationship;
    
  case Not_Equal: return ((relSoFar == Equal) ? Not_Equal : Equal);
    
  case No_Relationship:
  case Inconsistent:
  case Real_Valued:
    return No_Relationship;
#ifndef TEST_CASE_COVERAGE
  default: return No_Relationship;
#endif
  }
}


/*****************************************************************************
 *
 * FUNCTION: void cacheOrdering1(relation, qRel, justRel, q1, q2)
 *
 * DESCRIPTION:
 * qRel is the relationship found to exist between q1 and q2.  
 * justRel (between q3 and q2) is the last relation on the path between q1 and
 * q2.  Support the relation between q1 and q2 by the justRel and the relation
 * between q1 and q3.
 *
 * INPUTS:
 * QRELATION_TYPE qRel;
 * QUANTITY_PTR q1, q2;
 * CELL_PTR relation, jusRel;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void cacheOrdering1(CELL_PTR relation, QRELATION_TYPE qRel,
			   CELL_PTR justRel, QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  CELL_PTR relQ1Q3;
  
  LIST_PTR supList;
  
  QUANTITY_PTR q3;
  QRELATION_TYPE sqRel;
  TMS_NODE_PTR q3Node, node;
  
  if (relation != justRel) {
    if (IS_ALREADY_TRUE(relationship(relation, q1), qRel)) {
      tcaError("Trying to cache an order that is already known to be true.");
    }
    else {
      q3 = relation_other_arg(justRel, q2);
      
      relQ1Q3 = relationGetOrMake(q1, q3);
      
      sqRel = sufficient_relationship(qRel, relationship(justRel, q3));
      
      q3Node = relationshipSupporter1(sqRel, relQ1Q3, q1);
      
      supList = listMake2((char *)currentValueSupporter(justRel),
			  (char *)q3Node);
      
      node = relationshipSupporter1(qRel, relation, q1);
      tmsJustifyNode(node, tmsCreateJustification1("TRANSITIVITY", supList));
      
      cacheOrderingSearch(q1, q3, relQ1Q3);
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void cacheOrderingSearch(q1, q2, relation)
 *
 * DESCRIPTION:
 * Cache the result of the search so it won't have to be done again.
 * Remembers the justification of how the path through the lattice was found
 * (for retraction and explanation purposes).
 *
 * For an inferrence of the form "q1 rel1 q2 rel2 q3 rel3 q4", 
 * the justification is "q1 rel5 q3" and "q3 rel3 q4"; and "q1 rel5 q3" is 
 * justified by "q1 rel1 q2" and  " q2 rel2 q3".
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * CELL_PTR relation;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void cacheOrderingSearch(QUANTITY_PTR q1, QUANTITY_PTR q2,
				CELL_PTR relation)
{ 
  QRELATION_TYPE relationship;
  
  if (!relation) 
    relation = relationGetOrMake(q1, q2);
  
  relationship = q2->mark;
  
  if (q2->searchJust.justRel1) {
    if (q2->searchJust.justRel2) { 
      /* two justifiers for the node */
      
      cacheOrdering1(relation, q2->searchJust.qRel1, 
		     q2->searchJust.justRel1, q1, q2);
      
      cacheOrdering1(relation, q2->searchJust.qRel1, 
		     q2->searchJust.justRel1, q1, q2);
    }
    else 
      cacheOrdering1(relation, relationship, q2->searchJust.justRel1, q1, q2);
  }
}


/*****************************************************************************
 *
 * FUNCTION: void find_ordering2(quantity, relSoFar)
 *
 * DESCRIPTION:
 * For each relation which is transitive with the "rel_so_far" add the 
 * other argument of the relation to the search queue (unless it has already 
 * been added, and the new relationship is not more restrictive).
 *
 * INPUTS:
 * QUANTITY_PTR quantity; 
 * QRELATION_TYPE relSoFar;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void find_ordering2(QUANTITY_PTR quantity, QRELATION_TYPE relSoFar)
{ 
  LIST_PTR list;
  CELL_PTR relation;
  QUANTITY_PTR qnext;
  QRELATION_TYPE incRel, oldRel;
  
  list = quantity->relations;
  
  relation = (CELL_PTR)listFirst(list);
  while (relation) {
    incRel = TRANSITIVITY(relSoFar, relationship(relation, quantity));
    if (IS_RELATIONSHIP(incRel)) {
      qnext = relation_other_arg(relation, quantity);
      if ((qnext != quantity) && !IS_REAL_VALUED(qnext)) {
	oldRel = SEARCH_RELATIONSHIP(qnext);
	if (IS_NO_RELATIONSHIP(oldRel)) {
	  qnext->mark = incRel;
	  qnext->searchJust.justRel1 = relation;
	  qnext->searchJust.justRel2 = NULL;
	  Remember((char *)qnext, GET_S_GLOBAL(Search_Memory));
	  Enqueue((char *)qnext, GET_S_GLOBAL(FO_Queue));
	}
	else if (oldRel != updateSearchPath(incRel, qnext, oldRel, relation))
	  Head_Enqueue((char *)qnext, GET_S_GLOBAL(FO_Queue));
      }
    }
    relation = (CELL_PTR)listNext(list);
  }
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE find_ordering1(q1, q2, searchingFor)
 *
 * DESCRIPTION:
 *
 * Search the lattice, looking for a relationship between q1 and q2 which is 
 * equal to or more restrictive than the "searching_for" relationship.
 * Does a breadth_first search starting from q1.  Often there is more than one 
 * path from q1 to a node. Need to remember which quantities (nodes) have been 
 * searched, and avoid searching the same node twice, UNLESS the second path is
 * more restrictive (e.g. "<" over "<=") than the existing path (it will never
 * need to be searched more than twice).
 *
 * The "mark" of the quantity is the relationship found so far from the start
 * of the search to the node. If a path is found, the inference and its 
 * justification is cached.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * QRELATION_TYPE searchingFor;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

static QRELATION_TYPE find_ordering1(QUANTITY_PTR q1, QUANTITY_PTR q2,
				     QRELATION_TYPE searchingFor)
{ 
  QRELATION_TYPE relp, oldRel, relSoFar, foundOrdering;
  CELL_PTR existingRelation;
  QUANTITY_PTR node;
  
  q1->mark = Equal;
  q1->searchJust.justRel1 = NULL;
  q1->searchJust.justRel2 = NULL;
  Remember((char *)q1, GET_S_GLOBAL(Search_Memory));
  
  for (node=q1; node; node=(QUANTITY_PTR)Dequeue(GET_S_GLOBAL(FO_Queue))) {
    existingRelation = get_relation(node, q2);
    relSoFar = SEARCH_RELATIONSHIP(node);
    if (existingRelation) {
      relp = TRANSITIVITY(relSoFar, relationship(existingRelation,node));
      if (IS_RELATIONSHIP(relp)) {
	oldRel =  SEARCH_RELATIONSHIP(q2);
	if (oldRel != No_Relationship) 
	  relp = updateSearchPath(relp, q2, oldRel, existingRelation);
	else {
	  q2->mark = relp;
	  q2->searchJust.justRel1 = existingRelation;
	  q2->searchJust.justRel2 = NULL;
	  Remember((char *)q2, GET_S_GLOBAL(Search_Memory));
	}
	if (IS_UNDER_CONSTRAINED(relp))
	  find_ordering2(node, relSoFar);
	else 
	  /* The desired relationship has been found */
	  Clear_Queue(GET_S_GLOBAL(FO_Queue)); 
      } else if (IS_UNDER_CONSTRAINED(relationship(existingRelation,node))) {
	find_ordering2(node, relSoFar);
      }
    }
    else 
      find_ordering2(node, relSoFar);
  }
  
  foundOrdering = SEARCH_RELATIONSHIP(q2);
  if (IS_RELATIONSHIP(foundOrdering) &&
      (IS_NO_RELATIONSHIP(searchingFor) || 
       !IS_UNDER_CONSTRAINED(foundOrdering))) {
    cacheOrderingSearch(q1, q2, (CELL_PTR)NULL);
    unmark_search();
    LRUNQ();
  }
  else 
    unmark_search();
  
  return foundOrdering;
}

/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE find_ordering(q1, q2, searchingFor)
 *
 * DESCRIPTION:
 * Search through the lattice for a path between q1 and q2.  
 * For efficiency, start the search with the quantity with the 
 * smallest number of relations.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * QRELATION_TYPE searchingFor;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

static QRELATION_TYPE find_ordering(QUANTITY_PTR q1, QUANTITY_PTR q2,
				    QRELATION_TYPE searchingFor)
{ 
  QRELATION_TYPE foundRelationship;
  
  if (IS_REAL_VALUED(q1) || IS_REAL_VALUED(q2)) 
    return No_Relationship;
  else if (NUM_RELATIONS(q2) < NUM_RELATIONS(q1)) {
    foundRelationship = find_ordering1(q2, q1, 
				       SYMMETRIC_RELATIONSHIP(searchingFor));
    return SYMMETRIC_RELATIONSHIP(foundRelationship);
  }
  else 
    return find_ordering1(q1, q2, searchingFor);
}


/*****************************************************************************
 *
 * FUNCTION: int32 strictness_of(relationship) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QRELATION_TYPE relationship;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 strictness_of(QRELATION_TYPE relationship)
{ 
  return (((relationship == Less) || (relationship == Greater))
	  ? STRICT : NOT_STRICT);
}


/*****************************************************************************
 *
 * FUNCTION: int32 supportsNeither(supporter, node1, node2)
 *
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR supporter, node1, node2;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 supportsNeither(TMS_NODE_PTR supporter, 
			   TMS_NODE_PTR node1, TMS_NODE_PTR node2)
{ 
  return !(listMemberItem((char *)supporter,  tmsSupport(node1)) ||
	   listMemberItem((char *)supporter,  tmsSupport(node2)));
}


/*****************************************************************************
 *
 * FUNCTION: int32 numericRelationshipSupports(relation, relationship, 
 *                                           uboundCV, lboundCV)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR relation; 
 * QRELATION_TYPE relationship; 
 * TMS_NODE_PTR uboundCV, lboundCV;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 numericRelationshipSupports(CELL_PTR relation, 
				       QRELATION_TYPE relationship,
				       TMS_NODE_PTR uboundCV, TMS_NODE_PTR lboundCV)
{ 
  int32 doJustify;
  JUSTIFICATION_PTR justifier;
  TMS_NODE_PTR relationshipSupporter;
  
  relationshipSupporter = relationshipSupporter1(relationship, relation,
						 QUANTITY_OF_QBOUND(uboundCV));
  
  justifier = NODE_SUPPORT_JUSTIFICATION(relationshipSupporter);
  
  doJustify = (IS_OUT(relationshipSupporter) ||
	       (supportsNeither(relationshipSupporter, lboundCV, uboundCV) &&
		STREQ(justifier->informant, "Subsumption") &&
		supportsNeither((TMS_NODE_PTR)listFirst(justifier->supporters),
				lboundCV, uboundCV)));
  
  if (doJustify)
    tmsJustifyNode(relationshipSupporter,
		   tmsCreateJustification1("Numeric_Qrelations",
					   listMake2((char *)uboundCV, 
						     (char *)lboundCV)));
  return doJustify;
}


/*****************************************************************************
 *
 * FUNCTION: int32 is_illegal_interval(lowerQbound, upperQbound) 
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QBOUND_PTR lowerQbound, upperQbound;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 is_illegal_interval(QBOUND_PTR lowerQbound, QBOUND_PTR upperQbound)
{
  return (is_less_real(upperQbound->value, lowerQbound->value) &&
	  end_relationship(lowerQbound, upperQbound, FALSE, FALSE) == Less);
}


/*****************************************************************************
 *
 * FUNCTION: void justify_bound(bound, newValue, newStrictness, isLowerBound,
 *                              informant, supporters)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR bound; 
 * float newValue;
 * int32 newStrictness, isLowerBound; 
 * const char *informant; 
 * LIST_PTR supporters;
 *
 * OUTPUTS: none.
 *
 * NOTE: Uses "tmsCreateJustification1" which does *not* copy the list of 
 *       supporters.  Therefore, the supporters cannot be used elsewhere 
 *       otherwise memory management will screw up.  Currently, all users of
 *       justify_bound make a new list.
 *
 *****************************************************************************/

static void justify_bound(CELL_PTR bound, float newValue, int32 newStrictness,
			  int32 isLowerBound, const char *informant,
			  LIST_PTR supporters)
{ 
  QBOUND_PTR newQbound;
  QUANTITY_PTR quantity;
  
  newQbound = createQbound(newValue, newStrictness);
  quantity = QUANTITY_OF_BOUND(bound);
  justify(bound, (char *)newQbound, 
	  tmsCreateJustification1(informant, supporters));
  if ((isLowerBound && 
       is_illegal_interval(newQbound, quantityUpperQbound(quantity))) ||
      (!isLowerBound && 
       is_illegal_interval(quantityLowerQbound(quantity), newQbound))) {
    tcaError("Illegal Interval Propagated (%s)\n", qName(quantity));
  }
}


/*****************************************************************************
 *
 * FUNCTION: void upper_bound_propagation1(relation, uboundCV, qbound, 
 *                                         quantity,
 *                                         relationship, true_relationship)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR relation; 
 * TMS_NODE_PTR uboundCV; 
 * QBOUND_PTR qbound;
 * QUANTITY_PTR quantity; 
 * QRELATION_TYPE relationship, true_relationship;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void upper_bound_propagation1(CELL_PTR relation, TMS_NODE_PTR uboundCV,
				     QBOUND_PTR qbound, QUANTITY_PTR quantity,
				     QRELATION_TYPE relationship,
				     QRELATION_TYPE true_relationship)
{
  int32 strictness;
  CELL_PTR bound2;
  TMS_NODE_PTR relationshipSupporter;
  
  if (GREATER_OR_EQUAL_RELP(relationship)) {
    relationshipSupporter = (TMS_NODE_PTR)getValueNode(relation, 
						       (char *)true_relationship);
    strictness = strictness_of(relationship);
    
    if (!(listMemberItem((char *)uboundCV,
			 tmsSupport(relationshipSupporter)) ||
	  listMemberItem((char *)relationshipSupporter,
			 tmsSupport(uboundCV)))) {
      /* If neither the numeric bound or the relationship supported the 
       * value of the other, try to constrain the bound of the other 
       * quantity.*/
      bound2 = qval_upper(relation_other_arg(relation, quantity));
      if (do_propagate_ubound(strictness, qbound, THE_UPPER_QBOUND(bound2))) {
	justify_bound(bound2, qbound->value, 
		      NEW_LIMIT(strictness, qbound->limit),
		      FALSE, "Numeric_Constraint_Propagation",
		      listMake2((char *)uboundCV, (char *)relationshipSupporter));
      }
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void upper_bound_propagation(name, uboundCV) 
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * TMS_NODE_PTR uboundCV;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static void upper_bound_propagation(const char *name, TMS_NODE_PTR uboundCV,
				    const char *ignore )
{ 
#ifdef applec
#pragma unused(name,ignore)
#endif
  LIST_PTR list;
  CELL_PTR relation;
  QBOUND_PTR qbound;
  QRELATION_TYPE relp;
  QUANTITY_PTR quantity, otherArg;
  
  quantity = QUANTITY_OF_QBOUND(uboundCV);
  qbound = CV_QBOUND(uboundCV);
  
  if ((qbound->value != GET_S_GLOBAL(Default_Upper_Qbound)->value) ||
      (qbound->limit != GET_S_GLOBAL(Default_Upper_Qbound)->limit)) {
    
    list = quantity->relations;
    relation = (CELL_PTR)listFirst(list);
    while (relation) {
      otherArg = relation_other_arg(relation, quantity);
      relp = upper_lower_relationship(qbound, quantityLowerQbound(otherArg));
      
      if (IS_NO_RELATIONSHIP(relp) || 
	  !numericRelationshipSupports(relation, relp, uboundCV, 
				       currentValueSupporter(qval_lower(otherArg))))
	upper_bound_propagation1(relation, uboundCV, qbound, quantity,
				 relationship(relation, quantity),
				 relationRelationship(relation));
      
      relation = (CELL_PTR)listNext(list);
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void lower_bound_propagation1(relation, lboundCV, qbound,
 *                                         quantity,
 *                                         relationship, trueRelationship)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR relation; 
 * TMS_NODE_PTR lboundCV; 
 * QBOUND_PTR qbound;
 * QUANTITY_PTR quantity; 
 * QRELATION_TYPE relationship, trueRelationship;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void lower_bound_propagation1(CELL_PTR relation, TMS_NODE_PTR lboundCV,
				     QBOUND_PTR qbound, QUANTITY_PTR quantity,
				     QRELATION_TYPE relationship, 
				     QRELATION_TYPE trueRelationship)
{ 
  int32 strictness;
  CELL_PTR bound2;
  TMS_NODE_PTR relationshipSupporter;
  
  if (LESS_OR_EQUAL_RELP(relationship)) {
    /*printf("lower_bound_propagation1 (%s -> %s)\n", qName(quantity),
      qName(relation_other_arg(relation, quantity)));*/
    relationshipSupporter = (TMS_NODE_PTR)getValueNode(relation, 
						       (char *)trueRelationship);
    strictness = strictness_of(relationship);
    if (!(listMemberItem((char *)lboundCV,
			 tmsSupport(relationshipSupporter)) ||
	  listMemberItem((char *)relationshipSupporter,
			 tmsSupport(lboundCV)))) {
      /* If neither the numeric bound or the relationship supported the 
       * value of the other, try to constrain the bound of the other quantity.
       */
      bound2 = qval_lower(relation_other_arg(relation, quantity));
      if (do_propagate_lbound(strictness, qbound, THE_LOWER_QBOUND(bound2))) {
	justify_bound(bound2, qbound->value, 
		      NEW_LIMIT(strictness, qbound->limit),
		      TRUE, "Numeric_Constraint_Propagation",
		      listMake2((char *)lboundCV, (char *)relationshipSupporter));
      }
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void lower_bound_propagation(name, lboundCV) 
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * TMS_NODE_PTR lboundCV;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static void lower_bound_propagation(const char *name, TMS_NODE_PTR lboundCV,
				    const char *ignore)
{ 
#ifdef applec
#pragma unused(name,ignore)
#endif
  LIST_PTR list;
  CELL_PTR relation;
  QBOUND_PTR qbound;
  QRELATION_TYPE relp;
  QUANTITY_PTR quantity, otherArg;
  
  qbound = CV_QBOUND(lboundCV);
  quantity = QUANTITY_OF_QBOUND(lboundCV);
  
  if ((qbound->value != GET_S_GLOBAL(Default_Lower_Qbound)->value) ||
      (qbound->limit != GET_S_GLOBAL(Default_Lower_Qbound)->limit)) {
    
    list = quantity->relations;
    relation = (CELL_PTR)listFirst(list);
    while (relation) {
      otherArg = relation_other_arg(relation, quantity);
      relp = upper_lower_relationship(quantityUpperQbound(otherArg), qbound);
      
      if (IS_NO_RELATIONSHIP(relp) ||
	  !numericRelationshipSupports(relation, relp, 
				       currentValueSupporter(qval_upper(otherArg)), lboundCV))
	lower_bound_propagation1(relation, lboundCV, qbound, quantity,
				 relationship(relation, quantity),
				 relationRelationship(relation));
      
      relation = (CELL_PTR)listNext(list);
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void upper_bound_repropagation(name, out_uboundCV)
 *
 * DESCRIPTION:
 *
 * The basic repropagation algorithm:
 *
 * If value of the out node is more constrained than the current value of the 
 * bound then current_value <_ current_value of bound 
 *
 * for each relation
 *    if it is the right type of relation for constraining (less_or_equal_relp)
 *       and neither the supporter nor the other bound is supported 
 *           by the current value 
 *       and the current value is not supported by the other bound 
 *       and the value with the strictness of the relation is more 
 *           constraining than the current value 
 *    then
 *       current_value <_ new qbound, supporters <_ bound_cv, relation_cv
 *
 * if a new value has been found then justify_bound
 *
 * INPUTS:
 * const char *name; 
 * TMS_NODE_PTR out_uboundCV;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static void upper_bound_repropagation(const char *name,
				      TMS_NODE_PTR out_uboundCV,
				      const char *ignore)
{ 
#ifdef applec
#pragma unused(name,ignore)
#endif
  LIST_PTR list;
  QBOUND_PTR curQbound;
  QUANTITY_PTR quantity;
  CELL_PTR other_qt_bound, relation;
  
  TMS_NODE_PTR new_bound_supporter=NULL, new_relp_supporter=NULL;
  TMS_NODE_PTR relp_supporter, other_quantity_supporter;
  
  quantity = QUANTITY_OF_QBOUND(out_uboundCV);
  curQbound = quantityUpperQbound(quantity);
  
  if (is_upper_bound_more_constrained(CV_QBOUND(out_uboundCV), curQbound)) {
    
    list = quantity->relations;
    
    relation = (CELL_PTR)listFirst(list);
    while (relation) {
      
      new_bound_supporter = getValueNode(qval_upper(quantity),
					 (char *)curQbound);
      new_relp_supporter = NULL;
      
      if (LESS_OR_EQUAL_RELP(relationship(relation, quantity))) {
	relp_supporter = currentValueSupporter(relation);
	other_qt_bound = qval_upper(relation_other_arg(relation, quantity));
	other_quantity_supporter = currentValueSupporter(other_qt_bound);
	if (!listMemberItem((char *)other_quantity_supporter, 
			    tmsSupport(new_bound_supporter))
	    && supportsNeither(new_bound_supporter, relp_supporter,
			       other_quantity_supporter) &&
	    do_propagate_ubound(strictness_of(CV_RELP(relp_supporter)),
				CV_QBOUND(other_quantity_supporter),
				curQbound)) {
	  curQbound = CV_QBOUND(other_quantity_supporter);
	  new_bound_supporter = other_quantity_supporter;
	  new_relp_supporter = relp_supporter;
	}
      }
      
      relation = (CELL_PTR)listNext(list);
    }
    
    if (new_relp_supporter)
      justify_bound(qval_upper(quantity), curQbound->value,
		    NEW_LIMIT(strictness_of(CV_RELP(new_relp_supporter)),
			      curQbound->limit),
		    FALSE, "Numeric_Constraint_Propagation",
		    listMake2((char *)new_bound_supporter,
			      (char *)new_relp_supporter));
  }
}

/*
   (bound_repropagation T out_lboundCV) 
   (setq quantity_qbound_fn 'QUANTITY_LOWER_QBOUND
   constrained_fn 'LOWER_BOUND_MORE_CONSTRAINED? qval_fn 'QVAL_LOWER
   is_a_relationship 'GREATER_OR_EQUAL_RELP
   propagation?_fn 'PROPAGATE_LOWER_BOUND?))
   */


/*****************************************************************************
 *
 * FUNCTION: void lower_bound_repropagation(name, out_lboundCV) 
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * TMS_NODE_PTR out_lboundCV;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static void lower_bound_repropagation(const char *name,
				      TMS_NODE_PTR out_lboundCV,
				      const char *ignore)
{ 
#ifdef applec
#pragma unused(name,ignore)
#endif
  LIST_PTR list;
  QUANTITY_PTR quantity;
  QBOUND_PTR current_qbound;
  CELL_PTR other_qt_bound, relation;
  TMS_NODE_PTR new_bound_supporter, new_relp_supporter;
  TMS_NODE_PTR relp_supporter, other_quantity_supporter;
  
  quantity = QUANTITY_OF_QBOUND(out_lboundCV);
  current_qbound = quantityLowerQbound(quantity);
  if (is_lower_bound_more_constrained(CV_QBOUND(out_lboundCV), 
				      current_qbound)) {
    new_bound_supporter = getValueNode(qval_lower(quantity), 
				       (char *)current_qbound);
    new_relp_supporter = NULL;
    
    list = quantity->relations;
    relation = (CELL_PTR)listFirst(list);
    while (relation) {
      if (GREATER_OR_EQUAL_RELP(relationship(relation, quantity))) {
	
	relp_supporter = currentValueSupporter(relation);
	other_qt_bound = qval_lower(relation_other_arg(relation, quantity));
	other_quantity_supporter = currentValueSupporter(other_qt_bound);
	
	if (!listMemberItem((char *)other_quantity_supporter, 
			    tmsSupport(new_bound_supporter))
	    && supportsNeither(new_bound_supporter, relp_supporter,
			       other_quantity_supporter) 
	    && do_propagate_lbound(strictness_of(CV_RELP(relp_supporter)),
				   CV_QBOUND(other_quantity_supporter),
				   current_qbound)) {
	  current_qbound = CV_QBOUND(other_quantity_supporter);
	  new_bound_supporter = other_quantity_supporter;
	  new_relp_supporter = relp_supporter;
	}
      }
      
      relation = (CELL_PTR)listNext(list);
    }
    
    if (new_relp_supporter)
      justify_bound(qval_lower(quantity), current_qbound->value,
		    NEW_LIMIT(strictness_of(CV_RELP(new_relp_supporter)),
			      current_qbound->limit),
		    TRUE, "Numeric_Constraint_Propagation",
		    listMake2((char *)new_bound_supporter,
			      (char *)new_relp_supporter));
  }
}


/*****************************************************************************
 *
 * FUNCTION: void relation_propagation1(quantity1, relation, true_relationship)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QUANTITY_PTR quantity1; 
 * CELL_PTR relation; 
 * QRELATION_TYPE true_relationship;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void relation_propagation1(QUANTITY_PTR quantity1, CELL_PTR relation,
				  QRELATION_TYPE true_relationship)
{ 
  QRELATION_TYPE relp;
  QBOUND_PTR current_upper_qbound, current_lower_qbound;
  
  relp = relationship(relation, quantity1);
  current_upper_qbound = quantityUpperQbound(quantity1);
  current_lower_qbound = quantityLowerQbound(quantity1);
  
  if ((current_upper_qbound != GET_S_GLOBAL(Default_Upper_Qbound)) 
      && GREATER_OR_EQUAL_RELP(relp))
    upper_bound_propagation1(relation, getValueNode(qval_upper(quantity1),
						    (char *)current_upper_qbound),
			     current_upper_qbound, quantity1, 
			     relp, true_relationship);
  
  if ((current_lower_qbound != GET_S_GLOBAL(Default_Lower_Qbound)) 
      && LESS_OR_EQUAL_RELP(relp))
    lower_bound_propagation1(relation, getValueNode(qval_lower(quantity1),
						    (char *)current_lower_qbound),
			     current_lower_qbound, quantity1, 
			     relp, true_relationship);
}

/*****************************************************************************
 *
 * FUNCTION: void relation_propagation(name, relationshipSupporter)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * const char *name; 
 * TMS_NODE_PTR relationshipSupporter;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

/*ARGSUSED*/
static void relation_propagation(const char *name,
				 TMS_NODE_PTR relationshipSupporter,
				 const char *ignore)
{ 
#ifdef applec
#pragma unused(name, ignore)
#endif
  CELL_PTR relation;
  QRELATION_TYPE true_relationship;
  QUANTITY_PTR arg1, arg2;
  
  relation = (CELL_PTR)relationshipSupporter->cell;
  true_relationship = CV_RELP(relationshipSupporter);
  arg1 = REL_ARG1(relation);
  arg2 = REL_ARG2(relation);
  /* This line should be included when the 
     file "QINCREMENTAL_INTERFACE" is loaded.
     Try_To_Establish(relation, relationshipSupporter, arg1, arg2) */
  if (!IS_REAL_VALUED(arg2))
    relation_propagation1(arg1, relation, true_relationship);
  if (!IS_REAL_VALUED(arg1))
    relation_propagation1(arg2, relation, true_relationship);
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE numeric_qrelationship(q1, q2)
 *
 * DESCRIPTION
 * numeric_qrelationship(q1, q2)
 * Returns the relationship between "q1" and "q2"
 * based only on their interval values.
 * Returns "No_Relationship" if no relationship exists, or if the
 * numbers are both exactly infinity (same value).
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

static QRELATION_TYPE numeric_qrelationship(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  CELL_PTR relation;
  QRELATION_TYPE relationship1, relationship2;
  
  if (q1 == q2)
    return Equal;
  
  relationship1 = upper_lower_relationship(quantityUpperQbound(q1),
					   quantityLowerQbound(q2));
  relationship2 = upper_lower_relationship(quantityUpperQbound(q2),
					   quantityLowerQbound(q1));
  
  if (IS_RELATIONSHIP(relationship1) || IS_RELATIONSHIP(relationship2)) {
    relation = relationGetOrMake(q1, q2);
    if (IS_RELATIONSHIP(relationship1))
      (void)numericRelationshipSupports(relation, relationship1,
					currentValueSupporter(qval_upper(q1)),
					currentValueSupporter(qval_lower(q2)));
    if (IS_RELATIONSHIP(relationship2))
      (void)numericRelationshipSupports(relation, relationship2,
					currentValueSupporter(qval_upper(q2)),
					currentValueSupporter(qval_lower(q1)));
    LRUNQ();
    
    return REL_UNION(relationship1, SYMMETRIC_RELATIONSHIP(relationship2));
  } 
  else
    return No_Relationship;
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE find_existing_relation(q1, q2)
 *
 * DESCRIPTION:
 * If a relationship is cached use it only if it is the most
 * constrained relationship.
 * Otherwise more information might have been added since the
 * relationship was found to make it more constrained.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

static QRELATION_TYPE find_existing_relation(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  CELL_PTR existingRelation;
  QRELATION_TYPE relp;
  
  existingRelation = get_relation(q1, q2);
  if (existingRelation) {
    relp = relationship(existingRelation, q1);
    if (((relp != No_Relationship) && 
	 !IS_UNDER_CONSTRAINED(relp)) ||
	IS_REAL_VALUED(q1) || IS_REAL_VALUED(q2))
      return relp;
  }
  
  return No_Relationship;
}


/*****************************************************************************
 *
 * FUNCTION: QRELATION_TYPE qrelationship1(q1, q2, searchingFor)
 *
 * DESCRIPTION
 * Finds all the (internal numeric) relationship between the quantities. 
 * Typically, "searchingFor" should be No_Relationship.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * QRELATION_TYPE searchingFor;
 *
 * OUTPUTS: QRELATION_TYPE
 *
 *****************************************************************************/

QRELATION_TYPE qrelationship1(QUANTITY_PTR q1, QUANTITY_PTR q2,
			      QRELATION_TYPE searchingFor)
{ 
  QRELATION_TYPE relationship;
  
  if (q1 == q2) 
    return Equal;
  else {
    relationship = find_existing_relation(q1, q2);
    
    if (IS_RELATIONSHIP(relationship)) 
      return relationship;
    
    relationship = numeric_qrelationship(q1, q2);
    
    if (IS_RELATIONSHIP(relationship)) 
      return relationship;
    
    return find_ordering(q1, q2, searchingFor);
  }
}

/*****************************************************************************
 *
 * FUNCTION: const char *QRelationship(q1, q2)
 *
 * DESCRIPTION
 * Returns the string that represents the relationship between the two values.
 *
 * INPUTS: QUANTITY_PTR q1, q2;
 *
 * OUTPUTS: const char *
 *
 *****************************************************************************/

const char *QRelationship(QUANTITY_PTR q1, QUANTITY_PTR q2)
{ 
  return(relToSymb(qrelationship1(q1, q2, No_Relationship)));
}


/*****************************************************************************
 *
 * FUNCTION: TRI_VALUED_TYPE QIs(q1, relationship, q2)
 *
 * DESCRIPTION:
 * Finds the relationship between "q1" and "q2" and sees if
 *"relationship" is implied or inconsistent with the inferred relationship.
 * Return "Is_True" only if the specified relation holds between the 
 * two quantities, return "Is_False" if it does not hold, otherwise 
 * return "Is_Unknown".
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * const char *relationship; 
 *
 * OUTPUTS: TRI_VALUED_TYPE
 *
 *****************************************************************************/

TRI_VALUED_TYPE QIs(QUANTITY_PTR q1, const char *relationship,
		    QUANTITY_PTR q2)
{ 
  QRELATION_TYPE desired_relationship, actual_relationship;
  
  desired_relationship = symbToRel(relationship);
  actual_relationship = qrelationship1(q1, q2,  No_Relationship);
  
  if (IS_ALREADY_TRUE(actual_relationship, desired_relationship)) 
    return Is_True;
  else if (IS_IMPLIED_RELATION(actual_relationship, desired_relationship))
    return Is_Unknown;
  else 
    return Is_False;
}


/*****************************************************************************
 *
 * FUNCTION: int32 QIs_True(q1, relationship, q2)
 *
 * DESCRIPTION:
 * Returns TRUE only if the specified relation holds between the two
 * quantities;
 * return FALSE if it does not hold or it is not known if it holds.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * const char *relationship; 
 *
 * OUTPUTS: int32 TRUE or FALSE
 *
 *****************************************************************************/

int32 QIs_True(QUANTITY_PTR q1, const char *relationship, QUANTITY_PTR q2)
{ 
  QRELATION_TYPE desired_relationship;
  
  desired_relationship = symbToRel(relationship);
  return IS_ALREADY_TRUE(qrelationship1(q1, q2, desired_relationship),
			 desired_relationship);
}


/*****************************************************************************
 *
 * FUNCTION: void qassert1(q1, relp, q2, justification)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * QRELATION_TYPE relp; 
 * JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void qassert1(QUANTITY_PTR q1, QRELATION_TYPE relp, QUANTITY_PTR q2,
		     JUSTIFICATION_PTR justification)
{
  if (GET_S_GLOBAL(qassertCheckGlobal)) {
    if (IS_CONSISTENT(relp, q1, q2))
      UPDATE_RELATIONSHIP(relp, relationGetOrMake(q1, q2), q1, justification);
    else {
      Log("ERROR: Inconsistent assertion:\n");
      Print_Quantity(q1, FALSE);
      Log(" %s ", relToSymb(relp));
      Print_Quantity(q2, TRUE);
      tcaError((char *)NULL);
    }
  } else {
    UPDATE_RELATIONSHIP(relp, relationGetOrMake(q1, q2), q1, justification);
  }
}


/*****************************************************************************
 *
 * FUNCTION: JUSTIFICATION_PTR qlattice_justification(informant, justifiers)
 *
 * DESCRIPTION
 *
 * INPUTS:
 * const char *informant; 
 * LIST_PTR justifiers;
 *
 * OUTPUTS: JUSTIFICATION_PTR
 *
 *****************************************************************************/

static JUSTIFICATION_PTR qlattice_justification(const char *informant,
						LIST_PTR justifiers)
{
  if (!informant) 
    return tmsCreateJustification("Premise", (LIST_PTR)NULL);
  else if (STREQ(informant, "Assumption")) 
    return GET_S_GLOBAL(tmsAssumpJustificationGlobal);
  else 
    return tmsCreateJustification(informant, justifiers);
}


/*****************************************************************************
 *
 * FUNCTION: void QAssert(q1, relationship, q2, informant)
 *
 * DESCRIPTION:
 * Add to the lattice the specified relation between the two quantities.
 * Check (to a degree) that the relationship is consistent with the
 * current state of the quantity lattice.
 * Propagate all the numerical constraints imposed by the new relation.
 * If "informant" is NULL, makes it a premise ; if "informant" is
 * "Assumption", makes it an assumption; else uses the given "informant" 
 * to justify the assertion.
 * 
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * const char *relationship, *informant; 
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void QAssert(QUANTITY_PTR q1, const char *relationship, 
	     QUANTITY_PTR q2, const char *informant)
{
  if (q1 == q2) {
    if (STREQ(relationship, "<") || STREQ(relationship, ">")) {
      Log("ERROR: Quantity asserted to be less-than or greater-than itself ");
      Print_Quantity(q1, TRUE);
      tcaError((char *)NULL);
    }
  }
  else {
    qassert1(q1, symbToRel(relationship), q2,
	     qlattice_justification(informant, (LIST_PTR)NULL));
    LRUNQ();
  }
}


/*****************************************************************************
 *
 * FUNCTION: int32 qretract1(q1, relp, q2, informant, justifiers)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * QRELATION_TYPE relp; 
 * const char *informant; 
 * LIST_PTR justifiers;
 *
 * OUTPUTS: int32
 *
 *****************************************************************************/

static int32 qretract1(QUANTITY_PTR q1, QRELATION_TYPE relp, QUANTITY_PTR q2, 
		     const char *informant, LIST_PTR justifiers)
{ 
  TMS_NODE_PTR relp_node;
  
  relp_node = relationship_supporter(relp, q1, q2);
  if (IS_IN(relp_node)) {
    if (!justifiers && (IS_ASSUMED(relp_node) || tmsIsPremise(relp_node))) {
      if (IS_ASSUMED(relp_node)) 
	tmsUnassumeNode(relp_node);
      else 
	tmsUnassertNode(relp_node, (informant ? informant : "Premise"));
      return TRUE;
    }
    else {
      tcaError("Can only retract premises and assumptions (%s %s %s)",
	       qName(q1), relToSymb(relp), qName(q2));
    }
  }
  return FALSE;
}


/*****************************************************************************
 *
 * FUNCTION: void QRetract(q1, relationship, q2, informant)
 *
 * DESCRIPTION:
 * Remove from the lattice the specified relation between the two quantities.
 * Check that the relationship was actually asserted (using the "informant").
 * Remove all the constraints on interval values that were propagated
 * by the assertion of this relation.
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * const char *relationship, *informant; 
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void QRetract(QUANTITY_PTR q1, const char *relationship,
	      QUANTITY_PTR q2, const char *informant)
{ 
  if (q1 != q2)
    if (qretract1(q1, symbToRel(relationship), q2, informant, (LIST_PTR)NULL))
      LRUNQ();
}

/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR QNode(q1, relationship, q2)
 *
 * DESCRIPTION:
 * Returns the TMS node that encodes the information that "q1" stands in the
 * "relationship" to "q2".  Note that the node is returned even if it is "out" 
 * (ie, the relationship does not currently hold).
 *
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * const char relationship; 
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

TMS_NODE_PTR QNode(QUANTITY_PTR q1, const char *relationship, QUANTITY_PTR q2)
{ 
  QRELATION_TYPE relp;
  CELL_PTR relation;
  
  relp = symbToRel(relationship);
  (void)qrelationship1(q1, q2, relp);
  relation = relationGetOrMake(q1, q2);
  return relationshipSupporter1(relp, relation, q1);
}


/*****************************************************************************
 *
 * FUNCTION: void FreeQuantity(quantity)
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR quantity;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void FreeQuantity(QUANTITY_PTR *quantity)
{
  CELL_PTR relation;

  if ((*quantity) == NULL) return;
  
/*  if ((*quantity)->lower_bound) {*/
/*    Log("Not freeing lower bound of %s", qName((*quantity)));*/
/*  }*/
/*  if ((*quantity)->upper_bound) {*/
/*    Log("Not freeing upper bound of %s", qName((*quantity)));*/
/*  }*/
/*  if ((*quantity)->arithmetic_fields) {*/
/*    Log("Not freeing arithmetic fields of %s", qName((*quantity)));*/
/*  }*/
  
  relation = (CELL_PTR)listFirst((*quantity)->relations);
  while (relation) {
    listDeleteItem((char *)relation,
		   (relation_other_arg(relation, (*quantity)))->relations);
    (void)hashTableRemove((char *)RELATION_DATA(relation), 
			  GET_S_GLOBAL(Relation_Hash_Table));
    tcaFree((char *)relation->datum);
    cellFree(&relation);
    relation = (CELL_PTR)listNext((*quantity)->relations);
  }
  listFree(&((*quantity)->relations));
  tcaFree((char *)(*quantity)->name); 
  tcaFree((char *)(*quantity));
  *quantity = NULL;
}

/*****************************************************************************
 *
 * FUNCTION: void removeQuantityRules()
 *
 * DESCRIPTION: Remove all the rules for all the relations associated
 *              with the quantity.
 *
 * INPUTS: QUANTITY_PTR quantity;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void removeQuantityRules (QUANTITY_PTR quantity)
{
  CELL_PTR relation;
  
  relation = (CELL_PTR)listFirst(quantity->relations);
  while (relation) {
    rulesFree(relation->rules);
    relation->rules = NULL;
    rulesFree(relation->inRules);
    relation->inRules = NULL;
    rulesFree(relation->outRules);
    relation->outRules = NULL;
    relation = (CELL_PTR)listNext(quantity->relations);
  }
}

/*****************************************************************************
 *
 * FUNCTION: void set_up_initial_propagation_rules()
 *
 * DESCRIPTION:
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void set_up_initial_propagation_rules(void)
{
  GET_S_GLOBAL(Upper_Bound_Propagation_Rule) = 
    ruleCreateCell("GET_S_GLOBAL(Upper_Bound_Propagation_Rule)",
		   (CELL_RULE_FN) upper_bound_propagation, 
		   0, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(Upper_Bound_Propagation_Rule)->used = 1000;
  
  GET_S_GLOBAL(Lower_Bound_Propagation_Rule) = 
    ruleCreateCell("GET_S_GLOBAL(Lower_Bound_Propagation_Rule)",
		   (CELL_RULE_FN) lower_bound_propagation, 
		   0, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(Lower_Bound_Propagation_Rule)->used = 1000;
  
  GET_S_GLOBAL(Repropagate_Upper_Bound_Rule) = 
    ruleCreateOut("Upper_Bound_Repropagation",
		  (NODE_RULE_FN) upper_bound_repropagation, 
		  0, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(Repropagate_Upper_Bound_Rule)->used = 1000;
  
  GET_S_GLOBAL(Repropagate_Lower_Bound_Rule) = 
    ruleCreateOut("Lower_Bound_Repropagation",
		  (NODE_RULE_FN) lower_bound_repropagation, 
		  0, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(Repropagate_Lower_Bound_Rule)->used = 1000;
  
  GET_S_GLOBAL(Relation_Propagation_Rule) = 
    ruleCreateIn("Relation_Propagation",
		 (NODE_RULE_FN) relation_propagation, 0, (char *)NULL);
  /* Never free, Reid: 10-Jul-91 */
  GET_S_GLOBAL(Relation_Propagation_Rule)->used = 1000;
}


/*****************************************************************************
 *
 * FUNCTION: void set_up_initial_qlattice_values()
 *
 * DESCRIPTION:
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void set_up_initial_qlattice_values(void)
{
  GET_S_GLOBAL(Default_Lower_Qbound) = 
    createQbound(GET_S_GLOBAL(MINUS_INFINITY), CLOSED);
  GET_S_GLOBAL(Default_Upper_Qbound) = 
    createQbound(GET_S_GLOBAL(PLUS_INFINITY), CLOSED);
  GET_S_GLOBAL(Default_Finite_Lower_Qbound) = 
    createQbound(GET_S_GLOBAL(MINUS_INFINITY), OPEN);
  GET_S_GLOBAL(Default_Finite_Upper_Qbound) = 
    createQbound(GET_S_GLOBAL(PLUS_INFINITY), OPEN);
  
  GET_S_GLOBAL(Zero_Qbound) = createQbound(0.0, CLOSED);
  GET_S_GLOBAL(One_Qbound) = createQbound(1.0, CLOSED);
  GET_S_GLOBAL(Minus_One_Qbound) = createQbound(-1.0, CLOSED);
}


/*****************************************************************************
 *
 * FUNCTION: void print_quantity_value(quant) 
 *
 * DESCRIPTION:
 *
 * INPUTS: QUANTITY_PTR quant;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void print_quantity_value(QUANTITY_PTR quant)
{ 
  print_lower_qbound(quantityLowerQbound(quant));
  Log(" ... ");
  print_upper_qbound(quantityUpperQbound(quant));
}


/*****************************************************************************
 *
 * FUNCTION: void Print_Quantity(quant, endLine)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * QUANTITY_PTR quant; 
 * int endLine;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void Print_Quantity(QUANTITY_PTR quant, int32 endLine)
{
  Log("#<QUANTITY %s ", qName(quant));
  print_quantity_value(quant);
  Log(">");
  if (endLine) 
    Log("\n");
}

/*****************************************************************************
 *
 * FUNCTION: void Print_Relationship(q1, q2, end_line)
 *
 * DESCRIPTION: Print the qrelationship between the two quantities.
 * 
 * INPUTS:
 * QUANTITY_PTR q1, q2; 
 * int32 end_line;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void Print_Relationship(QUANTITY_PTR q1, QUANTITY_PTR q2, int32 end_line)
{ 
  Log("%s %s %s", qName(q1), QRelationship(q1, q2), qName(q2));
  if (end_line) 
    Log("\n");
}

/*****************************************************************************
 *
 * FUNCTION: void qlatticeInit()
 *
 * DESCRIPTION: Initialize the quantity lattice. 
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void qlatticeInit(void)
{
  GET_S_GLOBAL(Relation_Hash_Table) = 
    hashTableCreate(RELATION_HASH_TABLE_SIZE, (HASH_FN) rel_hash_key,
		    (EQ_HASH_FN) same_rel_data);
  
  GET_S_GLOBAL(FO_Queue) = Create_Queue();
  GET_S_GLOBAL(Search_Memory) = Create_Memory();
  
  tmsInit();
  rulesInit(5);
  
  GET_S_GLOBAL(numQuantitiesGlobal) = 0;
  GET_S_GLOBAL(numRelationsGlobal) = 0;
  GET_S_GLOBAL(numBoundsGlobal) = 0;
  set_up_initial_qlattice_values();
  set_up_initial_propagation_rules();
  
  GET_S_GLOBAL(Zero_Quant) = createRealQuantity(0.0);
  GET_S_GLOBAL(One_Quant) = createRealQuantity(1.0);
  GET_S_GLOBAL(Minus_One_Quant) = createRealQuantity(-1.0);
}


/*****************************************************************************
  
 * Returns the justification used if the relationship between q1 and q2 was 
 *   asserted as a premise (the default relationship is the current most 
 *   constrained one).
 
 JUSTIFICATION_PTR QJustifier (q1 relationship q2)
 (if (is_relationship (qrelationship1 q1 q2 No_Relationship))
 (setq relationship (symb_to_rel relationship))
 (let* ((relation (get_or_make_relation q1 q2))
 (support (relationshipSupporter1 (or relationship (relationship relation q1))
 relation q1)))
 (when (and support (not (derived? support)))
 (node_support_justification support->informant)))))
 
 ;;; Find the support of the relationship between quantities q1 and q2.
 ;;;   The relationship defaults to the current most constrained one.
 ;;;   The "support" is a list of tms_nodes.
 LIST_PTR QSupports (q1 relationship q2)
 (if (is_relationship (qrelationship1 q1 q2 No_Relationship))
 (setq relationship (symb_to_rel relationship))
 (let ((relation (get_or_make_relation q1 q2)))
 (qsupports1 (relationshipSupporter1 (or relationship 
 (relationship relation q1))
 relation q1)))))
 
 void qsupports1 (support)
 (cond ((not (derived? support)) (list support))
 (T (let (all_supports)
 (dolist (support (support support) all_supports)
 (setq all_supports (nunion (qsupports1 support) all_supports :TEST #'eql)))))))
 
 ;;; QSUPPORTS_READABLE is like QSUPPORTS except that it returns a list of elements
 ;;;   each of the form `(Q1 RELATIONSHIP Q2)' or `(Bound Q)'
 void QSUPPORTS_READABLE (q1 q2 &optional relationship)
 (mapcar #'(lambda (support) 
 (let ((cell (cv_cell support)))
 (cond ((is_relation cell)
 (list (rel_arg1 cell) (rel_to_symb (cv_value support)) (rel_arg2 cell)))
 (T cell))))
 (qsupports q1 q2 relationship)))
 
 ;;; Returns a list of pairs "(RELATION . QUANTITY)", for all quantities related to QUANT.
 ;;;   If "INCLUDE_INFERRED?" is NULL, then only asserted relations will be returned.
 ;;;   If non_NULL, then both asserted and inferred relations will be returned.
 ;;;
 ;;; Modified version of QNEIGHBOR_RELATIONS found in QLATTICE;QLATTICE.LISP
 ;;;    Bug fix:  Using a quantity as an argument in QPLUS caused bad relations
 ;;;    to be created (i.e. (QPLUS #<QUANTITY A> #<QUANTITY B>) causes #<QUANTITY A>
 ;;;    to have bad relations.
 void QNEIGHBOR_RELATIONS (quantity &optional include_inferred?)
 (let (neighboring_relations)
 (dolist (relation (quantity_relations quantity))
 (when (and (find_in_value relation)
 (or include_inferred?
 (not (derived? (Current_Value_Supporter relation)))))
 (push (cons (rel_to_symb (relationship relation quantity))
 (relation_other_arg relation quantity))
 neighboring_relations)))
 neighboring_relations))
 
 ;;; Prints out the description of a quantity in a readable way.
 void describe_quantity (quant)
 (print quant)
 (format T "~%VALUE :~15T %s" (quantity_value quant))
 (format T "~%RELATIONS :~15T %s" (quantity_relations quant))
 (format T "~%QPLIST :~15T %s" (quantity_qplist quant)))
 
 ;;;
 ;;; FOR INTERACTIVELY LOOKING AT DEPENDENCIES __ SEE THE "WHY" FUNCTION IN 'CONTEXTUAL_VALUES'
 ;;;
 void QWHY (q1 q2 &optional relationship)
 (qrelationship q1 q2)
 (let ((relation (get_relation q1 q2)))
 (why relation (when relationship
 (true_relationship (symb_to_rel relationship) relation q1)))))
 *****************************************************************************/
