/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: Formatters
 *
 * FILE: parseFmttrs.c
 *
 * ABSTRACT:
 * Parser and Printer for Formatter Data Structures
 *
 * REVISION HISTORY
 *
 * $Log: parseFmttrs.c,v $
 * Revision 1.18  1996/05/09  18:31:14  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.17  1996/03/15  21:18:11  reids
 * Added support for "enum" format type.
 *   Also, printData was not counting characters per line correctly.
 *
 * Revision 1.16  1996/03/05  05:04:35  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.15  1996/03/02  03:21:50  rich
 * Fixed memory leaks found using purify.
 *
 * Revision 1.14  1996/01/27  21:53:40  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.13  1995/07/10  16:18:06  rich
 * Interm save.
 *
 * Revision 1.12  1995/04/04  19:42:42  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.11  1995/01/18  22:41:38  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.10  1994/10/25  17:10:18  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.9  1994/05/05  00:46:28  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.8  1994/04/16  19:42:52  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1994/01/31  18:28:34  reids
 * Several major changes (and some minor ones)
 * 1. tcaFreeData and tcaFreeReply now work even if the data or message format
 *    is NULL
 * 2. Using the "-t" option in central, message taps are added as a child of
 *    the task tree node that was tapped.
 * 3. Named formatters are now expanded only when needed
 * For more details, see ../doc/tca-7-4.release.notes
 *
 * Revision 1.6  1993/12/14  17:34:25  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/11/21  20:18:46  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/27  08:38:50  fedor
 * Pass 2 aat a V7+V6+VxWorks merge. Many many problems with pointless casting.
 *
 * Revision 1.3  1993/08/27  07:15:56  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:30  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:28  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:26  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:54  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 10-Feb-89 Reid Simmons, School of Computer Science, CMU
 * Created.
 *
 * $Revision: 1.18 $
 * $Date: 1996/05/09 18:31:14 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"

static void FreeTokenList (TokenPtr TokenList)
{ 
  TokenPtr LastToken;
  
  while (TokenList) {
    LastToken = TokenList;
    TokenList = TokenList->next;
    if (LastToken->Type == STR_TOK)
      tcaFree(LastToken->value.str);
    tcaFree((char *)LastToken);
  }
}

static void ParserError(TokenPtr bad_token, char *expecting)
{ 
  int i;
  
  tcaModWarning( "Format Parsing Error: Expected %s.\n%s\n", 
		expecting, GET_SM_GLOBAL(ParseString));
  for (i=0; i<bad_token->Loc; i++) tcaModWarning( " ");
/*  tcaModError("^\n");*/
}

static FORMAT_PTR new_f_formatter(FORMAT_CLASS_TYPE type, FORMAT_PTR formatter)
{ 
  FORMAT_PTR new_formatter;
  
  new_formatter = NEW_FORMATTER();
  new_formatter->type = type;
  new_formatter->formatter.f = formatter;
  return new_formatter;
}

/* First element in a FORMAT_ARRAY_PTR is the size of the array. */
static FORMAT_PTR new_a_formatter(FORMAT_CLASS_TYPE type, int array_size)
{ 
  FORMAT_PTR new_formatter;
  
  new_formatter = NEW_FORMATTER();
  new_formatter->type = type;
  new_formatter->formatter.a = NEW_FORMAT_ARRAY(array_size);
  new_formatter->formatter.a[0].i = array_size;
  return new_formatter;
}

static FORMAT_PTR new_n_formatter(char *name)
{ 
  FORMAT_PTR new_formatter;
  
  new_formatter = NEW_FORMATTER();
  new_formatter->type = NamedFMT;
  new_formatter->formatter.name = strdup(name);
  return new_formatter;
}

/* Needed by Named_Format */
static FORMAT_ARRAY_PTR copyFormatArray (CONST_FORMAT_PTR format)
{
  int size;
  FORMAT_ARRAY_PTR copiedFormatArray;

  size = format->formatter.a[0].i;
  copiedFormatArray = NEW_FORMAT_ARRAY(size);
  copiedFormatArray[0].i = size;
  return copiedFormatArray;
}

static FORMAT_PTR copyFormatter (CONST_FORMAT_PTR format)
{
  int i;
  FORMAT_ARRAY_PTR format_array;
  FORMAT_PTR copiedFormat = NULL;

  if (format) {
    copiedFormat = NEW_FORMATTER();
    *copiedFormat = *format;
    switch (format->type) {
    case PrimitiveFMT:
    case LengthFMT: 
    case BadFormatFMT:
      break;

    case PointerFMT: 
      copiedFormat->formatter.f = copyFormatter(format->formatter.f);
      break;

    case StructFMT:
      copiedFormat->formatter.a = format_array = copyFormatArray(format);
      for (i=1; i<format_array[0].i; i++) {
	format_array[i].f = copyFormatter(format->formatter.a[i].f);
      }
      break;

    case FixedArrayFMT:
    case VarArrayFMT:
      copiedFormat->formatter.a = format_array = copyFormatArray(format);
      format_array[1].f = copyFormatter(format->formatter.a[1].f);
      for (i=2; i<format_array[0].i; i++) {
	format_array[i].i = format->formatter.a[i].i;
      }
      break;

    case NamedFMT: 
      copiedFormat->formatter.name = strdup(format->formatter.name);
      break;

    case EnumFMT:
      copiedFormat->formatter.a = format_array = copyFormatArray(format);
      format_array[1].i = format->formatter.a[1].i;
      for (i=2; i<format_array[0].i; i++) {
	format_array[i].f = copyFormatter(format->formatter.a[i].f);
      }
      break;
    }
  }
  return copiedFormat;
}

static FORMAT_PTR Struct_Format(BOOLEAN *error)
{ 
  FORMAT_PTR Form, subform;
  LIST_PTR format_list;
  int num_formats, i;
  
  format_list = listCreate();
  listInsertItem((char *)Parse(TRUE, error), format_list);
  num_formats = 1;
  
  while (GET_SM_GLOBAL(TokenList)->Type == COMMA_TOK) {
    (void)NextToken();
    listInsertItem((char *)Parse(TRUE,error), format_list);
    num_formats++;
  }
  
  Form = new_a_formatter(StructFMT, num_formats+1);
  /* Index from high to low since "format_list" 
     has formatters in reverse order */
  subform = (FORMAT_PTR)listFirst(format_list);
  for(i=num_formats;i>0;i--) {
    Form->formatter.a[i].f = subform;
    subform = (FORMAT_PTR)listNext(format_list);
  }
  
  listFree(&format_list);
  return Form;
}

static FORMAT_PTR Enum_Format(BOOLEAN *error)
{ 
  TokenPtr Token;
  FORMAT_PTR Form, subform;
  LIST_PTR format_list;
  int num_formats, i, maxVal;
  
  num_formats = 0;
  Token = NextToken();
  if (Token->Type == COLON_TOK) {
    Token = NextToken();
    if (Token->Type != INT_TOK) {
      *error = TRUE;
      ParserError(Token, "an integer");
      return NULL;
    } else {
      maxVal = Token->value.num;
    }
  } else {
    format_list = listCreate();
    do {
      if (num_formats > 0) Token = NextToken();
      if (Token->Type != STR_TOK) {
	*error = TRUE;
	ParserError(Token, "a string");
	return NULL;
      } else {
	Form = new_n_formatter(Token->value.str);
	listInsertItem((char *)Form, format_list);
	num_formats++;
      }
      Token = NextToken();
    } while (Token->Type == COMMA_TOK);
    UngetToken(Token);
    maxVal = num_formats - 1;
  }
  
  Form = new_a_formatter(EnumFMT, num_formats+2);
  Form->formatter.a[1].i = maxVal;
  if (num_formats > 0) {
    /* Index from high to low since "format_list" 
       has formatters in reverse order */
    subform = (FORMAT_PTR)listFirst(format_list);
    for(i=num_formats;i>0;i--) {
      Form->formatter.a[i+1].f = subform;
      subform = (FORMAT_PTR)listNext(format_list);
    }
    listFree(&format_list);
  }
  return Form;
}

static FORMAT_PTR Fixed_Array_Format(BOOLEAN *error)
{ 
  FORMAT_PTR Form, next_format;
  TokenPtr Token, tmp;
  int NumberOfIndexes, Continue, array_index;
  
  next_format = Parse(FALSE,error); /* Formatter */
  
  Token = NextToken();
  if (Token->Type != COLON_TOK) {
    ParserError(Token, "':'");
    return NULL;
  }
  
  tmp = GET_SM_GLOBAL(TokenList);
  NumberOfIndexes = 0;
  Continue = TRUE;
  do {
    if (tmp->Type != INT_TOK) {
      ParserError(tmp, "an integer value");
      return NULL;
    }
    else {
      NumberOfIndexes++;
      tmp = tmp->next;
      if (tmp->Type == COMMA_TOK) {
	tmp = tmp->next;
	Continue = TRUE;
      } else if (tmp->Type == RBRACK_TOK) {
	Continue = FALSE;
      } else {
	ParserError(tmp, "a ',' or ']'");
	return NULL;
      }
    }
  } while (Continue);
  
  Form = new_a_formatter(FixedArrayFMT, NumberOfIndexes+2);
  Form->formatter.a[1].f = next_format;
  
  /* this time munch tokens */
  
  NumberOfIndexes += 2;
  for (array_index=2; array_index < NumberOfIndexes; array_index++) {
    Token = NextToken(); /* This is an INT_TOK */
    Form->formatter.a[array_index].i = Token->value.num;
    
    Token = NextToken(); /* This is a COMMA_TOK or a RBRACK_TOK */
  }
  
  return Form;
}

static FORMAT_PTR Var_Array_Format(BOOLEAN *error)
{ 
  FORMAT_PTR Form, next_format;
  TokenPtr Token, tmp;
  int NumberOfIndexes, Continue, array_index;
  
  next_format = Parse(FALSE,error); /* Formatter */
  
  Token = NextToken();
  if (Token->Type != COLON_TOK) {
    ParserError(Token, "':'");
    return NULL;
  }
  
  tmp = GET_SM_GLOBAL(TokenList);
  NumberOfIndexes = 0;
  Continue = TRUE;
  do {
    if (tmp->Type != INT_TOK) {
      ParserError(tmp, "an integer value");
      return NULL;
    } else {
      NumberOfIndexes++;
      tmp = tmp->next;
      if (tmp->Type == COMMA_TOK) {
	tmp = tmp->next;
	Continue = TRUE;
      } else if (tmp->Type == GT_TOK) {
	Continue = FALSE;
      } else {
	ParserError(tmp, "a ',' or '>'");
	return NULL;
      }
    }
  } while (Continue);
  
  Form = new_a_formatter(VarArrayFMT, NumberOfIndexes+2);
  Form->formatter.a[1].f = next_format;
  
  /* this time munch tokens */
  
  NumberOfIndexes += 2;
  for (array_index=2; array_index < NumberOfIndexes; array_index++) {
    Token = NextToken(); /* This is an INT_TOK */
    Form->formatter.a[array_index].i = Token->value.num;
    
    Token = NextToken(); /* This is a COMMA_TOK or a GT_TOK */
  }
  
  return Form;
}

static FORMAT_PTR SelfPtr_Format(void)
{ 
  return new_f_formatter(PointerFMT, (FORMAT_PTR)NULL);
}

static FORMAT_PTR Ptr_Format(BOOLEAN *error)
{ 
  return new_f_formatter(PointerFMT, Parse(FALSE,error));
}

static FORMAT_PTR Length_Format(TokenPtr Token)
{ 
  return createIntegerFormat(LengthFMT, Token->value.num);
}

static FORMAT_PTR Named_Format(TokenPtr Token)
{ 
  static STR_LIST_PTR formatStack = NULL;
  NAMED_FORMAT_PTR namedFormat;
  
  if (formatStack == NULL) {
    formatStack = strListCreate();
  }
  
  if (strListMemberItem(Token->value.str, formatStack)) {
    /* found a recursive definition. just return pointer */
    return new_n_formatter(Token->value.str);
  }
  
  strListPush(strdup(Token->value.str),formatStack);

  namedFormat = (NAMED_FORMAT_PTR)
    hashTableFind(Token->value.str,
		  GET_M_GLOBAL(formatNamesTable));
  if (!namedFormat) {
    tcaModWarning("Warning: Named Format %s is not registered\n%s\n",
		Token->value.str, GET_SM_GLOBAL(ParseString));
    return new_f_formatter(BadFormatFMT, NULL);
  }
  /* Need to use the named formatter -- parse it now */
  if (!namedFormat->parsed) {
    namedFormat->format = 
      (FORMAT_PTR)ParseFormatString(namedFormat->definition);
    namedFormat->parsed = TRUE;
  }
  
  tcaFree((void *)strListPopItem(formatStack));
  return copyFormatter(namedFormat->format);
}

static FORMAT_PTR Primitive_Format(TokenPtr Token)
{ 
  return Named_Format(Token);
}

/*********************************************/

/* RGS 11/12/93: Made format string parser re-entrant: 
 *               Needed for lazy evaluation of named formatters
 */
CONST_FORMAT_PTR ParseFormatString(const char *FormString)
{ 
  CONST_FORMAT_PTR Formatter;
  const char *savedParseString;
  TokenPtr savedTokenList, savedTokenListHead;
  BOOLEAN error = FALSE;
  
  savedParseString = GET_SM_GLOBAL(ParseString);
  savedTokenList = GET_SM_GLOBAL(TokenList);
  savedTokenListHead = GET_SM_GLOBAL(TokenListHead);
  
  GET_SM_GLOBAL(ParseString) = FormString;
  LexString(FormString);
  GET_SM_GLOBAL(TokenListHead) = GET_SM_GLOBAL(TokenList);
  Formatter = Parse(FALSE, &error);
  if (error) Formatter = new_f_formatter(BadFormatFMT, NULL);
  FreeTokenList(GET_SM_GLOBAL(TokenListHead));
  
  GET_SM_GLOBAL(ParseString) = savedParseString;
  GET_SM_GLOBAL(TokenList) = savedTokenList;
  GET_SM_GLOBAL(TokenListHead) = savedTokenListHead;
  
  return Formatter;
}


/* "within_struct_flag" is TRUE if the enclosing format is "{...}" */
FORMAT_PTR Parse(int32 within_struct_flag, BOOLEAN *error)
{ 
  TokenPtr Token;
  FORMAT_PTR ReturnForm=NULL;
  
  Token = NextToken();
  
  switch(Token->Type) {
  case LBRACE_TOK:
    Token = NextToken();
    if (Token->Type == STR_TOK && !strcmp(Token->value.str, "enum")) {
      ReturnForm = Enum_Format(error);
    } else {
      UngetToken(Token);
      ReturnForm = Struct_Format(error);
    }
    Token = NextToken();
    if (Token->Type != RBRACE_TOK) {
      ParserError(Token, "'}'");
      *error = TRUE;
      return NULL;
    }
    break;
  case LBRACK_TOK:
    ReturnForm = Fixed_Array_Format(error);
    break;
  case LT_TOK:
    if (!within_struct_flag) {
      ParserError(Token,
		  "var array format '<..>' not embedded within a structure '{..}'");
      *error = TRUE;
      return NULL;
    } else {
      ReturnForm = Var_Array_Format(error);
    }
    break;
  case STAR_TOK:
    if (GET_SM_GLOBAL(TokenList)->Type == BANG_TOK) {
      Token = NextToken();	/* eat BANG_TOK */ 
      if (!within_struct_flag) {
	ParserError(Token, 
		    "self pointer '*!' embedded within a structure '{..}'");
	*error = TRUE;
	return NULL;
      } else {
	ReturnForm = SelfPtr_Format();
      }
    }
    else
      ReturnForm = Ptr_Format(error);
    break;
  case INT_TOK:
    ReturnForm = Length_Format(Token);
    break;
  case STR_TOK:
    ReturnForm = Named_Format(Token);
    if (ReturnForm->type == BadFormatFMT) {
      *error = TRUE;
      return NULL;
    }
    break;
  case PRIMITIVE_TOK:
    ReturnForm = Primitive_Format(Token);
    if (ReturnForm->type == BadFormatFMT) {
      *error = TRUE;
      return NULL;
    }
    break;
  case EOS_TOK: 
    ParserError(Token, 
		"additional tokens; premature end of string encountered");
    *error = TRUE;
    return NULL;
    break;

  default:
    ParserError(Token, "a different token type");
    *error = TRUE;
    return NULL;
  }
  
  return ReturnForm;
}
