/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: tca
 *
 * FILE: msgTapMon.c
 *
 * ABSTRACT: Provide the user-interface functions for the "wire-tap" facility.
 *
 * Messages can be tapped when certain conditions hold.  
 * The tapping "condition" is one of the following enumerated constants
 * (defined in the file tca.h):
 *
 * * WhenSent: Send as soon as TCA receives the tapped message.
 *
 * * BeforeHandling: Send when the tapped message is ready to be sent
 *   (i.e., its
 *   temporal constraints are met and its resource is available).  If both the
 *   listening and tapped messages are either goal, command, or monitor 
 *   messages, the listening message temporally precedes the tapped message 
 *   (i.e. the tapped message is delayed until the listening message is
 *   handled)
 *
 * * WhileHandling: Send when the tapped message is sent. The listening message
 *   may be handled concurrently with the tapped message.  No guarantee is made
 *   that the listening message will complete before the tapped message does.
 *
 * * AfterHandled: Send after the tapped message has been handled.  The 
 *   listening message may be handled concurrently with succeeding messages.
 *
 * * AfterReplied: Send after the reply to the tapped message has been sent
 *   out.  The tapped message must a query-class message.  This condition
 *   differs from "AfterHandled" in that the listening message is sent the
 *   reply data of the tapped message; with the "AfterHandled" condition, it
 *   would be sent the original query data.
 *
 * * AfterSuccess: Send after the tapped message has indicated "Success".  The
 *   tapped message must be a goal, command, or monitor message.  This is 
 *   similar to the "AfterHandled" condition.
 *
 * * AfterFailure: Send after the tapped message has indicated "Failure".  The
 *   tapped message must be a goal, command, or monitor message.  This is 
 *   similar to the "AfterHandled" condition.  Note, that this condition is
 *   just included for completeness: the exception handling mechanisms should
 *   be used to cover this case.
 *
 * * BeforeAchieving: Send when the tapped message is ready to be achieved
 *   (according to the temporal constraints).  The tapped message must be a 
 *   goal, command, or monitor.  If the listening message is also a goal, 
 *   command, or monitor, then the listening message will be achieved before 
 *   the tapped message starts to be achieved.
 *
 * * WhileAchieving: Send when the tapped message has started to be achieved.
 *   The listening message may be achieved concurrently with the achievement of
 *   the tapped message.
 *
 * * AfterAchieved: Send after the tapped message (and all its submessages) has
 *   been achieved.  The listening message does not necessarily have precedence
 *   over succeeding messages -- for that, use the "WhenAchieved" condition.
 *
 * * WhenAchieved: Send the listening message after the tapped message has been
 *   achieved.  If the listening message is a goal, command, or monitor, its 
 *   achievement is considered part of the achievement of the tapped message
 *   (i.e., the end of the listening message's achievement interval precedes
 *   the end of the tapped message's achievement interval).  This means that
 *   succeeding messages whose temporal constraints are related to the
 *   achievement of the tapped message will be delayed until the listening
 *   message is achieved.
 *
 * * BeforePlanning: Similar to "BeforeAchieving", except based on the planning
 *   interval of the tapped message.  The tapped message must be a goal
 *   message.
 *
 * * WhilePlanning: Similar to "WhileAchieving", except based on the planning
 *   interval of the tapped message.  The tapped message must be a goal
 *    message.
 *
 * * AfterPlanned: Similar to "AfterAchieved", except based on the planning 
 *   interval of the tapped message.  The tapped message must be a goal
 *   message.
 *
 * * WhenPlanned: Similar to "WhenAchieved", except based on the planning
 *   interval of the tapped message.  The tapped message must be a goal
 *   message.
 *
 * * WhenKilled: Send when the tapped message is killed.  Note that if the
 *   tapped message is being handled when it is killed, then TCA allows it to 
 *   continue to completion.  In such cases, the listening message is sent when
 *   the tapped message has actually completed.
 *
 * REVISION HISTORY
 *
 * $Log: msgTapMod.c,v $
 * Revision 1.11  1996/08/22  16:35:57  rich
 * Check the return code on tcaQueryCentral calls.
 *
 * Revision 1.10  1995/10/07  19:07:38  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.9  1995/04/08  02:06:27  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.8  1995/01/18  22:41:36  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.7  1994/04/28  16:16:35  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.6  1994/04/16  19:42:50  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.5  1993/11/21  20:18:44  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/30  21:54:01  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:15:54  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:26  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:28  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:22  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:52  fedor
 * Added Logging.
 *
 * 12-Mar-91 Reid Simmons, School of Computer Science, CMU
 * Created.
 *
 * $Revision: 1.11 $
 * $Date: 1996/08/22 16:35:57 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include "msgTap.h"


/******************************************************************************
 *
 * FUNCTION: tcaTapMessage(condition, tappedMsg, listeningMsg)
 *
 * DESCRIPTION: Send the listening message when the condition holds on the
 *              tapped message.
 *
 * INPUTS: TAP_CONDITION_TYPE condition; 
 *         char *tappedMsg, 
 *         char *listeningMsg;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * ABSTRACT: The listening message is sent with the same data as the tapped
 *  message, whenever the indicated condition of the tapped message holds.  
 *  The listening message cannot be a query-class message -- it must be the 
 *  name of a constraint, goal, command, or monitor message.  The constraints 
 *  on the class of the tapped message depends on the particular condition (see
 *  the previous page).  Note that tcaTapMessage applies to all instances of
 *  the tapped message.  To tap a particular instance of a message, use
 *  tcaTapReference.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaTapMessage(TAP_CONDITION_TYPE condition,
				    const char *tappedMsg, 
				    const char *listeningMsg)
{
  TAP_MSG_TYPE tapData;
  
  tapData.condition = condition;
  tapData.tappedMsg = tappedMsg;
  tapData.listeningMsg = listeningMsg;
  
  return tcaInform(TCA_TAP_MSG_INFORM, (void *)&tapData);
}


/****************************************************************
 *
 * FUNCTION: tcaTapReference(condition, tappedRef, listeningMsg)
 *
 * DESCRIPTION: Send the listening message when the condition holds on the
 *              message associated with the tapped reference pointer
 *
 * INPUTS: TAP_CONDITION_TYPE condition; 
 *         TCA_REF_PTR tappedRef; 
 *         char *listeningMsg;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * The listening message is sent with the same data as that of the tappedRef,
 * when the indicated condition of the tapped reference holds.  Note that,
 * unlike tcaTapMessage, tcaTapReference applies to just a single message.  The
 * tappedRef must refer to a goal, command, or monitor message.  The
 * listening message must be the name of a constraint, goal, command, or
 * monitor message.  The effects are unpredictable if the condition has
 * already occurred when the tap is initiated (e.g., if a condition of
 * "BeforeHandling" is given when the tappedRef has already been handled).
 *
 ****************************************************************/

TCA_RETURN_VALUE_TYPE tcaTapReference(TAP_CONDITION_TYPE condition,
				      TCA_REF_PTR tappedRef, 
				      const char *listeningMsg)
{
  TAP_REF_TYPE tapData;
  
  tapData.condition = condition;
  tapData.tappedRef = tappedRef;
  tapData.listeningMsg = listeningMsg;
  
  return tcaInform(TCA_TAP_REF_INFORM, (void *)&tapData);
}


/****************************************************************
 *
 * FUNCTION: tcaFindTappedReference(listeningRef)
 *
 * DESCRIPTION: Return a reference to the message that was tapped by 
 *              the listening reference.
 * 
 * INPUTS: TCA_REF_PTR listeningRef;
 *
 * OUTPUTS: TCA_REF_PTR (reference to the tapped message, or NULL)
 *
 ****************************************************************/

TCA_REF_PTR tcaFindTappedReference(TCA_REF_PTR listeningRef)
{
  TCA_REF_PTR tappedRef=NULL;
  
  if (tcaQueryCentral(TCA_FIND_TAP_INFORM, (void *)&(listeningRef->refId), 
		      (void *)&tappedRef) == NullReply) {
    return NULL;
  } else {
    return tappedRef;
  }
}


/****************************************************************
 *
 * FUNCTION: tcaRemoveTap(condition, tappedMsg, listeningMsg)
 * 
 * DESCRIPTION: Remove the listeningMsg/condition pair from the set of taps
 *              on the tapped message.  Undoes the effect of tcaTapMessage.
 *
 * INPUTS: TAP_CONDITION_TYPE condition; 
 *         char *tappedMsg;
 *         char *listeningMsg;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaRemoveTap(TAP_CONDITION_TYPE condition,
				   const char *tappedMsg, 
				   const char *listeningMsg)
{
  TAP_MSG_TYPE tapData;
  
  tapData.condition = condition;
  tapData.tappedMsg = tappedMsg;
  tapData.listeningMsg = listeningMsg;
  
  return tcaInform(TCA_REMOVE_TAP_INFORM, (void *)&tapData);
}


/****************************************************************
 *
 * FUNCTION: tcaCleanUpAfterAchieved(msgName)
 * 
 * DESCRIPTION: After each instance of the message msgName has been achieved,
 *              kill the task tree rooted at the message node.
 *              msgName must be a goal, command, or monitor class.
 *
 * INPUTS: char *msgName;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES: Currently does not check if the message is of the correct class.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaCleanUpAfterAchieved(const char *msgName)
{
  return tcaTapMessage(AfterAchieved, msgName, TCA_CLEANUP_AFTER_ACHIEVED_INFORM);
}
