/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: monitors
 *
 * FILE: monMod.c
 *
 * ABSTRACT:
 * 
 * module monitor code.
 *
 * REVISION HISTORY
 *
 * $Log: monMod.c,v $
 * Revision 1.20  1996/06/25  20:51:01  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.19  1996/02/10  16:50:14  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.18  1996/02/06  19:05:00  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.17  1995/11/03  03:04:38  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.16  1995/10/25  22:48:36  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.15  1995/07/10  16:17:59  rich
 * Interm save.
 *
 * Revision 1.14  1995/05/31  19:35:58  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.13  1995/04/04  19:42:34  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.12  1995/03/28  01:14:50  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.11  1995/03/19  19:39:36  rich
 * Implemented direct connections using tcaDirectResouce call.
 * Also made the basics.h file a module include.
 * Changed class in the interval structure to be interval_class to avoid a
 * conflict with C++.
 *
 * Revision 1.10  1995/01/18  22:41:24  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.9  1994/10/25  17:10:06  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.8  1994/05/17  23:16:44  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.7  1994/05/05  00:46:26  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.6  1994/04/16  19:42:44  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.5  1993/12/14  17:34:18  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.4  1993/11/21  20:18:35  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:15:40  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:14  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:27  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:09  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:40  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 13-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Added extra checks for user supplied monitor options.
 *
 *  3-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * created from monitor module routines that were in behviors.c
 *
 * $Revision: 1.20 $
 * $Date: 1996/06/25 20:51:01 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"

#include "monMod.h" 


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaPointMonitorWithConstraints(ref, name, 
 *                                                                data, tpl)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * char *name;
 * void *data; 
 * int tpl;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaPointMonitorWithConstraints(TCA_REF_PTR ref, 
						     const char *name,
						     const void *data,
						     int32 tpl)
{ 
  MSG_PTR msg;
  
  if (testInconsistent(tpl)) {
    tcaModError("ERROR: Inconsistent temporal constraints for monitor %s\n", name);
    return Failure;
  }
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, PointMonitorClass);
  
  (void)sendMessage(ref, msg, (char *)data, (char *)&tpl);
  
  return Success;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaPointMonitor(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaPointMonitor(const char *name, const void *data)
{ 
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, PointMonitorClass);
  
  return tcaPointMonitorWithConstraints((TCA_REF_PTR)NULL, name, data, 
					SEQ_ACH);
}

/*************************************************************/

void tcaRegisterDemonMonitor(const char *monitorName, 
			     const char *setUpName, 
			     const char *actName,
			     const char *cancelName)
{
  MON_REG_DATA_TYPE monRegData;
  
  monRegData.msg_class = DemonMonitorClass;
  
  monRegData.monitorName = monitorName;
  monRegData.conditionName = cancelName;
  monRegData.actionName = actName;
  monRegData.setUpName = setUpName;
  
  (void)tcaInform(TCA_REG_MONITOR_INFORM, (void *)&monRegData);
}


/*************************************************************/

void tcaRegisterPointMonitor(const char *monitorName, 
			     const char *condName, 
			     const char *actName)
{
  MON_REG_DATA_TYPE monRegData;
  
  monRegData.msg_class = PointMonitorClass;
  
  monRegData.monitorName = monitorName;
  monRegData.conditionName = condName;
  monRegData.actionName = actName;
  monRegData.setUpName = NULL;
  
  (void)tcaInform(TCA_REG_MONITOR_INFORM, (void *)&monRegData);
}


/*****************************************************************************
 *
 * FUNCTION: void tcaRegisterPollingMonitor(monitorName, condName, actName)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * char *monitorName, *condName, *actName;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaRegisterPollingMonitor(const char *monitorName, 
			       const char *condName, 
			       const char *actName)
{
  MON_REG_DATA_TYPE monRegData;
  
  monRegData.msg_class = PollingMonitorClass;
  
  monRegData.monitorName = monitorName;
  monRegData.conditionName = condName;
  monRegData.actionName = actName;
  monRegData.setUpName = NULL; 
  
  (void)tcaInform(TCA_REG_MONITOR_INFORM, (void *)&monRegData);
}


/*****************************************************************************
 *
 * FUNCTION: void initializeMonitorOptions(options)
 *
 * DESCRIPTION:
 * Sets default values for interval monitors.
 * Options and defaults are: 
 * maxFire:     maximum times to fire action message; -1 indicates infinite
 *              (default is -1)
 * fireEvery:   fire action message every n times condition is satisfied.
 *              (default is 1)
 * duration:    maximum time, in seconds, that the monitor can run;
 *	       -1 indicates infinite time (default is -1)
 * period:      time between polls, in seconds (default is 10)
 * initialWait: if TRUE, don't poll immediately (default is FALSE)
 *
 * INPUTS: INTERVAL_MONITOR_OPTIONS_PTR options;
 *
 * OUTPUTS: void
 *
 * NOTES:
 * as a side effect the fields of options is initialized.
 *
 *****************************************************************************/

static void initializeMonitorOptions(INTERVAL_MONITOR_OPTIONS_PTR options)
{
  options->maxFire = -1;
  options->fireEvery = 1;
  options->duration = -1;
  options->period = 10;
  options->initialWait = FALSE;
}


/*****************************************************************************
 *
 * FUNCTION: INTERVAL_MONITOR_OPTIONS_PTR tcaCreateMonitorOptions()
 *
 * DESCRIPTION: Creates and initializes a monitor option.
 *
 * INPUTS: none.
 *
 * OUTPUTS: INTERVAL_MONITOR_OPTIONS_PTR
 *
 *****************************************************************************/

INTERVAL_MONITOR_OPTIONS_PTR tcaCreateMonitorOptions(void)
{
  INTERVAL_MONITOR_OPTIONS_PTR options;
  
  options = NEW(INTERVAL_MONITOR_OPTIONS_TYPE);
  
  initializeMonitorOptions(options);
  
  return options;
}


/*****************************************************************************
 * 
 * FUNCTION void checkForDeadLock(monitorName, expirationTime, options)
 *
 * DESCRIPTION: Display warning message about possible deadlock.
 *
 * INPUTS:
 * char *monitorName;
 * TCA_TIME_POINT_PTR expirationTime;
 * INTERVAL_MONITOR_OPTIONS_PTR options;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void checkForDeadlock(const char *monitorName,
			     TCA_TIME_POINT_PTR expirationTime,
			     INTERVAL_MONITOR_OPTIONS_PTR options)
{
  if (expirationTime == &(GET_M_GLOBAL(tcaDefaultTimeGlobal))) {
    tcaModWarning( "\nWARNING: interval monitor %s %s\n", monitorName,
		  (((options->maxFire <= 0) && (options->duration < 0)) ? 
		   "will last forever unless explicitly killed." :
		   "may last forever."));
  }
}


/*****************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaIntervalMonitorWithConstraints(ref, name,
 *                                                data, start, expire, options)
 *
 * DESCRIPTION:
 * IF ref == NULL, use the issuer as the monitor's parent.
 * IF start == DEFAULT_TIME, the start time of the monitor is the 
 *    start time of its parent.
 * IF expire == DEFAULT_TIME, the monitor could last forever unless
 *    appropriate options are specified.
 * IF options == NULL, see initializeMonitorOptions() for default values.
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * char *name;
 * void *data;
 * TCA_TIME_POINT_TYPE start, expire;
 * INTERVAL_MONITOR_OPTIONS_PTR options;
 *
 * OUTPUTS: TCA_RETURN_VALUE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaIntervalMonitorWithConstraints
(TCA_REF_PTR ref, const char *name, const void *data,
 TCA_TIME_POINT_TYPE start, TCA_TIME_POINT_TYPE expire,
 INTERVAL_MONITOR_OPTIONS_PTR options)
{
  MSG_PTR msg;
  INTERVAL_MON_CLASS_TYPE intMonOptions;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  
  if ((msg->msgData->msg_class != PollingMonitorClass) &&
      (msg->msgData->msg_class != DemonMonitorClass)) {
    tcaModError("ERROR: Monitor %s must be a polling or demon-invoked monitor\n",
		name);
    return WrongMsgClass;
  }
  
  if (options) {
    intMonOptions.options = options;
    if (options->maxFire < -1) {
      tcaModWarning(
		    "\nWARNING: Max Fire of monitor %s set to default of -1\n",
		    name);
      intMonOptions.options->maxFire = -1;
    }
    if (options->fireEvery < 1) {
      tcaModWarning(
		    "\nWARNING: Fire Every of monitor %s set to default of 1\n",
		    name);
      intMonOptions.options->fireEvery = 1;
    }
    if (options->duration < -1) {
      tcaModWarning(
		    "\nWARNING: Duration of monitor %s set to default of -1.\n",
		    name);
      intMonOptions.options->duration = -1;
    }
    if (options->period < 1) {
      tcaModWarning( 
		    "\nWARNING: Polling period of monitor %s rounded up to 1 second\n",
		    name);
      intMonOptions.options->period = 1;
    }
    if (options->initialWait != FALSE && options->initialWait != TRUE) {
      tcaModWarning( "\nWARNING: Initial wait of monitor %s set to FALSE\n",
		    name);
      intMonOptions.options->initialWait = FALSE;
    }
  }
  else
    intMonOptions.options = tcaCreateMonitorOptions();
  
  checkForDeadlock(name, &expire, intMonOptions.options);
  
  intMonOptions.start = start;
  intMonOptions.expire = expire;
  
  return sendMessage(ref, msg, (char *)data, (char *)&intMonOptions);
}

#if defined(LISP)

TCA_RETURN_VALUE_TYPE tcaIntMonWithConstraintsLISPC
(TCA_REF_PTR ref, char *name, void *data,
 TCA_POINT_CLASS_TYPE startClass, TCA_INTERVAL_CLASS_TYPE startINTclass,
 int startINTmsgRef,
 TCA_POINT_CLASS_TYPE expireClass, TCA_INTERVAL_CLASS_TYPE expireINTclass,
 int expireINTmsgRef, 
 int maxFire, int fireEvery, int duration, int period, int initialWait);

TCA_RETURN_VALUE_TYPE tcaIntMonWithConstraintsLISPC
(TCA_REF_PTR ref, char *name, void *data,
 TCA_POINT_CLASS_TYPE startClass, TCA_INTERVAL_CLASS_TYPE startINTclass,
 int startINTmsgRef,
 TCA_POINT_CLASS_TYPE expireClass, TCA_INTERVAL_CLASS_TYPE expireINTclass,
 int expireINTmsgRef, 
 int maxFire, int fireEvery, int duration, int period, int initialWait)
{
  TCA_TIME_POINT_TYPE start, expire;
  TCA_INTERVAL_TYPE startInt, expireInt;
  
  INTERVAL_MONITOR_OPTIONS_TYPE options;
  
  options.maxFire = maxFire;
  options.fireEvery = fireEvery;
  options.duration = duration;
  options.period = period;
  options.initialWait = initialWait;
  
  startInt.interval_class = startINTclass;
  startInt.msgRef = startINTmsgRef;
  
  expireInt.interval_class = expireINTclass;
  expireInt.msgRef = expireINTmsgRef;
  
  start.point_class = startClass;
  start.interval = startInt;
  
  expire.point_class = expireClass;
  expire.interval = expireInt;
  
  return(tcaIntervalMonitorWithConstraints(ref, name, data, 
					   start, expire, &options));
}

#endif /* LISP */


/*****************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaIntervalMonitor(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: char *name;
 *         void *data; 
 *
 * OUTPUTD: TCA_RETURN_VALUE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaIntervalMonitor(const char *name, const void *data)
{
  return tcaIntervalMonitorWithConstraints((TCA_REF_PTR)NULL, name, data,
					   GET_M_GLOBAL(tcaDefaultTimeGlobal), 
					   GET_M_GLOBAL(tcaDefaultTimeGlobal), 
					   (INTERVAL_MONITOR_OPTIONS_PTR)NULL);
}


/*****************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaFireDemon(name, idValue, data)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * char *name;
 * int idValue;
 * void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaFireDemon(const char *name, int32 idValue,
				   const void *data)
{ 
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  
  return sendResponse((TCA_REF_PTR)NULL, msg, (char *)data, FireDemonClass, 
		      (char *)&idValue,
		      GET_C_GLOBAL(tcaServerWriteGlobal));
}
