/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: monitors
 *
 * FILE: mon.c
 *
 * ABSTRACT:
 *
 * Monitors
 *
 * REVISION HISTORY:
 *
 * $Log: mon.c,v $
 * Revision 1.31  1996/07/19  18:14:16  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.30  1996/06/30  20:17:47  reids
 * Handling of polling monitors was severely broken.
 *
 * Revision 1.29  1996/06/25  20:50:59  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.28  1996/05/09  18:31:12  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.27  1996/03/15  21:22:59  reids
 * Fixed bug relating to re-registering messages -- now does not crash central.
 *
 * Revision 1.26  1996/02/10  16:50:12  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.25  1996/01/05  16:31:34  rich
 * Added windows NT port.
 *
 * Revision 1.24  1995/11/03  03:04:36  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.23  1995/10/29  18:26:52  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.22  1995/10/10  00:42:57  rich
 * Added more system messages to ignore.
 *
 * Revision 1.21  1995/10/07  19:07:30  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.20  1995/08/06  16:44:02  reids
 * A bug existed in that two demon monitors that sent the same ID number
 * would conflict (causing the wrong one to fire).  This has been fixed, and
 * in the process, one of the hash-key functions was made a bit more general.
 *
 * Revision 1.19  1995/07/12  04:54:55  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.18  1995/07/10  16:17:48  rich
 * Interm save.
 *
 * Revision 1.17  1995/06/14  03:21:39  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.16  1995/05/31  19:35:56  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.15  1995/04/19  14:28:22  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.14  1995/03/16  18:05:33  rich
 * Merged in changes to the 7.9 branch.
 * Changed the VERSION_ to TCA_VERSION_
 *
 * Revision 1.13  1995/01/18  22:41:18  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.12  1994/10/25  17:10:03  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.11  1994/05/20  23:36:12  rich
 * Fixed release date.  Removed centralSuccess from the delay handler.
 * Added DEPEND_PREFIX for creating dependencies for object files in subdirs.
 *
 * Revision 1.10  1994/05/17  23:16:37  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.9  1994/04/28  16:16:24  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.8  1994/04/16  19:42:38  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1993/12/14  17:34:14  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/11/21  20:18:33  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:14:01  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/30  21:53:48  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:15:34  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:09  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:51  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:31:04  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:33  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 17-Aug-91 Christopher Fedor, School of Computer Science, CMU
 * Updated actMsg for command class actions to issue the correct class data.
 * Changed global tplConstr value to simply malloc storage for class data.
 *
 * 15-Aug-91 Christopher Fedor, School of Computer Science, CMU
 * Moved tplConstr from in reply monitor procedures to be a global
 * so when it is used as class data it will not vanish with the stack.
 *
 *  9-Jul-91 Christopher Fedor, School of Computer Science, CMU
 * Changed format tests to test with format string because the message
 * may not have been parsed yet when the test is done.
 *
 * 19-Mar-91 Christopher Fedor, School of Computer Science, CMU
 * Goofed on implementation of fireEvery. Reimplemented.
 * fireEvery: fire action message every n times condition is satisfied.
 *
 * 16-Nov-90 Christopher Fedor, School of Computer Science, CMU
 * Added cancelIfAttendingIntervalMonitor call for taskTree.c
 * WARNING: Need to check interaction between stopping an interval
 * monitor that is also a durable item.
 *
 * 24-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Removed unnecessary queue include files.
 *
 * 13-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Reimplemented fireEvery - to fire the action message multiple times.
 *
 * 12-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Reimplemented polling monitors.
 *
 *  9-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Reimplemented durable items.
 *
 *  8-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Reimplemented demon monitor.
 *
 *  6-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * Replaced module monitor code with monitor behavior routines now
 * in behaviors.c Reunited this split module to a single central
 * monitor module.
 *
 * 12-Apr-90 Long-Ji Lin at School of Computer Science, CMU
 * Added code to check for potential deadlocks caused by inappropriate 
 * expiration time specified for interval monitors.
 *
 * 30-Mar-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to software standards.
 *
 * 26-Mar-90 Long-Ji Lin, School of Computer Science, CMU
 * Modified code to allow options of interval monitors to be specified at run
 * time rather than at registration time.
 * Changed the way interval monitors are handled, so we can use interval
 * monitors just like goals to specify temporal constraints of other
 * messages.
 * Reimplemented tcaIntervalMonitor() and added
 * tcaIntervalMonitorWithConstraints();
 *
 * 12-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Changed way monitors are registered, so the name of the monitor is used, 
 * instead of the generic name (e.g. "PointMonitor").
 *
 *  9-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Fix to kill point monitors correctly; no longer necessary to register
 * messages before registering the monitor that uses them.
 *
 * 20-Oct-89 Reid Simmons, School of Computer Science, CMU
 * Added a "duration" loop to time the extent of dispatches.  
 * Used to implement a "tcaDelay" command message, and adding a "duration"
 * option to interval monitors.
 *
 * 26-Sep-89 Reid Simmons, School of Computer Science, CMU
 * Streamlined calling of monitors.
 * Added options to interval monitors.
 *
 *  2-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Changed "printf"s to use logging facility.
 *
 * 12-May-89 Reid Simmons, School of Computer Science, CMU
 * Split off module monitor code from monitors.c
 *
 *  1-May-89 Reid Simmons, School of Computer Science, CMU
 *
 * $Revision: 1.31 $
 * $Date: 1996/07/19 18:14:16 $
 * $Author: reids $
 * created.
 *
 *****************************************************************************/



#include "globalS.h"

/* Defined below; forward reference */
static void terminatePollingMonitor(char *ruleName, TMS_NODE_PTR relNode,
				    INTERVAL_MONITOR_DATA_PTR intMon);
static void terminateDemonMonitor(char *ruleName, TMS_NODE_PTR relNode,
				  INTERVAL_MONITOR_DATA_PTR intMon);


/******************************************************************************
 *
 * FUNCTION: DURABLE_ITEM_PTR addDurableItem(duration, intMon, dispatch)
 *
 * DESCRIPTION: Creates a durableItem adds it to the list and returns it.
 *
 * INPUTS: 
 * int duration;
 * INTERVAL_MON intMon;
 * DISPATCH_PTR dispatch;
 *
 * OUTPUTS: DURABLE_ITEM_PTR
 *
 *****************************************************************************/

static DURABLE_ITEM_PTR addDurableItem(int duration,
				       INTERVAL_MONITOR_DATA_PTR intMon,
				       DISPATCH_PTR dispatch)
{
  DURABLE_ITEM_PTR durableItem;
  
  durableItem = NEW(DURABLE_ITEM_TYPE);
  
  durableItem->endTime = timeInMsecs() + 1000*duration;
  durableItem->dispatch = dispatch;
  durableItem->intMon = intMon;
  
  listInsertItem((char *)durableItem, GET_S_GLOBAL(durableItemList));
  
  return durableItem;
}


/******************************************************************************
 *
 * FUNCTION: void spliceOutDurableItem(durableItem)
 *
 * DESCRIPTION: Remove durableItem from durableItemList and reclaim memory.
 *
 * INPUTS: DURABLE_ITEM_PTR durableItem;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void spliceOutDurableItem(DURABLE_ITEM_PTR durableItem)
{
  listDeleteItem((char *)durableItem, GET_S_GLOBAL(durableItemList));
  tcaFree((char *)durableItem);
}


/******************************************************************************
 *
 *  POLLING MONITOR
 *
 *****************************************************************************/

/******************************************************************************
 *
 * FUNCTION: void spliceOutPollingMonitor(intMon)
 *
 * DESCRIPTION: Remove intMon from the list of polling monitors and free it.
 *
 * INPUTS: INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void spliceOutPollingMonitor(INTERVAL_MONITOR_DATA_PTR intMon)
{
  listDeleteItem((char *)intMon, GET_S_GLOBAL(pollingMonitorList));
  tcaFree((char *)intMon);
}


/******************************************************************************
 *
 * FUNCTION: void stopPollingMonitor(intMon)
 *
 * DESCRIPTION: 
 * Remove the instance of the polling monitor intMon and mark the monitor
 * as handled.
 *
 * INPUTS: INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void stopPollingMonitor(INTERVAL_MONITOR_DATA_PTR intMon)
{ 
  DISPATCH_PTR dispatchPtr = intMon->dispatch;

  if (intMon->expirationTime)
    removeOutRule((NODE_RULE_FN) terminatePollingMonitor, 
		  get_relation(GET_S_GLOBAL(Now), intMon->expirationTime));
  
  if (intMon->fireTime == ALREADY_FIRED) {
    /* keep poller around until the current condition message returns */
    intMon->monInfo = NULL;
  }
  else
    spliceOutPollingMonitor(intMon);
  
  dispatchUpdateAndDisplay(HandledDispatch, dispatchPtr);
  CompletionConstraints(dispatchPtr);
}


/******************************************************************************
 *
 * FUNCTION: void terminatePollingMonitor(ruleName, relNode, intMon)
 *
 * DESCRIPTION: Rule function to remove a polling monitor.
 *
 * INPUTS: 
 * char *ruleName; 
 * TMS_NODE_PTR relNode; 
 * INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void terminatePollingMonitor(char *ruleName, TMS_NODE_PTR relNode,
				    INTERVAL_MONITOR_DATA_PTR intMon)
{ 
#ifdef applec
#pragma unused(ruleName)
#endif
  if (Less_Than_Now_Relp(relNode) && Relp_Still_Out(relNode)) {
    if (intMon->intMonOps->options->duration >= 0)
      spliceOutDurableItem(intMon->durableItem);
    stopPollingMonitor(intMon);
  }
}


/******************************************************************************
 *
 * FUNCTION: void replyPollingMonitor(dispatch, intMon)
 *
 * DESCRIPTION: Reply function from a polling monitor condition message.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void replyPollingMonitor(DISPATCH_PTR dispatch,
				INTERVAL_MONITOR_DATA_PTR intMon)
{
  int32 *tplConstr;
  MSG_PTR actMsg;
  BLOCK_COM_PTR comData;
  DISPATCH_PTR parentDispatch;
  
  if (!intMon->monInfo) {
    /* monitor already ended */
    spliceOutPollingMonitor(intMon);
    /* 12-May-91: fedor: can not free here becuase it will interfere with
       resourceProcessPending in recvMsg.c */
    /* dispatchFree(dispatch);*/
    return;
  }
  
  if (dataMsgTestMsgData(DISPATCH_RES_DATA(dispatch))) {
    
    intMon->condSatisfiedCounter++;
    
    parentDispatch = DISPATCH_FROM_ID(DISPATCH_PARENT_REF(dispatch));
    
    actMsg = GET_MESSAGE(intMon->monInfo->actionName);
    
    if (!actMsg) {
      tcaError("ERROR: replyPointMonitor: missing action message: %s\n",
	       intMon->monInfo->actionName);
    }
    
    if (intMon->condSatisfiedCounter == intMon->intMonOps->options->fireEvery){
      intMon->condSatisfiedCounter = 0; /* reset counter */
      --intMon->maxFireCounter;
      
      /* Made monitor actions be sequential, rather than no constraints 
	 between successive monitor actions (Reid, 3/9/95) */
      if (actMsg->msgData->msg_class == CommandClass) {
	comData = NEW(BLOCK_COM_TYPE);
	comData->waitFlag = FALSE;
	comData->tplConstr = SEQ_ACH;
	sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)comData,
		    parentDispatch);
      } else {
	tplConstr = NEW(int32);
	*tplConstr = SEQ_ACH;
	sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)tplConstr, 
		    parentDispatch);
      }
    }
  }
  
  
  /* 12-May-91: fedor: can not free here becuase it will interfere with
     resourceProcessPending in recvMsg.c */
  /* dispatchFree(dispatch);*/
  
  if (intMon->maxFireCounter == 0) 
    stopPollingMonitor(intMon);
  else {
    intMon->fireTime = timeInMsecs() + 1000*intMon->intMonOps->options->period;
  }
}


/******************************************************************************
 *
 * FUNCTION: void execPollingMonitor(intMon)
 *
 * DESCRIPTION: Start the condition message for the polling monitor intMon.
 *
 * INPUTS: INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void execPollingMonitor(INTERVAL_MONITOR_DATA_PTR intMon)
{
  MSG_PTR condMsg;
  
  condMsg = GET_MESSAGE(intMon->monInfo->conditionName);
  
  if (!condMsg) {
    tcaError("ERROR: execPollingMonitor: missing condition message: %s\n",
	     intMon->monInfo->conditionName);
  }
  
  intMon->fireTime = ALREADY_FIRED;
  
  sendDataMsgWithReply(condMsg, DISPATCH_MSG_DATA(intMon->dispatch), 
		       intMon->dispatch, 
		       (RESP_PROC_FN) replyPollingMonitor, (char *)intMon);
}


/******************************************************************************
 *
 * FUNCTION: void pollingMonStart(intMon)
 *
 * DESCRIPTION: 
 * Create the termination rule and initialize the polling monitor intMon.
 *
 * INPUTS: 
 * INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void pollingMonStart(INTERVAL_MONITOR_DATA_PTR intMon)
{
  if (intMon->expirationTime)
    addOutRule(ruleCreateOut("Terminate Monitor", 
			     (NODE_RULE_FN) terminatePollingMonitor, 3, 
			     (char *)intMon), 
	       relationGetOrMake(GET_S_GLOBAL(Now), intMon->expirationTime));
  
  listInsertItem((char *)intMon, GET_S_GLOBAL(pollingMonitorList));
  
  if (intMon->intMonOps->options->initialWait) {
    intMon->fireTime = timeInMsecs() + 1000*intMon->intMonOps->options->period;
  } else {
    execPollingMonitor(intMon);
  }
}      


/******************************************************************************
 *
 *  DEMON MONITOR
 *
 *****************************************************************************/

/******************************************************************************
 *
 * FUNCTION: void stopDemonMonitor(intMon)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void stopDemonMonitor(INTERVAL_MONITOR_DATA_PTR intMon)
{ 
  MSG_PTR cancelMsg;
  static int32 cancelKey = ALREADY_CANCELLED; 
  INTERVAL_MONITOR_DATA_PTR intMon1;
  INT_STR_KEY_TYPE key;

  if (intMon->expirationTime)
    removeOutRule((NODE_RULE_FN) terminateDemonMonitor, 
		  get_relation(GET_S_GLOBAL(Now), intMon->expirationTime));
  
  if (*(intMon->id) == NO_KEY)
    /* have yet to receive the reply to setUpMsg */
    intMon->id = &cancelKey;
  else {
    cancelMsg = GET_MESSAGE(intMon->monInfo->conditionName);
    if (!cancelMsg) {
      tcaError("ERROR: stopDemonMonitor: missing cancel message: %s\n",
	       intMon->monInfo->conditionName);
    }
    
    if (cancelMsg->msgData->msg_class != InformClass) {
      tcaError("ERROR: Demon Cancel Message: %s: Should be InformClass.\n",
	       cancelMsg->msgData->name);
    }
    
    if (!strKeyEqFunc(INT_FMT_NAME, cancelMsg->msgFormatStr)) {
      tcaError("ERROR: Demon Cancel Message: %s: Message format is not int\n",
	       cancelMsg->msgData->name);
    }
    
    if (!cancelMsg->parsedFormats) {
      parseMsg(cancelMsg);
    }
    
    (void)centralSendMessage((TCA_REF_PTR)NULL, cancelMsg, 
			     (char *)intMon->id, (char *)NULL);
    
    dispatchUpdateAndDisplay(HandledDispatch, intMon->dispatch);
    CompletionConstraints(intMon->dispatch);
    
    key.num = *(intMon->id); key.str = intMon->monInfo->monitorName;
    intMon1 = ((INTERVAL_MONITOR_DATA_PTR)
	       hashTableRemove((char *)&key, GET_S_GLOBAL(demonMonitorTable)));
    if (intMon != intMon1) {
      tcaError("ERROR: demonMonitorTable is screwed up (%s)!\n",
	       cancelMsg->msgData->name);
    }

    tcaFree((char *)intMon);
  }
}


/******************************************************************************
 *
 * FUNCTION: void terminateDemonMonitor(ruleName, relNode, intMon)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static void terminateDemonMonitor(char *ruleName, TMS_NODE_PTR relNode,
				  INTERVAL_MONITOR_DATA_PTR intMon)
{ 
#ifdef applec
#pragma unused(ruleName)
#endif
  if (Less_Than_Now_Relp(relNode) && Relp_Still_Out(relNode)) {
    if (intMon->intMonOps->options->duration >= 0)
      spliceOutDurableItem(intMon->durableItem);
    stopDemonMonitor(intMon);
  }
}


/******************************************************************************
 *
 * FUNCTION: void fireDemonHnd(dispatch, id)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * NOTES:
 * 6-Oct-90: fedor:
 * test for data for action message first?
 *
 * 8-Oct-90: fedor
 * problems if maxFireCounter wraps - blah!
 *
 *****************************************************************************/

void fireDemonHnd(DISPATCH_PTR dispatch, int32 *id)
{ 
  int32 *tplConstr;
  MSG_PTR actMsg;
  BLOCK_COM_PTR comData;
  INTERVAL_MONITOR_DATA_PTR intMon;
  INT_STR_KEY_TYPE key;

  key.num = *id; key.str = dispatch->msg->msgData->name;
  intMon = ((INTERVAL_MONITOR_DATA_PTR)
	    hashTableFind((char *)&key, GET_S_GLOBAL(demonMonitorTable)));
  
  if (intMon) {
    intMon->condSatisfiedCounter++;
    
    actMsg = GET_MESSAGE(intMon->monInfo->actionName);
    if (!actMsg) {
      tcaError("ERROR: fireDemonHnd: missing action message: %s\n",
	       intMon->monInfo->actionName);
    }
    
    if (intMon->condSatisfiedCounter == intMon->intMonOps->options->fireEvery){
      intMon->condSatisfiedCounter = 0; /* reset counter */
      --intMon->maxFireCounter;
      
      /* Made monitor actions be sequential, rather than no constraints 
	 between successive monitor actions (Reid, 3/9/95) */
      if (actMsg->msgData->msg_class == CommandClass) {
	comData = NEW(BLOCK_COM_TYPE);
	comData->waitFlag = FALSE;
	comData->tplConstr = SEQ_ACH;
	sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)comData,
		    intMon->dispatch);
      } else {
	tplConstr = NEW(int32);
	*tplConstr = SEQ_ACH;
	sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)tplConstr, 
		    intMon->dispatch);
      }
    }
    
    if (intMon->maxFireCounter == 0) 
      stopDemonMonitor(intMon);
  }
  
  /* 12-May-91: fedor: not sure yet if this one interfers 
     with resourceProcessPending */
  dispatchFree(dispatch);
}


/******************************************************************************
 *
 * FUNCTION: void replyDemonMonitor(dispatch, data)
 *
 * DESCRIPTION: 
 * Store intMon in the demonMonitorTable under the returned key from setUpMsg.
 *
 * INPUTS:
 * DISPATCH_PTR dispatch; 
 * INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void replyDemonMonitor(DISPATCH_PTR dispatch,
			      INTERVAL_MONITOR_DATA_PTR intMon)
{
  INT_STR_KEY_TYPE key;

  if (*intMon->id != ALREADY_CANCELLED) {
    intMon->id = (int32 *)dispatchDecodeResponse(dispatch);
    key.num = *(intMon->id); key.str = intMon->monInfo->monitorName;
    (void)hashTableInsert((char *)&key, sizeof(INT_STR_KEY_TYPE), 
			  (char *)intMon, GET_S_GLOBAL(demonMonitorTable));
  }
  /* 12-Oct-90: fedor: more memory problems ... 
     free dispatch here causes it to crash when replying from the setUpMsg 
     in deliverResponse - the dispatch is NULL */
}


/******************************************************************************
 *
 * FUNCTION: void demonMonStart(dispatch, intMon)
 *
 * DESCRIPTION:
 * Sets up an instance of a demon monitor and call set up message.
 *
 * INPUTS:
 * DISPATCH_PTR dispatch; 
 * INTERVAL_MONITOR_DATA_PTR intMon;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void demonMonStart(DISPATCH_PTR dispatch, 
			  INTERVAL_MONITOR_DATA_PTR intMon)
{
  MSG_PTR setUpMsg;
  
  if (intMon->expirationTime)
    addOutRule(ruleCreateOut("Terminate Monitor", 
			     (NODE_RULE_FN) terminateDemonMonitor, 3, intMon),
	       relationGetOrMake(GET_S_GLOBAL(Now), intMon->expirationTime));
  
  setUpMsg = GET_MESSAGE(intMon->monInfo->setUpName);  
  if (!setUpMsg) {
    tcaError("ERROR: demonMonStart: missing set up message: %s\n",
	     intMon->monInfo->setUpName);
  }
  
  sendDataMsgWithReply(setUpMsg, DISPATCH_MSG_DATA(dispatch), dispatch,
		       (RESP_PROC_FN) replyDemonMonitor, (char *)intMon);
}


/******************************************************************************
 *
 * FUNCTION: void intervalMonitorHnd(dispatch, intMonOps)
 *
 * DESCRIPTION:
 * Sets up an instance of an interval monitor.
 *
 * INPUTS:
 * DISPATCH_PTR dispatch; 
 * INT_MON_CLASS_PTR intMonOps;
 *
 * OUTPUTS: void.
 *
 * NOTES:
 *
 *  5-Oct-90: fedor
 * AttendingConstraints were set in the original after the temporally inactive
 * test. AttendingConstraints already got set before this handler was called.
 * This may be a problem.
 *
 *  9-Oct-90: fedor
 * Reid does not think it will be a problem if the monitor is never set up
 * because expiration already occured even though AttendingConstraints have
 * already been called. Could be another potential memory leak.
 *
 *****************************************************************************/

static void intervalMonitorHnd(DISPATCH_PTR dispatch,
			       INTERVAL_MON_CLASS_PTR intMonOps)
{
  static int32 key = NO_KEY;
  TCA_TIME_POINT_PTR CEnd; 
  INTERVAL_MONITOR_DATA_PTR intMon;
  
  intMon = NEW(INTERVAL_MONITOR_DATA_TYPE);
  intMon->id = &key;
  intMon->fireTime = 0;
  intMon->maxFireCounter = 0;
  intMon->condSatisfiedCounter = 0;
  intMon->dispatch = dispatch;
  
  /* 16-Nov-90: fedor: for taskTree kills - store the intMon on the 
     monitor dispatch. This has the potential for messing up 
     future calls to sendDataMsg. */
  dispatch->respData = (char *)intMon;
  
  intMon->durableItem = NULL;
  intMon->monInfo = 
    (MON_REG_DATA_PTR)hashTableFind(dispatch->msg->msgData->name, 
				    GET_S_GLOBAL(monitorTable));
  
  if (!intMon->monInfo) {
    tcaError("ERROR: intervalMonitorHnd: missing monitor info: %s\n",
	     dispatch->msg->msgData->name);
  }
  
  CEnd = &(intMonOps->expire);
  intMon->expirationTime = (VALID_TIME(*CEnd)) ? Time_Of(CEnd) : NULL;
  
  intMon->intMonOps = intMonOps;
  
  if (intMon->expirationTime && QIs_True(GET_S_GLOBAL(Now), ">=", 
					 intMon->expirationTime)) {
    Log_Status("Monitor %s ", dispatch->msg->msgData->name);
    Log_RefId(dispatch, LOGGING_STATUS);
    Log_Status(" is not scheduled because it is temporally inactive\n");
    
  } else {
    
    intMon->maxFireCounter = intMonOps->options->maxFire;
    
    if (intMonOps->options->duration > 0)
      intMon->durableItem = addDurableItem(intMonOps->options->duration, 
					   intMon, dispatch);
    
    switch(dispatch->msg->msgData->msg_class) {
    case DemonMonitorClass:
      demonMonStart(dispatch, intMon);
      break;
    case PollingMonitorClass:
      pollingMonStart(intMon);
      break;
    default:
      tcaError("ERROR: intervalMonitorHnd: Unknown Class: %d\n",
	       dispatch->msg->msgData->msg_class);
      break;
    }
  }
}

/* 
   07-Aug-91: Reid: Need to set the interval monitor start time *before*
   "processInactiveTests" is called in recvMessage 
   17-May-93: Reid: Changed Qassert from "=" to ">=".  Don't know why it
   was initially "=", but that does not always do the right thing.
   */

void setIntervalMonitorStartTime (DISPATCH_PTR dispatch, 
				  INTERVAL_MON_CLASS_PTR intMonOps)
{
  TCA_TIME_POINT_PTR CStart; 
  TASK_TREE_NODE_PTR monitorNode;
  struct _QUANTITY_TYPE *MStart;
  
  monitorNode = dispatch->treeNode;
  
  MStart = monitorNode->handlingInterval->start;
  CStart = &(intMonOps->start);
  
  if (VALID_TIME(*CStart))
    QAssert(MStart, ">=", Time_Of(CStart), "Monitor Constraints");
  else if (monitorNode->parent->handlingInterval)
    QAssert(MStart, ">=", monitorNode->parent->handlingInterval->start, 
	    "Monitor Constraints");
}


/******************************************************************************
 *
 *  POINT MONITOR
 *
 *****************************************************************************/

/******************************************************************************
 *
 * FUNCTION: void replyPointMonitor(dispatch, monInfo)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * DISPATCH_PTR dispatch;
 * MON_REG_DATA_PTR monInfo;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void replyPointMonitor(DISPATCH_PTR dispatch, MON_REG_DATA_PTR monInfo)
{
  int32 *tplConstr;
  MSG_PTR actMsg;
  BLOCK_COM_PTR comData;
  DISPATCH_PTR parentDispatch;
  
  parentDispatch = DISPATCH_FROM_ID(DISPATCH_PARENT_REF(dispatch));
  
  if (!monInfo){
    tcaError("ERROR: replyPointMonitor: missing monitor info: %s\n",
	     dispatch->msg->msgData->name);
  }
  
  if (dataMsgTestMsgData(DISPATCH_RES_DATA(dispatch))) {
    actMsg = GET_MESSAGE(monInfo->actionName);
    if (!actMsg) {
      tcaError("ERROR: replyPointMonitor: missing action message: %s\n",
	       monInfo->actionName);
    }
    
    if (actMsg->msgData->msg_class == CommandClass) {
      comData = NEW(BLOCK_COM_TYPE);
      comData->waitFlag = FALSE;
      comData->tplConstr = NO_TPLCONSTR;
      sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)comData,
		  parentDispatch);
    } else {
      tplConstr = NEW(int32);
      *tplConstr = NO_TPLCONSTR;
      sendDataMsg(actMsg, DISPATCH_RES_DATA(dispatch), (char *)tplConstr, 
		  parentDispatch);
    }
  }
  
  processHandledMessage(parentDispatch);
}


/******************************************************************************
 *
 * FUNCTION: void pointMonitorHnd(dispatch, data)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * DISPATCH_PTR dispatch;
 * char *data;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void pointMonitorHnd(DISPATCH_PTR dispatch, void *data)
{
#ifdef applec
#pragma unused(data)
#endif
  MSG_PTR msg, condMsg;
  MON_REG_DATA_PTR monInfo;
  
  msg = dispatch->msg;
  
  monInfo = (MON_REG_DATA_PTR)hashTableFind(msg->msgData->name,
					    GET_S_GLOBAL(monitorTable));
  
  if (!monInfo) {
    tcaError("ERROR: pointMonitorHnd: missing monitor info: %s\n",
	     msg->msgData->name);
  }
  
  condMsg = GET_MESSAGE(monInfo->conditionName);
  if (!condMsg) {
    tcaError("ERROR: pointMonitorHnd: missing condition message: %s\n",
	     monInfo->conditionName);
  }
  
  sendDataMsgWithReply(condMsg, DISPATCH_MSG_DATA(dispatch), dispatch,
		       (RESP_PROC_FN) replyPointMonitor, (char *)monInfo);
}


/******************************************************************************
 *
 *  DELAY COMMAND
 *  Use the duration loop to implement a delay command -- 
 *  waits for a given duration, before returning success.
 *
 *  9-Oct-90: fedor: this whole delay needs more thought.
 *
 *****************************************************************************/

static void delayCommandHnd(DISPATCH_PTR dispatch, int32 *duration)
{ 
  (void)addDurableItem(*duration, (INTERVAL_MONITOR_DATA_PTR)NULL, dispatch);
}

static int sameDurableItem(DISPATCH_PTR dispatch, DURABLE_ITEM_PTR durableItem)
{
  return (dispatch == durableItem->dispatch);
}

static BOOLEAN abortDelayCommand(DISPATCH_PTR dispatch)
{
  DURABLE_ITEM_PTR delayItem;
  
  delayItem = (DURABLE_ITEM_PTR)
    listMemReturnItem((LIST_ITER_FN) sameDurableItem, 
		      (char *)dispatch, 
		      GET_S_GLOBAL(durableItemList));
  if (delayItem) {
    spliceOutDurableItem(delayItem);
    return TRUE;
  }
  
  return FALSE;
}


/******************************************************************************
 *
 *  MONITOR REGISTRATION AND INITIALIZATION
 *
 *****************************************************************************/

/******************************************************************************
 *
 * FUNCTION: void parseMonitorMsg(msg)
 *
 * DESCRIPTION: 
 * Check and parse message format information. This allows monitors to be
 * defined before used.
 *
 * INPUTS: 
 * MSG_PTR msg;
 *
 * OUTPUTS: void.
 *
 * NOTES: 
 * 20-Jun-91: fedor: wow a year to finally do this!
 *
 *****************************************************************************/

void parseMonitorMsg(MSG_PTR msg)
{
  MSG_PTR conditionMsg, setUpMsg, actMsg;
  
  if (msg->msgData->msg_class == DemonMonitorClass) {
    setUpMsg = msgFind(msg->msgFormatStr);
    if (setUpMsg == NULL) return;
    actMsg = msgFind(msg->resFormatStr);
    if (actMsg == NULL) return;
    
    if (!setUpMsg) {
      tcaError("ERROR: parseMonitorMsg: Missing Message Definition: %s\n",
	       msg->msgFormatStr);
    }
    
    if (setUpMsg->msgData->msg_class != QueryClass) {
      tcaError("ERROR: Demon Set Up Message: %s: Should be QueryClass.\n",
	       msg->msgFormatStr);
    }
    
    if (!strKeyEqFunc("int", setUpMsg->resFormatStr)) {
      tcaError("ERROR: Demon Set Up Message: %s: Reply format is not int.\n",
	       msg->msgFormatStr);
    }
    
    if (!actMsg) {
      tcaError("ERROR: parseMonitorMsg: Missing Message Definition: %s\n",
	       msg->resFormatStr);
    }
    
    if (!setUpMsg->parsedFormats)
      parseMsg(setUpMsg);
    
    if (!actMsg->parsedFormats)
      parseMsg(actMsg);
    
    msg->msgData->msgFormat = setUpMsg->msgData->msgFormat;
    msg->msgData->resFormat = actMsg->msgData->msgFormat;
  }
  else {
    conditionMsg = msgFind(msg->msgFormatStr);
    
    if (!conditionMsg) {
      tcaError("ERROR: parseMonitorMsg: Missing Message Definition: %s\n",
	       msg->msgFormatStr);
    }
    
    if (!conditionMsg->parsedFormats)
      parseMsg(conditionMsg);
    
    msg->msgData->msgFormat = conditionMsg->msgData->msgFormat;
    msg->msgData->resFormat = conditionMsg->msgData->resFormat;
  }
  
  msg->parsedFormats = TRUE;
}

/******************************************************************************
 *
 * FUNCTION: void regMonitorHnd(dispatch, regData)
 *
 * DESCRIPTION: 
 * Handler for tcaRegisterPointMonitor, tcaRegisterPollingMonitor and
 * tcaRegisterDemonMonitor.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch; 
 * MON_REG_DATA_PTR regData;
 *
 * OUTPUTS: none.
 *
 * NOTES:
 *
 * 19-Jun-90: fedor: 
 * arrrrrgggg! have the system register the message and update the format 
 * info later - since we no longer have the string format info to parse. 
 *
 * Need to allow message def to come later requires some 
 * module information query.
 *
 * Need to test formats of key messages that they return what they are
 * supposed to - setUpMsg for demons should return int.
 *
 * PointMonitor: 
 * PollingMonitor:
 * msgFormat = condition message's msgFormat
 * resFormat = condition message's resFormat
 *
 * DemonMonitor:
 * msgFormat = setup message's msgFormat - setUp QueryClass and reply format "int".
 * resFormat = action message's msgFormat
 * cancelMsg - ConstraintClass whose format is "int".
 *
 *****************************************************************************/

static void regMonitorHnd(DISPATCH_PTR dispatch, MON_REG_DATA_PTR regData)
{
#ifdef applec
#pragma unused(dispatch)
#endif
  MSG_PTR monMsg;
  MON_REG_DATA_PTR oldRegData;
  const char *monitorName;
  
  monitorName = regData->monitorName;
  centralRegisterMessage(monitorName, regData->msg_class, NULL, NULL);
  
  monMsg = GET_MESSAGE(monitorName);
  
  /* 22-Jun-91: fedor: reset monitor parse flag because
     centralRegisterMessage will flag it as parsed. */
  monMsg->parsedFormats = FALSE;
  
  switch (regData->msg_class) {
  case PointMonitorClass:
    centralRegisterHandler(monitorName, "pointMonitorHnd", pointMonitorHnd);
    break;
  case DemonMonitorClass:
  case PollingMonitorClass:
    centralRegisterHandler(monitorName, "intervalMonitorHnd",
			   intervalMonitorHnd);
    break;
  default:
    tcaError("ERROR: regMonitorHnd: unknown monitor class.\n");
    break;
  }
  
  /* 20-Jun-91: fedor: msgFormatStr and resFormatStr are not byte
     copied because regData is saved to check a defintion change. */
  
  if (regData->msg_class == DemonMonitorClass) {
    monMsg->msgFormatStr = regData->setUpName;
    monMsg->resFormatStr = regData->actionName;
  }
  else {
    monMsg->msgFormatStr = regData->conditionName;
    monMsg->resFormatStr = regData->conditionName;
  }
  
  oldRegData = (MON_REG_DATA_PTR)hashTableFind(monitorName,
					       GET_S_GLOBAL(monitorTable));
  if (oldRegData) {
    Log("\nWARNING: regMonitorHnd: Registration for %s updated.\n",
	monitorName);
    tcaFree((char *)oldRegData->monitorName);
    if (oldRegData->conditionName) tcaFree((char *)oldRegData->conditionName);
    if (oldRegData->actionName) tcaFree((char *)oldRegData->actionName);
    if (oldRegData->setUpName) tcaFree((char *)oldRegData->setUpName);
    oldRegData->monitorName = regData->monitorName;
    oldRegData->conditionName = regData->conditionName;
    oldRegData->actionName = regData->actionName;
    oldRegData->setUpName = regData->setUpName;
    tcaFree((char *)regData);
  } else {
    hashTableInsert(monitorName, strlen(regData->monitorName)+1,
		    (char *)regData, GET_S_GLOBAL(monitorTable));
  }
}

/******************************************************************************
 *
 * FUNCTION: int pollFn(currentTime, intMon)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static int pollFn(unsigned long *currentTime, INTERVAL_MONITOR_DATA_PTR intMon)
{
  if (intMon->fireTime != ALREADY_FIRED && intMon->fireTime <= *currentTime) 
    execPollingMonitor(intMon);
  
  return TRUE;
}


/******************************************************************************
 *
 * FUNCTION: int durableFn(currentTime, durablleItem)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

static int32 durableFn(unsigned long *currentTime, DURABLE_ITEM_PTR durableItem)
{
  if (durableItem->endTime <= *currentTime) {
    switch (durableItem->dispatch->msg_class) {
    case CommandClass: 
      /* Currently, this will only be used by the "delay" command 
         If we make it more general, then look out for calling success twice */
      /* Set the class to make TCA know that the command is ending */
      /* NULL out the classData, because it is already stored in blockCom
	 (which gets freed within recvSuccessFailureMessage) */
      durableItem->dispatch->msg_class = SuccessClass;
      durableItem->dispatch->classData = NULL;
      recvSuccessFailureMessage(durableItem->dispatch, SuccessClass);
      break;
    case DemonMonitorClass:
      stopDemonMonitor(durableItem->intMon);
      break;
    case PollingMonitorClass:
      stopPollingMonitor(durableItem->intMon);
      break;
    default: 
      tcaError("Only Commands and Interval Monitors can have durations");
    }
    spliceOutDurableItem(durableItem);
  }
  return TRUE;
}


/******************************************************************************
 *
 * FUNCTION: void monitorPoll()
 *
 * DESCRIPTION: 
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 * NOTES:
 *  9-Oct-90: fedor:
 * all time values in danger of wrapping - switch to unsigned?
 *
 *****************************************************************************/

void monitorPoll(void)
{ 
  unsigned long currentTime;
  
  if (listLength(GET_S_GLOBAL(pollingMonitorList)) || 
      listLength(GET_S_GLOBAL(durableItemList))) {
    currentTime = timeInMsecs();
    if (currentTime > GET_S_GLOBAL(lastTime)) {
      GET_S_GLOBAL(lastTime) = currentTime;
      (void)listIterate((LIST_ITER_FN) pollFn, (char *)&currentTime,
			GET_S_GLOBAL(pollingMonitorList));
      (void)listIterate((LIST_ITER_FN) durableFn, (char *)&currentTime,
			GET_S_GLOBAL(durableItemList));
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: void cancelIfAttendingIntervalMonitor(dispatch)
 *
 * DESCRIPTION: 
 * Given a dispatch, test if it is attending and an interval monitor and
 * cancel it. An interval monitor is a DemonMonitorClass or
 * PollingMonitorClass.
 *
 * This provides an interface to taskTree for stopping monitors.
 *
 * INPUTS: DISPATCH_PTR dispatch;
 *
 * OUTPUTS: none.
 *
 * NOTES:
 * 16-Nov-90: fedor: questions remain if you stop a monitor that
 * is also a durable item.
 *
 *****************************************************************************/

void cancelIfAttendingIntervalMonitor(DISPATCH_PTR dispatch)
{
  INTERVAL_MONITOR_DATA_PTR intMon;
  
  intMon = (INTERVAL_MONITOR_DATA_PTR)dispatch->respData;
  
  if (dispatch->status == AttendingDispatch && dispatch->respData) {
    if (dispatch->msg->msgData->msg_class == DemonMonitorClass)
      stopDemonMonitor(intMon);
    else if (dispatch->msg->msgData->msg_class == PollingMonitorClass)
      stopPollingMonitor(intMon);
  }
}


/******************************************************************************
 *
 * FUNCTION: void monitorInitialize()
 *
 * DESCRIPTION: Initialize central monitors.
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void monitorInitialize(void)
{
  GET_S_GLOBAL(durableItemList) = listCreate();
  GET_S_GLOBAL(pollingMonitorList) = listCreate();
  
  GET_S_GLOBAL(demonMonitorTable) = hashTableCreate(11, (HASH_FN)intStrHashFunc,
						    (EQ_HASH_FN)intStrKeyEqFunc);
  GET_S_GLOBAL(monitorTable) = hashTableCreate(11, (HASH_FN) strHashFunc,
					       (EQ_HASH_FN) strKeyEqFunc);
  
  centralRegisterInform(TCA_REG_MONITOR_INFORM,
			TCA_REG_MONITOR_INFORM_FORMAT,
			regMonitorHnd);
  Add_Message_To_Ignore(TCA_REG_MONITOR_INFORM);
  
  centralRegisterInform(TCA_REG_MONITOR_INFORM_OLD,
			TCA_REG_MONITOR_INFORM_FORMAT,
			regMonitorHnd);
  Add_Message_To_Ignore(TCA_REG_MONITOR_INFORM_OLD);
  
  /* 13-Oct-90: fedor: this is a message for users to delay for a certain
     number of seconds - there should be a tca routine for this instead. */
  centralRegisterCommand(TCA_DELAY_CMD, 
			 TCA_DELAY_CMD_FORMAT,
			 delayCommandHnd);
  centralRegisterCommand(TCA_DELAY_CMD_OLD, 
			 TCA_DELAY_CMD_FORMAT,
			 delayCommandHnd);
  (void)addAbortHandler(TCA_DELAY_CMD, abortDelayCommand);
  (void)addAbortHandler(TCA_DELAY_CMD_OLD, abortDelayCommand);
}
