/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1994.
 * 
 * MODULE: communications
 *
 * FILE: modVar.c
 *
 * ABSTRACT:
 * 
 * Provides a system global variables.
 *
 * REVISION HISTORY
 *
 * $Log: modVar.c,v $
 * Revision 1.15  1996/03/09  06:13:15  rich
 * Fixed problem where lisp could use the wrong byte order if it had to
 * query for a message format.  Also fixed some memory leaks.
 *
 * Revision 1.14  1996/02/10  16:50:10  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.13  1995/12/17  20:21:42  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.12  1995/10/25  22:48:35  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.11  1995/10/07  19:07:27  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.10  1995/08/14  21:31:56  rich
 * Got rid of the "sharedBuffers" flag on the dataMessages.  It was not the
 * right solution, and possibly caused a memory leak.
 * Limit pending for TCA_TAPPED_MSG_VAR to 1.
 *
 * Revision 1.9  1995/06/05  23:59:05  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.8  1995/04/08  02:06:26  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.7  1995/04/07  05:03:27  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.6  1995/04/04  19:42:31  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.5  1995/03/30  15:43:31  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.4  1995/03/28  01:14:46  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.3  1995/01/18  22:41:15  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.2  1994/05/25  04:57:48  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.1  1994/05/17  23:16:33  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 *
 * $Revision: 1.15 $
 * $Date: 1996/03/09 06:13:15 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalS.h"
#include "hash.h"
#include "modVar.h"

/* Global Defines */

/* Private Global types */

typedef struct _var {
  const char *varName;
  const char *format;
  void *value;
} GLOBAL_VAR_TYPE, *GLOBAL_VAR_PTR;



/*****************************************************************************
 *
 * FUNCTION: void tcaRegisterVar(const char *varName, const char *format)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaRegisterVar(const char *varName, const char *format)
{
  GLOBAL_VAR_TYPE var;
  
  var.varName = varName;
  var.format = format;
  tcaInform(TCA_REGISTER_VAR_INFORM, &var);
}


/*****************************************************************************
 *
 * FUNCTION: void _tcaSetVar(const char *varName, const void *value)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE _tcaSetVar(const char *varName, const void *value)
{
  char *setMsgName;
  TCA_RETURN_VALUE_TYPE result;
  
  setMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_SET_PREFIX));
  strcpy(setMsgName,VAR_SET_PREFIX);
  strcat(setMsgName,varName);
  
  /*  tcaExecuteCommandAsync(setMsgName, value);*/
  result = tcaInform(setMsgName, value);

  tcaFree(setMsgName);
  return result;
}


/*****************************************************************************
 *
 * FUNCTION: void _tcaGetVar(const char *varName, void *value)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE _tcaGetVar(const char *varName, void *value)
{
  char *getMsgName;
  TCA_RETURN_VALUE_TYPE result;
  
  getMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_GET_PREFIX));
  strcpy(getMsgName,VAR_GET_PREFIX);
  strcat(getMsgName,varName);
  
  result = tcaQuery(getMsgName, NULL, value);

  tcaFree(getMsgName);
  return result;
}


/*****************************************************************************
 *
 * FUNCTION: void _tcaGetSetVar(const char *varName, void *value)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE _tcaGetSetVar(const char *varName, 
				    void *setValue, void *getValue)
{
  char *getSetMsgName;
  TCA_RETURN_VALUE_TYPE result;
  
  getSetMsgName = (char *)tcaMalloc(1+strlen(varName)+
				    strlen(VAR_GET_SET_PREFIX));
  strcpy(getSetMsgName,VAR_GET_SET_PREFIX);
  strcat(getSetMsgName,varName);
  
  result = tcaQuery(getSetMsgName, setValue, getValue);
  tcaFree(getSetMsgName);
  
  return result;
}


/*****************************************************************************
 *
 * FUNCTION: void _tcaWatchVar(const char *varName, const char *format, 
 *                             TCA_HND_FN watchFn)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void _tcaWatchVar(const char *varName, const char *format, TCA_HND_FN watchFn)
{
  char *watchMsgName;
  
  watchMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,varName);
  
  /* Make sure the variable is registered first. */
  /*  tcaRegisterVar(varName, format);*/
  
  /* Register the tap messages and handler */
  tcaRegisterHandler(watchMsgName,watchMsgName,watchFn);
  tcaFree(watchMsgName);
}


/*****************************************************************************
 *
 * FUNCTION: void tcaUnwatchVar(const char *varName)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaUnwatchVar(const char *varName)
{
  char *watchMsgName;
  
  watchMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,varName);
  
  /* Unregister the tap messages and handler */
  tcaDeregisterHandler(watchMsgName,watchMsgName);
  tcaFree(watchMsgName);
}

void tcaIgnoreVarLogging(const char *varName)
{
  char *setMsgName;
  char *getMsgName;
  char *getSetMsgName;
  char *watchMsgName;
  
  setMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_SET_PREFIX));
  strcpy(setMsgName,VAR_SET_PREFIX);
  strcat(setMsgName,varName);
  tcaIgnoreLogging(setMsgName);
  
  getMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_GET_PREFIX));
  strcpy(getMsgName,VAR_GET_PREFIX);
  strcat(getMsgName,varName);
  tcaIgnoreLogging(getMsgName);
  
  getSetMsgName = (char *)tcaMalloc(1+strlen(varName)+ 
				    strlen(VAR_GET_SET_PREFIX));
  strcpy(getSetMsgName,VAR_GET_SET_PREFIX);
  strcat(getSetMsgName,varName);
  tcaIgnoreLogging(getSetMsgName);
  
  watchMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,varName);
  tcaIgnoreLogging(watchMsgName);

  tcaFree(setMsgName);
  tcaFree(getMsgName);
  tcaFree(getSetMsgName);
  tcaFree(watchMsgName);
}

void tcaFreeVar(const char *varName, void *varData)
{
  char *getMsgName;

  getMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_GET_PREFIX));
  strcpy(getMsgName,VAR_GET_PREFIX);
  strcat(getMsgName,varName);

  tcaFreeReply(getMsgName, varData);

  tcaFree(getMsgName);

}

static void fileLogVarHnd(TCA_REF_PTR Ref, LOG_PTR log)
{
  if (GET_M_GLOBAL(logList)[0] != NULL)
    free(GET_M_GLOBAL(logList)[0]);
  GET_M_GLOBAL(logList)[0] = log;
  GET_M_GLOBAL(logList)[2] = NULL;
}

static void terminalLogVarHnd(TCA_REF_PTR Ref, LOG_PTR log)
{
  if (GET_M_GLOBAL(logList)[1] != NULL)
    free(GET_M_GLOBAL(logList)[1]);
  GET_M_GLOBAL(logList)[1] = log;
  GET_M_GLOBAL(logList)[2] = NULL;
}

static void tappedMsgVarHnd(TCA_REF_PTR Ref, STR_LIST_PTR tapped)
{
  STR_LIST_PTR temp;
  MSG_PTR msg;

  strListFree(&GET_C_GLOBAL(tappedMsgs),TRUE);
  
  GET_C_GLOBAL(tappedMsgs) = tapped;
  temp = tapped->next;
  while (temp != NULL) {
    msg = GET_MESSAGE(temp->item);
    if (msg != NULL) {
      msg->direct = FALSE;
    }
    temp = temp->next;
  }
}

static void broadcastMsgVarHnd(TCA_REF_PTR Ref, STR_LIST_PTR broadcast)
{
  strListFree(&GET_C_GLOBAL(broadcastMsgs),TRUE);
  
  GET_C_GLOBAL(broadcastMsgs) = broadcast;
}


/******************************************************************************
 *
 * FUNCTION: void modVarInitialize()
 *
 * DESCRIPTION: 
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void modVarInitialize(void)
{

  GET_M_GLOBAL(logList)[0] = (LOG_PTR) tcaMalloc(sizeof(LOG_TYPE));
  GET_M_GLOBAL(logList)[1] = (LOG_PTR) tcaMalloc(sizeof(LOG_TYPE));
  GET_M_GLOBAL(logList)[2] = NULL;

  /* Register Handlers for the logging status. */
  tcaGetVar(TCA_TERMINAL_LOG_VAR, (GET_M_GLOBAL(logList)[0]));
  tcaGetVar(TCA_FILE_LOG_VAR, (GET_M_GLOBAL(logList)[1]));
  /* Register Handlers for tapped messages. */
  GET_C_GLOBAL(tappedMsgs) = strListCreate();
  tcaGetVar(TCA_TAPPED_MSG_VAR, GET_C_GLOBAL(tappedMsgs));
  /* Register Handlers for broadcast messages. */
  GET_C_GLOBAL(broadcastMsgs) = strListCreate();
  tcaGetVar(TCA_BROADCAST_MSG_VAR, GET_C_GLOBAL(broadcastMsgs));

  /* Only want to tap if the module is also listening for other messages. */
  if (((GET_C_GLOBAL(willListen) > 0) ||
       ((GET_C_GLOBAL(willListen) == -1) &&
	(hashTableCount(GET_C_GLOBAL(handlerTable)) > 0))) &&
      (GET_C_GLOBAL(directDefault))) {
    tcaWatchVar(TCA_TERMINAL_LOG_VAR,TCA_LOG_VAR_FORMAT, terminalLogVarHnd);
    tcaWatchVar(TCA_FILE_LOG_VAR, TCA_LOG_VAR_FORMAT, fileLogVarHnd);
    tcaWatchVar(TCA_TAPPED_MSG_VAR,TCA_STR_LIST_FORMAT, tappedMsgVarHnd);
    tcaLimitPendingMessages(TCA_TERMINAL_LOG_VAR, 
			    GET_M_GLOBAL(modNameGlobal), 1);
    tcaLimitPendingMessages(TCA_FILE_LOG_VAR, 
			    GET_M_GLOBAL(modNameGlobal), 1);
    tcaLimitPendingMessages(TCA_TAPPED_MSG_VAR, 
			    GET_M_GLOBAL(modNameGlobal), 1);
  }

  /* Only want to tap is the module is also listening for other messages. */
  if ((GET_C_GLOBAL(willListen) > 0) ||
      ((GET_C_GLOBAL(willListen) == -1) &&
       (hashTableCount(GET_C_GLOBAL(handlerTable)) > 0))) {
    tcaWatchVar(TCA_BROADCAST_MSG_VAR,TCA_STR_LIST_FORMAT, broadcastMsgVarHnd);
    tcaLimitPendingMessages(TCA_BROADCAST_MSG_VAR, 
			    GET_M_GLOBAL(modNameGlobal), 1);
  } else {
    strListFree(&GET_C_GLOBAL(broadcastMsgs), TRUE);
    GET_C_GLOBAL(broadcastMsgs) = NULL;
  }
}

