/*****************************************************************************
 * 
 * PROJECT: Task Control Architecture.
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: logging
 *
 * FILE: logging.h
 *
 * ABSTRACT:
 * External declarations for message and status logging facilities.
 *
 * REVISION HISTORY
 *
 * $Log: logging.h,v $
 * Revision 1.14  1996/07/19  18:14:14  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.13  1995/05/31  19:35:54  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.12  1995/03/30  15:43:27  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.11  1995/03/28  01:14:44  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.10  1995/01/18  22:41:12  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.9  1994/10/25  17:10:01  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.8  1994/05/17  23:16:20  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.7  1994/04/28  22:16:58  rich
 * Added very simple stdin interface to central.  You can type command line
 * argument into standard in while central is running.  One option per
 * line.  In addition, it understands "quit" and "help".
 *
 * Revision 1.6  1994/04/28  16:16:20  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.5  1993/12/01  18:03:51  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.4  1993/11/21  20:18:18  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:15:31  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:18:06  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:45  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:30:31  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:31  fedor
 * Added Logging.
 *
 *  5-Jan-93 Domingo Gallardo, School of Computer Science, CMU
 * Added field parentId to LOG_TYPE.
 * 
 *  6-Oct-89 Christopher Fedor, School of Computer Science, CMU
 * Added message handle time summary.
 *
 *  3-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added code for ignoring registration msgs.
 *
 *  2-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Created.
 *
 * $Revision: 1.14 $
 * $Date: 1996/07/19 18:14:14 $
 * $Author: reids $
 *
 *****************************************************************************/

/*********************************************************************
 *                                                                   *
 * logging.h                                                         *
 *                                                                   *
 *                                                                   *
 * Functions:                                                        *
 *   The following three functions take a variable number of         *
 *   arguments, just like printf                                     *
 *                                                                   *
 *   Log (format_string, ...<args>) :                                *
 *     logs unconditionally onto the terminal and into the log file  *
 *     file (if it is open).                                         *
 *                                                                   *
 *   Log_Message (format_string, ...<args>) :                        *
 *     logs onto the terminal if Terminal_Log_Message is not 0;      *
 *     logs into the Log_File if File_Log_Message is not 0;          *
 *                                                                   *
 *   Log_Status (format_string, ...<args>) :                         *
 *     logs onto the terminal if Terminal_Log_Status is not 0;       *
 *     logs into the Log_File if File_Log_Status is not 0;           *
 *                                                                   *
 *   Log_Time (indent) :                                             *
 *     logs the time (hh:mm:sec.milli), indenting "indent" spaces.   *
 *     Logs onto the terminal if Terminal_Log_Time is not 0;         *
 *     Logs into the Log_File if File_Log_Time is not 0.             *
 *                                                                   *
 *   Log_Data (format, data, indent) :                               *
 *     logs the "data" using the given "format", indenting each line *
 *     by "indent".  Assumes a line length of 80 and truncates after *
 *     5 lines of output (these can be changed by setting the        *
 *     variables Line_Length and Print_Length, respectively).        *
 *     Logs onto the terminal if Terminal_Log_Data is not 0;         *
 *     Logs into the Log_File if File_Log_Data is not 0;             *
 *                                                                   *
 *   Start_Ignore_Logging () :                                       *
 *     starting now, don't log messages or status reports            *
 *     on the terminal if Terminal_Log_Ignore is TRUE, and           *
 *     don't log in file if File_Log_Ignore is TRUE.                 *
 *                                                                   *
 *   End_Ignore_Logging () :                                         *
 *     reinstate logging messages and status reports                 *
 *                                                                   *
 *   Start_Terminal_Logging () :                                     *
 *     Prints out the TCA header and version number.                 *
 *                                                                   *
 *   Start_File_Logging () :                                         *
 *     If either File_Log_Messages or File_Log_Status are non-zero,  *
 *     opens the file given by Log_File_Name and prints out the TCA  *
 *     header and version number.                                    *
 *     Sets signal functions to trap errors and close log file       *
 *     Gives user option to enter initial comments to file.          *
 *                                                                   *
 *   End_File_Logging () :                                           *
 *     Gives user option to enter final comments to file.            *
 *     Close the Log_File (if open).                                 *
 *                                                                   *
 *                                                                   *
 *********************************************************************/

#ifndef INClogging
#define INClogging

/* What type of message is this a part of */
typedef enum { LOGGING_ALWAYS, LOGGING_MESSAGE, LOGGING_STATUS} LOG_STATUS_ENUM;

typedef struct {
  FILE *theFile;
  int32 messages;
  int32 status;
  int32 time;
  int32 data;
  int32 summary;
  int32 refId;
  int32 parentId;
  int32 quiet;
  int32 flush;
  int32 ignore;
  int32 ignoreNow;
} LOG_TYPE, *LOG_PTR; 

#define MAX_LOG_FILE_NAME_LENGTH 60

/*extern char Log_File_Name[MAX_LOG_FILE_NAME_LENGTH];*/

void Log(const char *format_string, ...);
void Log_Message(const char *format_string, ...);
void Log_Status(const char *format_string, ...);

void LogHandleSummary(long arg1, float arg2, float arg3, float arg4,
		      long arg5, long arg6, long arg7, long arg8);

void Log_Time(int32 indent);
void Log_ParentId (int32 parent_id, LOG_STATUS_ENUM logStatus);
void Log_RefId (DISPATCH_PTR dispatch, LOG_STATUS_ENUM logStatus);
void Log_Data(CONST_FORMAT_PTR Format, const void *Data, int32 indent);

extern int32 Ignore_Logging_Message (MSG_PTR Message);
extern void Add_Message_To_Ignore(const char *name);
extern void ignoreLoggingHnd(DISPATCH_PTR dispatch, char **msgName);
extern void resumeLoggingHnd(DISPATCH_PTR dispatch, char **msgName);

void Start_Ignore_Logging(void);
void End_Ignore_Logging(void);
void Start_Terminal_Logging(void);
void Start_File_Logging(void);
void End_File_Logging(void);

#endif /* INClogging */
