/**************************************************************************
 * 
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: list
 *
 * FILE: list4.c
 *
 * ABSTRACT:
 * The list module provides basic list creation and manipulation
 * routines for doubly linked lists.
 *
 * The include file list.h provides the top level routines for other modules.
 *
 * EXPORTS:
 *
 * LIST_PTR listCreate()
 * Creates an empty list.
 *
 * void listFree(list)
 * LIST_PTR list;
 * Frees storage associated with a list.
 *
 * void listInsertItemFirst(item, list)
 * char *item;
 * LIST_PTR list;
 * Adds item as the first item in the list.
 *
 * void listInsertItemLast(item, list)
 * char *item;
 * LIST_PTR list;
 * Adds item as the last item in the list.
 *
 * void listDeleteItem(item, list)
 * char *item;
 * LIST_PTR list;
 * Removes item from list.
 *
 * void listDeleteItemAll(item, list)
 * char *item;
 * LIST_PTR list;
 * Removes all elements containing item from list.
 *
 * void listTestDeleteItem(func, param, list)
 * int (*func)();
 * char *param;
 * LIST_PTR list;
 * Removes the first item in the list found such that func(param, item)
 * returns 1 (TRUE).
 *
 * void listTestDeleteItemAll(func, param, list)
 * int (*func)();
 * char *param;
 * LIST_PTR list;
 * Removes all items in the list found such that func(param, item)
 * returns 1 (TRUE).
 *
 * int listMemberItem(item, list)
 * char *item;
 * LIST_PTR list;
 * Returns 1 (TRUE) if item is in the list, otherwise 0 (FALSE) is returned.
 *
 * char *listMemReturnItem(func, param, list)
 * int (*func)();
 * char *param;
 * LIST_PTR list;
 * listMemReturnItem is a more general form of listMemberItem.
 * listMemReturnItem will return the item (or one of the items) in list
 * for which func(param, item) is non-zero, i.e. is TRUE.
 * The function takes two arguments, the first is the param and the second is 
 * an item of the list and returns an integer value. int func(param, item).
 * If the functions does not satisfy any of the items in the list NULL
 * is returned.
 *
 * int listIterateFromFirst(func, param, list)
 * int (*func)();
 * char *param;
 * LIST_PRT list;
 *
 * int listIterateFromLast(func, param, list)
 * int (*func)();
 * char *param;
 * LIST_PRT list;
 *
 * listIterateFromFirst and listIterateFromLast will call the function
 * func with param on all of its elements stoping when the list is
 * finished or when func returns 0 (ie FALSE). The function func will
 * take two arguments the first is param the second is an item of the
 * set. func(param, item).  listIterate returns 0 (FALSE) if the function
 * func returns 0 (FALSE).  Otherwise listIterate returns 1 (TRUE).
 * listIterate will return 0 (FASLE) if the list is NULL.
 *
 * listIterateFromFirst starts iteration from the first item in the list going
 * forward through the list.
 *
 * listIterateFromLast starts iteration from the last item in the list going
 * backwards through the list.
 *
 * char *listPopItem(list)
 * LIST_PTR list
 * Removes and returns first item from list.
 *
 * REVISION HISTORY:
 *
 * $Log: list.c,v $
 * Revision 1.16  1996/06/25  20:50:49  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.15  1996/03/15  21:19:50  reids
 * Update "next" pointer in list when doing deleting items.
 *
 * Revision 1.14  1996/02/10  16:50:06  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.13  1996/01/12  01:02:37  rich
 * Need to return created list.
 *
 * Revision 1.12  1996/01/12  00:53:15  rich
 * Simplified GNUmakefiles.  Fixed some dbmalloc problems.
 *
 * Revision 1.11  1995/12/17  20:21:35  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.10  1995/12/15  01:23:16  rich
 * Moved Makefile to Makefile.generic to encourage people to use
 * GNUmakefile.
 * Fixed a memory leak when a module is closed and some other small fixes.
 *
 * Revision 1.9  1995/07/06  21:16:47  rich
 * Solaris and Linux changes.
 *
 * Revision 1.8  1995/04/07  05:03:21  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.7  1995/04/04  19:42:29  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.6  1995/01/18  22:41:03  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.5  1993/12/14  17:33:59  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.4  1993/11/21  20:18:11  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  08:38:44  fedor
 * Pass 2 aat a V7+V6+VxWorks merge. Many many problems with pointless casting.
 *
 * Revision 1.2  1993/08/27  07:15:24  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:17:58  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:25  rich
 * Importing tca version 8
 *
 * Revision 1.2  1993/05/19  17:24:24  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 8-Dec-91  Christopher Fedor at School of Computer Science, CMU
 * Added nextTmp so that listDelete can be used from within listMemReturnItem.
 *
 * 21-Aug-91  Christopher Fedor at School of Computer Science, CMU
 * Added the routine listInsertItemAfter.
 *
 * 22-Dec-90  Christopher Fedor at School of Computer Science, CMU
 * Added tests so that nill can not be inserted in a list.
 * This avoids possible confusion in list iterate and delete functions.
 *
 * 11-Dec-90  Christopher Fedor at School of Computer Science, CMU
 * Added item tests in list iterate so that holes created by listDelete 
 * do no kill the func call. - this may still need work.
 *
 *  3-Oct-90  Christopher Fedor at School of Computer Science, CMU
 * Added nextTmp and previousTmp to list iterate routines so that
 * they terminate gracefully if there test function calls listDelete.
 * This was needed for the removeConnection routine which is called from
 * within a listIterateFromFirst. Just goes to show that there is
 * still much trouble with interactions among generic data structure 
 * routines.
 *
 *  5-Apr-90  Christopher Fedor at School of Computer Science, CMU
 * Added listFreeListGlobal as a list of free top level list elements.
 * Also added warning code for those routines who insist on freeing
 * already freed lists.
 *
 * 13-Nov-89  Christopher Fedor at School of Computer Science, CMU
 * Added listCellFreeListGlobal as a list element free list.
 *
 * 16-Oct-89  Christopher Fedor at School of Computer Science, CMU
 * Added listPushItem, in list.h,  and listPopItem.
 *
 * 13-Oct-89  Christopher Fedor at School of Computer Science, CMU
 * Added listTestDeleteItem and listTestDeleteItemAll routines.
 *
 * 10-Oct-89  Christopher Fedor at School of Computer Science, CMU
 * Added membership and iteration routines so this list abstract 
 * data type can be used as a simple set abstract data type.
 *
 *  9-Oct-89  Christopher Fedor at School of Computer Science, CMU
 * Rewrote to form basis for sets and queues. Updated to Software Standards.
 * Based routine names on functionality, got rid of lispish names.
 * Removed Long-Ji's test for empty list because it was a redundant test
 * - sounds like the real problem was soleved by a correction elsewhere.
 *
 * 29-Aug-89  Long-Ji Lin at School of Computer Science, CMU
 * Modified "DoList" and "DoList1" to make "body" go 
 * after "list_var = Cdr(list_var)"
 *
 * 24-Aug-89  Long-Ji Lin at School of Computer Science, CMU
 * Added a check to return from listDeleteItem (was SetDelete) if the list
 * exists but is empty.
 *
 * Ancient    Reid Simmons at School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.16 $
 * $Date: 1996/06/25 20:50:49 $
 * $Author: rich $
 *
 **************************************************************************/

#include "globalM.h"

/******************************************************************************
 * Forward Declarations
 *****************************************************************************/

#if !defined(DBMALLOC)
LIST_PTR listDBCreate(const char* file, int line);
#else
#undef listCreate
LIST_PTR listCreate(void);
#endif


/**************************************************************************
 *
 * FUNCTION: listIncFreeList()
 *
 * DESCRIPTION: Increment list top level free list.
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: If malloc returns NULL, call tcaError.
 *
 * DESIGN: Malloc LIST_INC_AMOUNT number of list elments.
 *
 * NOTES: listFreeListGlobal points to the newly created elements.
 *
 **************************************************************************/

#if defined(DBMALLOC)
static void listDBIncFreeList(const char* file, int line)
#define listIncFreeList() listDBIncFreeList(__FILE__,__LINE__)
#else
static void listIncFreeList(void)
#endif
{
  int32 i;
  LIST_PTR newList;
  
  for(i=1;i < LIST_INC_AMOUNT;i++) {
#if defined(DBMALLOC)
    newList = NEW_DB(file,line,LIST_TYPE);
#else
    newList = NEW(LIST_TYPE);
#endif
    
    if (!newList) {
      tcaModError("Error: Can not increment list top level free list.");
      return;
    }
    
    newList->length = 0;
    newList->first = NULL;
    newList->last = NULL;
    newList->next = NULL;
    
    newList->freeList = GET_M_GLOBAL(listFreeListGlobal);
    GET_M_GLOBAL(listFreeListGlobal) = newList;
  }
}


/**************************************************************************
 *
 * FUNCTION: listIncCellFreeList()
 *
 * DESCRIPTION: Increment list element free list.
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: If malloc returns NULL, call tcaError.
 *
 * DESIGN: Malloc LIST_CELL_INC_AMOUNT number of list elments.
 *
 * NOTES: listCellFreeListGlobal points to the newly created elements.
 *
 **************************************************************************/

static void listIncCellFreeList(void)
{
  int32 i;
  LIST_ELEM_PTR newCell;
  
  for(i=1;i<LIST_CELL_INC_AMOUNT;i++) {
    newCell = NEW(LIST_ELEM_TYPE);
    
    if (!newCell) {
      tcaModError("Error: Can not increment list element free list.");
      return;
    }
    
    newCell->item = NULL;
    newCell->previous = NULL;
    
    newCell->next = GET_M_GLOBAL(listCellFreeListGlobal);
    GET_M_GLOBAL(listCellFreeListGlobal) = newCell;
  }
}


/**************************************************************************
 *
 * FUNCTION: void listFreeTop(list)
 *
 * DESCRIPTION: Initializes a list top level and returns it to the free list.
 *
 * INPUTS: LIST_PTR list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Initializes the cell and inserts it as the first item to 
 * listFreeListGlobal.
 *
 * NOTES:
 *
 **************************************************************************/

static void listFreeTop(LIST_PTR list)
{
  list->length = 0;
  list->first = NULL;
  list->last = NULL;
  list->next = NULL;
  
  list->freeList = GET_M_GLOBAL(listFreeListGlobal);
  GET_M_GLOBAL(listFreeListGlobal) = list;
}


/**************************************************************************
 *
 * FUNCTION: void listFreeCell(listCell)
 *
 * DESCRIPTION: Initializes a list element and returns it to the free list.
 *
 * INPUTS: LIST_ELEM_PTR listCell.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Initializes the cell and inserts it as the first item to 
 * listCellFreeListGlobal.
 *
 * NOTES:
 *
 **************************************************************************/

static void listFreeCell(LIST_ELEM_PTR listCell)
{
  listCell->item = NULL;
  listCell->previous = NULL;
  
  listCell->next = GET_M_GLOBAL(listCellFreeListGlobal);
  GET_M_GLOBAL(listCellFreeListGlobal) = listCell;
}


/**************************************************************************
 *
 * FUNCTION: void listFree(list)
 *
 * DESCRIPTION: Frees a list.
 *
 * INPUTS: LIST_PTR list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Call listFreeCell on each list element.
 *
 * NOTES: 
 *
 **************************************************************************/

void listFree(LIST_PTR *list)
{
  LIST_ELEM_PTR tmpA, tmpB;
  
  if (!(*list))
    return;
  
  if ((*list)->freeList) {
    tcaModWarning(
		  "\nlistFree: OP IGNORED WARNING: list already on free list.\n");
    return;
  }
  
  tmpA = (*list)->first;
  
  while (tmpA) {
    tmpB = tmpA;
    tmpA = tmpA->next;
    
    listFreeCell(tmpB);
  }
  
  listFreeTop((*list));
  *list = NULL;
}    


/**************************************************************************
 *
 * FUNCTION: void listFreeMaster(list)
 *
 * DESCRIPTION: Frees a master list of lists.
 *
 * INPUTS: LIST_PTR list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Call listFreeCell on each list element.
 *
 * NOTES: 
 *
 **************************************************************************/

static void listFreeMaster(LIST_PTR *list)
{
  if (!(*list))
    return;
  
  listFreeMaster(&(*list)->freeList);
  tcaFree((char *)*list);
  *list = NULL;
}    


/**************************************************************************
 *
 * FUNCTION: void listInsertItemFirst(item, list)
 *
 * DESCRIPTION: Adds item as the first item in the list.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a list element to store item and insert the element as the first item
 * in the list.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void listInsertItemFirst(const void *item, LIST_PTR list)
{
  LIST_ELEM_PTR element;
  
  if (!item || !list)
    return;
  
  if (!GET_M_GLOBAL(listCellFreeListGlobal))
    listIncCellFreeList();
  
  element = GET_M_GLOBAL(listCellFreeListGlobal);
  GET_M_GLOBAL(listCellFreeListGlobal) = 
    GET_M_GLOBAL(listCellFreeListGlobal)->next;
  
  element->item = item;
  element->next = list->first;
  element->previous = NULL;
  
  if (!list->first) {
    list->first = element;
  }
  else {
    list->first->previous = element;
  }
  
  if (!list->last) {
    list->last = element;
  }
  
  list->length++;
  list->first = element;
}


/**************************************************************************
 *
 * FUNCTION: void listInsertItemLast(item, list)
 *
 * DESCRIPTION: Adds item as the last item in the list.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a list element to store item and insert the element as the first item
 * in the list.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void listInsertItemLast(const void *item, LIST_PTR list)
{
  LIST_ELEM_PTR element;
  
  if (!item || !list)
    return;
  
  if (!GET_M_GLOBAL(listCellFreeListGlobal))
    listIncCellFreeList();
  
  element = GET_M_GLOBAL(listCellFreeListGlobal);
  GET_M_GLOBAL(listCellFreeListGlobal) = 
    GET_M_GLOBAL(listCellFreeListGlobal)->next;
  
  element->item = item;
  element->next = NULL;
  element->previous = list->last;
  
  if (!list->first) {
    list->first = element;
  }
  
  if (!list->last) {
    list->last = element;
  }
  else {
    list->last->next = element;
  }
  
  list->length++;
  list->last = element;
}



/**************************************************************************
 *
 * FUNCTION: void listInsertItemAfter(item, after, list)
 *
 * DESCRIPTION: Splices item into the list after, after.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * char *after - the item to be inserted after.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 * If after is NULL call listInsertItemFirst.
 * If after is not found call listInsertItemLast.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void listInsertItemAfter(const void *item, void *after, LIST_PTR list)
{
  LIST_ELEM_PTR element, tmp;
  
  if (!item || !list)
    return;
  
  if (!after) {
    listInsertItemFirst(item, list);
    return;
  }
  
  tmp = list->first;
  
  while (tmp && tmp->item != after) 
    tmp = tmp->next;
  
  if (!tmp) {
    listInsertItemLast(item, list);
    return;
  }
  
  if (!GET_M_GLOBAL(listCellFreeListGlobal))
    listIncCellFreeList();
  
  element = GET_M_GLOBAL(listCellFreeListGlobal);
  GET_M_GLOBAL(listCellFreeListGlobal) = 
    GET_M_GLOBAL(listCellFreeListGlobal)->next;
  
  element->item = item;
  element->next = tmp->next;
  element->previous = tmp;
  
  tmp->next = element;
  
  list->length++;
}

/**************************************************************************
 *
 * FUNCTION: char *listPopItem(list)
 *
 * DESCRIPTION: Removes and returns first item from list.
 *
 * INPUTS: 
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS:
 * The first item or NULL
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return NULL.
 *
 * DESIGN: 
 * Remove the first list element, return the item and free the element.
 *
 * NOTES:
 * might want to start a free list of list elements.
 *
 **************************************************************************/

const void *listPopItem(LIST_PTR list)
{
  const char *item;
  LIST_ELEM_PTR oldElement;
  
  item = NULL;
  
  if (list && list->first) {
    item = list->first->item;
    oldElement = list->first;
    list->first = list->first->next;
    if (list->first) {
      list->first->previous = NULL;
    }
    if (list->last == oldElement) {
      list->last = NULL;
    }
    list->length--;
    listFreeCell(oldElement);
  }
  
  return item;
}



void listTestDeleteItem(LIST_ITER_FN func, const void *param, LIST_PTR list)
{
  LIST_ELEM_PTR current, previous;
  
  if (!list || !list->first)
    return;
  
  current = previous = list->first;
  
  if ((*func)(param, current->item)) {
    /* item is the first element of the list */
    if (list->last == current) {
      list->last = NULL;
    }
    list->first = current->next;

      /* 15-Mar-96 RGS: Maintain the list->next pointer correctly */
    if (list->next == current) 
      list->next = current->next;

    if (current->next) {
      current->next->previous = NULL;
    }
    list->length--;
    listFreeCell(current);
    return;
  }
  
  current = current->next;
  
  while (current) {
    if ((*func)(param, current->item)) {
      if (list->last == current) {
	list->last = previous;
      }

      /* 15-Mar-96 RGS: Maintain the list->next pointer correctly */
      if (list->next == current)
	list->next = current->next;

      current->previous = previous;
      previous->next = current->next;
      
      if (current->next) {
	current->next->previous = previous;
      }
      
      list->length--;
      listFreeCell(current);
      return;
    }
    previous = current;
    current = current->next;
  }
}


/**************************************************************************
 *
 * FUNCTION: void listTestDeleteItemAll(func, param, list)
 *
 * DESCRIPTION: 
 * Removes all items in the list found such that func(param, item)
 * returns 1 (TRUE).
 *
 * INPUTS: 
 * LIST_ITER_FN func - pointer to a test function of the form func(param, item)
 * char *param - a pointer to a parameter for func.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return;
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item,
 * such that func(param, item) returns 1 (TRUE). If found
 * the list element is removed and freed. All the items of the list
 * are tested. Reset element's previous pointer and list->last if needed.
 *
 * NOTES: item is not freed. Use listDeleteItem to delete a single occurance
 * of item. Modified from listDeleteItem.
 *
 **************************************************************************/

void listTestDeleteItemAll(LIST_ITER_FN func, const void *param, LIST_PTR list)
{
  LIST_ELEM_PTR current, previous;
  
  if (!list || !list->first)
    return;
  
  while (list->first && (*func)(param, list->first->item)) {
    list->length--;
    current = list->first;
    list->first = current->next;

    /* 15-Mar-96 RGS: Maintain the list->next pointer correctly */    
    if (list->next == current)
      list->next = current->next;

    if (list->first) {
      list->first->previous = NULL;
    }
    if (list->last == current) {
      list->last = NULL;
    }
    listFreeCell(current);
  }
  
  if (!list->first)
    return;
  
  previous = list->first;
  current  = list->first->next;
  
  while (current) {
    if ((*func)(param, current->item)) {
      if (list->last == current) {
	list->last = previous;
      }
      
      /* 15-Mar-96 RGS: Maintain the list->next pointer correctly */    
      if (list->next == current)
	list->next = current->next;

      previous->next = current->next;
      
      if (current->next) {
	current->next->previous = previous;
      }
      
      list->length--;
      listFreeCell(current);
      
      current = previous->next;
    }
    else {
      previous = current;
      current = current->next;
    }
  }
}


/**************************************************************************
 *
 * FUNCTION: void listFreeAllItems(func, list)
 *
 * DESCRIPTION: 
 * Applies the func to each item in the list while removing it.
 *
 * INPUTS: 
 * LIST_ITER_FN func - pointer to a function of the form func(item).
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return;
 *
 * DESIGN: 
 * Based on calling listTestDeleteItemAll.
 *
 * NOTES: 
 * The list itself is not freed.
 *
 **************************************************************************/

static int32 listApplyFunc(void (* func)(void *item), void *item)
{
  (*func)(item);
  
  return 1;
}

void listFreeAllItems(LIST_FREE_FN func, LIST_PTR list)
{
  listTestDeleteItemAll((LIST_ITER_FN)listApplyFunc, (const void *)func, list);
}


/**************************************************************************
 *
 * FUNCTION: int32 listItemEq(a, b)
 *
 * DESCRIPTION: Simple Equal Test for listDeleteItem.
 *
 * INPUTS:
 * char *a, *b;
 *
 * OUTPUTS: Returns 1 TRUE or 0 FALSE.
 *
 * DESIGN: return(a == b);
 *
 * NOTES:
 *
 **************************************************************************/

int32 listItemEq(void *a, void *b)
{
  return(a == b);
}


/**************************************************************************
 *
 * FUNCTION: listDeleteItem(item, list)
 *
 * DESCRIPTION: Removes an item from list.
 *
 * INPUTS:
 * char *item; 
 * LIST_PTR *list;
 *
 * OUTPUTS: none.
 *
 * DESIGN: call listTestDeleteItem with listItemEq test.
 *
 * NOTES: list is modified.
 *
 **************************************************************************/

void listDeleteItem(const void *item, LIST_PTR list)
{
  listTestDeleteItem((LIST_ITER_FN) listItemEq, item, list);
}


/**************************************************************************
 *
 * FUNCTION: listDeleteItemAll(item, list)
 *
 * DESCRIPTION: Removes an all such item from list.
 *
 * INPUTS:
 * char *item; 
 * LIST_PTR *list;
 *
 * OUTPUTS: none.
 *
 * DESIGN: call listTestDeleteItemAll with listItemEq test.
 *
 * NOTES: list is modified.
 *
 **************************************************************************/

void listDeleteItemAll(const void *item, LIST_PTR list)
{
  listTestDeleteItemAll((LIST_ITER_FN) listItemEq, item, list);
}


/**************************************************************************
 *
 * FUNCTION: int32 listMemberItem(item, list)
 *
 * DESCRIPTION: Tests if item is an element of list.
 *
 * INPUTS: 
 * char *item - a pointer to an item of data.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: 
 * 0 - FALSE 
 * 1 - TRUE
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return;
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item. 
 * If found the value 1 is returned, else the value 0 is returned.
 *
 * NOTES:
 *
 **************************************************************************/

int32 listMemberItem(const void *item, LIST_PTR list)
{
  LIST_ELEM_PTR tmp;
  
  if (!list)
    return 0; /* False */
  
  tmp = list->first;
  
  while (tmp) {
    if ((const void *)(tmp->item) == item)
      return 1; /* TRUE */
    tmp = tmp->next;
  }
  
  return 0; /* False */
}


/**************************************************************************
 *
 * FUNCTION: char *listMemReturnItem(func, param, list)
 *
 * DESCRIPTION:
 * listMemReturnItem is a more general form of listMemberItem.
 * listMemReturnItem will return the item (or one of the items) in list
 * for which func(param, item) is non-zero, i.e. is TRUE.
 * The function takes two arguments, the first is the param and the second is 
 * an item of the list and returns an integer value. int func(param, item).
 * If the functions does not satisfy any of the items in the list NULL
 * is returned.
 *
 * INPUTS: 
 * LIST_ITER_FN func;
 * char *param - a pointer to a parameter that is passed to func.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: 
 * A pointer to an item in the list that satisfies func(param, item) 
 * or NULL if no such item exists. 
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, NULL is returned.
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item, such that
 * func(param, item) is non-zero. Then return the item.
 *
 * NOTES:
 *
 **************************************************************************/

const void *listMemReturnItem(LIST_ITER_FN func, 
			      const void *param, 
			      LIST_PTR list)
{
  const LIST_ELEM_TYPE *tmp, *nextTmp;
  
  if (!list)
    return NULL;
  else {
    tmp = list->first;
    while (tmp) {
      nextTmp = tmp->next;
      if ((*func)(param, tmp->item))
	return tmp->item;
      else
	tmp = nextTmp;
    }
    return NULL;
  }
}


/**************************************************************************
 *
 * FUNCTION: int32 listIterateFromFirst(func, param, list)
 *
 * DESCRIPTION:
 * listIterateFromFirst will call the function func with param on all of its
 * elements stoping when the list is finished or when func returns 0 (ie
 * FALSE). The function func will take two arguments the first is 
 * param the second is an item of the set. func(param, item).
 *
 * listIterateFromFirst starts from the first item in the list and iterates
 * forward through the items in the list.
 *
 * INPUTS: 
 * LSIT_ITER_FN func;
 * char *param - a pointer to a parameter that is passed to func.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: 
 * listIterate returns 0 (FALSE) if the function func returns 0 (FALSE).
 * Otherwise listIterate returns 1 (TRUE).
 *
 * EXCEPTIONS: 
 *
 * listIterate will return 0 (FASLE) if the list is NULL.
 *
 * DESIGN: 
 * iterate through the list of elements calling func on each item.
 * return when the list is finished or func has returned 0 (FALSE).
 *
 * NOTES:
 *
 * 11-Dec-90: fedor: 
 * Added item test so holes generated by listDelete do not kill func call
 *
 *  3-Oct-90: fedor: 
 * nextTmp allows the iteration to terminate gracefully if the list element
 * is removed if func calls listDelete.
 *
 **************************************************************************/

int32 listIterateFromFirst(LIST_ITER_FN func, const void *param, LIST_PTR list)
{
  const char *item;
  LIST_ELEM_PTR tmp, nextTmp;
  
  if (!list)
    return 0;
  
  tmp = list->first;
  
  while (tmp) {
    item = tmp->item;
    nextTmp = tmp->next;
    if (item && !(*func)(param, item))
      return 0;
    tmp = nextTmp;
  }
  
  return 1;
}


/**************************************************************************
 *
 * FUNCTION: int listIterateFromLast(func, param, list)
 *
 * DESCRIPTION:
 * listIterateFromLast will call the function func with param on all of its
 * elements stoping when the list is finished or when func returns 0 (ie
 * FALSE). The function func will take two arguments the first is 
 * param the second is an item of the set. func(param, item).
 *
 * listIterateFromLast starts with thelast item in the list and iterates
 * backwards through the list.
 *
 * INPUTS: 
 * LIST_ITER_FN func;
 * char *param - a pointer to a parameter that is passed to func.
 * LIST_PTR list - a pointer to a list.
 *
 * OUTPUTS: 
 * listIterateFromLast returns 0 (FALSE) if the function func returns 0 (FALSE)
 * Otherwise listIterateFromLast returns 1 (TRUE).
 *
 * EXCEPTIONS: 
 *
 * listIterate will return 0 (FASLE) if the list is NULL.
 *
 * DESIGN: 
 * iterate through the list of elements calling func on each item.
 * return when the list is finished or func has returned 0 (FALSE).
 *
 * NOTES:
 *
 * 11-Dec-90: fedor: 
 * Added item test so holes generated by listDelete do not kill func call
 *
 *  3-Oct-90: fedor: 
 * previousTmp allows the iteration to terminate gracefully if the list element
 * is removed if func calls listDelete.
 *
 **************************************************************************/

int32 listIterateFromLast(LIST_ITER_FN func, const void *param, LIST_PTR list)
{
  const char *item;
  LIST_ELEM_PTR tmp, previousTmp;
  
  if (!list)
    return 0;
  
  tmp = list->last;
  
  while (tmp) {
    item = tmp->item;
    previousTmp = tmp->previous;
    if (item && !(*func)(param, item))
      return 0;
    tmp = previousTmp;
  }
  
  return 1;
}


/**************************************************************************
 *
 * FUNCTION: LIST_PTR listCopy(list)
 *
 * DESCRIPTION: Copies the given list.
 *
 * INPUTS: LIST_PTR list
 *
 * OUTPUTS: A pointer to the newly created list of type LIST_PTR.
 * If there is an error NULL is returned.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: Iterate through the original list inserting the items into the
 * new list.
 *
 * NOTES:
 *
 **************************************************************************/

static int32 listCopyInsert(LIST_PTR list, const void *item)
{
  listInsertItemFirst(item , list);
  
  return 1;
}

LIST_PTR listCopy(LIST_PTR list)
{
  LIST_PTR newList;
  
  newList = listCreate();
  
  (void)listIterateFromLast((LIST_ITER_FN)listCopyInsert, 
			    (char *)newList, list);
  
  return newList;
}


/**************************************************************************/

/* List interface routines */

void listInsertItem(const void *item, LIST_PTR list)
{
  listInsertItemFirst(item, list);
}

int32 listIterate(LIST_ITER_FN func, const void *param, LIST_PTR list)
{
  return(listIterateFromFirst(func, param, list));
}

int32 listLength(LIST_PTR list)
{
  if (!list)
    return 0;
  
  return(list->length);
}


int32 listEqual(LIST_PTR list1, LIST_PTR list2)
{
  int32 good;
  LIST_ELEM_PTR a, b;
  
  if (list1 == list2)
    return 1;
  
  /* this is the same style test used in tms.c but it is not general ! */
  
  a = list1->first;
  b = list2->first;
  
  good = 1;
  while (good && a && b) {
    good = (a->item == b->item);
    a = a->next;
    b = b->next;
  }
  
  return(good && (a == NULL) && (b == NULL));
}


LIST_PTR listMake1(const void *item)
{
  LIST_PTR list;
  
  list = listCreate();
  
  listInsertItem(item, list);
  
  return list;
}


LIST_PTR listMake2(const void *item1, const void *item2)
{
  LIST_PTR list;
  
  list = listCreate();
  listInsertItem(item2, list);
  listInsertItem(item1, list);
  
  return list;
}


const void *listFirst(LIST_PTR list)
{
  if (list && list->first) {
    list->next = list->first;
    return(list->first->item);
  }
  else
    return NULL;
}

const void *listLast(LIST_PTR list)
{
  if (list && list->last) {
    return(list->last->item);
  }
  else
    return NULL;
}

const void *listNext(LIST_PTR list)
{
  if (list && list->next) {
    list->next = list->next->next;
    
    if (list->next)
      return(list->next->item);
  }
  
  return NULL;
}


/**************************************************************************
 *
 * FUNCTION: void listCleanup(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: 
 *
 * NOTES:
 *
 **************************************************************************/

static void listFreeElement(LIST_ELEM_PTR element)
{
  if (element == NULL) return;
  listFreeElement(element->next);
  tcaFree((char *)element);
}

void listCleanup(void)
{
  listFreeElement(GET_M_GLOBAL(listCellFreeListGlobal));
  GET_M_GLOBAL(listCellFreeListGlobal) = NULL;
  /* Now need to really free the master list of lists */
  listFreeMaster(&GET_M_GLOBAL(listFreeListGlobal));
}


/**************************************************************************
 *
 * FUNCTION: LIST_PTR listCreate()
 *
 * DESCRIPTION: Creates an empty list.
 *
 * INPUTS: none.
 *
 * OUTPUTS: A pointer to the newly created empty list of type LIST_PTR.
 * If there is an error NULL is returned.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: malloc up storage for the list, checking to see if space was
 * successfully allocated, and initialize the list structure.
 *
 * NOTES:
 *
 **************************************************************************/

#if defined(DBMALLOC)
LIST_PTR listCreate(void) 
{return listDBCreate("Unknown",-1);}

LIST_PTR listDBCreate(const char* file, int line)

#else
LIST_PTR listDBCreate(const char* file, int line)
{ return listCreate();}

LIST_PTR listCreate(void)
#endif
{
  LIST_PTR list;
  
  if (!GET_M_GLOBAL(listFreeListGlobal)) {
#if defined(DBMALLOC)
    listDBIncFreeList(file,line);
#else
    listIncFreeList();
#endif
  }
  
  list = GET_M_GLOBAL(listFreeListGlobal);
  GET_M_GLOBAL(listFreeListGlobal) = 
    GET_M_GLOBAL(listFreeListGlobal)->freeList;
  
  list->freeList = NULL;
  
  return list;
}
