/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: lex
 *
 * FILE: lex.c
 *
 * ABSTRACT:
 * 
 * primitve lexer
 *
 * REVISION HISTORY
 *
 * $Log: lex.c,v $
 * Revision 1.13  1996/05/09  18:31:04  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.12  1996/03/15  21:18:08  reids
 * Added support for "enum" format type.
 *   Also, printData was not counting characters per line correctly.
 *
 * Revision 1.11  1996/03/05  05:04:33  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.10  1996/01/27  21:53:32  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.9  1995/01/18  22:40:55  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.8  1994/04/16  19:42:24  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1993/12/14  17:33:54  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/11/21  20:18:05  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:13:58  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/27  07:15:18  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/06/13  23:28:11  rich
 * Made changes for lisp needed for vx works style global variables.
 * Fixed some random compiler warnings.
 * Moved test routines to test directory.
 *
 * Revision 1.2  1993/05/26  23:17:48  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:24  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:30:19  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:15  fedor
 * Added Logging.
 *
 * 13-Feb-89 Christopher Fedor, School of Computer Science, CMU
 * Changed to lex the whole string at once.
 *
 * 10-Feb-89 Christopher Fedor, School of Computer Science, CMU
 * Created.
 *
 * NOTES: 
 * 17-Jun-91: fedor:
 * Needs to be rewritten to take advantage of abstract data type routines
 * and to start looking at memory use. Blah this is a mess!
 *
 * $Revision: 1.13 $
 * $Date: 1996/05/09 18:31:04 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"

#include "lex.h"
#include "tcaMem.h"

static TokenPtr ReadString(const char *s)
{
  const char *t2;
  TokenPtr tmp;
  int i, amount;
  
  for (i=GET_SM_GLOBAL(Location);
       (s[i] != ',' && s[i] != '}' && s[i] != '{' && s[i] != '[' 
	&& s[i] != ']' && s[i] != ':' && s[i] != ' ' && s[i] != '>'
	&& s[i] != '<' && s[i] != '\0');
       i++);
  
  amount = i-GET_SM_GLOBAL(Location);
  NEW_TOKEN(tmp, STR_TOK);
  tmp->value.str = (char *)tcaMalloc(sizeof(char)*(unsigned)(amount+1));
  t2 = s+GET_SM_GLOBAL(Location);
  strncpy(tmp->value.str, t2, amount);
  tmp->value.str[amount] = '\0';
  
  GET_SM_GLOBAL(Location) += amount;
  
  return tmp;
}

static TokenPtr ReadInt(const char *s)
{
  char *t;
  const char *t2;
  TokenPtr tmp;
  int i, amount;
#if defined(VXWORKS)
  int num;
#endif
  
  for (i=GET_SM_GLOBAL(Location);
       (isdigit(s[i]) && s[i] != ',' && s[i] != '}' && s[i] != '{' 
	&& s[i] != '[' && s[i] != ']' && s[i] != ':' && s[i] != ' ' 
	&& s[i] != '>' && s[i] != '<' && s[i] != '\0');
       i++);
  
  amount = i-GET_SM_GLOBAL(Location);
  NEW_TOKEN(tmp, INT_TOK);
  t = (char *)tcaMalloc(sizeof(char)*(unsigned)(amount+1));
  t2 = s+GET_SM_GLOBAL(Location);
  strncpy(t, t2, amount);
  t[amount] = '\0';
  
#ifndef VXWORKS
  tmp->value.num = atoi(t);
#else
  sscanf(t, "%d", &num);
  tmp->value.num = num;
#endif
  
  tcaFree(t);
  
  GET_SM_GLOBAL(Location) += amount;
  
  return tmp;
}    

static TokenPtr BuildToken(const char *s)
{
  TokenPtr Token;
  
  if (!s) {
    NEW_TOKEN(Token, EOS_TOK);
    return Token;
  }
  
 Start:
  switch(s[GET_SM_GLOBAL(Location)]) {
  case ' ':
  case '\t':
  case '\f':
    GET_SM_GLOBAL(Location)++;
    goto Start;
  case '{':
    NEW_TOKEN(Token, LBRACE_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '}':
    NEW_TOKEN(Token, RBRACE_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '[':
    NEW_TOKEN(Token, LBRACK_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case ']':
    NEW_TOKEN(Token, RBRACK_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '*':
    NEW_TOKEN(Token, STAR_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '!':
    NEW_TOKEN(Token, BANG_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case ',':
    NEW_TOKEN(Token, COMMA_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '<':
    NEW_TOKEN(Token, LT_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '>':
    NEW_TOKEN(Token, GT_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case ':':
    NEW_TOKEN(Token, COLON_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  case '\0':
    NEW_TOKEN(Token, EOS_TOK);
    GET_SM_GLOBAL(Location)++;
    break;
  default:
    if (isdigit(s[GET_SM_GLOBAL(Location)]))
      Token = ReadInt(s);
    else 
      Token = ReadString(s);
  }
  
  return Token;
}

#if 0
/* No longer used */
static void PrintToken(TokenPtr Token)
{
  tcaModWarning( "Loc: %d: Type: ", Token->Loc);
  
  switch(Token->Type) {
  case LBRACE_TOK:
    tcaModWarning( "LBRACE_TOK\n");
    break;
  case RBRACE_TOK:
    tcaModWarning( "RBRACE_TOK\n");
    break;
  case LBRACK_TOK:
    tcaModWarning( "LBRACK_TOK\n");
    break;
  case RBRACK_TOK:
    tcaModWarning( "RBRACK_TOK\n");
    break;
  case STAR_TOK:
    tcaModWarning( "STAR_TOK\n");
    break;
  case BANG_TOK:
    tcaModWarning( "BANG_TOK\n");
    break;
  case COMMA_TOK:
    tcaModWarning( "COMMA_TOK\n");
    break;
  case LT_TOK:
    tcaModWarning( "LT_TOK\n");
    break;
  case GT_TOK:
    tcaModWarning( "GT_TOK\n");
    break;
  case COLON_TOK:
    tcaModWarning( "COLON_TOK\n");
    break;
  case EOS_TOK:
    tcaModWarning( "EOS_TOK\n");
    break;
  case INT_TOK:
    tcaModWarning( "INT_TOK: %d\n", Token->value.num);
    break;
  case STR_TOK:
    tcaModWarning( "STR_TOK: %s\n", Token->value.str);
    break;
  case PRIMITIVE_TOK:
    tcaModWarning( "PRIMITIVE_TOK: %s\n", Token->value.str);
    break;
#ifndef TEST_CASE_COVERAGE
  default:
    tcaModWarning( "Token of Unknown type: %d\n", Token->Type);
    break;
#endif
  }
}
#endif

#if 0
/* No longer used */
static void PrintTokenList(TokenPtr Token)
{
  TokenPtr tmp;
  
  tmp = Token;
  while (tmp) {
    PrintToken(tmp);
    tmp = tmp->next;
  }
}
#endif

void LexString(const char *s)
{
  TokenPtr Token;
  
  GET_SM_GLOBAL(Location) = 0;
  
  GET_SM_GLOBAL(TokenList) = BuildToken(s);
  
  Token = GET_SM_GLOBAL(TokenList);
  while (Token->Type != EOS_TOK) {
    Token->next = BuildToken(s);
    Token = Token->next;
  }
}

TokenPtr NextToken(void)
{
  TokenPtr tmp;
  
  tmp = GET_SM_GLOBAL(TokenList);
  GET_SM_GLOBAL(TokenList) = GET_SM_GLOBAL(TokenList)->next;
  
  return tmp;
}

void UngetToken(TokenPtr token)
{
  token->next = GET_SM_GLOBAL(TokenList);
  GET_SM_GLOBAL(TokenList) = token;
}
