/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: key
 *
 * FILE: key.c
 *
 * ABSTRACT:
 * 
 * Collection of hash and key functions for table lookups.
 *
 * REVISION HISTORY
 *
 * $Log: key.c,v $
 * Revision 1.10  1996/02/21  18:36:47  rich
 * Remove null keys.
 *
 * Revision 1.9  1996/02/21  18:30:25  rich
 * Created single event loop.
 *
 * Revision 1.8  1995/08/06  16:43:56  reids
 * A bug existed in that two demon monitors that sent the same ID number
 * would conflict (causing the wrong one to fire).  This has been fixed, and
 * in the process, one of the hash-key functions was made a bit more general.
 *
 * Revision 1.7  1995/07/10  16:17:34  rich
 * Interm save.
 *
 * Revision 1.6  1995/01/18  22:40:53  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.5  1994/05/17  23:16:07  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.4  1993/11/21  20:18:01  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:15:13  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:17:43  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:24  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:30:16  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:09  fedor
 * Added Logging.
 *
 * 16-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.10 $
 * $Date: 1996/02/21 18:36:47 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include "key.h"


/******************************************************************************
 *
 * FUNCTION: int intHashFunc(i)
 *
 * DESCRIPTION: Return int value as hash value.
 *
 * INPUTS: int *i;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 intHashFunc(int32 *i)
{
  return(*i);
}


/******************************************************************************
 *
 * FUNCTION: int intKeyEqFunc(a, b)
 *
 * DESCRIPTION: Return comparison of the value of int pointers a and b
 *
 * INPUTS: 
 * int32 *a, *b;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 intKeyEqFunc(int32 *a, int32 *b)
{
  return(*a == *b);
}


/******************************************************************************
 *
 * FUNCTION: int strHashFunc(s)
 *
 * DESCRIPTION: Calculate a simple hash of a NULL terminated string.
 *
 * INPUTS: char *s
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 strHashFunc(const char *s)
{
  int sum, i;
  char c;

  sum = 0;
  for (i=0;s[i] != '\0';i++) {
    c = s[i];
    sum += (isupper(c) ? tolower(c) : c);
  }

  return sum;
}


/******************************************************************************
 *
 * FUNCTION: int strKeyEqFunc(a, b)
 *
 * DESCRIPTION: 
 * Return 1 (TRUE) if strings a and b are equal. Returns 0 (FASLE) otherwise.
 * Case insensitive compare.
 *
 * INPUTS: char *a, *b
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 strKeyEqFunc(const char *a, const char *b)
{
  int i;
  char a1, b1;
  
  i = 0;
  while (a[i] != '\0' && b[i] != '\0') {
    a1 = a[i];
    if (isupper(a1)) a1 = tolower(a1);

    b1 = b[i];
    if (isupper(b1)) b1 = tolower(b1);

    if (a1 != b1)
      return FALSE;
    
    i++;
  }
  
  return (a[i] == b[i]);
}


/******************************************************************************
 *
 * FUNCTION: int intStrHashFunc(key)
 *
 * DESCRIPTION: Calculate a simple hash of an integer and a
 *              NULL terminated string.
 *
 * INPUTS: INT_STR_KEY_PTR key;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 intStrHashFunc(INT_STR_KEY_PTR key)
{
  return (key->num + strHashFunc(key->str));
}

/******************************************************************************
 *
 * FUNCTION: int intStrKeyEqFunc(keyA, keyB)
 *
 * DESCRIPTION: 
 * Return 1 (TRUE) if keyA and keyB are equal (case insensitive compare).
 * Returns 0 (FASLE) otherwise. 
 *
 * INPUTS: INT_STR_KEY_PTR keyA, keyB;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 intStrKeyEqFunc(INT_STR_KEY_PTR keyA, INT_STR_KEY_PTR keyB)
{
  return ((keyA->num == keyB->num) && strKeyEqFunc(keyA->str, keyB->str));
}

/******************************************************************************
 *
 * FUNCTION: int classHashFunc(className)
 *
 * DESCRIPTION: Class hash function.
 *
 * INPUTS: TCA_MSG_CLASS_TYPE *className;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 classHashFunc(TCA_MSG_CLASS_TYPE *className)
{
  return ((int32)*className);
}


/******************************************************************************
 *
 * FUNCTION: int classEqFunc(classA, classB)
 *
 * DESCRIPTION: Class key equal function.
 *
 * INPUTS: TCA_MSG_CLASS_TYPE *classA, *classB;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

int32 classEqFunc(TCA_MSG_CLASS_TYPE *classA, TCA_MSG_CLASS_TYPE *classB)
{
  return (*classA == *classB);
}
