/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: idTable
 *
 * FILE: idtable.h
 *
 * ABSTRACT:
 * 
 * Generic dynamic array abstract data type.
 *
 * REVISION HISTORY
 *
 * $Log: idtable.c,v $
 * Revision 1.9  1996/06/25  20:50:45  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.8  1995/08/14  21:31:53  rich
 * Got rid of the "sharedBuffers" flag on the dataMessages.  It was not the
 * right solution, and possibly caused a memory leak.
 * Limit pending for TCA_TAPPED_MSG_VAR to 1.
 *
 * Revision 1.7  1995/07/10  16:17:31  rich
 * Interm save.
 *
 * Revision 1.6  1995/07/06  21:16:38  rich
 * Solaris and Linux changes.
 *
 * Revision 1.5  1995/01/18  22:40:50  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.4  1994/05/25  04:57:36  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.3  1993/08/27  07:15:07  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:17:39  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:24  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:30:09  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:24:05  fedor
 * Added Logging.
 *
 * 25-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 *
 * 20-Feb-89 Christopher Fedor, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.9 $
 * $Date: 1996/06/25 20:50:45 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include "idtable.h"

ID_TABLE_PTR idTableCreate(char *name, int32 incSize)
{
  ID_TABLE_PTR idTable;
  
  idTable = NEW(ID_TABLE_TYPE);
  
  idTable->name = name;
  idTable->incSize = incSize;
  idTable->freeTotal = 0;
  idTable->currentSize = 0;
  idTable->table = NULL;
  idTable->freeList = NULL;
  
  return idTable;
}

void idTableFree(ID_TABLE_PTR *table)
{
  if (table == NULL) return;
  if (*table) {
    if ((*table)->table) {
      
      tcaFree((char *)(*table)->table);
    }
    tcaFree((char *)*table);
  }
  *table=NULL;
}

static void idTableIncrement(ID_TABLE_PTR table)
{
  int i, newTableSize, currentSize, newFreeIds;
  TELEM_PTR tmp, newTable, oldTable, newFreeList, lastNewFreeBlock;
  
  newFreeIds = table->incSize;
  currentSize = table->currentSize;
  
  newTableSize = currentSize+newFreeIds;
  
  oldTable = table->table;
  newTable = (TELEM_TYPE *)tcaMalloc(sizeof(TELEM_TYPE)*
				     (unsigned)newTableSize);
  
  i = 0;
  
  if (oldTable) {
    /* Copy Old Table */
    for (;i<currentSize;i++)
      newTable[i] = oldTable[i];
    tcaFree((char *)oldTable);
  }
  
  /* Update Free List and Initialize New Blocks */
  newFreeList = NULL;
  lastNewFreeBlock = newTable + i;
  for (;i<newTableSize;i++) {
    tmp = newTable + i;
    tmp->data = NULL;
    tmp->ref = i;
    tmp->next = newFreeList;
    newFreeList = tmp;
  }
  
  /* Adjust Table Entry */
  lastNewFreeBlock->next = table->freeList;
  table->freeList = newFreeList;
  
  table->freeTotal += newFreeIds;
  table->currentSize = newTableSize;
  
  table->table = newTable;
}

int32 idTableInsert(void *item, ID_TABLE_PTR table)
{
  TELEM_PTR tmp;
  
  if (!table->freeList)
    idTableIncrement(table);
  
  tmp = table->freeList;
  table->freeList = tmp->next;
  tmp->next = NULL;
  table->freeTotal--;
  
  tmp->data = item;
  
  return(tmp->ref);
}

void *idTableRemove(int32 id, ID_TABLE_PTR table)
{
  char *data;
  TELEM_PTR tmp;
  
  tmp = table->table+id;
  
  data = tmp->data;
  
  if (!tmp->next) {
    /* Not Already on Free List */
    tmp->data = NULL;
    tmp->next = table->freeList;
    table->freeList = tmp;
    table->freeTotal++;
  }
  
  return data;
}


void idTablePrintInfo(ID_TABLE_PTR table)
{
  if (table) {
    tcaModWarning( "idTable: %s\n", table->name);
    tcaModWarning( "free   : %d\n", table->freeTotal);
    tcaModWarning( "size   : %d\n", table->currentSize);
    tcaModWarning( "incSize: %d\n", table->incSize);
  }
  else
    tcaModWarning( "idTablePrintInfo: NULL Table\n");
}

int32 idTableSearch(void *item, ID_TABLE_PTR table)
{
  int i;
  
  for (i=0; i<table->currentSize; i++)
    {
      if (table->table == item)
	return i;
    }
  return -1;
}
