/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1994.
 * 
 * MODULE: communications
 *
 * FILE: globalVar.c
 *
 * ABSTRACT:
 * 
 * Provides a system global variables.
 *
 * REVISION HISTORY
 *
 * $Log: globalVar.c,v $
 * Revision 1.13  1996/06/25  20:50:41  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.12  1996/06/05  19:47:53  reids
 * Registering global vars freed memory that was later being used.
 *
 * Revision 1.11  1996/03/02  03:21:39  rich
 * Fixed memory leaks found using purify.
 *
 * Revision 1.10  1996/02/14  22:12:27  rich
 * Eliminate extra variable logging on startup.
 *
 * Revision 1.9  1995/10/07  19:07:22  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.8  1995/07/19  14:26:20  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.7  1995/07/06  21:16:33  rich
 * Solaris and Linux changes.
 *
 * Revision 1.6  1995/04/04  19:42:23  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.5  1995/03/30  15:43:08  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.4  1995/03/28  01:14:39  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.3  1994/10/25  17:09:58  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.2  1994/05/25  04:57:29  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.1  1994/05/17  23:16:00  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 *
 * $Revision: 1.13 $
 * $Date: 1996/06/25 20:50:41 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalS.h"
#include "modVar.h"

/* Private Global types */

typedef struct _var {
  const char *varName;
  const char *format;
  void *value;
} GLOBAL_VAR_TYPE, *GLOBAL_VAR_PTR;

typedef const GLOBAL_VAR_TYPE *CONST_GLOBAL_VAR_PTR;


/*****************************************************************************
 *
 * FUNCTION: static void setVarHnd(DISPATCH_PTR dispatch, void *varValue)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void setVarHnd(DISPATCH_PTR dispatch, void *varValue)
{
  GLOBAL_VAR_PTR var=NULL;
  const char *msgName = dispatch->msg->msgData->name;
  const char *varName;
  char *watchMsgName;
  
  varName = &(msgName[strlen(VAR_SET_PREFIX)]);
  
  var = (GLOBAL_VAR_PTR) hashTableFind(varName, GET_S_GLOBAL(varTable));
  if (var == NULL) {
    /* handle the error here. */
  } else {
    /* Free the old data if it exists */
    if (var->value != NULL)
      tcaFreeData(msgName,var->value);
    /* Store the pointer to the new data */
    var->value = varValue;
  }
  /* Need to do a tcaSuccess, if a command. */
  /*  centralSuccess(dispatch);*/
  /* Broadcast the result. */
  
  watchMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,varName);

  centralBroadcast(watchMsgName,var->value);
  tcaFree(watchMsgName);
}


/*****************************************************************************
 *
 * FUNCTION: static void getVarHnd(DISPATCH_PTR dispatch, void *empty)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void getVarHnd(DISPATCH_PTR dispatch, void *empty)
{
#ifdef applec
#pragma unused(empty)
#endif
  
  GLOBAL_VAR_PTR var=NULL;
  const char *msgName;
  const char *varName;
  
  msgName = dispatch->msg->msgData->name;
  varName = &(msgName[strlen(VAR_SET_PREFIX)]);
  var = (GLOBAL_VAR_PTR) hashTableFind(varName, GET_S_GLOBAL(varTable));
  if (var == NULL) {
    /* handle the error here. */
  } else {
    centralReply(dispatch, (char *)var->value);
  }
}


/*****************************************************************************
 *
 * FUNCTION: static void getSetVarHnd(DISPATCH_PTR dispatch, void *varValue)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void getSetVarHnd(DISPATCH_PTR dispatch, void *varValue)
{
  GLOBAL_VAR_PTR var=NULL;
  const char *msgName;
  const char *varName;
  
  msgName = dispatch->msg->msgData->name;
  varName = &(msgName[strlen(VAR_GET_SET_PREFIX)]);
  var = (GLOBAL_VAR_PTR) hashTableFind(varName, GET_S_GLOBAL(varTable));
  if (var == NULL) {
    /* handle the error here. */
  } else {
    /* Free the old data if it exists */
    if (var->value != NULL) {
      centralReply(dispatch, (char *)var->value);
      tcaFreeData(msgName,var->value);
    }
    /* Store the pointer to the new data */
    var->value = varValue;
  }
}


/*****************************************************************************
 *
 * FUNCTION: void registerVarHnd(DISPATCH_PTR dispatch, VAR_REG_PTR varRegData)
 
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

static void registerVarHnd(DISPATCH_PTR dispatch, VAR_REG_PTR varRegData)
{
#ifdef applec
#pragma unused(dispatch)
#endif
  
  GLOBAL_VAR_PTR var;  
  CONST_GLOBAL_VAR_PTR old_var;  
  char *getMsgName, *setMsgName, *getSetMsgName;
  char *watchMsgName;
  
  var = NEW(GLOBAL_VAR_TYPE);
  
  /* Create the data structure to hold information about the variable */
  var->varName = varRegData->varName;
  var->format = varRegData->format;
  var->value = NULL;
  old_var = hashTableInsert((void *)var->varName, 1+strlen(var->varName), 
			    (void *)var, GET_S_GLOBAL(varTable));
  
  if (old_var != NULL) {
    /* The mesage is already registered, just check the format.   */
    if (strcmp(old_var->format,var->format)) {
      tcaError("ERROR: centralRegisterVar: variable %s already registered with a differnt format.\n",
	       var->varName);
    }
  }
  
  /* Create the name strings for the set and get messages.*/
  
  setMsgName = (char *)tcaMalloc(1+strlen(var->varName)+
				 strlen(VAR_SET_PREFIX));
  strcpy(setMsgName,VAR_SET_PREFIX);
  strcat(setMsgName,var->varName);
  
  getMsgName = (char *)tcaMalloc(1+strlen(var->varName)+
				 strlen(VAR_GET_PREFIX));
  strcpy(getMsgName,VAR_GET_PREFIX);
  strcat(getMsgName,var->varName);
  
  getSetMsgName = (char *)tcaMalloc(1+strlen(var->varName)+
				    strlen(VAR_GET_SET_PREFIX));
  strcpy(getSetMsgName,VAR_GET_SET_PREFIX);
  strcat(getSetMsgName,var->varName);
  
  watchMsgName = (char *)tcaMalloc(1+strlen(var->varName)+
				   strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,var->varName);
  
  /* Register the functions to set and get the variable value.  */
  
  centralRegisterInform(setMsgName, var->format, setVarHnd);
  
  centralRegisterQuery(getMsgName, NULL, var->format, getVarHnd);
  
  centralRegisterQuery(getSetMsgName, var->format, var->format, getSetVarHnd);

  centralRegisterBroadcastMessage(watchMsgName, var->format);

  tcaFree(setMsgName);
  tcaFree(getMsgName);
  tcaFree(getSetMsgName);
  tcaFree(watchMsgName);
  /* Just free top level structure -- The strings need to be saved */
  tcaFree((char *) varRegData);
}


/*****************************************************************************
 *
 * FUNCTION: void centralRegisterVar
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
void centralRegisterVar(const char *varName, const char *format)
{
  GLOBAL_VAR_TYPE var;
  
  var.varName = varName;
  var.format = format;
  centralInform(TCA_REGISTER_VAR_INFORM, &var);
}


/*****************************************************************************
 *
 * FUNCTION: void centralSetVar
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
void centralSetVar(const char *varName, const char *value)
{
  char *setMsgName;

  if (!mGlobalp())
    return;
  
  setMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_SET_PREFIX));
  strcpy(setMsgName,VAR_SET_PREFIX);
  strcat(setMsgName,varName);
  
  /*  tcaExecuteCommandAsync(setMsgName, value);*/
  centralInform(setMsgName, value);
  tcaFree(setMsgName);
}


/*****************************************************************************
 *
 * FUNCTION: void centralIgnoreVarLogging
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
void centralIgnoreVarLogging(const char *varName)
{
  char *setMsgName;
  char *getMsgName;
  char *getSetMsgName;
  char *watchMsgName;
  
  setMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_SET_PREFIX));
  strcpy(setMsgName,VAR_SET_PREFIX);
  strcat(setMsgName,varName);
  Add_Message_To_Ignore(setMsgName);
  tcaFree(setMsgName);

  getMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_GET_PREFIX));
  strcpy(getMsgName,VAR_GET_PREFIX);
  strcat(getMsgName,varName);
  Add_Message_To_Ignore(getMsgName);
  tcaFree(getMsgName);
  
  getSetMsgName = (char *)tcaMalloc(1+strlen(varName)+ 
				    strlen(VAR_GET_SET_PREFIX));
  strcpy(getSetMsgName,VAR_GET_SET_PREFIX);
  strcat(getSetMsgName,varName);
  Add_Message_To_Ignore(getSetMsgName);
  tcaFree(getSetMsgName);
  
  watchMsgName = (char *)tcaMalloc(1+strlen(varName)+strlen(VAR_WATCH_PREFIX));
  strcpy(watchMsgName,VAR_WATCH_PREFIX);
  strcat(watchMsgName,varName);
  Add_Message_To_Ignore(watchMsgName);
  tcaFree(watchMsgName);
}


/*****************************************************************************
 *
 * FUNCTION: void globalVarInitialize(void)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
void globalVarInitialize(void)
{
  /* Create the table to hold the information for each global variable. */
  GET_S_GLOBAL(varTable) = hashTableCreate(11, (HASH_FN) strHashFunc, 
					   (EQ_HASH_FN)strKeyEqFunc);
  
  /* Register the handler for creating the global variable */
  
  centralRegisterInform(TCA_REGISTER_VAR_INFORM,
			TCA_REGISTER_VAR_INFORM_FORMAT,
			registerVarHnd);
  Add_Message_To_Ignore(TCA_REGISTER_VAR_INFORM);
  centralRegisterInform(TCA_REGISTER_VAR_INFORM_OLD,
			TCA_REGISTER_VAR_INFORM_FORMAT,
			registerVarHnd);
  Add_Message_To_Ignore(TCA_REGISTER_VAR_INFORM_OLD);

  centralRegisterVar(TCA_TERMINAL_LOG_VAR, TCA_LOG_VAR_FORMAT);
  centralRegisterVar(TCA_FILE_LOG_VAR, TCA_LOG_VAR_FORMAT);
  centralIgnoreVarLogging(TCA_TERMINAL_LOG_VAR);
  centralIgnoreVarLogging(TCA_FILE_LOG_VAR);
  centralSetVar(TCA_TERMINAL_LOG_VAR, (char *)&GET_S_GLOBAL(terminalLog));
  centralSetVar(TCA_FILE_LOG_VAR, (char *)&GET_S_GLOBAL(fileLog));
}
