/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: exception handling
 *
 * FILE: exception.c
 *
 * ABSTRACT:
 * 
 * exception code.
 *
 * REVISION HISTORY
 *
 * $Log: exception.c,v $
 * Revision 1.23  1996/07/19  18:13:59  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.22  1996/02/06  19:04:38  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.21  1995/10/29  18:26:40  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.20  1995/10/25  22:48:13  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.19  1995/10/10  00:42:54  rich
 * Added more system messages to ignore.
 *
 * Revision 1.18  1995/10/07  19:07:16  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.17  1995/08/06  16:43:54  reids
 * A bug existed in that two demon monitors that sent the same ID number
 * would conflict (causing the wrong one to fire).  This has been fixed, and
 * in the process, one of the hash-key functions was made a bit more general.
 *
 * Revision 1.16  1995/07/06  21:16:08  rich
 * Solaris and Linux changes.
 *
 * Revision 1.15  1995/05/31  19:35:26  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.14  1995/01/18  22:40:21  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.13  1994/10/25  17:09:49  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.12  1994/04/28  16:15:57  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.11  1994/04/16  19:42:05  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.10  1994/01/31  18:27:43  reids
 * Several major changes (and some minor ones)
 * 1. tcaFreeData and tcaFreeReply now work even if the data or message format
 *    is NULL
 * 2. Using the "-t" option in central, message taps are added as a child of
 *    the task tree node that was tapped.
 * 3. Named formatters are now expanded only when needed
 * For more details, see ../doc/tca-7-4.release.notes
 *
 * Revision 1.9  1993/12/14  17:33:33  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.8  1993/11/21  20:17:43  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.7  1993/10/21  16:13:51  rich
 * Fixed compiler warnings.
 *
 * Revision 1.6  1993/08/27  08:38:31  fedor
 * Pass 2 aat a V7+V6+VxWorks merge. Many many problems with pointless casting.
 *
 * Revision 1.5  1993/08/27  07:14:41  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.4  1993/08/20  23:06:53  fedor
 * Minor changes for merge. Mostly added htons and removed cfree calls.
 *
 * Revision 1.3  1993/08/20  00:26:04  fedor
 * commiting others changes
 *
 * Revision 1.2  1993/05/26  23:17:21  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:20  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:42  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:39  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 17-Feb-92 Reid Simmons, School of Computer Science, CMU
 * Fixed choosing exception handlers so that task tree is killed if no
 *  exception is found (fix of 31-Oct-91 just worked for byPass)
 *
 * 31-Dec-91 Reid Simmons, School of Computer Science, CMU
 * Added facility to add exception handlers to class of messages, 
 *  rather than just a single task tree node.
 *
 * 31-Oct-91 Christopher Fedor, School of Computer Science, CMU
 * If exception handler could not be found by byPass display warning
 * and kill SubTaskTree of tcaTaskTreeRootGlobal.
 *
 * 29-May-91 Christopher Fedor, School of Computer Science, CMU
 * Cut the temporal constraints between the node to be retried
 * and its children to prevent a confusion when moving "Now" backwards.
 *
 * 12-May-91 Christopher Fedor, School of Computer Science, CMU
 * Revised to verion tca 5.x
 *
 * 25-Nov-89 Long-Ji Lin, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.23 $
 * $Date: 1996/07/19 18:13:59 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"

/******************************************************************************
 *
 * FUNCTION: void killTopLevelTaskTreeHandler(ref, dummy)
 *
 * DESCRIPTION:
 * Kills the top-level task tree.
 *
 * INPUTS:  DISPATCH_PTR dispatch;
 *          char *dummy;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/
/*ARGSUSED*/
static void killTopLevelTaskTreeHandler (DISPATCH_PTR dispatch, void *dummy)
{
#ifdef applec
#pragma unused(dummy)
#endif
  Log_Message("\nWARNING: Unable to find a handler for the exception %s.\n",
	      dispatch->msg->msgData->name);
  Log_Message("WARNING: Killing task tree under root node.\n");
  
  killSubTaskTree(GET_S_GLOBAL(taskTreeRootGlobal));
  processHandledMessage(dispatch);
}    

/******************************************************************************
 *
 * FUNCTION: DISPATCH_HND_PTR chooseExceptionHandler(dispatch)
 *
 * DESCRIPTION:
 * Chooses the execption handler for the first call and for by passing
 * handlers.
 * chooseExceptionHandler will also create or update the context information.
 *
 * INPUTS:  DISPATCH_PTR dispatch;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static int findMatchingHnd(LIST_PTR hndList, DISPATCH_HND_PTR hnd)
{
  /* byPassHnd forces the exception handler selection to pass over
     previously executed handlers in the node exception handler list. */
  
  if (!GET_S_GLOBAL(byPassHnd)) {
    return(listMemberItem((char *)hnd, hndList));
  }
  else if (GET_S_GLOBAL(byPassHnd) == hnd) {
    GET_S_GLOBAL(byPassHnd) = NULL;
  }
  
  return 0;
}

static int findHnd(HND_KEY_PTR key, DISPATCH_HND_PTR hnd, char *findName)
{
  if (STREQ(key->str, findName)) {
    GET_S_GLOBAL(foundHnd) = hnd;
    return 0;
  }
  
  return 1;
}

/*********************************************************/

DISPATCH_HND_PTR chooseExceptionHandler(DISPATCH_PTR dispatch)
{
  DISPATCH_HND_PTR hnd;
  int size, nodeSize;
  LIST_PTR hndList, nodeHndList;
  TASK_TREE_NODE_PTR ENode, node;
  
  hndList = dispatch->msg->hndList;
  size = listLength(hndList);
  
  if (!size)
    return NULL;
  
  ENode = node = dispatch->treeNode->parent;
  
  /* test for byPass Exception to continue from the context node */
  if (dispatch->treeNode->ECNode) {
    node = dispatch->treeNode->ECNode->CNode;
  }
  
  while (node) {
    nodeHndList = node->excHndList;
    nodeSize = listLength(nodeHndList);
    
    if (nodeSize) {
      hnd = (DISPATCH_HND_PTR)listMemReturnItem((LIST_ITER_FN) findMatchingHnd,
						(char *)hndList, 
						nodeHndList);
      if (hnd) {
	/* set ENode and CNode */
	if (!dispatch->treeNode->ECNode) {
	  dispatch->treeNode->ECNode = NEW(ECNODE_TYPE);
	}
	dispatch->treeNode->ECNode->ENode = ENode;
	dispatch->treeNode->ECNode->CNode = node;
	
	return hnd;
      }
    }
    node = node->parent;
  }
  
  /* Added by reids, 17-Feb-92: Kill task tree if no handler found */
  GET_S_GLOBAL(foundHnd) = NULL;
  hashTableIterate((HASH_ITER_FN)findHnd, GET_C_GLOBAL(handlerTable), 
		   "killTopLevelTaskTreeHandler");
  return GET_S_GLOBAL(foundHnd);
}


/******************************************************************************
 *
 * FUNCTION: void addExceptionHnd(dispatch, addInfo)
 *
 * DESCRIPTION: 
 * Add an exception handler to a node.
 * If the node already has this handler then simply do not add it again,
 * this is needed if retries call hnd that add exception handlers.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * ADD_EXC_PTR addInfo;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void addExceptionHnd(DISPATCH_PTR dispatch, ADD_EXC_PTR addInfo)
{
  DISPATCH_PTR addDispatch;
  TASK_TREE_NODE_PTR addNode;
  
  addNode = NULL;
  
  if (addInfo->ref->refId == NO_REF)
    addNode = GET_S_GLOBAL(taskTreeRootGlobal);
  else {
    addDispatch = DISPATCH_FROM_ID(addInfo->ref->refId);
    
    /* 12-May-91: fedor: is this correct? */
    if (addDispatch)
      addNode = addDispatch->treeNode;
  }
  
  if (addNode) {
    /* 12-May-91: fedor: major bad news!- new handler key prevents
       using simple hash look up */
    GET_S_GLOBAL(foundHnd) = NULL;
    hashTableIterate((HASH_ITER_FN)findHnd, GET_C_GLOBAL(handlerTable),
		     (void *)(addInfo->hndName));
    
    if (!GET_S_GLOBAL(foundHnd))
      tcaError("ERROR: addExcepHnd: Missing handler %s\n.", addInfo->hndName);
    
    if (!addNode->excHndList)
      addNode->excHndList = listCreate();
    
    if (!listMemberItem((char *)GET_S_GLOBAL(foundHnd), addNode->excHndList))
      listInsertItem((char *)GET_S_GLOBAL(foundHnd), addNode->excHndList);
  }
  
  /* A bit more efficient than using tcaFreeData */
  freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)addInfo);
}

/******************************************************************************
 *
 * FUNCTION: void addExceptionToMsgHnd(dispatch, addInfo)
 *
 * DESCRIPTION: 
 * Add an exception handler to a class of messages.
 * If the message already has this handler then simply do not add it again.
 * Use "findOrRegisterMessage" in case the message is not yet registered.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * ADD_EXC_MSG_PTR addInfo;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/
static void addExceptionToMsgHnd(DISPATCH_PTR dispatch,
				 ADD_EXC_MSG_PTR addInfo)
{
  MSG_PTR msg;
  
  msg = findOrRegisterMessage(addInfo->msgName);
  
  GET_S_GLOBAL(foundHnd) = NULL;
  hashTableIterate((HASH_ITER_FN)findHnd, GET_C_GLOBAL(handlerTable),
		   (void *)(addInfo->hndName));
  
  if (!GET_S_GLOBAL(foundHnd))
    tcaError("ERROR: addExcepHnd: Missing handler %s\n.", addInfo->hndName);
  
  if (!msg->excepList) {
    msg->excepList = listCreate();
  }
  if (!listMemberItem((char *)GET_S_GLOBAL(foundHnd), msg->excepList)) {
    listInsertItemFirst((char *)GET_S_GLOBAL(foundHnd), msg->excepList);
  }
  /* A bit more efficient than using tcaFreeData */
  freeDataStructure(dispatch->msg->msgData->msgFormat, (char *)addInfo);
}

/******************************************************************************
 *
 * FUNCTION: void addMsgExceptions(msg, taskTreeNode)
 *
 * DESCRIPTION: 
 * If the message has any message-wide exceptions, add them to the exception
 * list of the individual task tree node.
 *
 * INPUTS: 
 * MSG_PTR msg;
 * TASK_TREE_NODE_PTR taskTreeNode;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/
void addMsgExceptions(MSG_PTR msg, TASK_TREE_NODE_PTR taskTreeNode)
{
  HND_PTR exceptionHnd;
  
  if (msg->excepList) {
    if (!taskTreeNode->excHndList) {
      taskTreeNode->excHndList = listCreate();
    }
    exceptionHnd = (HND_PTR)listFirst(msg->excepList);
    while (exceptionHnd) {
      /* Insert last so as to maintain the same order as original excepList */
      listInsertItemLast((char *)exceptionHnd, taskTreeNode->excHndList);
      exceptionHnd = (HND_PTR)listNext(msg->excepList);
    }
  }
}

/******************************************************************************
 *
 * FUNCTION: void reExecMsg(dispatch, hnd)
 *
 * DESCRIPTION:
 * Does the work of resending a message.
 *
 * The "Now" point is moved back and the node is temporally severed from its
 * children to prevent the confusion of moving time backwards.
 *
 * The sub tree of the node is killed. And the message is resent based
 * on the method in Long-Ji Lin's original code.
 *
 * If hnd is not NULL then it is reset. This allows this code to be
 * used to resend a retry and a by passed handler with the same dispatch.
 *
 * Selection of a handler is ignored in recvMsg for exception class 
 * dispatchs if it is already set.
 
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * HND_PTR hnd;
 *
 * OUTPUTS: void. 
 * calls processInactiveTests which will call processActiveMessage.
 *
 * NOTES:
 * 29-May-91: fedor: If the node to reExec is not an exception class
 * - most retries. Then recvMsg will reselect the handler - this may 
 * be bad if multiple handlers are registered for the same message.
 * Then again this may be good so I left it and wrote this cute note.
 *
 *****************************************************************************/

void reExecMsg(DISPATCH_PTR dispatch, DISPATCH_HND_PTR hnd)
{
  int retracted;
  
  resourceRemoveAttending(dispatch);
  
  if (hnd) {
    dispatch->hnd = hnd;
    dispatch->des = hnd->hndOrg;
    dispatch->desId = hnd->hndOrg->writeSd;
  }
  
  retracted = retractEndOfConstraints(dispatch->treeNode);
  
  /* RTG Oct-7-92 : do all the qlattice assertions inside the same 
   * DELAY_RULES call to avoid inconsistencies in the qlattice.
   */
  DELAY_RULES({ 
    QRetract(GET_S_GLOBAL(Now), ">", dispatch->treeNode->handlingInterval->end,
	     "Completed Message");
    QRetract(GET_S_GLOBAL(Now), ">=",
	     dispatch->treeNode->handlingInterval->start, 
	     "Scheduler");
    (void)listIterate((LIST_ITER_FN)removeParentChildRelations, 
		      (char *)dispatch->treeNode, 
		      dispatch->treeNode->children);
    
    AssertHoldingConstraint(dispatch->treeNode);});
  
  if (retracted) 
    restoreEndOfRelations(dispatch->treeNode);
  
  killSubTaskTree(dispatch->treeNode);
  
  DISPATCH_SET_STATUS(AllocatedDispatch, dispatch);
  
  dispatch->msg_class = dispatch->msg->msgData->msg_class;
  
  tapWhenSent(dispatch);
  setUpBeforeTaps(dispatch);
  
  /* 22-May-91: fedor: should note that the class information gets
     munged to the failure class and failure class data on multiple times
     through */ 
  
  processInactiveTests(dispatch);
  
  setUpDuringAndAfterTaps(dispatch);
}


/******************************************************************************
 *
 * FUNCTION: void retryExcepHnd(dispatch, ref)
 *
 * DESCRIPTION: 
 * Retry the specified ref. 
 *
 * If the node to retry is not the exception node (most cases) then mark
 * the exception as handled so that killing the sub tree at a parent node
 * will complete.
 *
 * If the retry node is the exception node then do not mark as handled
 * in case there are temporal constraints that will get activated
 * inadvertently (unlikely). This also avoid extra resource work (minor).
 *
 * The passed dispatch will be cleaned up as a constraint dispatch.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * TCA_REF_PTR *ref;
 *
 * OUTPUTS: void. calls reExecMsg.
 *
 *****************************************************************************/

static void retryExcepHnd(DISPATCH_PTR dispatch, TCA_REF_PTR *ref)
{
  DISPATCH_PTR retry, exception;
  
  retry = DISPATCH_FROM_ID((*ref)->refId);
  
  exception = DISPATCH_FROM_ID(dispatch->pRef);
  
  retry->treeNode->retries++;
  
  reExecMsg(retry, (DISPATCH_HND_PTR)NULL);
  
  Log_Message("  Retry  %15s", exception->msg->msgData->name);
  Log_RefId(dispatch, LOGGING_MESSAGE);
  Log_Message(":\n");
  
  if (retry != exception)
    processHandledMessage(exception);
  
  tcaRefFree(*ref); /* Assumes the data is a simple TCA_REF_PTR */
}


/******************************************************************************
 *
 * FUNCTION: void byPassExcepHnd(dispatch, ref)
 *
 * DESCRIPTION: 
 * By pass the current exception by selecting a different exception handler.
 * byPassHnd marks where in the list of handlers the previous search stopped.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * TCA_REF_PTR *ref;
 *
 * OUTPUTS: void. calls reExecMsg to continue.
 *
 *****************************************************************************/

static void byPassExcepHnd(DISPATCH_PTR dispatch, TCA_REF_PTR *ref)
{
  DISPATCH_HND_PTR hnd;
  DISPATCH_PTR exception;
  
  exception = DISPATCH_FROM_ID((*ref)->refId);
  
  GET_S_GLOBAL(byPassHnd) = exception->hnd;
  hnd = chooseExceptionHandler(exception);
  
  Log_Message("  ByPass %15s", exception->msg->msgData->name);
  Log_RefId(dispatch, LOGGING_MESSAGE);
  Log_Message(":\n");
  
  if (hnd) {
    reExecMsg(exception, hnd);
  }
  else {
    processHandledMessage(exception);
    
    Log_Message("\nWARNING: Unable to find a handler for the exception.\n");
    Log_Message("WARNING: Killing task tree under root node.\n");
    killSubTaskTree(GET_S_GLOBAL(taskTreeRootGlobal));
  }
  
  /* the exception for byPassExcepHnd will be freed because its a constraint */
  
  tcaRefFree(*ref); /* Assumes the data is a simple TCA_REF_PTR */
}


/******************************************************************************
 *
 * FUNCTION: void getInfoExcepHnd(dispatch, ref)
 *
 * DESCRIPTION:
 * Return information about an exception.
 *
 * INPUTS: 
 * DISPATCH_PTR dispatch;
 * TCA_REF_PTR *ref;
 *
 * OUTPUTS: void. calls centralReply.
 *
 * NOTES:
 * 29-May-91: fedor:
 * Another place where a consistent null TCA_REF_PTR would be useful.
 *
 *****************************************************************************/

static void getInfoExcepHnd(DISPATCH_PTR dispatch, TCA_REF_PTR *ref)
{
  ECNODE_PTR ECNode;
  TCA_REF_PTR nullRef;
  EXC_INFO_TYPE excInfo;
  DISPATCH_PTR exception;
  
  exception = DISPATCH_FROM_ID((*ref)->refId);
  
  ECNode = exception->treeNode->ECNode;
  
  if (ECNode) {
    excInfo.numOfRetries = ECNode->CNode->retries;
    
    excInfo.ERef = tcaRefCreate(ECNode->ENode->dispatch->msg,
				ECNode->ENode->dispatch->msg->msgData->name,
				ECNode->ENode->dispatch->locId);
    
    if (IS_ROOT_NODE(ECNode->CNode)) {
      excInfo.CRef = CREATE_NULL_REF();
    } else {
      excInfo.CRef = tcaRefCreate(ECNode->CNode->dispatch->msg,
				  ECNode->CNode->dispatch->msg->msgData->name,
				  ECNode->CNode->dispatch->locId);
    }
    
    centralReply(dispatch, (char *)&excInfo);
    
    tcaRefFree(excInfo.ERef);
    tcaRefFree(excInfo.CRef);
  } else {
    excInfo.numOfRetries = 0;
    
    /* 17-May-91: fedor: should implement null tca_ref_ptr transfer */
    nullRef = CREATE_NULL_REF();
    excInfo.ERef = nullRef;
    excInfo.CRef = nullRef;
    
    centralReply(dispatch, (char *)&excInfo);
    
    tcaRefFree(nullRef);
  }
  
  tcaRefFree(*ref); /* Assumes the data is a simple TCA_REF_PTR */
}


/******************************************************************************
 *
 * FUNCTION: void exceptionIntialize()
 *
 * DESCRIPTION: 
 * Initialize exception handling. Remeber to init byPassHnd.
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void exceptionIntialize(void)
{
  GET_S_GLOBAL(byPassHnd) = NULL;
  
  centralRegisterInform(TCA_ADD_EXCEP_INFORM,
			TCA_ADD_EXCEP_INFORM_FORMAT,
			addExceptionHnd);
  Add_Message_To_Ignore(TCA_ADD_EXCEP_INFORM);
  
  centralRegisterInform(TCA_ADD_EXCEP_INFORM_OLD,
			TCA_ADD_EXCEP_INFORM_FORMAT,
			addExceptionHnd);
  Add_Message_To_Ignore(TCA_ADD_EXCEP_INFORM_OLD);
  
  centralRegisterInform(TCA_ADD_EXCEPTIONS_INFORM,
			TCA_ADD_EXCEPTIONS_INFORM_FORMAT,
			addExceptionToMsgHnd);
  Add_Message_To_Ignore(TCA_ADD_EXCEPTIONS_INFORM);
  
  centralRegisterInform(TCA_ADD_EXCEPTIONS_INFORM_OLD,
			TCA_ADD_EXCEPTIONS_INFORM_FORMAT,
			addExceptionToMsgHnd);
  Add_Message_To_Ignore(TCA_ADD_EXCEPTIONS_INFORM_OLD);
  
  centralRegisterInform(TCA_RETRY_EXCEP_INFORM,
			TCA_RETRY_EXCEP_INFORM_FORMAT,
			retryExcepHnd);
  Add_Message_To_Ignore(TCA_RETRY_EXCEP_INFORM);
  
  centralRegisterInform(TCA_RETRY_EXCEP_INFORM_OLD,
			TCA_RETRY_EXCEP_INFORM_FORMAT,
			retryExcepHnd);
  Add_Message_To_Ignore(TCA_RETRY_EXCEP_INFORM_OLD);
  
  centralRegisterInform(TCA_BYPASS_EXCEP_INFORM,
			TCA_BYPASS_EXCEP_INFORM_FORMAT,
			byPassExcepHnd);
  Add_Message_To_Ignore(TCA_BYPASS_EXCEP_INFORM);
  
  centralRegisterInform(TCA_BYPASS_EXCEP_INFORM_OLD,
			TCA_BYPASS_EXCEP_INFORM_FORMAT,
			byPassExcepHnd);
  Add_Message_To_Ignore(TCA_BYPASS_EXCEP_INFORM_OLD);
  
  centralRegisterQuery(TCA_GET_INFO_EXCEP_QUERY,
		       "TCA_REF_PTR",
		       TCA_GET_INFO_EXCEP_QUERY_FORMAT,
		       getInfoExcepHnd);
  Add_Message_To_Ignore(TCA_GET_INFO_EXCEP_QUERY);
  
  centralRegisterQuery(TCA_GET_INFO_EXCEP_QUERY_OLD,
		       "TCA_REF_PTR",
		       TCA_GET_INFO_EXCEP_QUERY_FORMAT,
		       getInfoExcepHnd);
  Add_Message_To_Ignore(TCA_GET_INFO_EXCEP_QUERY_OLD);
  
  /* Added by reids, 17-Feb-92: Handles all exceptions that fall through 
     to top of task tree */
  centralRegisterException(TCA_KILL_TASK_TREE_EXCEP,
			   TCA_KILL_TASK_TREE_EXCEP_FORMAT,
			   killTopLevelTaskTreeHandler);
  centralRegisterException(TCA_KILL_TASK_TREE_EXCEP_OLD,
			   TCA_KILL_TASK_TREE_EXCEP_FORMAT,
			   killTopLevelTaskTreeHandler);
  
}
