/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: exception handling
 *
 * FILE: excepMod.c
 *
 * ABSTRACT:
 * 
 * module exception code.
 *
 * REVISION HISTORY
 *
 * $Log: excepMod.c,v $
 * Revision 1.16  1996/08/22  16:35:55  rich
 * Check the return code on tcaQueryCentral calls.
 *
 * Revision 1.15  1996/02/10  16:49:54  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.14  1996/01/27  21:53:21  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.13  1995/11/03  03:04:26  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.12  1995/10/25  22:48:12  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.11  1995/10/07  19:07:13  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.10  1995/04/08  02:06:23  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.9  1995/01/18  22:40:19  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.8  1994/04/16  19:42:03  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1994/03/28  02:22:52  rich
 * parseFmttrs needs to be in the server objects and not the module objects.
 *
 * Revision 1.6  1993/12/14  17:33:29  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.5  1993/11/21  20:17:41  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/30  21:53:22  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:14:37  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:17:19  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:20  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:40  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:36  fedor
 * Added Logging.
 *
 * 31-Dec-91 Reid Simmons, School of Computer Science, CMU
 * Added facility to add exception handlers to class of messages, 
 *  rather than just a single task tree node.
 *
 * 21-May-91 Christopher Fedor, School of Computer Science, CMU
 * Revised to verion tca 5.x
 *
 * 25-Nov-89 Long-Ji Lin, School of Computer Science, CMU
 * created.
 *
 * $Revision: 1.16 $
 * $Date: 1996/08/22 16:35:55 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include "taskTree.h"
#ifdef   DOS_FILE_NAMES
#include "exceptio.h"
#else
#include "exception.h"
#endif

#if defined(LISP)
/******************************************************************************
 * Forward Declarations
 *****************************************************************************/
EXC_INFO_PTR tcaGetExcepC(TCA_REF_PTR ref);
TCA_REF_PTR tcaGetERef(EXC_INFO_PTR info);
TCA_REF_PTR tcaGetCRef(EXC_INFO_PTR info);
int tcaGetRetries(EXC_INFO_PTR info);
void tcaFreeExcInfo(EXC_INFO_PTR info);

#endif


/******************************************************************************
 *
 * FUNCTION: void tcaAddExceptionHandler(ref, hndName)
 *
 * DESCRIPTION: Adds the named handler to the task tree at node ref.
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * char *hndName;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaAddExceptionHandler(TCA_REF_PTR ref, const char *hndName)
{
  MSG_PTR msg;
  ADD_EXC_TYPE addInfo;
  
  addInfo.ref = ref;
  addInfo.hndName = hndName;
  
  msg = msgFind(TCA_ADD_EXCEP_INFORM);
  if (msg == NULL) return;
  (void)sendMessage((TCA_REF_PTR)NULL, msg, (char *)&addInfo, (char *)NULL);
}


/******************************************************************************
 *
 * FUNCTION: void tcaAddExceptionHandlerToMessage(msgName, hndName)
 *
 * DESCRIPTION: Adds the named exception handler to *all* named message 
 *              instances.
 *
 * INPUTS:
 * char *msgName;
 * char *hndName;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaAddExceptionHandlerToMessage(const char *msgName, const char *hndName)
{
  MSG_PTR msg;
  ADD_EXC_MSG_TYPE addInfo;
  
  addInfo.msgName = msgName;
  addInfo.hndName = hndName;
  
  msg = msgFind(TCA_ADD_EXCEPTIONS_INFORM);
  if (msg == NULL) return;
  (void)sendMessage((TCA_REF_PTR)NULL, msg, (char *)&addInfo, (char *)NULL);
}


/******************************************************************************
 *
 * FUNCTION: void tcaGetExceptionInfo(ref, info, selection)
 *
 * DESCRIPTION: Return information about an exception.
 *
 * INPUTS:
 *
 * OUTPUTS: void.
 *
 * NOTES: 
 * 12-May-91: fedor: This should be passed in automaticaly on an exception.
 *
 *****************************************************************************/

void tcaGetExceptionInfo(TCA_REF_PTR ref, EXC_INFO_PTR info)
{
  if (info == NULL) return;
  bzero(info,sizeof(EXC_INFO_TYPE));
  (void)tcaQueryCentral(TCA_GET_INFO_EXCEP_QUERY, (void *)&ref, (void *)info);
}

#if defined(LISP)

EXC_INFO_PTR tcaGetExcepC(TCA_REF_PTR ref)
{
  EXC_INFO_PTR info;
  
  info = NEW(EXC_INFO_TYPE);
  bzero(info,sizeof(EXC_INFO_TYPE));
  (void)tcaQueryCentral(TCA_GET_INFO_EXCEP_QUERY, (void *)&ref, (void *)info);
  
  return info;
}


TCA_REF_PTR tcaGetERef(EXC_INFO_PTR info)
{
  return info->ERef;
}

TCA_REF_PTR tcaGetCRef(EXC_INFO_PTR info)
{
  return info->CRef;
}

int tcaGetRetries(EXC_INFO_PTR info)
{
  return info->numOfRetries;
}

void tcaFreeExcInfo(EXC_INFO_PTR info)
{
  free((char *)info);
}

#endif


/******************************************************************************
 *
 * FUNCTION: void tcaRetry(ref)
 *
 * DESCRIPTION: Kill the task tree under node ref and resend message at of ref.
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaRetry(TCA_REF_PTR ref)
{
  MSG_PTR msg;
  
  GET_C_GLOBAL(responseIssuedGlobal) = TRUE;
  
  msg = msgFind(TCA_RETRY_EXCEP_INFORM);
  if (msg == NULL) return;
  (void)sendMessage((TCA_REF_PTR)NULL, msg, (char *)&ref, (char *)NULL);
}


/******************************************************************************
 *
 * FUNCTION: void tcaByPassException(ref)
 *
 * DESCRIPTION: Notify the server that the exception can not be handled.
 * The exception will then be passed to the next exception or propagate up
 * the task tree.
 *
 * Should only be called from an exception handler.
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 *
 * OUTPUTS: void.
 *
 * NOTES: 12-May-91: fedor: How about a call to tcaFailure instead?
 *
 *****************************************************************************/

void tcaByPassException(TCA_REF_PTR ref)
{
  MSG_PTR msg;
  
  GET_C_GLOBAL(responseIssuedGlobal) = TRUE;
  
  msg = msgFind(TCA_BYPASS_EXCEP_INFORM);
  if (msg == NULL) return;
  (void)sendMessage((TCA_REF_PTR)NULL, msg, (char *)&ref, (char *)NULL);
}
