/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: datamsg
 *
 * FILE: datamsg.c
 *
 * ABSTRACT:
 * 
 * A datamsg represents the encoded form of a message and is used to perform
 * the actual sending and receiving of messages.
 *
 * REVISION HISTORY
 *
 * $Log: datamsg.c,v $
 * Revision 1.46  1996/07/19  18:13:53  reids
 * Record broadcast messages if handler is registered before message.
 * Transfer any pending messages to the new resource under "addHndToResource"
 * Fixed tcaDelayCommand (wrong time units).
 * Fixed logging of refid's (have to distinguish whether they are part of
 *   a status, message, or "always" log).
 * Sanity check for encoding/decoding messages.
 *
 * Revision 1.45  1996/06/25  20:50:26  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.44  1996/05/09  18:30:45  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.43  1996/05/07  16:49:20  rich
 * Changes for clisp.
 *
 * Revision 1.42  1996/03/09  06:13:13  rich
 * Fixed problem where lisp could use the wrong byte order if it had to
 * query for a message format.  Also fixed some memory leaks.
 *
 * Revision 1.41  1996/02/10  16:49:45  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.40  1996/02/06  19:04:31  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.39  1996/01/27  21:53:14  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.38  1996/01/10  03:16:20  rich
 * Fixed libtca_lisp.a to work with dbmalloc.  Added central commands to
 * show resource state and to unlock locked resouces.  Fixed a bug where
 * dispatches were not freed when handlers were cleared. Reset errno variable.
 *
 * Revision 1.37  1996/01/05  16:31:14  rich
 * Added windows NT port.
 *
 * Revision 1.36  1995/08/14  21:31:30  rich
 * Got rid of the "sharedBuffers" flag on the dataMessages.  It was not the
 * right solution, and possibly caused a memory leak.
 * Limit pending for TCA_TAPPED_MSG_VAR to 1.
 *
 * Revision 1.35  1995/08/05  21:11:41  reids
 * The "sharedBuffers" flag was not always being set.
 *
 * Revision 1.34  1995/08/05  18:16:27  rich
 * Removed debugging code.
 *
 * Revision 1.33  1995/08/05  18:13:14  rich
 * Fixed problem with writeNBuffers on partial writes.
 * Added "sharedBuffers" flag to the dataMsg structure, rather than
 * checking to see if the dataStruct pointer and the message data pointer
 * are the same.  This allows central to clear the dataStruc pointer so
 * that messages don't try to access old data structures that might have
 * changed since the  message was created.
 *
 * Revision 1.32  1995/08/04  16:40:59  rich
 * Fixed problem with in place translation of doubles from Big to little endian.
 *
 * Revision 1.31  1995/07/19  14:26:06  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.30  1995/07/12  04:54:37  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.29  1995/07/10  16:17:11  rich
 * Interm save.
 *
 * Revision 1.28  1995/07/06  21:39:36  rich
 * Fixes ported from 7.9.
 *
 * Revision 1.27  1995/07/06  21:15:59  rich
 * Solaris and Linux changes.
 *
 * Revision 1.26  1995/06/05  23:59:00  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.25  1995/04/19  14:27:58  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.24  1995/04/07  05:03:04  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.23  1995/04/04  19:42:00  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.22  1995/03/30  15:42:42  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.21  1995/03/16  18:05:15  rich
 * Merged in changes to the 7.9 branch.
 * Changed the VERSION_ to TCA_VERSION_
 *
 * Revision 1.20  1995/03/14  22:36:41  rich
 * Fixed problem with multiple read needed when doing a vector read. (Fix
 * to version 7.9)
 * Also fixed the data size problem from 7.9.
 *
 * Revision 1.19  1995/01/30  16:17:45  rich
 * Added indications of message byte order and alignment to the message
 * header in the upper two bytes of the classID.
 * Now the receiver translates the byte order but must receive data in
 * "PACKED" alignment.
 * Made -gstabs come after -g for i386_mach machines so the debugger will work.
 *
 * Revision 1.18.2.2  1995/03/14  03:57:26  rich
 * Added a data reference count to the dataMsg type.  It is used to decide
 * when to free the data associated with a message.  Messages can share
 * data buffers.
 * Fixed bug in the vector read routine (read2Buffers).  It would not
 * correctly update the buffer pointers if multiple reads were needed.
 *
 * Revision 1.18.2.1  1995/02/26  22:45:22  rich
 * I thought the class data came after the message data in transmission
 * like it does in the message data structure, but it does not.
 * The data transmission order is header->msgData->classData.
 * This has been changed in versin 8.0.
 *
 * Revision 1.18  1995/01/18  22:40:08  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.17  1994/05/31  03:23:47  rich
 * Removed CFLAGS_sun4.
 * Removed cfree and bzero from mem routines.
 * Set zero wait on both sides of the pipe.  Can connect to host using inet
 * number.
 *
 * Revision 1.16  1994/05/17  23:15:33  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.15  1994/05/11  01:57:18  rich
 * Now set an invalid tcaServerGlobal (a socket fd) to -1 rather than 0
 * which is stdout.
 * Added checks to make sure tcaServerGlobal is a valid socket before
 * sending messages or waiting for messages.
 *
 * Revision 1.14  1994/04/28  16:15:43  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.13  1994/04/26  16:23:23  rich
 * Now you can register an exit handler before anything else and it will
 * get called if connecting to central fails.
 * Also added code to handle pipe breaks during writes.
 *
 * Revision 1.12  1994/04/16  19:41:56  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.11  1994/04/04  16:01:10  reids
 * Fixed the way data transfer occurred from/to big and little Endian machines
 *
 * Revision 1.10  1993/12/14  17:33:18  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.9  1993/12/01  18:03:11  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.8  1993/11/21  20:17:30  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.7  1993/10/20  19:00:30  rich
 * Fixed bug with self registed messages in the lisp version.
 * Added new routine : tcaGetServerGlobal to get the server socket.
 * Fixed some bad global references for the lisp version.
 * Updated some prototypes.
 *
 * Revision 1.6  1993/08/30  23:13:45  fedor
 * Added SUN4 as well as sun4 compile flag.
 * Corrected Top level failure message name display with a define in dispatch.c
 *
 * Revision 1.4  1993/08/27  07:14:29  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/08/20  23:06:46  fedor
 * Minor changes for merge. Mostly added htons and removed cfree calls.
 *
 * Revision 1.2  1993/05/26  23:17:06  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:19  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:26  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:25  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 6-May-92 Christopher Fedor, School of Computer Science, CMU
 * Added EWOULDBLOCK detection - for now simply continue to try to write
 * where it left off. THis may not be correct for all applications and can
 * cause the central server to simply wedge.
 *
 *  9-Jul-1991 Reid Simmons, School of Computer Science, CMU
 * Commented out stuff to store data message buffers.
 *
 *  8-Jul-1991 Reid Simmons, School of Computer Science, CMU
 * Added routine to pre-allocate a datamsg buffer
 *
 * 25-Jun-91 Christopher Fedor, School of Computer Science, CMU
 * Replaced fixed number of datamsg buffers with Reid's datamsg code.
 * Just postponing the eventual memory lossage.
 *
 * 11-Jun-91 Christopher Fedor, School of Computer Science, CMU
 * Set a fixed number of datamsg buffers.
 *
 * 30-Jan-91 Christopher Fedor, School of Computer Science, CMU
 * Added fflush(stdout) to printf for module code calls from lisp
 *
 * 17-Sep-90 Christopher Fedor, School of Computer Science, CMU
 * Added stats for tracking datamsg memory usage.
 *
 *  5-Jul-90 Christopher Fedor, School of Computer Science, CMU
 * Added refCount for dealing with datamsg garbage collection.
 *
 *  4-Jun-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 *
 * 29-May-90 Christopher Fedor, School of Computer Science, CMU
 * Revised double encoding of datamsg to support class data for tca 5.x
 *
 * 26-Apr-89 Christopher Fedor, School of Computer Science, CMU
 * Changed to support double encoding.
 *
 * 23-Feb-89 Christopher Fedor, School of Computer Science, CMU
 * Interface to Communications by dataMsgs
 *
 *    Dec-88 Christopher Fedor, School of Computer Science, CMU
 * Created.
 *
 * 15-Nov-88 Christopher Fedor  created readNBytes
 *
 * NOTES:
 *
 * 26-Apr-89 Christopher Fedor, School of Computer Science, CMU
 * Need to insure that central will not block on socket reads and writes. 
 *
 * $Revision: 1.46 $
 * $Date: 1996/07/19 18:13:53 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalM.h"

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE readNBytes(sd, buf, nbytes)
 *
 * DESCRIPTION:
 * Read nbytes of data from sd into buf. Continue to read until
 * nbytes have been read.
 *
 * INPUTS:
 * int sd;
 * char *buf;
 * int32 nbytes;
 *
 * OUTPUTS: TCA_RETURN_STATUS
 *
 * NOTES:
 *
 * buf is a preallocated pointer to storage equal to nbytes.
 * Propagation of low level errors - failures on read/writes still an issue.
 *
 *****************************************************************************/

TCA_RETURN_STATUS_TYPE readNBytes(int sd, char *buf, int32 nbytes)
{
  int32 amountRead, amountToRead;
  
  amountToRead = nbytes;
  for(;;){
#ifdef _WINSOCK_
    amountRead = recv(sd, buf, amountToRead, 0);
#else
    amountRead = read(sd, buf, amountToRead);
#endif
    if (amountRead < 0)
      return StatError;
    if (!amountRead)
      return StatEOF;
    amountToRead -= amountRead;
    if (!amountToRead)
      return StatOK;
    buf += amountRead;
    /*    printf(".");*/
  }
}

#if defined(NEED_READV)

#define readv my_readv
#define writev my_writev

int32 my_readv (int fd, struct iovec *iov, int32 iovcnt)
{
  int32 i, totalCharsRead=0, charsRead;
  int32 readResult;
  for(i=0;i<iovcnt; i++){
    charsRead = 0;
    while (charsRead < iov[i].iov_len) {
#ifdef _WINSOCK_
      readResult = recv(fd,iov[i].iov_base+charsRead,
			min(iov[i].iov_len-charsRead,MAX_SOCKET_PACKET), 0);
#else
      readResult = read(fd,iov[i].iov_base+charsRead,
			MIN(iov[i].iov_len-charsRead,MAX_SOCKET_PACKET));
#endif
      if (readResult < 0) {
	/* Errror, return and  */
	return readResult;
      } else {
	/* Read some, update amounts. */
	charsRead += readResult;
	totalCharsRead += readResult;
      }
    }
  }
  return totalCharsRead;
}

int32 my_writev (int fd, struct iovec *iov, int32 iovcnt)
{
  int32 i, totalCharsWrite=0, charsWrite;
  int32 writeResult;
  for(i=0;i<iovcnt; i++){
    charsWrite = 0;
    while (charsWrite <  iov[i].iov_len) {
#ifdef _WINSOCK_
      writeResult = send(fd,iov[i].iov_base+charsWrite,
			 min(iov[i].iov_len-charsWrite,MAX_SOCKET_PACKET),0);
#else
      writeResult = write(fd,iov[i].iov_base+charsWrite,
			  MIN(iov[i].iov_len-charsWrite,MAX_SOCKET_PACKET));
#endif
      if (writeResult < 0) {
	/* Errror, return and  */
	return writeResult;
      } else {
	/* Write some, update amounts. */
	charsWrite += writeResult;
	totalCharsWrite += writeResult;
      }
    }
  }
  return totalCharsWrite;
  
}

#endif


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE read2Buffers(sd, buf1, nbytes,
 *                                               buf2, nbytes)
 *
 * DESCRIPTION:
 * Read nbytes of data from sd into buf. Continue to read until
 * nbytes have been read.
 *
 * INPUTS:
 * int sd;
 * char *buf1;
 * int32 nbytes1;
 * char *buf2;
 * int32 nbytes2;
 *
 * OUTPUTS: TCA_RETURN_STATUS
 *
 * NOTES:
 *
 * buf is a preallocated pointer to storage equal to nbytes.
 * Propagation of low level errors - failures on read/writes still an issue.
 *
 *****************************************************************************/

static TCA_RETURN_STATUS_TYPE read2Buffers(int sd,
					   char *buf1, int32 nbytes1,
					   char *buf2, int32 nbytes2)
{
  int32 amountRead, amountToRead;
  int32 numBuffers = 2;
  
  struct iovec vec[2];
  
  vec[0].iov_base = buf1;
  vec[0].iov_len = nbytes1;
  vec[1].iov_base = buf2;
  vec[1].iov_len = nbytes2;
  
  amountToRead = nbytes1 + nbytes2;
  for(;;){
    amountRead = readv(sd, vec, numBuffers);
    if (amountRead < 0)
      return StatError;
    if (!amountRead)
      return StatEOF;
    amountToRead -= amountRead;
    if (!amountToRead)
      return StatOK;
    /* Need to adjust buffers */
    if (amountToRead <= vec[1].iov_len) {
      /* Only need to use one buffer */
      numBuffers = 1;
      vec[0].iov_base = buf2 + nbytes2 - amountToRead;
      vec[0].iov_len = amountToRead;
    } else {
      /* Still need both buffers. */
      vec[0].iov_base += amountRead;
      vec[0].iov_len -= amountRead;
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE writeNBytes(sd, buf, nbytes)
 *
 * DESCRIPTION: This routine primarily calls the system function write.
 *
 * INPUTS:
 * int sd;
 * char *buf;
 * int32 nbytes;
 *
 * OUTPUTS: TCA_RETURN_STATUS_TYPE
 *
 *****************************************************************************/

TCA_RETURN_STATUS_TYPE writeNBytes(int sd, char *buf, int32 nbytes)
{
  int32 amountWritten = 0;
  
  while (nbytes > 0) {
    GET_M_GLOBAL(pipeBroken) = FALSE;
    errno = 0;
#ifdef _WINSOCK_
    amountWritten = send(sd, buf, nbytes,0);
#else
    amountWritten = write(sd, buf, nbytes);
#endif
    if ((GET_M_GLOBAL(pipeBroken)) || (errno == EPIPE)) {
      tcaModWarning( "\nWARNING: pipe broken!\n");
      return StatError;
    } else if (amountWritten < 0) {
#if defined(VXWORKS) || defined(THINK_C) || defined(applec)
      return StatError;
#else
#ifdef _WINSOCK_
      if (WSAGetLastError() == WSAEWOULDBLOCK)
#else
	if (errno == EWOULDBLOCK)
#endif
	  {
	    tcaModWarning(
			  "\nWARNING: writeNBytes: EWOULDBLOCK: trying again!\n");
	    PAUSE_MIN_DELAY();
	} else {
	  return StatError;
	}
#endif
    } else {
      nbytes -= amountWritten;
      buf += amountWritten;
    }
  }
  return StatOK;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE
 *           writeNBuffers(sd, struct iovec *vec)
 *
 * DESCRIPTION: This routine primarily calls the system function write.
 *
 * INPUTS:
 * int sd;
 * char *buf;
 * int32 nbytes;
 *
 * OUTPUTS: TCA_RETURN_STATUS_TYPE
 *
 *****************************************************************************/

static TCA_RETURN_STATUS_TYPE writeNBuffers(int sd, struct iovec *vec,
					    int32 amount)
{
  int32 amountWritten = 0;
  int32 numBuffers=0;
  int32 amountToWrite=0;
  int32 i, start=0;
  
  for (i=0; (vec[i].iov_base != NULL); i++) {
    numBuffers++;
    amountToWrite += vec[i].iov_len;
  }
  
  if (amountToWrite != amount) {
    tcaModError("Internal Error: Amounts incorrect in msg send.\n");
    return StatError;
  }
  
  while (amountToWrite > 0) {
    GET_M_GLOBAL(pipeBroken) = FALSE;
    errno = 0;
    amountWritten = writev(sd, &vec[start], numBuffers);
    if ((GET_M_GLOBAL(pipeBroken)) || (errno == EPIPE)) {
      tcaModWarning( "\nWARNING: pipe broken!\n");
      tcaFree((char *)vec);
      return StatError;
    } else if (amountWritten < 0) {
#if defined(VXWORKS) || defined(THINK_C) || defined(applec)
      tcaFree((char *)vec);
      return StatError;
#else
#ifdef _WINSOCK_
      if (WSAGetLastError() == WSAEWOULDBLOCK)
#else
	if (errno == EWOULDBLOCK)
#endif
	  {
	    tcaModWarning(
			  "\nWARNING: writeNBytes: EWOULDBLOCK: trying again!\n");
	    PAUSE_MIN_DELAY();
	} else {
	  tcaFree((char *)vec);
	  return StatError;
	}
#endif
    } else {
      amountToWrite -= amountWritten;
      if (amountToWrite > 0) {
	while (amountWritten > 0) {
	  if (vec[start].iov_len <= amountWritten) {
	    amountWritten -= vec[start].iov_len;
	    start++;
	    numBuffers--;
	  } else if (vec[start].iov_len > amountWritten) {
	    vec[start].iov_len -= amountWritten;
	    vec[start].iov_base += amountWritten;
	    amountWritten = 0;
	  }
	}
      }
    }
  }
  tcaFree((char *)vec);
  return StatOK;
}

/*****************************************************************************/

/*****************************************************************************/

void dataMsgInitialize(void)
{
  GET_M_GLOBAL(DMFree) = 0;
  GET_M_GLOBAL(DMTotal) = 0;
  GET_M_GLOBAL(DMmin) = 2000000000;
  GET_M_GLOBAL(DMmax) = -1;
  
  GET_M_GLOBAL(dataMsgBufferList) = listCreate();
}

/*****************************************************************************/

/*****************************************************************************/

static DATA_MSG_PTR dataMsgAlloc(int32 size)
{
  DATA_MSG_PTR dataMsg;
  
  dataMsg = (DATA_MSG_PTR)tcaMalloc((unsigned)size);
  
  (GET_M_GLOBAL(DMTotal))++;
  
  if (size > (GET_M_GLOBAL(DMmax))) (GET_M_GLOBAL(DMmax)) = size;
  if (size < (GET_M_GLOBAL(DMmin))) (GET_M_GLOBAL(DMmin)) = size;
  
  return (dataMsg);
}


/******************************************************************************
 *
 * FUNCTION: void dataMsgFree(dataMsg)
 *
 * DESCRIPTION: Recycle a datamsg.
 *
 * INPUTS: DATA_MSG_PTR dataMsg
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

DATA_MSG_PTR dataMsgFree(DATA_MSG_PTR dataMsg)
{
  if (dataMsg) {
    (dataMsg->refCount)--;
    if (dataMsg->refCount < 1) {
      if (dataMsg->dataRefCountPtr != NULL) {
	(*(dataMsg->dataRefCountPtr))--;
	if (*(dataMsg->dataRefCountPtr) < 1) {
	  /* Can now free the data. */
	  tcaFree((char *)dataMsg->dataRefCountPtr);
	  dataMsg->dataRefCountPtr = NULL;
 	  if ((dataMsg->msgData != NULL) &&
 	      (dataMsg->msgData != dataMsg->dataStruct)) {
	    tcaFree((char *)dataMsg->msgData);
	  }
	}
      }
      if (dataMsg->vec != NULL)
	tcaFree((char *)dataMsg->vec);
      dataMsg->vec = NULL;
      
      tcaFree((char *)dataMsg);
      (GET_M_GLOBAL(DMFree))++;
      dataMsg = NULL;
    }
  }
  return dataMsg;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE dataMsgRecv(sd, dataMsg)
 *
 * DESCRIPTION:
 * Creates a dataMsg and initialize it with information from the socket
 * specified by the value sd. If there is an error in reading the
 * information dataMsg will be initialized to NULL. The communication
 * status is returned.
 *
 * INPUTS:
 * int sd;
 * DATA_MSG_PTR *dataMsg;
 *
 * OUTPUTS: TCA_RETURN_STATUS_TYPE
 *
 *****************************************************************************/

TCA_RETURN_STATUS_TYPE dataMsgRecv(int sd, DATA_MSG_PTR *dataMsg,
				   int32 replyRef, void *replyBuf, 
				   int32 replyLen)
{
  TCA_RETURN_STATUS_TYPE status;
  
  DATA_MSG_TYPE header;
  
  *dataMsg = NULL;
  
  status = readNBytes(sd, (char *)&(header.classTotal),
		      sizeof(DATA_MSG_TYPE) -
		      offsetof(DATA_MSG_TYPE, classTotal) );
  if (status != StatOK) {
    *dataMsg = NULL;
    return status;
  }
  
  NET_INT_TO_INT(header.classTotal);
  NET_INT_TO_INT(header.msgTotal);
  
  *dataMsg = dataMsgAlloc(header.classTotal + sizeof(DATA_MSG_TYPE));
  **dataMsg = header;
  
  NET_INT_TO_INT((*dataMsg)->parentRef);
  NET_INT_TO_INT((*dataMsg)->intent);
  NET_INT_TO_INT((*dataMsg)->classId);
  NET_INT_TO_INT((*dataMsg)->dispatchRef);
  NET_INT_TO_INT((*dataMsg)->msgRef);
  
  if( header.msgTotal > 0) {
    (*dataMsg)->dataRefCountPtr = tcaMalloc(sizeof(int32));
    *((*dataMsg)->dataRefCountPtr) = 1;
  } else {
    (*dataMsg)->dataRefCountPtr = NULL;
  }
  (*dataMsg)->refCount = 0;
  (*dataMsg)->dataStruct = NULL;
  (*dataMsg)->dataByteOrder = GET_DATA_ENDIAN((*dataMsg)->classId);
  (*dataMsg)->classByteOrder = GET_CLASS_ENDIAN((*dataMsg)->classId);
  (*dataMsg)->alignment = GET_ALIGNMENT((*dataMsg)->classId);
  (*dataMsg)->classId = GET_CLASSID((*dataMsg)->classId);
#if defined(LISP)
  GET_M_GLOBAL(byteOrder) = (*dataMsg)->dataByteOrder;
  GET_M_GLOBAL(alignment) = (*dataMsg)->alignment;
#endif
  
  if (header.classTotal > 0)
    (*dataMsg)->classData = ((char *)*dataMsg + sizeof(DATA_MSG_TYPE));
  else
    (*dataMsg)->classData = NULL;
  
  /*  For now, we only handle packed data. */
  if ((*dataMsg)->alignment != ALIGN_PACKED) {
    tcaModError("ERROR: received message with data that is not packed.");
    return StatError;
  }
  
  /* Want to be able to use the already allocated buffer, if possible. */
  
  if (((*dataMsg)->msgRef == replyRef) && (replyBuf != NULL) &&
      (replyLen == header.msgTotal)) {
    (*dataMsg)->msgData = replyBuf;
    (*dataMsg)->dataStruct = replyBuf;
  } else if (header.msgTotal > 0)
    (*dataMsg)->msgData = tcaMalloc((unsigned) header.msgTotal);
  else
    (*dataMsg)->msgData = NULL;
  
  if ((header.msgTotal > 0) && (header.classTotal >0)) {
    status = read2Buffers(sd,
			  (*dataMsg)->classData, header.classTotal,
			  (*dataMsg)->msgData, header.msgTotal
			  );
  } else if (header.classTotal >0) {
    status = readNBytes(sd, (*dataMsg)->classData, header.classTotal);
  } else if (header.msgTotal >0) {
    status = readNBytes(sd, (*dataMsg)->msgData, header.msgTotal);
    
  }
  
  /* Need to create the vector here.  */
  (*dataMsg)->vec = (struct iovec *) tcaMalloc(2 * sizeof(struct iovec));
  
  (*dataMsg)->vec[0].iov_base = (*dataMsg)->msgData;
  (*dataMsg)->vec[0].iov_len = (*dataMsg)->msgTotal;
  
  (*dataMsg)->vec[1].iov_base = NULL;
  (*dataMsg)->vec[1].iov_len = 0;
#if defined(LISP)
  /*  { int i;*/
  /*    printf("Received\n ");*/
  /*    for (i=0; i<(*dataMsg)->msgTotal; i++)*/
  /*      printf("%d ",(int32)((*dataMsg)->msgData[i]));*/
  /*    printf("\n ");*/
  /*    FLUSH_IF_NEEDED(stdout);*/
  /*  }*/
#endif
  
  return status;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_STATUS_TYPE dataMsgSend(sd, dataMsg)
 *
 * DESCRIPTION:
 * Sends the information in dataMsg to the socket specified by sd. The
 * status of the communication will be returned.
 *
 * INPUTS:
 * int sd;
 * DATA_MSG_PTR dataMsg;
 *
 * OUTPUTS: TCA_RETURN_STATUS_TYPE
 * NOTES: Switch byte order for 486 and pmax machines, so they can talk
 *        to SUN machines.
 *
 *****************************************************************************/

TCA_RETURN_STATUS_TYPE dataMsgSend(int sd, DATA_MSG_PTR dataMsg)
{
  int32 headerAmount, classAmount, dataAmount;
  TCA_RETURN_STATUS_TYPE res;
  char *sendInfo;
  struct iovec *tmpVec;
  
  headerAmount = 7*sizeof(int32);
  classAmount = dataMsg->classTotal;
  dataAmount = dataMsg->msgTotal;
  
  sendInfo = (char *)&(dataMsg->classTotal);
  dataMsg->classId = SET_DATA_ENDIAN(dataMsg->classId,dataMsg->dataByteOrder);
  dataMsg->classId = SET_CLASS_ENDIAN(dataMsg->classId,
				      dataMsg->classByteOrder);
  dataMsg->classId = SET_ALIGNMENT(dataMsg->classId);
  
  INT_TO_NET_INT(dataMsg->classTotal);
  INT_TO_NET_INT(dataMsg->msgTotal);
  INT_TO_NET_INT(dataMsg->parentRef);
  INT_TO_NET_INT(dataMsg->intent);
  INT_TO_NET_INT(dataMsg->classId);
  INT_TO_NET_INT(dataMsg->dispatchRef);
  INT_TO_NET_INT(dataMsg->msgRef);
  
  if (classAmount > 0)  {
    tmpVec = copyVectorization(dataMsg->vec,2);
    tmpVec[0].iov_base = sendInfo;
    tmpVec[0].iov_len = headerAmount;
    tmpVec[1].iov_base = dataMsg->classData;
    tmpVec[1].iov_len = classAmount;
    res = writeNBuffers(sd, tmpVec,headerAmount+classAmount+dataAmount);
  } else if (dataAmount > 0) {
    tmpVec = copyVectorization(dataMsg->vec,1);
    tmpVec[0].iov_base = sendInfo;
    tmpVec[0].iov_len = headerAmount;
    res = writeNBuffers(sd, tmpVec,headerAmount+classAmount+dataAmount);
  } else {
    res = writeNBytes(sd, sendInfo, headerAmount);
  }
  
  NET_INT_TO_INT(dataMsg->classTotal);
  NET_INT_TO_INT(dataMsg->msgTotal);
  NET_INT_TO_INT(dataMsg->parentRef);
  NET_INT_TO_INT(dataMsg->intent);
  NET_INT_TO_INT(dataMsg->classId);
  NET_INT_TO_INT(dataMsg->dispatchRef);
  NET_INT_TO_INT(dataMsg->msgRef);
  
  dataMsg->classId = GET_CLASSID(dataMsg->classId);
  
  return res;
}


/******************************************************************************
 *
 * FUNCTION:
 *
 * DESCRIPTION:
 * Creates a new dataMsg and encodes data for sending based on Format. If
 * data, or Format or both are NULL then a new dataMsg is still created
 * but only Header Information is initialized.
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 *****************************************************************************/

DATA_MSG_PTR dataMsgCreate(int32 parentRef, int32 intent, int32 classId,
			   int32 dispatchRef, int32 msgRef,
			   CONST_FORMAT_PTR msgFormat,
			   const void *msgData,
			   CONST_FORMAT_PTR classFormat,
			   const void *classData)
{
#if defined(LISP)
  BUFFER_TYPE buffer;
#endif
  DATA_MSG_PTR dataMsg;
  int32 classTotal, msgTotal;
  
  msgTotal = 0;
  
  if (msgData && msgFormat) {
    if (msgFormat->type == BadFormatFMT) return NULL;
#if defined(LISP)
    if (msgData == &(GET_M_GLOBAL(lispFlagGlobal)))
      (*(GET_M_GLOBAL(lispBufferSizeGlobal)))(&msgTotal, msgFormat,
					      lisp_value(0));
    else
#endif
      msgTotal = bufferSize(msgFormat, msgData);
  }
  
  if (classData && classFormat)
    classTotal = bufferSize(classFormat, classData);
  else
    classTotal = 0;
  
  dataMsg = dataMsgAlloc(classTotal + sizeof(DATA_MSG_TYPE));
  
  dataMsg->dataRefCountPtr = (int32 *)tcaMalloc(sizeof(int32));
  *(dataMsg->dataRefCountPtr) = 1;
  dataMsg->refCount = 0;
  dataMsg->vec = NULL;
  
  dataMsg->msgTotal = msgTotal;
  dataMsg->classTotal = classTotal;
  
  dataMsg->parentRef = parentRef;
  dataMsg->intent = intent;
  dataMsg->classId = classId;
  dataMsg->dispatchRef = dispatchRef;
  dataMsg->msgRef = msgRef;
  
#if (LISP)
  if ((msgTotal != 0) && (msgData == &(GET_M_GLOBAL(lispFlagGlobal)))) {
    dataMsg->msgData = tcaMalloc(msgTotal);
    buffer.buffer = dataMsg->msgData;
    buffer.bstart = 0;
    (*(GET_M_GLOBAL(lispEncodeMsgGlobal)))(msgFormat,
					   lisp_value(0), &buffer);
    dataMsg->vec = createVectorization(msgFormat,msgData,dataMsg->msgData,
				       msgTotal);
  } else {
    encodeMsgData(msgFormat, msgData, dataMsg, 0);
  }
#else
  encodeMsgData(msgFormat, msgData, dataMsg, 0);
#endif
  
  if (classTotal) {
    dataMsg->classData = (char *)dataMsg + sizeof(DATA_MSG_TYPE);
    encodeData(classFormat, classData, dataMsg->classData, 0, classTotal);
  }
  else
    dataMsg->classData = NULL;
  
  /* Include the pointer to the original data. */
  dataMsg->dataStruct = msgData;
  dataMsg->dataByteOrder = BYTE_ORDER;
  dataMsg->classByteOrder = BYTE_ORDER;
  dataMsg->alignment = ALIGN;
  
  return dataMsg;
}

void encodeMsgData(CONST_FORMAT_PTR Format, const void *DataStruct,
		   DATA_MSG_PTR dataMsg, int32 BStart)
{
  dataMsg->dataStruct = DataStruct;
  
  if (Format == NULL) {
    dataMsg->msgData = NULL;
  } else {
    
#if (!defined(LISP))
    if (sameFixedSizeDataBuffer(Format)) {
      dataMsg->msgData = (char *)DataStruct;
    } else {
      dataMsg->msgData = (char *)tcaMalloc((unsigned)dataMsg->msgTotal);
      encodeData(Format, DataStruct, dataMsg->msgData, BStart, 
		 dataMsg->msgTotal);
    }
#else
    dataMsg->msgData = (char *)tcaMalloc((unsigned)dataMsg->msgTotal);
    encodeData(Format, DataStruct, dataMsg->msgData, BStart, 
	       dataMsg->msgTotal);
#endif
  }
  dataMsg->vec = createVectorization(Format,DataStruct,dataMsg->msgData,
				     dataMsg->msgTotal);
}

/*************************************************************/

void *decodeMsgData(CONST_FORMAT_PTR Format,  DATA_MSG_PTR dataMsg,
		    BOOLEAN keepData)
{
  char *DataStruct=NULL;
#if (!defined(LISP))
  if (sameFixedSizeDataBuffer(Format)) {
    dataMsg->dataStruct = dataMsg->msgData;
    DataStruct = dataMsg->msgData;
    if (dataMsg->dataByteOrder == BYTE_ORDER)
      /* This is now a no op */
      return ((void *)dataMsg->msgData);
  }
#endif
  /* Have to keep it anyway, probably for sending after printing.*/
  if (keepData) {
    DataStruct = NULL;
    dataMsg->dataStruct = NULL;
  }
  
  return decodeData(Format, dataMsg->msgData, 0, DataStruct,
		    dataMsg->dataByteOrder, dataMsg->alignment, 
		    dataMsg->msgTotal);
}


/******************************************************************************
 *
 * FUNCTION: char *dataMsgDecodeMsg(msgFormat, dataMsg)
 *
 * DESCRIPTION:
 * Returns the message data defined by msgFormat from the information
 * encoded in dataMsg. If there is no data to decode, dataMsgDecodeMsg
 * returns NULL.
 *
 * INPUTS:
 * CONST_FORMAT_PTR msgFormat;
 * DATA_MSG_PTR dataMsg;
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

void *dataMsgDecodeMsg(CONST_FORMAT_PTR msgFormat, DATA_MSG_PTR dataMsg,
		       BOOLEAN keepData)
{
  if (!msgFormat || (msgFormat == BAD_FORMAT) || !dataMsg->msgTotal)
    return NULL;
  else
    return decodeMsgData(msgFormat, dataMsg, keepData);
}


/******************************************************************************
 *
 * FUNCTION: char *dataMsgDecodeClass(classFormat, dataMsg)
 *
 * DESCRIPTION:
 * Returns the message data defined by classFormat from the information
 * encoded in dataMsg. If there is no data to decode, dataMsgDecodeMsg
 * returns NULL.
 *
 * INPUTS:
 * CONST_FORMAT_PTR classFormat;
 * DATA_MSG_PTR dataMsg;
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

void *dataMsgDecodeClass(CONST_FORMAT_PTR classFormat, DATA_MSG_PTR dataMsg)
{
  if (!classFormat || !dataMsg->classTotal)
    return NULL;
  else
    return decodeData(classFormat, dataMsg->classData, 0, NULL,
		      dataMsg->classByteOrder, dataMsg->alignment,
		      dataMsg->classTotal);
}


/******************************************************************************
 *
 * FUNCTION: int32 dataMsgTestMsgData(dataMsg)
 *
 * DESCRIPTION:
 * Returns TRUE if the dataMsg number of message bytes is greater than zero.
 * Returns FALSE otherwise or if dataMsg is NULL.
 *
 * INPUTS: DATA_MSG_PTR dataMsg
 *
 * OUTPUTS: int32.
 *
 *****************************************************************************/

BOOLEAN dataMsgTestMsgData(DATA_MSG_PTR dataMsg)
{
  if (dataMsg && dataMsg->msgTotal > 0)
    return TRUE;
  else
    return FALSE;
}


/******************************************************************************
 *
 * FUNCTION: DATA_MSG_PTR dataMsgReplaceClassData(format, data, dataMsg)
 *
 * DESCRIPTION:
 * Replaces class information on an existing dataMsg with data.
 *
 * INPUTS:
 * CONST_FORMAT_PTR format;
 * char *data;
 * DATA_MSG_PTR dataMsg;
 *
 * OUTPUTS: DATA_MSG_PTR
 *
 *****************************************************************************/

DATA_MSG_PTR dataMsgReplaceClassData(CONST_FORMAT_PTR classFormat,
				     void *data,
				     DATA_MSG_PTR dataMsg,
				     CONST_FORMAT_PTR msgFormat)
{
  DATA_MSG_PTR newDataMsg;
  int32 classTotal=0;
  
  if (data && classFormat)
    classTotal = bufferSize(classFormat, data);
  
  newDataMsg = dataMsgAlloc(sizeof(DATA_MSG_TYPE)+classTotal);
  
  /* 3-Sep-90: fedor: this should work because class info is in the
     dataMsg struture as the last item. */
  
  BCOPY((char *)dataMsg, (char *)newDataMsg, sizeof(DATA_MSG_TYPE));
  
  if (classTotal) {
    newDataMsg->classData = ((char *)newDataMsg + sizeof(DATA_MSG_TYPE));
    encodeData(classFormat, data, newDataMsg->classData, 0, classTotal);
    newDataMsg->classByteOrder = BYTE_ORDER;
  }
  else
    newDataMsg->classData = NULL;
  
  /* reset msgData pointer */
  newDataMsg->msgData = dataMsg->msgData;
  newDataMsg->dataStruct = dataMsg->dataStruct;
  dataMsg->msgData = NULL;
  
  /* Need to copy the vector data. */
  if (dataMsg->vec != NULL)
    newDataMsg->vec = copyVectorization(dataMsg->vec,0);
  else {
    tcaModError("Internal Error: Missing data Vector\n");
    return NULL;
  }
  
  /* set refCount */
  newDataMsg->dataRefCountPtr = dataMsg->dataRefCountPtr;
  if ( newDataMsg->dataRefCountPtr != NULL)
    (*(newDataMsg->dataRefCountPtr))++;
  newDataMsg->refCount = 0;
  
  newDataMsg->classTotal = classTotal;
  
  if (dataMsgFree(dataMsg) != NULL)
    dataMsg->msgData = newDataMsg->msgData;
  
  return newDataMsg;
}


/******************************************************************************
 *
 * FUNCTION: void dataMsgDisplayStats(stream)
 *
 * DESCRIPTION:
 * Display datamsg memory usage stats.
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static int32 countDataMessages(int32 allocatedP, DATA_MSG_BUF_PTR dataMsgBuf)
{
  if (dataMsgBuf->allocated == allocatedP) {
    GET_M_GLOBAL(numAllocatedDM)++;
    GET_M_GLOBAL(sizeDM) += dataMsgBuf->size;
  }
  return TRUE;
}

void dataMsgDisplayStats(FILE *stream)
{
  fprintf(stream,"Data Msg Buffer Stats\n");
  fprintf(stream,"Total Alloc  : %d\n", GET_M_GLOBAL(DMTotal));
  fprintf(stream,"Total Freed  : %d\n", GET_M_GLOBAL(DMFree));
  fprintf(stream,"Min Request  : %d\n", GET_M_GLOBAL(DMmin));
  fprintf(stream,"Max Request  : %d\n", GET_M_GLOBAL(DMmax));
  fprintf(stream,"Buf List Size: %d\n",
	  listLength(GET_M_GLOBAL(dataMsgBufferList)));
  
  GET_M_GLOBAL(numAllocatedDM) = GET_M_GLOBAL(sizeDM) = 0;
  listIterateFromFirst((LIST_ITER_FN)countDataMessages, (char *)TRUE,
		       (GET_M_GLOBAL(dataMsgBufferList)));
  fprintf(stream,"  %d allocated, with %d total bytes\n",
	  GET_M_GLOBAL(numAllocatedDM),
	  GET_M_GLOBAL(sizeDM));
  GET_M_GLOBAL(numAllocatedDM) = GET_M_GLOBAL(sizeDM) = 0;
  listIterateFromFirst((LIST_ITER_FN)countDataMessages, FALSE,
		       (GET_M_GLOBAL(dataMsgBufferList)));
  fprintf(stream,"  %d deallocated, with %d total bytes\n",
	  GET_M_GLOBAL(numAllocatedDM),
	  GET_M_GLOBAL(sizeDM));
  
  FLUSH_IF_NEEDED(stream);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaAllocateDataBuffer()
 *
 * DESCRIPTION:
 * Pre-allocate a data message buffer of the requested size
 *
 * INPUTS: int32 size -- number of bytes to pre-allocate.
 *
 * OUTPUTS: Returns "Success" if the buffer could be allocated,
 *          "Failure" otherwise
 *
 * NOTES: Currently, since "dataMsgBufferCreate" has no return value,
 *        "tcaAllocateDataBuffer" always returns Success (or bombs
 *        out within "dataMsgBufferCreate").
 *****************************************************************************/

#if 0
/* no longer used */
static TCA_RETURN_VALUE_TYPE tcaAllocateDataBuffer(int32 size)
{
  DATA_MSG_BUF_PTR dataMsgBuf;
  
  /* Creates an unallocated data message buffer */
  dataMsgBuf = dataMsgBufferCreate(size);
  listInsertItem((char *)dataMsgBuf, (GET_M_GLOBAL(dataMsgBufferList)));
  
  if (size > (GET_M_GLOBAL(DMmax))) (GET_M_GLOBAL(DMmax)) = size;
  if (size < (GET_M_GLOBAL(DMmin))) (GET_M_GLOBAL(DMmin)) = size;
  
  return Success;
}
#endif
