/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: communications
 *
 * FILE: comModule.c
 *
 * ABSTRACT:
 * 
 * Module Communications
 *
 * REVISION HISTORY 
 *
 * $Log: comModule.c,v $
 * Revision 1.78  1996/08/22  16:35:51  rich
 * Check the return code on tcaQueryCentral calls.
 *
 * Revision 1.77  1996/07/25  22:24:09  rich
 * Fixed some memory leaks with handlers and removed some problems where a
 * disconnection caused a cleanup, but a variable would be assumed to still
 * be valid.
 *
 * Revision 1.76  1996/07/03  21:43:28  reids
 * Have "tcaWaitUntilReady" print out any required resources it is waiting for
 *
 * Revision 1.75  1996/06/27  15:40:01  rich
 * Added tcaGetAcceptFds.
 *
 * Revision 1.74  1996/06/25  20:50:16  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.73  1996/05/14  22:40:03  rich
 * Need the check for valid context before dereferencing.
 *
 * Revision 1.72  1996/05/09  18:30:31  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.71  1996/03/29  21:28:48  reids
 * Fixed a problem with having a NULL format as the reply format of a query.
 *
 * Revision 1.70  1996/03/15  21:27:52  reids
 * Added tcaQueryNotify, tcaAddEventHandler, tcaRemoveEventHandler.
 *   Fixed re-registration bug; Plugged memory leaks; Fixed way task trees
 *   are killed; Added support for "enum" format type.
 *
 * Revision 1.69  1996/03/05  05:04:18  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.68  1996/03/04  18:32:08  rich
 * Fixed problem with recursive queries.
 *
 * Revision 1.67  1996/03/02  03:21:24  rich
 * Fixed memory leaks found using purify.
 *
 * Revision 1.66  1996/02/21  18:30:10  rich
 * Created single event loop.
 *
 * Revision 1.65  1996/02/12  17:42:02  rich
 * Handle direct connection disconnect/reconnect.
 *
 * Revision 1.64  1996/02/11  21:34:54  rich
 * Updated GNUmakefiles for faster complilation.  Use FAST_COMPILE=1 for
 * routine recompiles.
 *
 * Revision 1.63  1996/02/10  16:49:37  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.62  1996/02/06  19:04:20  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.61  1996/01/27  21:53:01  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.60  1996/01/23  00:06:24  rich
 * Fixed memory leak when a module connects and disconnects.  Also fixed a
 * problem with using the direct connection flag.  This was introduced when
 * we added contexts for keeping track of the central server.
 *
 * Revision 1.59  1996/01/05  16:31:05  rich
 * Added windows NT port.
 *
 * Revision 1.58  1995/12/21  19:17:35  reids
 * For safety's sake, copy strings when registering messages, handlers and
 *    when connecting.
 *
 * Revision 1.57  1995/12/17  20:21:13  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.56  1995/12/15  01:23:06  rich
 * Moved Makefile to Makefile.generic to encourage people to use
 * GNUmakefile.
 * Fixed a memory leak when a module is closed and some other small fixes.
 *
 * Revision 1.55  1995/11/03  03:04:18  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.54  1995/10/29  18:26:29  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 * 
 * Revision 1.53.2.3  1995/12/19  04:26:37  reids
 * For safety's sake, copy strings when registering messages, handlers and
 *   when connecting.
 *
 * Revision 1.53.2.2  1995/11/02  22:42:13  reids
 * Fixed a problem with incorrectly setting modData.hostName
 *
 * Revision 1.53.2.1  1995/10/27  15:16:40  rich
 * Fixed problems with connecting to multiple central servers.
 *
 * Revision 1.53  1995/10/25  22:48:00  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.52  1995/10/07  19:07:02  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.51  1995/08/14  21:31:19  rich
 * Got rid of the "sharedBuffers" flag on the dataMessages.  It was not the
 * right solution, and possibly caused a memory leak.
 * Limit pending for TCA_TAPPED_MSG_VAR to 1.
 *
 * Revision 1.50  1995/08/06  16:43:43  reids
 * A bug existed in that two demon monitors that sent the same ID number
 * would conflict (causing the wrong one to fire).  This has been fixed, and
 * in the process, one of the hash-key functions was made a bit more general.
 *
 * Revision 1.49  1995/08/05  17:16:16  reids
 * Several important bug fixes:
 *   a) Found a memory leak in the tms (when nodes are unasserted)
 *   b) Fixed a problem with direct connections that would cause TCA to crash
 *      when a command or goal message was sent from an inform or a query.
 *      As part of that fix, all command and goal messages that are sent from
 *      informs or queries are now added to the root node of the task tree.
 *
 * Revision 1.48  1995/07/19  14:25:52  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.47  1995/07/12  04:54:24  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.46  1995/07/10  16:16:57  rich
 * Interm save.
 *
 * Revision 1.45  1995/07/06  21:15:47  rich
 * Solaris and Linux changes.
 *
 * Revision 1.44  1995/06/14  03:21:26  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.43  1995/06/05  23:58:46  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.42  1995/04/21  03:53:12  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.41  1995/04/19  14:27:50  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.40  1995/04/09  20:30:07  rich
 * Added /usr/local/include and /usr/local/lib to the paths for compiling
 * for sunOS machines. (Support for new vendor OS).
 * Create a tca directory in /tmp and put the socket in there so others can
 * delete dead sockets.  The /tmp directory has the sticky bit set so you
 * can't delete files even if you have write permission on the directory.
 * Fixes to libc.h to use the new declarations in the gcc header files and
 * avoid problems with dbmalloc.
 *
 * Revision 1.39  1995/04/08  02:06:19  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.38  1995/04/07  05:02:55  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.37  1995/04/05  19:10:36  rich
 * Adding Reid's Changes.
 * Consolidating the matrix code.
 *
 * Revision 1.36  1995/04/04  19:41:50  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.35  1995/03/30  15:42:32  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.34  1995/03/28  01:14:22  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.33  1995/03/19  19:39:22  rich
 * Implemented direct connections using tcaDirectResouce call.
 * Also made the basics.h file a module include.
 * Changed class in the interval structure to be interval_class to avoid a
 * conflict with C++.
 *
 * Revision 1.32  1995/03/18  15:11:00  rich
 * Fixed updateVersion script so it can be run from any directory.
 *
 * Revision 1.31  1995/02/06  14:46:16  reids
 * Removed the "Global" suffix from tcaRootNode, tcaServer and tcaDefaultTime
 *
 * Revision 1.30  1995/01/30  16:17:40  rich
 * Added indications of message byte order and alignment to the message
 * header in the upper two bytes of the classID.
 * Now the receiver translates the byte order but must receive data in
 * "PACKED" alignment.
 * Made -gstabs come after -g for i386_mach machines so the debugger will work.
 *
 * Revision 1.29  1995/01/25  00:00:59  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Fixed problems with little endian translation.
 *
 * Revision 1.28  1995/01/18  22:39:52  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.27  1994/11/02  21:34:00  rich
 * Now works for linux machines (i486).
 * Got afs to work on alpha (and hopefully other vendor OS's)
 * Added generic Makefile.
 * Made libc.h and tcaMatrix.h module includes.
 * Reduced the size of libc.h by using more system includes.
 *
 * Revision 1.26  1994/11/02  16:14:26  reids
 * Fixed connectSocket to handle server hosts specified as <host>:<port>
 *
 * Revision 1.25  1994/10/25  17:09:41  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.24  1994/06/07  02:30:22  rich
 * Include the top level README and other files in the tarfile.
 * Include extra header infomation to get rid of warnings.
 *
 * Revision 1.23  1994/05/31  03:23:44  rich
 * Removed CFLAGS_sun4.
 * Removed cfree and bzero from mem routines.
 * Set zero wait on both sides of the pipe.  Can connect to host using inet
 * number.
 *
 * Revision 1.22  1994/05/25  04:57:06  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.21  1994/05/24  13:48:31  reids
 * Fixed so that messages are not sent until a tcaWaitUntilReady is received
 * (and the expected number of modules have all connected)
 *
 * Revision 1.20  1994/05/17  23:15:20  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.19  1994/05/11  01:57:12  rich
 * Now set an invalid tcaServerGlobal (a socket fd) to -1 rather than 0
 * which is stdout.
 * Added checks to make sure tcaServerGlobal is a valid socket before
 * sending messages or waiting for messages.
 *
 * Revision 1.18  1994/05/05  00:46:03  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.17  1994/04/28  16:15:25  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.16  1994/04/26  16:23:15  rich
 * Now you can register an exit handler before anything else and it will
 * get called if connecting to central fails.
 * Also added code to handle pipe breaks during writes.
 *
 * Revision 1.15  1994/04/16  19:41:45  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.14  1994/04/15  17:09:48  reids
 * Changes to support vxWorks version of TCA 7.5
 *
 * Revision 1.13  1994/04/04  16:42:48  reids
 * Fixed a bug found by Rich in waitForReply2
 *
 * Revision 1.12  1994/03/27  22:50:14  rich
 * Fixed problem with lisp version not working because of some compiler
 * flags used for the shared library version.
 * TCA now compiles for alphas, but does not run.
 *
 * Revision 1.11  1993/12/14  17:33:02  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.10  1993/12/01  18:57:44  rich
 * Changed TCP to  IPPROTO_TCP, a system defined constant.
 *
 * Revision 1.9  1993/12/01  18:02:53  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.8  1993/11/21  20:17:21  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.7  1993/10/21  16:13:36  rich
 * Fixed compiler warnings.
 *
 * Revision 1.6  1993/10/20  19:00:25  rich
 * Fixed bug with self registed messages in the lisp version.
 * Added new routine : tcaGetServerGlobal to get the server socket.
 * Fixed some bad global references for the lisp version.
 * Updated some prototypes.
 *
 * Revision 1.5  1993/08/30  21:53:12  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.4  1993/08/27  07:14:19  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/08/20  23:06:37  fedor
 * Minor changes for merge. Mostly added htons and removed cfree calls.
 *
 * Revision 1.2  1993/05/26  23:16:59  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:16  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:18  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:19  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 * 6-May-92 Christopher Fedor, School of Computer Science, CMU
 * Added while loop to select if select dies on an EINTR then retry select.
 *
 * 17-Feb-92 Christopher Fedor, School of Computer Science, CMU
 * Generalized call to a registered exit handler tcaExitHnd for all modules
 * except lisp. VxWorks no longer has a special tcaVxExit routine.
 *
 * 28-Oct-91 Christopher Fedor, School of Computer Science, CMU
 * Moved call for tcaVxExit so that it will be called when a close is detected
 * by both a direct connection and central disconnect. Currently the VxWorks 
 * part is not set up for direct connections and the exit call was only 
 * happening for direct connections.
 *
 * 14-Aug-91 Christopher Fedor, School of Computer Science, CMU
 * Changed VxWorks include files to vx5.0
 *
 * 25-Mar-91 Christopher Fedor, School of Computer Science, CMU
 * Added lispExitGlobal as the VxWorks way of exiting lisp.
 * Should eventually add a general abort handler and exit handler.
 *
 * 30-Jan-91 Christopher Fedor, School of Computer Science, CMU
 * Added fflush(stdout) to printf for module code calls from lisp
 *
 *  6-Dec-90 Christopher Fedor, School of Computer Science, CMU
 * Removed signal.h because the signal to call tcaClose on various signals
 * was commeted out and signal.h for sunos 4.1 eventually includes
 * sys/stdtypes which defines size_t as an int - this conflicts with the
 * vxworks includes which eventually include types.h which defines
 * size_t as a long.
 * Move sys/time.h to non VxWorks and added vxworks utime.h because the
 * timeval struct is needed and the size_t conflict exits.
 *
 *  7-Apr-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 *
 *  6-Mar-89 Christopher Fedor, School of Computer Science, CMU
 * Created.  
 * fedor: 6-Mar-89 problem with many defined globals from com_server and lots
 * of repetitive code needs to be removed!  perhaps a com_com file for
 * common communications between the two - actually i want to avoid 
 * differences - module and server should be alike as much as possible.
 *
 * $Revision: 1.78 $
 * $Date: 1996/08/22 16:35:51 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"
#include <stdarg.h>

#ifdef NMP_IPC
#include "parseFmttrs.h"
#include "printData.h"
#endif

#if defined(LISP)
/******************************************************************************
 * Forward Declarations
 *****************************************************************************/
TCA_REF_PTR returnRootGlobal(void);
#endif
static TCA_RETURN_VALUE_TYPE processQueryReply(DATA_MSG_PTR dataMsg, 
					       MSG_PTR msg, BOOLEAN sel,
					       void *reply);
TCA_RETURN_VALUE_TYPE waitForReplyFromTime(TCA_REF_PTR ref, void *reply, 
					   BOOLEAN sel, struct timeval *time,
					   int fd);


/******************************************************************************
 *
 * FUNCTION: void printDataModuleInitialize();
 *
 * DESCRIPTION: 
 * Sets print data routines for modules.
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

#ifndef NMP_IPC
/* ARGSUSED */
static void sampleModulePrintData(FILE *stream, const void *item, int32 next)
{
#ifdef applec
#pragma unused(stream, item, next)
#endif
  tcaModWarning( "\nWARNING: DPrint called!\n");
}

/* ARGSUSED */
static void sampleModuleMapPrintData(FILE *stream, void *map, 
				     CONST_FORMAT_PTR format,
				     int32 next, void *mapElements)
{
#ifdef applec
#pragma unused(stream, map, format, next, mapElements)
#endif
  tcaModWarning( "\nWARNING: DPrint on maps called!\n");
}

static void printDataModuleInitialize(void)
{
  GET_M_GLOBAL(dPrintSTR_FN) = (void (*)(FILE *, const char *, int32)) 
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintFORMAT_FN) = (void (*)(FILE *, CONST_FORMAT_PTR, int32))
    sampleModulePrintData;
#ifndef NMP_IPC
  GET_M_GLOBAL(dPrintMAP_FN) = (void (*)(FILE *, const genericMatrix *,
					 CONST_FORMAT_PTR,
					 int, const void *))
    sampleModuleMapPrintData;
#endif
  GET_M_GLOBAL(dPrintTCA_FN) = (void (*)(FILE *, const TCA_REF_TYPE *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintCHAR_FN) = (void (*)(FILE *, const char *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintSHORT_FN) = (void (*)(FILE *, const int16 *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintLONG_FN) = (void (*)(FILE *, const long *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintINT_FN) = (void (*)(FILE *, const int32 *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintBOOLEAN_FN) = (void (*)(FILE *, const int32 *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintFLOAT_FN) = (void (*)(FILE *, const float *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintDOUBLE_FN) = (void (*)(FILE *, const double *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintBYTE_FN) = (void (*)(FILE *, const char *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintUBYTE_FN) = (void (*)(FILE *, const char *, int))
    sampleModulePrintData;
  GET_M_GLOBAL(dPrintTWOBYTE_FN) = (void (*)(FILE *, const char *, int))
    sampleModulePrintData;
}
#else
static void printDataModuleInitialize(void)
{
  GET_M_GLOBAL(byteFormat) = ParseFormatString("byte");
  GET_M_GLOBAL(charFormat) = ParseFormatString("char");
  GET_M_GLOBAL(shortFormat) = ParseFormatString("short");
  GET_M_GLOBAL(intFormat) = ParseFormatString("int");
  GET_M_GLOBAL(longFormat) = ParseFormatString("long");
  GET_M_GLOBAL(floatFormat) = ParseFormatString("float");
  GET_M_GLOBAL(doubleFormat) = ParseFormatString("double");
  
  GET_M_GLOBAL(dPrintSTR_FN) = dPrintSTR;
  GET_M_GLOBAL(dPrintFORMAT_FN) = dPrintFORMAT;
#ifndef NMP_IPC
  GET_M_GLOBAL(dPrintMAP_FN) = mapPrint;
#endif
  GET_M_GLOBAL(dPrintTCA_FN) = dPrintTCA;
  GET_M_GLOBAL(dPrintCHAR_FN) = printChar;
  GET_M_GLOBAL(dPrintSHORT_FN) = printShort;
  GET_M_GLOBAL(dPrintLONG_FN) = printLong;
  GET_M_GLOBAL(dPrintINT_FN) = printInt;
  GET_M_GLOBAL(dPrintBOOLEAN_FN) = printBoolean;
  GET_M_GLOBAL(dPrintFLOAT_FN) = printFloat;
  GET_M_GLOBAL(dPrintDOUBLE_FN) = printDouble;
  GET_M_GLOBAL(dPrintBYTE_FN) = printByte;
  GET_M_GLOBAL(dPrintUBYTE_FN) = printUByte;
  GET_M_GLOBAL(dPrintTWOBYTE_FN) = printTwoByte;
}
#endif

/******************************************************************************
 *
 * FUNCTION: void tcaModuleInitialize()
 *
 * DESCRIPTION: 
 * Sets globals and calls main initialize routines for initializing a module.
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleInitialize(void)
{
  if (mGlobalp() && GET_C_GLOBAL(valid))
    /* already have initialized the data structures */
    return;
  
  globalMInit();
  
  GET_M_GLOBAL(tcaExitHnd) = NULL;
  
  GET_M_GLOBAL(sizeDM) = 0;
  GET_M_GLOBAL(numAllocatedDM) = 0;
  
  GET_M_GLOBAL(directFlagGlobal) = FALSE;
  GET_C_GLOBAL(moduleConnectionTable) = 
    hashTableCreate(11,
		    (HASH_FN)intHashFunc,
		    (EQ_HASH_FN)intKeyEqFunc);
  FD_ZERO(&GET_M_GLOBAL(externalMask));
  GET_M_GLOBAL(externalFdTable) = hashTableCreate(11, (HASH_FN)intHashFunc, 
						  (EQ_HASH_FN)intKeyEqFunc);
  
  GET_M_GLOBAL(versionGlobal).tcaMajor = TCA_VERSION_MAJOR;
  GET_M_GLOBAL(versionGlobal).tcaMinor = TCA_VERSION_MINOR;
  
  dataMsgInitialize();
  
  tcaRefInitialize();
  formatInitialize();
  
  GET_C_GLOBAL(msgIdTable) = idTableCreate("Message Table", 10);
  GET_C_GLOBAL(hndIdTable) = idTableCreate("Handler Table", 10);
  
  GET_C_GLOBAL(messageTable) = hashTableCreate(101, (HASH_FN)strHashFunc, 
					       (EQ_HASH_FN)strKeyEqFunc);
  GET_C_GLOBAL(handlerTable) = hashTableCreate(101, (HASH_FN)intStrHashFunc, 
					       (EQ_HASH_FN)intStrKeyEqFunc);
  
  GET_M_GLOBAL(classFormatTable) = hashTableCreate(11, (HASH_FN)classHashFunc, 
						   (EQ_HASH_FN) classEqFunc);
  
  printDataModuleInitialize();
  
  GET_M_GLOBAL(Message_Ignore_Set) = listCreate();
  
  GET_C_GLOBAL(valid) = TRUE;
}


/******************************************************************************
 *
 * FUNCTION: char *tcaServerMachine()
 *
 * DESCRIPTION: Simply calls getenv to read the server machine from CENTRALHOST
 *
 * INPUTS: none.
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

const char *tcaServerMachine(void)
{
  char *servHost;
  
  servHost = getenv("CENTRALHOST");
  if (!servHost) {
    tcaModWarning("Warning: CENTRALHOST environment variable not set.\n");
    return NULL;
  };
  
  return servHost;
}

/******************************************************************************
 *
 * FUNCTION: void tcaIgnoreLogging()
 *
 * DESCRIPTION: Inform central to avoid logging a particular message from 
 *              now on
 *
 * INPUTS: char *msgName to ignore.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaIgnoreLogging(char *msgName)
{
  (void)tcaInform(TCA_IGNORE_LOGGING_INFORM, (void *)&msgName);
}

/******************************************************************************
 *
 * FUNCTION: void tcaResumeLogging()
 *
 * DESCRIPTION: Inform central to resume logging a particular message 
 *              from now on
 *
 * INPUTS: char *msgName to resume logging.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaResumeLogging(char *msgName)
{
  (void)tcaInform(TCA_RESUME_LOGGING_INFORM, (void *)&msgName);
}

/******************************************************************************
 *
 * FUNCTION: void tcaClose()
 *
 * DESCRIPTION: 
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaClose(void)
{
  /* inform central that we are shutting down, if a connection exists */
  if (isValidServerConnection() &&
      (GET_C_GLOBAL(tcaServerReadGlobal) != CENTRAL_SERVER_ID)) {
    (void)shutdown((GET_C_GLOBAL(tcaServerReadGlobal)), 2);
#ifdef _WINSOCK_
    closesocket((GET_C_GLOBAL(tcaServerReadGlobal)));
#else
    close((GET_C_GLOBAL(tcaServerReadGlobal)));
#endif
    FD_CLR((GET_C_GLOBAL(tcaServerReadGlobal)),
	   &(GET_C_GLOBAL(tcaConnectionListGlobal)));
  }
#ifndef NO_UNIX_SOCKETS
  closeSocket(GET_C_GLOBAL(listenPortNum));
#endif
  globalMInvalidate();
}

#if defined(LISP)

TCA_REF_PTR returnRootGlobal(void)
{
  return(tcaRootNode());
}

#endif




/******************************************************************************
 *
 * FUNCTION: int resetDirect(const char *msgName, MSG_PTR msg, int sd)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/
static int resetDirect(const char *msgName, MSG_PTR msg, int sd)
{
  if ((msg != NULL) && (msg->direct) && (msg->sd == sd)) {
    msg->sd = NO_FD;
  }
  return TRUE;
}

static void tcaHandleClosedConnection(int sd, DIRECT_INFO_PTR directInfo)
{
  FD_CLR(directInfo->readSd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));
  FD_CLR(directInfo->readSd, &(GET_C_GLOBAL(tcaListenMaskGlobal)));
#ifndef VXWORKS
  (void)shutdown(directInfo->readSd, 2);
#endif /*VXWORKS*/
#ifdef _WINSOCK_
  closesocket(directInfo->readSd);
#else
  close(directInfo->readSd);
  if (directInfo->readSd != directInfo->writeSd) {
    close(directInfo->writeSd);
  }
#endif /*_WINSOCK_*/
  if (!DIRECT_CONNECTION(directInfo)) {
    tcaModError("Closed Connection Detected from: Central Server %s\n",
		GET_C_GLOBAL(servHostGlobal));
    /* May have already cleaned up with the tcaModError.   */
    if (hashTableFind((void *)&sd, GET_C_GLOBAL(moduleConnectionTable))) {
      hashTableRemove((char *)&(directInfo->readSd),
		      (GET_C_GLOBAL(moduleConnectionTable)));
      globalMInvalidate();
      tcaFree((char *)directInfo);
    }
#if defined(LISP)
    if (GET_M_GLOBAL(lispExitGlobal) != NULL)
      (*(GET_M_GLOBAL(lispExitGlobal)))();
#else
    if (!(GET_M_GLOBAL(tcaExitHnd))) {
      
#ifdef _WINSOCK_
      WSACleanup();
#endif /* Unload Winsock DLL */
      
      exit(-1);
    } else {
      (*(GET_M_GLOBAL(tcaExitHnd)))();
    }
    
#endif /* defined(LISP) */
  } else {
    tcaModWarning("Closed Connection Detected from: Module: sd: %d:\n",
		  directInfo->readSd);
    hashTableRemove((char *)&sd,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    /* Need to reset the direct info for messages */
    hashTableIterate((HASH_ITER_FN)resetDirect,
		     GET_C_GLOBAL(messageTable), (void *)sd);
    tcaFree((char *)directInfo);
  }
}


/******************************************************************************
 *
 * FUNCTION: void tcaSetContext(TCA_CONTEXT_PTR context)
 *
 * DESCRIPTION: 
 *
 * INPUTS: char *modName, *serverHost;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/
void tcaSetContext(TCA_CONTEXT_PTR context)
{
  if (context != NULL)
    GET_M_GLOBAL(currentContext) = context;
}


/******************************************************************************
 *
 * FUNCTION: TCA_CONTEXT_PTR tcaGetContext(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

TCA_CONTEXT_PTR tcaGetContext(void)
{
  return GET_M_GLOBAL(currentContext);
}


/******************************************************************************
 *
 * FUNCTION: TCA_CONTEXT_PTR tcaResetContext(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

static void tcaResetContext(void)
{
  GET_M_GLOBAL(currentContext) = 
    (TCA_CONTEXT_PTR)malloc(sizeof(TCA_CONTEXT_TYPE));
  
  FD_ZERO(&GET_C_GLOBAL(tcaConnectionListGlobal));
  FD_ZERO(&GET_C_GLOBAL(tcaListenMaskGlobal));

  GET_C_GLOBAL(servHostGlobal) = NULL;
  GET_C_GLOBAL(moduleConnectionTable) = 
    hashTableCreate(11,
		    (HASH_FN)intHashFunc, 
		    (EQ_HASH_FN)intKeyEqFunc);
  GET_C_GLOBAL(msgIdTable) = idTableCreate("Message Table", 10);
  GET_C_GLOBAL(hndIdTable) = idTableCreate("Handler Table", 10);
  GET_C_GLOBAL(messageTable) = hashTableCreate(101, (HASH_FN)strHashFunc,
					       (EQ_HASH_FN)strKeyEqFunc);
  GET_C_GLOBAL(handlerTable) = hashTableCreate(101, (HASH_FN)intStrHashFunc, 
					       (EQ_HASH_FN)intStrKeyEqFunc);
  GET_C_GLOBAL(resourceTable) = hashTableCreate(11, (HASH_FN)strHashFunc, 
						(EQ_HASH_FN)strKeyEqFunc);
  GET_C_GLOBAL(tcaPendingRepliesGlobal) = listCreate();
  GET_C_GLOBAL(queryNotificationList) = listCreate();
  GET_C_GLOBAL(tcaRefFreeList) = listCreate();
  
  GET_C_GLOBAL(tcaMaxConnection) = 0;
  GET_C_GLOBAL(parentRefGlobal) = -1; 
  GET_C_GLOBAL(responseIssuedGlobal) = FALSE;
  GET_C_GLOBAL(sendMessageRefGlobal) = 1;
  GET_C_GLOBAL(listenPortNum) = 0;
  GET_C_GLOBAL(tcaListenPort) = NO_FD;
  GET_C_GLOBAL(tcaListenSocket) = NO_FD;
  GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
  GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
  GET_C_GLOBAL(willListen) = -1;
  GET_C_GLOBAL(valid) = FALSE;
  GET_C_GLOBAL(tappedMsgs) = NULL;
  GET_C_GLOBAL(broadcastMsgs) = NULL;
  GET_C_GLOBAL(directDefault) = FALSE;
}


/******************************************************************************
 *
 * FUNCTION: void tcaModuleProvides(resourceName, ...)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleProvides(const char* resourceName, ...)
{
  const char *res;
  STR_LIST_PTR resources;
  va_list args;
  
  if (resourceName == NULL) 
    return;
  
  resources = strListCreate();
  res = resourceName;
  va_start(args, resourceName);
  while (res != NULL) {
    strListPushUnique(res, resources);
    res = va_arg(args,const char *);
  }
  va_end(args);
  
  tcaInform(TCA_PROVIDES_INFORM, resources);
  strListFree(&resources,FALSE);
}


/******************************************************************************
 *
 * FUNCTION: void tcaModuleProvidesArgv(resourceName, ...)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleProvidesArgv(const char** resourceNames)
{
  int i=0;
  STR_LIST_PTR resources;
  
  if (resourceNames == NULL) 
    return;
  
  resources = strListCreate();
  for (i=0; resourceNames[i] != NULL; i++){
    strListPushUnique(resourceNames[i], resources);
  }
  
  tcaInform(TCA_PROVIDES_INFORM, resources);
  strListFree(&resources,FALSE);
}


/******************************************************************************
 *
 * FUNCTION: void tcaModuleRequires(resourceName, ...)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleRequires(const char *resourceName, ...)
{
  const char *res;
  STR_LIST_PTR resources;
  va_list args;
  
  if (resourceName == NULL) return;
  
  resources = strListCreate();
  res = resourceName;
  va_start(args, resourceName);
  while (res != NULL) {
    strListPushUnique(res, resources);
    res = va_arg(args,const char *);
  }
  va_end(args);
  
  tcaInform(TCA_REQUIRES_INFORM, resources);
  strListFree(&resources,FALSE);
}


/******************************************************************************
 *
 * FUNCTION: void tcaModuleRequiresArgv(char **resourceNames)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleRequiresArgv(const char **resourceNames)
{
  int i;
  
  if (resourceNames == NULL) return;
  
  for (i=0; resourceNames[i] != NULL; i++){
    strListPushUnique(strdup(resourceNames[i]),
		      GET_M_GLOBAL(requiredResources));
  }
  
  tcaInform(TCA_REQUIRES_INFORM, GET_M_GLOBAL(requiredResources));
}


/******************************************************************************
 *
 * FUNCTION: const char **tcaQueryAvailable(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

char **tcaQueryAvailable(void)
{
  STR_LIST_PTR resources,tmp;
  int length,i;
  char **res;
  
  resources = strListCreate();
  tcaQuery(TCA_AVAILABLE_QUERY, NULL, resources);
  length = strListLength(resources);
  res = (char **)tcaMalloc((length + 1) * sizeof(char *));
  tmp = resources->next;
  for (i=0; i<length; i++)
    res[i] = tmp->item;
  strListFree(&resources,FALSE);
  res[length] = NULL;
  return res;
}


/******************************************************************************
 *
 * FUNCTION: void tcaConnectModule(modName, serverHost)
 *
 * DESCRIPTION: 
 *
 * INPUTS: char *modName, *serverHost;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaConnectModule(const char *modName, const char *serverHost)
{
  MOD_DATA_TYPE modData;
  MOD_START_TYPE vData;
  DIRECT_INFO_PTR directInfo;
  TCA_RETURN_VALUE_TYPE statusV;
  char thisHost[HOST_NAME_SIZE+1];
  BOOLEAN useThisHost = FALSE;
  
  /* Allow for connections to multiple central servers on different machines */
  /* Determine if already connected to another machine. */
  if (isValidServerConnection()) {
    if ((GET_C_GLOBAL(servHostGlobal) != NULL) &&
	strcmp(GET_C_GLOBAL(servHostGlobal), serverHost) == 0) {
      tcaModWarning("tcaConnectModule: already connected to %s\n", serverHost);
    } else {
      /* Connecting to another central server. */
      (GET_M_GLOBAL(modNameGlobal)) = strdup(modName);
      
      /* Need to reset the context. */
      tcaResetContext();
      if (serverHost != NULL)
	GET_C_GLOBAL(servHostGlobal) = strdup(serverHost);
      GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
      GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
      
      if (serverHost) {
	if (!connectSocket(serverHost, &(GET_C_GLOBAL(tcaServerReadGlobal)),
			   &(GET_C_GLOBAL(tcaServerWriteGlobal)))) {
	  tcaModError("ERROR: Can not connect to the server on %s.",
		      serverHost);
	  return;
	}
      } else {
	GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
	GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
	bzero(&(thisHost[0]),HOST_NAME_SIZE+1);
	gethostname(thisHost,HOST_NAME_SIZE);
	useThisHost = TRUE;
	tcaModWarning("Warning: trying to connect to local host %s.\n",
		      thisHost);
	if (!connectSocket(thisHost, &(GET_C_GLOBAL(tcaServerReadGlobal)),
			   &(GET_C_GLOBAL(tcaServerWriteGlobal)))) {
	  tcaModError("ERROR: Can not connect to the server on %s.",
		      thisHost);
	  GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
	  GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
	  return;
	}
      }
      
      FD_SET((GET_C_GLOBAL(tcaServerReadGlobal)), 
	     &(GET_C_GLOBAL(tcaConnectionListGlobal)));
      
      /* Need to keep track of the maximum.  */
      GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					    GET_C_GLOBAL(tcaServerReadGlobal));
      GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					    GET_C_GLOBAL(tcaServerWriteGlobal));
      
      directInfo = NEW(DIRECT_INFO_TYPE);
      bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
      directInfo->readSd = (GET_C_GLOBAL(tcaServerReadGlobal));
      directInfo->writeSd = (GET_C_GLOBAL(tcaServerWriteGlobal));
      
      hashTableInsert((char *)&(directInfo->readSd),
		      sizeof(directInfo->readSd),
		      (char *)directInfo,
		      (GET_C_GLOBAL(moduleConnectionTable)));
      
      msgInfoMsgInitialize();
      classModInitialize();
      
      modData.modName = modName; /* No need to copy; not saved */
      modData.hostName = (char *)tcaMalloc(sizeof(char)*HOST_NAME_SIZE+1);
      bzero((char *)modData.hostName,HOST_NAME_SIZE+1);
      gethostname((char *)modData.hostName,HOST_NAME_SIZE);  
      
      statusV = tcaQueryCentral(TCA_CONNECT_QUERY, (void *)&modData,
				(void *)&vData);
      tcaFree((void *)modData.hostName);
      modData.hostName = NULL;
      
      if (statusV != Success) {
	tcaModError("ERROR: ConnectModule Query failed.");
	return;
      }
    }
  } else {
    /* First connection. */
    tcaModuleInitialize();
    
    GET_M_GLOBAL(modNameGlobal) = strdup(modName);
    if (serverHost != NULL)
      GET_C_GLOBAL(servHostGlobal) = strdup(serverHost);
    
    FD_ZERO(&(GET_C_GLOBAL(tcaConnectionListGlobal)));
    FD_ZERO(&(GET_C_GLOBAL(tcaListenMaskGlobal)));
    GET_C_GLOBAL(tcaMaxConnection) = 0;
    
    if (serverHost) {
      if (!connectSocket(serverHost, &(GET_C_GLOBAL(tcaServerReadGlobal)),
			 &(GET_C_GLOBAL(tcaServerWriteGlobal)))) {
	GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
	GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
#ifndef NMP_IPC
	tcaModError("ERROR: Can not connect to the server on %s.",
		    serverHost);
#endif
	return;
      }
    } else {
      GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
      GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
      bzero(&(thisHost[0]),HOST_NAME_SIZE+1);
      gethostname(thisHost,HOST_NAME_SIZE);
      tcaModWarning("Warning: trying to connect to local host %s.\n",
		    thisHost);
      if (!connectSocket(thisHost, &(GET_C_GLOBAL(tcaServerReadGlobal)),
			 &(GET_C_GLOBAL(tcaServerWriteGlobal)))) {
	tcaModError("ERROR: Can not connect to the server on %s.",
		    thisHost);
	GET_C_GLOBAL(tcaServerReadGlobal) = NO_SERVER_GLOBAL;
	GET_C_GLOBAL(tcaServerWriteGlobal) = NO_SERVER_GLOBAL;
	return;
      }
    }
    
    FD_SET((GET_C_GLOBAL(tcaServerReadGlobal)), 
	   &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  GET_C_GLOBAL(tcaServerReadGlobal));
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  GET_C_GLOBAL(tcaServerWriteGlobal));
    
    directInfo = NEW(DIRECT_INFO_TYPE);
    bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
    directInfo->readSd = (GET_C_GLOBAL(tcaServerReadGlobal));
    directInfo->writeSd = (GET_C_GLOBAL(tcaServerWriteGlobal));
    
    hashTableInsert((char *)&(directInfo->readSd), sizeof(directInfo->readSd),
		    (char *)directInfo,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    
    msgInfoMsgInitialize();
    classModInitialize();
    
    if (!GET_M_GLOBAL(tcaRootNodeGlobal)) {
      GET_M_GLOBAL(tcaRootNodeGlobal) = CREATE_NULL_REF();
    }
    
    GET_C_GLOBAL(tcaPendingRepliesGlobal) = listCreate();
    GET_C_GLOBAL(queryNotificationList) = listCreate();
    
    modData.modName = modName; /* No need to copy; not saved */
    modData.hostName = (char *)tcaMalloc(sizeof(char)*HOST_NAME_SIZE+1);
    bzero((void *)modData.hostName,HOST_NAME_SIZE+1);
    gethostname((char *)modData.hostName, HOST_NAME_SIZE);
    
    statusV = tcaQueryCentral(TCA_CONNECT_QUERY, (void *)&modData,
			      (void *)&vData);
    tcaFree((void *)modData.hostName);
    modData.hostName = NULL;
    if (statusV != Success) {
      tcaModError("ERROR: ConnectModule Query failed.");
      return;
    }
    
    GET_C_GLOBAL(directDefault) = vData.direct;
    
    if (vData.version.tcaMajor != TCA_VERSION_MAJOR) {
      tcaModWarning( "*** ERROR ***\n");
      tcaModWarning( 
		    "Major Version Number is different from Task Control Server.\n");
      tcaModWarning( "Module Version     : %d.%d\n", 
		    (GET_M_GLOBAL(versionGlobal)).tcaMajor,
		    (GET_M_GLOBAL(versionGlobal)).tcaMinor);
      tcaModWarning( "Task Control Server: %d.%d\n",
		    vData.version.tcaMajor, vData.version.tcaMinor);
      tcaModError(NULL);
      return;
    }
#ifndef NMP_IPC
    else if (vData.version.tcaMinor != TCA_VERSION_MINOR) {
      tcaModWarning( "\n*** WARNING ***\n");
      tcaModWarning( "Minor Version Number is different from Task Control Server.\n");
      tcaModWarning( "Module Version     : %d.%d\n", 
		    (GET_M_GLOBAL(versionGlobal)).tcaMajor,
		    (GET_M_GLOBAL(versionGlobal)).tcaMinor);
      tcaModWarning( "Task Control Server: %d.%d\n",
		    vData.version.tcaMajor, vData.version.tcaMinor);
      
    } else {
      tcaModWarning( "Task Control Connected %d.%d\n",
		    vData.version.tcaMajor, vData.version.tcaMinor);
    }
#endif
    
    /* RTG: Moved to tcaWaitUntilReady */
    /* modVarInitialize();*/
    
    /* dataMsgDisplayStats(); */
    
    if(GET_C_GLOBAL(directDefault)) {
      /* Make the default module direct. */
      tcaDirectResource(modName);
    }
  }
}


/******************************************************************************
 *
 * FUNCTION: void tcaWaitUntilReady()
 *
 * DESCRIPTION: 
 *
 * INPUTS: none.
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaWaitUntilReady(void)
{
  int go;
  TCA_RETURN_VALUE_TYPE statusV;
  
  if (!isValidServerConnection()) {
    tcaModError("ERROR: tcaWaitUntilReady: Must connect to server first\n");
    return;
  }
  
  /* Now initialize the tap messages. */
  modVarInitialize();
  
#ifndef NMP_IPC
  if (strListEmpty(GET_M_GLOBAL(requiredResources))) {
    tcaModWarning( "TCA Waiting ...\n");
  } else {
    STR_LIST_PTR element = GET_M_GLOBAL(requiredResources)->next;

    tcaModWarning( "TCA Waiting For:");
    while (element) {
      tcaModWarning(" %s", (char *)element->item);
      element = element->next;
    }
    tcaModWarning(" ...\n");
  }
#endif
  statusV = tcaQueryCentral(TCA_WAIT_QUERY, (void *)NULL, (void *)&go);
  if (statusV != Success) {
    tcaModError("ERROR: tcaWaitUntilReady Query failed.\n");
    return;
  }
  
#ifndef NMP_IPC
  tcaModWarning( "... TCA Done.\n");
#endif
  GET_M_GLOBAL(expectedWaitGlobal) = TRUE;
}

/******************************************************************************
 * FUNCTION: handleQueryNotification
 *
 * DESCRIPTION: Find an applicable notification handler and handle the reply.
 *
 * INPUTS: DATAMSG_PTR dataMsg; QUERY_REPLY_INTENT data message
 *
 * OUTPUTS: BOOLEAN (whether dataMsg was handled within)
 *
 *****************************************************************************/

#ifdef NMP_IPC
/* We have to "fool" the receiver of the response into using this
   message for encoding/decoding, rather than the query message.
   Not pretty, but it works.  Depends on the fact that ReplyClass has
   no class data.  Later, rework central so that it knows how to handle
   this case explicitly. */ 
MSG_PTR getResponseMsg (DATA_MSG_PTR dataMsg)
{ 
  int resClass1 = ExecHndClass;
  CLASS_FORM_PTR classForm;
  MSG_PTR msg;
  char *name, **pointerToName;

  /* Format for ExecHndClass is "string" -- used to send the message name */
  classForm = GET_CLASS_FORMAT(&resClass1);
  /* Get the message name, and find it */
  pointerToName = (char **)dataMsgDecodeClass(classForm->format, dataMsg);
  name = (char *)(*pointerToName);
  msg = msgFind(name);
  classDataFree(resClass1, (char *)pointerToName);

  return msg;
}
#endif

static BOOLEAN testNotificationRef(int *refId, QUERY_NOTIFICATION_PTR element)
{
  return (*refId == element->ref->refId);
}

static BOOLEAN handleQueryNotification (DATA_MSG_PTR dataMsg)
{
  QUERY_NOTIFICATION_PTR queryNotif;
  CONST_FORMAT_PTR decodeFormat;
  void *reply;
 
  queryNotif = ((QUERY_NOTIFICATION_PTR)
		listMemReturnItem((LIST_ITER_FN)testNotificationRef,
				  (char *)&(dataMsg->msgRef),
				  GET_C_GLOBAL(queryNotificationList)));
  if (queryNotif) {
#ifdef NMP_IPC
    /* For IPC, need to use the message that was responded to, not the
       original message that was queried */
    queryNotif->ref->msg = getResponseMsg(dataMsg);
    queryNotif->ref->name = strdup(queryNotif->ref->msg->msgData->name);
    decodeFormat = queryNotif->ref->msg->msgData->msgFormat;
#else
    decodeFormat = queryNotif->ref->msg->msgData->resFormat;
#endif
    reply = (!dataMsg->msgTotal ? NO_CLIENT_DATA /* Represents NullReply */
	     : decodeDataInLanguage(dataMsg, decodeFormat, 
				    queryNotif->language));
#ifdef NMP_IPC
    queryNotif->ref->responded = TRUE; /* Prevent it from responding */
    reply = ipcData(queryNotif->ref->msg->msgData->msgFormat, reply);
    (*((TCA_HND_DATA_FN)queryNotif->handler))(queryNotif->ref, reply,
					      queryNotif->clientData);
#else
    (*queryNotif->handler)(reply, queryNotif->clientData);
#endif
    listDeleteItem((char *)queryNotif, GET_C_GLOBAL(queryNotificationList));
    tcaRefFree(queryNotif->ref);
    tcaFree((void *)queryNotif);
    dataMsgFree(dataMsg);
  }
  return (queryNotif != NULL);
}


/******************************************************************************
 *
 * FUNCTION: int iterateModuleConnection(int readSd,
 *                                       DIRECT_INFO_PTR directInfo)
 *
 * DESCRIPTION: 
 *
 * INPUTS: 
 *
 * OUTPUTS:
 *
 * NOTES:
 *
 *****************************************************************************/

static void handleModuleInput(int readSd,
			      DIRECT_INFO_PTR directInfo)
{
  DATA_MSG_PTR dataMsg;
  
  if ((directInfo->readSd > 0) && ( directInfo->readSd == readSd)) {
    bzero((void *)&dataMsg,sizeof(dataMsg));
    
    switch(dataMsgRecv(directInfo->readSd, &dataMsg, 0, NULL, 0)){
    case StatOK:
      if (dataMsg->intent != QUERY_REPLY_INTENT) {
	execHnd(directInfo, dataMsg);
      } else if (!handleQueryNotification(dataMsg)) {
	listInsertItem(dataMsg, GET_C_GLOBAL(tcaPendingRepliesGlobal));
      }
      break;
    case StatError:
      tcaModError("ERROR: Reading Data Message.");
      break;
    case StatEOF:
      tcaHandleClosedConnection(directInfo->readSd, directInfo);
      break;
    case StatSendEOF:
    case StatSendError:
    case StatRecvEOF:
    case StatRecvError:
#ifndef TEST_CASE_COVERAGE
    default:
#endif
      tcaModError("Internal Error: iterateModuleConnections: UNKNOWN INTENT\n");
    }
  }
  return;
}

/******************************************************************************
 *
 * FUNCTION: static void acceptConnections(void)
 *
 * DESCRIPTION: 
 *   Accept requests for new connections.
 *
 * INPUTS: 
 *
 * OUTPUTS: 
 *
 * NOTES:
 *
 *****************************************************************************/
#ifndef VXWORKS
static void acceptConnections(fd_set *readMask)
{
  int sd;
  DIRECT_INFO_PTR directInfo;

  if ((GET_M_GLOBAL(directFlagGlobal)) &&
      (GET_C_GLOBAL(tcaListenPort) != NO_FD) &&
      FD_ISSET((GET_C_GLOBAL(tcaListenPort)), readMask)) {
    sd = accept((GET_C_GLOBAL(tcaListenPort)), 
		(struct sockaddr *)NULL, (int *)NULL);
    if (sd <= 0) {
      tcaModError("Accept Failed\n");
      return;
    }
    FD_SET(sd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  sd);
    
    directInfo = NEW(DIRECT_INFO_TYPE);
    bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
    directInfo->readSd = sd;
    directInfo->writeSd = sd;
    
    tcaModWarning("New TCP Connection Established for a Module at sd: %d\n",
		  directInfo->readSd);
    
    hashTableInsert((char *)&(directInfo->readSd), sizeof(directInfo->readSd),
		    (char *)directInfo,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    /* 25-Jun-91: fedor: set sd in readMask as well? */
  } else if ((GET_M_GLOBAL(directFlagGlobal)) &&
	     (GET_C_GLOBAL(tcaListenSocket) != NO_FD) &&
	     FD_ISSET((GET_C_GLOBAL(tcaListenSocket)), readMask)) {
    sd = accept((GET_C_GLOBAL(tcaListenSocket)), 
		(struct sockaddr *)NULL, (int *)NULL);
    if (sd <= 0) {
      tcaModError("Accept Failed\n");
      return;
    }
    
    directInfo = NEW(DIRECT_INFO_TYPE);
    bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
    directInfo->readSd = sd;
    directInfo->writeSd = sd;
    
    tcaModWarning("New UNIX Connection Established for a Module at sd: %d\n",
		  directInfo->readSd);
    
    hashTableInsert((char *)&(directInfo->readSd), sizeof(directInfo->readSd),
		    (char *)directInfo,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    FD_SET(sd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection), sd);
    /* 25-Jun-91: fedor: set sd in readMask as well? */
  } 
}

#else
static void acceptConnections(fd_set *readMask)
{
  int readSd, writeSd;
  DIRECT_INFO_PTR directInfo;
  char socketName[80]; 
  char portNum[80];
  char modName[80];
  struct sockaddr addr;
  int len = sizeof(struct sockaddr);
  
  if ((GET_M_GLOBAL(directFlagGlobal)) &&
      (GET_C_GLOBAL(tcaListenPort) != NO_FD) &&
      FD_ISSET((GET_C_GLOBAL(tcaListenPort)), readMask)) {
    readSd = accept((GET_C_GLOBAL(tcaListenPort)), &addr, &len);
    if (readSd <= 0) {
      tcaModError("Accept Failed\n");
      return;
    }
    FD_SET(readSd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  readSd);
    
    directInfo = NEW(DIRECT_INFO_TYPE);
    bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
    directInfo->readSd = readSd;
    directInfo->writeSd = readSd;
    
    tcaModWarning("New TCP Connection Established for a Module at sd: %d\n",
		  directInfo->readSd);
    
    hashTableInsert((char *)&(directInfo->readSd), sizeof(directInfo->readSd),
		    (char *)directInfo,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    /* 25-Jun-91: fedor: set sd in readMask as well? */
  } else if ((GET_M_GLOBAL(directFlagGlobal)) && 
	     (GET_C_GLOBAL(tcaListenSocket) != NO_FD) &&
	     FD_ISSET((GET_C_GLOBAL(tcaListenSocket)), readMask)) {
    /* Using Vx pipes for local communication. */
    
#ifdef DEBUG
    fprintf(stderr, "Reading\n");
#endif
    readNBytes(GET_C_GLOBAL(tcaListenSocket),modName, 80);
#ifdef DEBUG
    fprintf(stderr, "Read %s\n",modName);
#endif
    
    sprintf(portNum,"%d",GET_C_GLOBAL(listenPortNum));
    sprintf(socketName,VX_PIPE_NAME,portNum,modName);
#ifdef DEBUG
    fprintf(stderr, "opening %s\n", socketName);
#endif
    readSd = open(socketName, O_RDONLY, 0644);
    if (readSd < 0) {
      tcaModError("Open pipe Failed\n");
      return;
    }

    sprintf(socketName,VX_PIPE_NAME,modName,portNum);
#ifdef DEBUG
    fprintf(stderr, "opening %s\n", socketName);
#endif
    writeSd = open(socketName, O_WRONLY, 0644);
    if (writeSd < 0) {
      tcaModError("Open pipe Failed\n");
      return;
    }
    
#ifdef DEBUG
    fprintf(stderr, "Writting %s to accept pipe\n", portNum);
#endif
    writeNBytes(writeSd,portNum, 80);
#ifdef DEBUG
    fprintf(stderr, "Write done\n");
#endif
    
    directInfo = NEW(DIRECT_INFO_TYPE);
    bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
    directInfo->readSd = readSd;
    directInfo->writeSd = writeSd;
    
    tcaModWarning("New VX PIPE Connection Established for a Module at sd: %d\n",
		  directInfo->readSd);
    
    hashTableInsert((char *)&(directInfo->readSd), sizeof(directInfo->readSd),
		    (char *)directInfo,
		    (GET_C_GLOBAL(moduleConnectionTable)));
    FD_SET(readSd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  readSd);
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  writeSd);
    /* 25-Jun-91: fedor: set sd in readMask as well? */
  } 
}
#endif /* VXWORKS */

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaSingleMsgHnd(time)
 *
 * DESCRIPTION: 
 *
 * INPUTS: struct timeval *time;
 *
 * OUTPUTS: int
 *
 * NOTES:
 * Assumes that this connection is only listening to the central server
 * not very general here but it fits our current tca model of control.
 *
 * tcaConnectionListGlobal was inited from tcaServerGlobal
 *
 *****************************************************************************/

static TCA_RETURN_VALUE_TYPE tcaSingleMsgHnd(struct timeval *time)
{
  struct timeval forever;

  if (time == NULL) {
    forever.tv_usec = 0;
    forever.tv_sec = WAITFOREVER;
    time = &forever;
  }
  return waitForReplyFromTime(NULL, NULL, 0, time, NO_FD);
}

/******************************************************************************
 *
 * FUNCTION: int tcaHandleFdInput(int fd)
 *
 * DESCRIPTION: 
 *
 * INPUTS: struct timeval *time;
 *
 * OUTPUTS: int
 *
 * NOTES:
 * Assumes that this connection is only listening to the central server
 * not very general here but it fits our current tca model of control.
 *
 * tcaConnectionListGlobal was inited from tcaServerGlobal
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaHandleFdInput(int fd)
{
  fd_set readMask;
  DIRECT_INFO_PTR directInfo = NULL;
  
  FD_ZERO(&readMask);
  FD_SET(fd, &readMask);
  
  acceptConnections(&readMask);
  
  directInfo = (DIRECT_INFO_PTR) 
    hashTableFind((void *)&fd, GET_C_GLOBAL(moduleConnectionTable));
  if (directInfo) {
    handleModuleInput(fd, directInfo);
  } else {
    (void)execFdHnd(fd);
  }
  
  return 1; /* Success */
}


/******************************************************************************
 *
 * FUNCTION: void *decodeDataInLanguage (dataMsg, decodeFormat, language)
 *
 * DESCRIPTION: Decode dataMsg using format information, creating (and
 *              returning) a new data structure.
 *
 * INPUTS:
 * DATA_MSG_PTR dataMsg;
 * CONST_FORMAT_PTR decodeFormat;
 * HND_LANGUAGE_ENUM language (C, LISP, RAW)
 *
 * OUTPUTS: void * (the newly created, decoded data)
 *
 *****************************************************************************/

void *decodeDataInLanguage (DATA_MSG_PTR dataMsg, CONST_FORMAT_PTR decodeFormat,
			    HND_LANGUAGE_ENUM language)
{
  void *data;

  switch (language) {
  case C_LANGUAGE:
    data = dataMsgDecodeMsg(decodeFormat, dataMsg, FALSE); break;

#ifdef LISP
  case LISP_LANGUAGE: 
    { BUFFER_TYPE buffer;
      buffer.bstart = 0;
      buffer.buffer = NULL;
      
      if (decodeFormat && dataMsg->msgTotal) {
	buffer.buffer = dataMsg->msgData;
	data = (void *)(*(GET_M_GLOBAL(lispDecodeMsgGlobal)))(decodeFormat, 
							      &buffer);
      } else {
	data = (void *)(*(GET_M_GLOBAL(lispDecodeMsgGlobal)))(0, NULL);
      }
      break;
    }
#endif
  default: 
    data = NULL;
    tcaModError("Decode in unknown language (%d)\n", language);
  }
  return data;
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE processQueryReply(dataMsg, msg, sel, reply)
 *
 * DESCRIPTION: 
 * Decodes dataMsg using msg format information and storing it in reply. 
 * Reclaims dataMsg storage.
 *
 * INPUTS:
 * DATA_MSG_PTR dataMsg;
 * MSG_PTR msg;
 * int sel; (which format to choose)
 * void *reply;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

static TCA_RETURN_VALUE_TYPE processQueryReply(DATA_MSG_PTR dataMsg, 
					       MSG_PTR msg, BOOLEAN sel,
					       void *reply)
{
  char *r2= NULL;
  CONST_FORMAT_PTR decodeFormat;
  int amount;
  
  if (dataMsg->msgTotal == 0) {
    dataMsgFree(dataMsg);
    return NullReply;
  }
  
  decodeFormat = (sel ? msg->msgData->msgFormat : msg->msgData->resFormat);
#if defined(LISP)
  if (reply == &(GET_M_GLOBAL(lispFlagGlobal))) {
    r2 = (char *)decodeDataInLanguage(dataMsg, decodeFormat, LISP_LANGUAGE);
    (*(GET_M_GLOBAL(lispReplyMsgGlobal)))(r2);
  } else
#endif
    {
      r2 = (char *)decodeDataInLanguage(dataMsg, decodeFormat, C_LANGUAGE);
      amount = dataStructureSize(decodeFormat);

      /* Free only if it was allocated. */
      if (r2 != reply) {
	BCOPY(r2, reply, amount);
	tcaFree((char *)r2);
      }
    }
  
  dataMsgFree(dataMsg);
  return Success;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE waitForReply(ref, reply, timeout)
 *
 * DESCRIPTION: Loops to handle messages while waiting for a query reply.
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * char *reply;
 * long timeout;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES: Modified from tcaSingleMsgHnd
 *
 *****************************************************************************/

static int testMsgRef(int *refId, DATA_MSG_PTR dataMsg)
{
  return (dataMsg->msgRef == *refId);
}

static BOOLEAN handleModuleReply(int fd, BOOLEAN sel,
				 TCA_REF_PTR ref,
				 char *replyData,
				 DIRECT_INFO_PTR directInfo,
				 TCA_RETURN_VALUE_TYPE *status)
{
  int refId;
  int sd = directInfo->readSd;
  DATA_MSG_PTR dataMsg;
  int replySize;
  
  refId = (ref ? ref->refId : NO_REF);
  replySize = (ref ? dataStructureSize(ref->msg->msgData->resFormat) : 0);

  if (directInfo->readSd == fd) {
    switch(dataMsgRecv(directInfo->readSd, &dataMsg, refId,
		       replyData, replySize)){
    case StatOK:
      if (dataMsg->intent == QUERY_REPLY_INTENT) {
	if (dataMsg->msgRef == refId) {
	  *status = processQueryReply(dataMsg, ref->msg, sel, replyData);
	  return TRUE;
	} else if (!handleQueryNotification(dataMsg)) {
	  listInsertItem((char *)dataMsg, 
			 GET_C_GLOBAL(tcaPendingRepliesGlobal));
	}
      }
      else
	execHnd(directInfo, dataMsg);
      return FALSE;
    case StatError:
      tcaModError("ERROR: Reading Data Message in waitForReplyFrom %d.\n",
		  directInfo->readSd);
    case StatEOF:
      /* If we have disconnected, then just return.  */
      if (!hashTableFind((void *)&sd, GET_C_GLOBAL(moduleConnectionTable))) {
	return FALSE;
      }
      tcaHandleClosedConnection(directInfo->readSd, directInfo);
      break;
    case StatSendEOF:
    case StatSendError:
    case StatRecvEOF:
    case StatRecvError:
#ifndef TEST_CASE_COVERAGE
    default:
#endif
      tcaModError("Internal Error: waitForReplyFrom: UNKNOWN INTENT\n");
    }
  }
  return FALSE;
}

/* Returns TRUE if externalMask is not zero */
static BOOLEAN externalFdSet (void)
{
  fd_set externalMask; 
  int i;

  externalMask = GET_M_GLOBAL(externalMask);
  for (i=0; i<=GET_C_GLOBAL(tcaMaxConnection); i++) {
    if (FD_ISSET(i, &externalMask)) 
      return TRUE;
  }
  return FALSE;
}

/******************************************************************************
 *
 * FUNCTION: void subTime (struct timeval *t1, struct timeval *t2)
 *
 * DESCRIPTION: 
 * subtracts t2 from t1.
 *
 * INPUTS:
 *
 * OUTPUTS: 
 *
 *****************************************************************************/

static void subTime(struct timeval *t1, struct timeval *t2)
{
  t1->tv_usec -= t2->tv_usec;
  t1->tv_sec -= t2->tv_sec;
  /* do a borrow if needed */
  if (t1->tv_usec < 0) {
    t1->tv_usec += 1000000;
    t1->tv_sec -= 1;
  }
}

/*****************
 * Returns Timeout if "timeout" seconds elapse without receiving a message.
 *  WAITFOREVER always waits until reply received (a very long time, at least).
 ****************/
TCA_RETURN_VALUE_TYPE waitForReplyFromTime(TCA_REF_PTR ref, void *reply, 
					   BOOLEAN sel, struct timeval *time,
					   int fd)
{
  DATA_MSG_PTR dataMsg = NULL;
  TCA_RETURN_VALUE_TYPE status = Success;
  fd_set readMask; 
  int ret, i, firstTime;
  DIRECT_INFO_PTR directInfo=NULL;
  struct timeval start, now, tmpTime;
  
  do {
    if (ref != NULL) {
      dataMsg = (DATA_MSG_PTR)listMemReturnItem((LIST_ITER_FN) testMsgRef,
						(char *)&ref->refId, 
						(GET_C_GLOBAL(tcaPendingRepliesGlobal)));
    }
    if (dataMsg) {
      listDeleteItem((char *)dataMsg, (GET_C_GLOBAL(tcaPendingRepliesGlobal)));
      return processQueryReply(dataMsg, ref->msg, sel, reply);
    }
    
#ifndef VXWORKS
    if (!GET_M_GLOBAL(directFlagGlobal) && (time->tv_sec == WAITFOREVER) &&
	!externalFdSet()) {
      /* Don't need to do the select, just pretend it return as it will and
       * set the return to 1.
       */
      FD_ZERO(&readMask);
      if (GET_C_GLOBAL(tcaServerReadGlobal) == NO_FD) {
	ret = 0;
      } else {
	FD_SET((GET_C_GLOBAL(tcaServerReadGlobal)), &readMask);
	ret = 1;
      }
    } else
#endif
      {
	if (fd == NO_FD) {
	  readMask = (GET_C_GLOBAL(tcaConnectionListGlobal));
	} else {
	  FD_ZERO(&readMask);
	  FD_SET(fd,&readMask);
	}
	firstTime = TRUE;
	if (time->tv_sec != WAITFOREVER) {
	  gettimeofday(&start, NULL);
	}
	do {
	  /* Fixed a bug where this loop would never exit if signals kept 
	     getting generated, causing the select to exit */
	  if (!firstTime && time->tv_sec != WAITFOREVER) {
	    gettimeofday(&now, NULL);
	    tmpTime = now;
	    subTime(&now, &start);
	    subTime(time, &now);
	    if (time->tv_sec < 0 || (time->tv_sec == 0 && time->tv_usec == 0)){
	      ret = 0;
	      break;
	    }
	    start = tmpTime;
	  }
	  ret = select(FD_SETSIZE, &readMask, (fd_set *)NULL, (fd_set *)NULL,
		       ((time->tv_sec == WAITFOREVER) ? (struct timeval *)NULL 
			: time));
	  firstTime = FALSE;
	}
#ifdef _WINSOCK_
	while (ret == SOCKET_ERROR && WSAGetLastError() == WSAEINTR);
#else
	while (ret < 0 && errno == EINTR);
#endif
      }
    if (ret == 0) {
      return TimeOut;
    } else if (ret < 1) {
      tcaModError("ERROR: on select in WaitForReplyFrom.\n");
      return Failure;
    }
    
    acceptConnections(&readMask);
    
    for (i=0; i<=GET_C_GLOBAL(tcaMaxConnection); i++) {
      if (FD_ISSET(i, &readMask)) {
	directInfo = (DIRECT_INFO_PTR) 
	  hashTableFind((void *)&i, GET_C_GLOBAL(moduleConnectionTable));
	if (directInfo) {
	  FD_CLR(i, &readMask);
	  if (handleModuleReply(i, sel, ref, reply, directInfo, &status)) {
	    return status;
	  }
	} else {
	  if (execFdHnd(i)) {
	    FD_CLR(i, &readMask);
	  }
	}
      }
    }
  } while (ref != NULL);
  return status;
}

TCA_RETURN_VALUE_TYPE waitForReplyFrom(TCA_REF_PTR ref, void *reply, 
				       BOOLEAN sel, long timeout,
				       int fd)
{
  struct timeval time;
  
  time.tv_usec = 0;
  time.tv_sec = timeout;
  return waitForReplyFromTime(ref, reply, sel, &time, fd);
}

TCA_RETURN_VALUE_TYPE waitForReply(TCA_REF_PTR ref, void *reply)
{
  return waitForReplyFrom(ref, reply, FALSE, WAITFOREVER, NO_FD);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaHandleMessage(sec)
 *
 * DESCRIPTION: 
 *
 * INPUTS: long sec;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

/* For NMP_IPC, the time argument is taken to be msecs, not seconds !! */

TCA_RETURN_VALUE_TYPE tcaHandleMessage(long sec)
{
  struct timeval time;
  
  if (!isValidServerConnection()) {
    tcaModError("ERROR: Must connect to server first.\n");
    return Failure;
  } else if (!GET_M_GLOBAL(expectedWaitGlobal)) {
    tcaModWarning(
		  "WARNING: call tcaWaitUntilReady before listening for messages\n");
  }
  
  if (sec == WAITFOREVER) {
    return tcaSingleMsgHnd((struct timeval *)NULL);
  } else {
#ifndef NMP_IPC
    time.tv_sec = sec;
    time.tv_usec = 0;
#else /* NMP_IPC treats the argument as msecs */
    time.tv_sec = sec/1000;
    time.tv_usec = 1000*(sec%1000);
#endif
    return tcaSingleMsgHnd(&time);
  }
}


/******************************************************************************
 *
 * FUNCTION: void tcaModuleListen()
 *
 * DESCRIPTION: 
 *
 * INPUTS: none.
 * 
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaModuleListen(void)
{
  
  if (!isValidServerConnection()) {
    tcaModError("ERROR: Must connect to server first (tcaModuleListen).\n");
    return;
  } else if (!GET_M_GLOBAL(expectedWaitGlobal)) {
    tcaModWarning(
		  "WARNING: call tcaWaitUntilReady before listening for messages\n");
  }
  
  for (;;) {
    (void)tcaSingleMsgHnd((struct timeval *)NULL);
  }
}


/******************************************************************************
 *
 * FUNCTION: void tcaRegisterExitHandler(proc)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 * int (*proc)();
 * 
 * OUTPUTS: void.
 *
 *****************************************************************************/

void tcaRegisterExitProc(void (*proc)(void))
{
  /* make sure the data structures are initialzied. */
  tcaModuleInitialize();
  (GET_M_GLOBAL(tcaExitHnd)) = proc;
}

/****************************************************************************/

void tcaDirectResource(const char *name)
{
  int port;
  DIRECT_TYPE direct;
  
  if (GET_C_GLOBAL(tcaListenPort) == NO_FD) {
    port = LISTEN_PORT;
    while ((port < LISTEN_PORT_MAX) && 
	   !listenAtPort(&port, &(GET_C_GLOBAL(tcaListenPort))))
      port++;
    
    if (port == LISTEN_PORT_MAX) {
      tcaModError( "ERROR: Exceeded available ports for direct connection.\n");
      return;
    }
    (GET_M_GLOBAL(directFlagGlobal)) = TRUE;
    FD_SET((GET_C_GLOBAL(tcaListenPort)), 
	   &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    FD_SET((GET_C_GLOBAL(tcaListenPort)), 
	   &(GET_C_GLOBAL(tcaListenMaskGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  (GET_C_GLOBAL(tcaListenPort)));
#if !defined(NO_UNIX_SOCKETS) || defined(VX_PIPES)
    listenAtSocket(port, &(GET_C_GLOBAL(tcaListenSocket)));
    FD_SET((GET_C_GLOBAL(tcaListenSocket)), 
	   &(GET_C_GLOBAL(tcaConnectionListGlobal)));
    FD_SET((GET_C_GLOBAL(tcaListenSocket)), 
	   &(GET_C_GLOBAL(tcaListenMaskGlobal)));
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  (GET_C_GLOBAL(tcaListenSocket)));
#endif
    
    GET_C_GLOBAL(listenPortNum) = port;
  }
  
  direct.port = GET_C_GLOBAL(listenPortNum);
  direct.name = name;
  
  tcaInform(TCA_DIRECT_RES_INFORM, &direct);
}

/****************************************************************************/

void establishDirect(MSG_PTR msg)
{
  DIRECT_INFO_PTR directInfo, oldInfo=NULL;
  int i;
  TCA_RETURN_VALUE_TYPE status;

  directInfo = NEW(DIRECT_INFO_TYPE);
  bzero((void *)directInfo,sizeof(DIRECT_INFO_TYPE));
  
  status = tcaQueryCentral(TCA_DIRECT_INFO_QUERY, (void *)&msg->msgData->name,
			   (void *)directInfo);
  if (status == NullReply) {
    tcaFreeReply(TCA_DIRECT_INFO_QUERY, (void *)directInfo);
    tcaFree((void *)directInfo);
    tcaModWarning( "Using central connection for %s to %s.\n",
		  msg->msgData->name, GET_M_GLOBAL(modNameGlobal));
    return;
  } else if (status != Success) {
    tcaFreeReply(TCA_DIRECT_INFO_QUERY, (void *)directInfo);
    tcaFree((void *)directInfo);
    return;
  }
 
  if (strcmp(GET_M_GLOBAL(modNameGlobal),directInfo->module) == 0) {
    tcaFreeReply(TCA_DIRECT_INFO_QUERY, (void *)directInfo);
    tcaFree((void *)directInfo);
    tcaModWarning( "Using central connection for %s to %s.\n",
		  msg->msgData->name, GET_M_GLOBAL(modNameGlobal));
    return;
  } else {
    msg->direct = TRUE;
    msg->intent = directInfo->intent;

    for (i=0; i<=GET_C_GLOBAL(tcaMaxConnection); i++) {
      oldInfo = (DIRECT_INFO_PTR) 
	hashTableFind((void *)&i, GET_C_GLOBAL(moduleConnectionTable));
      if (oldInfo != NULL) {
	if ((directInfo->port == oldInfo->port) &&
	    strKeyEqFunc(directInfo->host, oldInfo->host)) {
	  break;
	}
      }
      oldInfo = NULL;
    }
    
    if (oldInfo) {
      msg->sd = oldInfo->writeSd;
      tcaFreeReply(TCA_DIRECT_INFO_QUERY, (void *)directInfo);
      tcaFree((void *)directInfo);
      tcaModWarning( "Using existing direct connection for %s to %s.\n",
		    msg->msgData->name, oldInfo->module);
      return;
    } 
    
    if (!connectAt(directInfo->host, directInfo->port,
		   &(directInfo->readSd), &(directInfo->writeSd))) {
      tcaModError("ERROR: Can not establish direct connection to %s at %ld.\n",
		  directInfo->host, (long)directInfo->port);
      return;
    } else {
      (GET_M_GLOBAL(directFlagGlobal)) = TRUE;
      tcaModWarning( "Direct Connection Established for %s to %s.\n", 
		     msg->msgData->name, directInfo->module);
    }
    FD_SET(directInfo->readSd, &(GET_C_GLOBAL(tcaConnectionListGlobal)));  
    /* Need to keep track of the maximum.  */
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  directInfo->readSd);
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(GET_C_GLOBAL(tcaMaxConnection),
					  directInfo->writeSd);
  }
  
  /* 26-Jun-91: fedor
   * attempt to make the server the first item
   * otherwise things get overrun on the server socket
   * in waitForReply - going to have to redesign this nonsense!
   */
  hashTableInsert((void *)&(directInfo->readSd),  sizeof(directInfo->readSd),
		  (char *)directInfo,
		  (GET_C_GLOBAL(moduleConnectionTable)));
  
  msg->sd = directInfo->writeSd;
}


/******************************************************************************
 *
 * FUNCTION: int tcaGetVersionMajor(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 * 
 * OUTPUTS: version number.
 *
 *****************************************************************************/

int32 tcaGetVersionMajor(void)
{
  return TCA_VERSION_MAJOR;
}


/******************************************************************************
 *
 * FUNCTION: int tcaGetVersionMinor(void)
 *
 * DESCRIPTION: 
 *
 * INPUTS:
 * 
 * OUTPUTS: version number.
 *
 *****************************************************************************/

int32 tcaGetVersionMinor(void)
{
  return TCA_VERSION_MINOR;
}

/******************************************************************************
 *
 * FUNCTION: void tcaAddEventHandr(int fd, TCA_FD_HND_FN hnd, void *clientData)
 *
 * DESCRIPTION: Add the handler to handle inputs on the file descriptor fd;
 *              The handler function is called with the file descriptor and
 *              the client data (which can be a pointer to any user-specified 
 *              data).
 *
 * INPUTS:
 * 
 * OUTPUTS:
 *
 *****************************************************************************/

void tcaAddEventHandler(int fd, TCA_FD_HND_FN hnd, void *clientData)
{
  FD_HND_PTR fdHndData;

  fdHndData = (FD_HND_PTR)GET_FD_HANDLER(fd);
  if (fdHndData) {
    if (fdHndData->hndProc != hnd) {
      tcaModWarning("WARNING: Handler function replaced for fd %d\n", fd);
      fdHndData->hndProc = hnd;
    }
  } else {
    fdHndData = NEW(FD_HND_TYPE);
    fdHndData->fd = fd;
    fdHndData->hndProc = hnd;
    fdHndData->clientData = clientData;

    hashTableInsert((const void *)&fd, sizeof(fd), fdHndData,
		    GET_M_GLOBAL(externalFdTable));
    FD_SET(fd,&GET_M_GLOBAL(externalMask));
    FD_SET(fd,&GET_C_GLOBAL(tcaConnectionListGlobal));
    GET_C_GLOBAL(tcaMaxConnection) =  MAX(fd, GET_C_GLOBAL(tcaMaxConnection));
  }
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaRemoveEventHandler(int fd)
 *
 * DESCRIPTION: Remove the handler to handle events for the file descriptor fd
 *
 * INPUTS: int fd; 
 * 
 * OUTPUTS: none
 *
 *****************************************************************************/

void tcaRemoveEventHandler(int fd)
{
  FD_HND_PTR fdHndData;

  fdHndData = (FD_HND_PTR)hashTableRemove((const void *)&fd,
					  GET_M_GLOBAL(externalFdTable));
  if (fdHndData) tcaFree((char *)fdHndData);
  FD_CLR(fd,&GET_M_GLOBAL(externalMask));
  FD_CLR(fd,&GET_C_GLOBAL(tcaConnectionListGlobal));
}
