/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: central
 *
 * FILE: centralIO.c
 *
 * ABSTRACT:
 *
 * This file contains i/o routines for central.
 *
 * REVISION HISTORY:
 *
 * $Log: centralIO.c,v $
 * Revision 1.19  1996/05/09  18:30:23  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.18  1996/02/07  00:27:34  rich
 * Add prefix to VERSION_DATE and COMMIT_DATE.
 *
 * Revision 1.17  1996/01/30  15:03:52  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.16  1996/01/23  00:06:20  rich
 * Fixed memory leak when a module connects and disconnects.  Also fixed a
 * problem with using the direct connection flag.  This was introduced when
 * we added contexts for keeping track of the central server.
 *
 * Revision 1.15  1996/01/10  03:16:10  rich
 * Fixed libtca_lisp.a to work with dbmalloc.  Added central commands to
 * show resource state and to unlock locked resouces.  Fixed a bug where
 * dispatches were not freed when handlers were cleared. Reset errno variable.
 *
 * Revision 1.14  1995/12/17  20:21:10  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.13  1995/10/25  22:47:55  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.12  1995/04/21  03:53:08  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.11  1995/04/07  05:02:46  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.10  1995/04/04  19:41:47  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.9  1995/03/30  15:42:26  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.8  1995/03/28  01:14:18  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.7  1995/03/18  15:10:58  rich
 * Fixed updateVersion script so it can be run from any directory.
 *
 * Revision 1.6  1995/01/18  22:39:49  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.5  1994/10/27  14:43:06  reids
 * Change the default handling of wiretap messages.  Now the default is to
 * place them as children of the tapped message; the "-t" option changes the
 * default to place them all under the task tree root node.
 *
 * Revision 1.4  1994/05/25  04:56:58  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.3  1994/05/17  23:15:14  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.2  1994/05/11  01:57:07  rich
 * Now set an invalid tcaServerGlobal (a socket fd) to -1 rather than 0
 * which is stdout.
 * Added checks to make sure tcaServerGlobal is a valid socket before
 * sending messages or waiting for messages.
 *
 * Revision 1.1  1994/05/06  04:47:01  rich
 * Put central io routines in a new file.
 * Fixed GNUmakefile.
 *
 *
 * $Revision: 1.19 $
 * $Date: 1996/05/09 18:30:23 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"


/*****************************************************************************
 *
 * FUNCTION: void setOptions(option, logPtr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * char *option;
 * LOG_PTR logPtr;
 *
 * OUTPUTS: void
 *
 * HISTORY:
 *
 *  5-Jan-93 Domingo Gallardo at School of Computer Science, CMU
 *  New option: log parent's ID
 *
 *  2-Dec-92 Richard Goodwin at School of Computer Science, CMU
 *  Cleaned it up a little.
 *
 *  6-Oct-89 Christopher Fedor at School of Computer Science, CMU
 *  eeeks -- this setting options is a mess! 
 *
 *****************************************************************************/

BOOLEAN checkOccurrence(char *option, char *item)
{
#ifndef VXWORKS
  return (strpbrk(option, item) != NULL);
#else
  /* use index - vxworks is missing strpbrk */
  
  while(item && item[0] != '\0') {
    if (index(option, item[0])) {
      return TRUE;
    }
    item++;
  }
  
  return FALSE;
#endif
}

void setOptions(char *option, LOG_PTR logPtr)
{
  if ((strlen(option) == 1) || (strlen(option) == 2)) { 
    /* -l or -L -- use all the options, except i */
    logPtr->messages = logPtr->status = logPtr->time = TRUE;
    logPtr->data = logPtr->summary = TRUE;
    logPtr->ignore = logPtr->refId = FALSE;
  }
  else if (checkOccurrence(option, "xX")) {
    /* no options */
    logPtr->messages = logPtr->status = logPtr->time = logPtr->data = FALSE;
    logPtr->data = logPtr->summary = logPtr->ignore = FALSE;
  }
  else {
    logPtr->messages = checkOccurrence(option, "mM");
    logPtr->status = checkOccurrence(option, "sS");
    logPtr->time = checkOccurrence(option, "tT");
    logPtr->data = checkOccurrence(option, "dD");
    logPtr->ignore = checkOccurrence(option, "iI");
    logPtr->summary = checkOccurrence(option, "hH");
    logPtr->refId = checkOccurrence(option, "rR");
    logPtr->parentId = checkOccurrence(option, "pP");
    
    if (checkOccurrence(option, "fF")) /* leave default if not specified. */
      logPtr->flush = FALSE;
  }
}

static INLINE void printTorF(int item)
{
  if (item) {
    printf("TRUE\n");
  } else {
    printf("FALSE\n");
  }
}

void printOps(LOG_PTR logPtr)
{
  printf("messages: ");
  printTorF(logPtr->messages);
  
  printf("status: ");
  printTorF(logPtr->status);
  
  printf("time: ");
  printTorF(logPtr->time);
  
  printf("data: ");
  printTorF(logPtr->data);
  
  printf("ignore: ");
  printTorF(logPtr->ignore);
  
  printf("summary: ");
  printTorF(logPtr->summary);
  
  printf("refId: ");
  printTorF(logPtr->refId);
  
  printf("parentId: ");
  printTorF(logPtr->parentId);
  
  printf("flush: ");
  printTorF(logPtr->flush);
}

void printPrompt(void)
{
  printf("> ");
  fflush(stdout);
}


/*****************************************************************************
 *
 * FUNCTION: void parseCommandLineOptions(argc, argv)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * int argc;
 * char **argv;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void parseOption(char *option, BOOLEAN started)
{
  switch(option[1]) {
  case 'l':
    /*    printf("Terminal Logging Status\n");*/
    setOptions(option, &GET_S_GLOBAL(terminalLog));
    if (started) {
      centralSetVar(TCA_TERMINAL_LOG_VAR, (char *)&GET_S_GLOBAL(terminalLog));
    }
    /*    printOps(GET_S_GLOBAL(terminalLog));*/
    break;
  case 'L': 
    /*    printf("File Logging Status\n");*/
    setOptions(option, &GET_S_GLOBAL(fileLog));
    if (started) {
      centralSetVar(TCA_FILE_LOG_VAR, (char *)&GET_S_GLOBAL(fileLog));
      /*    printOps(GET_S_GLOBAL(fileLog));*/
    }
    break;
  case 'f':
    if (strlen(option) > 2) {
      (void)strncpy(GET_S_GLOBAL(Log_File_Name), option+2, 
		    MAX_LOG_FILE_NAME_LENGTH);
      printf("Logging File Name: %s\n",GET_S_GLOBAL(Log_File_Name));
      fflush(stdout);
    }
    break;
  case 'p':
    if (strlen(option) > 2) 
      (void)sscanf(option+2, "%d", &(GET_S_GLOBAL(serverPortGlobal)));
    break;
  case 'd':
    GET_S_GLOBAL(tcaDebugGlobal) = TRUE;
    break;
  case 'r':
    GET_S_GLOBAL(resendAfterCrashGlobal) = TRUE;
    break;
  case 'q':
    GET_S_GLOBAL(qassertCheckGlobal) = TRUE;
    break;
  case 't':
    GET_S_GLOBAL(tapsUnderRoot) = TRUE;
    break;
  case 'u':
    GET_S_GLOBAL(listenToStdin) = FALSE;
    break;
  case 'c':
    GET_S_GLOBAL(directDefault) = TRUE;
    break;
  default: 
    tcaModWarning("Ignoring unknown option '%s'\n", option);
  }
}

void parseCommandLineOptions(int argc, char **argv)
{ 
  int i;
  
  for (i=1; i<argc; i++)
    if (argv[i][0] == '-') {
      parseOption(argv[i],FALSE);
    }
}

void parseOpsFromStr(char *str, int *expectedMods, BOOLEAN started)
{
  char *op;
  int i, j, len;
  
  if (!str) {
    return;
  }
  
  len = strlen(str);
  
  i = 0;
  while (i != len) {
    if (str[i] == ' ' || str[i] == '\0') {
      i++;
    } else {
      op = str+i;
      
      for(j=0;op[j] != ' ' && op[j] != '\0';j++);
      op[j] = '\0';
      
      if (op[0] == '-') {
	parseOption(op, started);
      } else if (expectedMods && isdigit(op[0])) {
	*expectedMods = atoi(op);
      }
      
      i += j;
    }
  }
}


/*****************************************************************************
 *
 * FUNCTION: void displayOptions(name)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * char *name;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void displayOptions(char *name)
{
#if defined(VXWORKS)
  printf("Usage: central(\"[expected-modules] -l[options] -L[options] -f[filename]\")\n");
#else
  printf("Usage: %s [expected-modules] -l[options] -L[options] -f[filename]\n",
	 name);
#endif
  printf("\n expected-modules: minimum number of modules being run\n");
  printf(" -l: logging onto terminal.\n");
  printf("     options are: m (message traffic)\n");
  printf("                  s (status of TCA)\n");
  printf("                  t (time messages are handled)\n");
  printf("                  d (data associated with message)\n");
  printf("                  i (ignore logging registration and deregistration messages)\n");
  printf("                  h (incoming message handle time summary)\n");
  printf("                  r (log the reference Id as well as the msg name)\n");
  printf("                  p (log the reference Id of the message's parent)\n");
  printf("                  x (no logging)\n");
  printf("     -l is equivalent to -lmstdh; the default is -lmsi\n");
  printf(" -L: logging into file.\n");
  printf("     options are the same as above with the addition of F.\n");
  printf("                  F (don't flush file after each line)\n");
  printf("     the default is -Lx\n");
  printf(" -f: filename to use for logging.\n");
  printf("     If not specified, name is automatically generated.\n");
  printf(" -v: display server version info.\n");
  printf(" -r: try resending non-completed messages when modules crash\n");
  printf(" -p: specify the server port - the compiled port is: %d\n", 
	 SERVER_PORT);
  printf(" -d: flag random debugging.\n");
  printf(" -q: test quantity lattice insertions for task tree nodes.\n");
  printf(" -t: put all wiretap messages under the task tree root node\n");
  printf("     (the default is that they are children of the tapped message\n");
  printf(" -s: silent running, don't print anything to stdout.\n");
  printf(" -u: Don't run the user interface (stdin).\n");
  printf(" -c: Use direct (not via central) conntections where possible.\n");
  fflush(stdout);
}


/*****************************************************************************
 *
 * FUNCTION: void displayHelp()
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * char *name;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void displayHelp(void)
{
  
  printf("\nThe following command are available:\n");
  printf("help: print this message.\n");
  printf("kill: Kill the task tree. Removes all pending and complete messages.\n");
  printf("display : Display the task tree and non-task tree messages.\n");
  printf("status : Display the known modules and their status.\n");
  printf("close <module>: Close a connection to a module.\n");
  printf("unlock <resource>: Unlock a locked resouce.\n");
  printf("The following command line options can also be used as commands\n");
  printf(" -l: logging onto terminal.\n");
  printf("     options are: m (message traffic)\n");
  printf("                  s (status of TCA)\n");
  printf("                  t (time messages are handled)\n");
  printf("                  d (data associated with message)\n");
  printf("                  i (ignore logging registration and deregistration messages)\n");
  printf("                  h (incoming message handle time summary)\n");
  printf("                  r (log the reference Id as well as the msg name)\n");
  printf("                  p (log the reference Id of the message's parent)\n");
  printf("                  x (no logging)\n");
  printf("     -l is equivalent to -lmstdh; the default is -lmsi\n");
  printf(" -L: logging into file.\n");
  printf("     options are the same as above with the addition of F.\n");
  printf("                  F (don't flush file after each line)\n");
  printf("     the default is -Lx\n");
  printf(" -f: filename to use for logging.\n");
  printf("     If not specified, name is automatically generated.\n");
  printf(" -v: display server version info.\n");
  printf(" -r: try resending non-completed messages when modules crash\n");
  printf(" -d: flag random debugging.\n");
  printf(" -q: test quantity lattice insertions for task tree nodes.\n");
  printf(" -t: put all wiretap messages under the task tree root node\n");
  printf("     (the default is that they are children of the tapped message\n");
  printf(" -s: silent running, don't print anything to stdout.\n");
  printf(" -u: Don't run the user interface (stdin).\n");
  printf(" -c: Use direct (not via central) contections where possible.\n");
  fflush(stdout);
}


/*****************************************************************************
 *
 * FUNCTION: void displayVersion()
 *
 * DESCRIPTION: displays the central server banner.
 *
 * INPUTS: none.
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

void displayVersion(void)
{
  tcaModWarning( "Task Control Server %d.%d.%d \n",
		TCA_VERSION_MAJOR, TCA_VERSION_MINOR, TCA_VERSION_MICRO);
  tcaModWarning( " Released : %s\n", TCA_VERSION_DATE);
  tcaModWarning( " Commited : %s\n", TCA_COMMIT_DATE);
  tcaModWarning( " Compiled : %s %s\n", __DATE__, __TIME__);
}
