/*****************************************************************************
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: central
 *
 * FILE: central.c
 *
 * ABSTRACT:
 *
 * This file contains the "main" of the central tca server.
 *
 * REVISION HISTORY:
 *
 * $Log: central.c,v $
 * Revision 1.34  1996/05/09  18:30:21  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.33  1996/02/11  04:11:58  rich
 * Fixed warning message.
 *
 * Revision 1.32  1996/02/10  16:49:32  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.31  1996/02/06  19:04:09  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.30  1996/01/30  15:03:50  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.29  1996/01/05  16:31:00  rich
 * Added windows NT port.
 *
 * Revision 1.28  1995/12/17  20:21:08  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.27  1995/12/15  01:23:02  rich
 * Moved Makefile to Makefile.generic to encourage people to use
 * GNUmakefile.
 * Fixed a memory leak when a module is closed and some other small fixes.
 *
 * Revision 1.26  1995/07/12  04:54:17  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.25  1995/07/10  16:16:48  rich
 * Interm save.
 *
 * Revision 1.24  1995/07/08  18:24:39  rich
 * Change all /afs/cs to /afs/cs.cmu.edu to get ride of conflict problems.
 *
 * Revision 1.23  1995/07/06  21:15:42  rich
 * Solaris and Linux changes.
 *
 * Revision 1.22  1995/04/21  03:53:07  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.21  1995/04/07  05:02:44  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.20  1995/03/28  01:14:16  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.19  1995/03/14  22:36:37  rich
 * Fixed problem with multiple read needed when doing a vector read. (Fix
 * to version 7.9)
 * Also fixed the data size problem from 7.9.
 *
 * Revision 1.18  1995/01/25  00:00:54  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Fixed problems with little endian translation.
 *
 * Revision 1.17  1995/01/18  22:39:46  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.16  1994/05/25  04:56:55  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.15  1994/05/06  04:46:58  rich
 * Put central io routines in a new file.
 * Fixed GNUmakefile.
 *
 * Revision 1.14  1994/05/05  00:46:00  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.13  1994/04/28  22:16:40  rich
 * Added very simple stdin interface to central.  You can type command line
 * argument into standard in while central is running.  One option per
 * line.  In addition, it understands "quit" and "help".
 *
 * Revision 1.12  1994/04/28  16:15:22  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.11  1994/04/26  16:23:08  rich
 * Now you can register an exit handler before anything else and it will
 * get called if connecting to central fails.
 * Also added code to handle pipe breaks during writes.
 *
 * Revision 1.10  1994/04/16  19:41:42  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.9  1994/03/28  02:22:33  rich
 * parseFmttrs needs to be in the server objects and not the module objects.
 *
 * Revision 1.8  1994/01/31  18:27:30  reids
 * Several major changes (and some minor ones)
 * 1. tcaFreeData and tcaFreeReply now work even if the data or message format
 *    is NULL
 * 2. Using the "-t" option in central, message taps are added as a child of
 *    the task tree node that was tapped.
 * 3. Named formatters are now expanded only when needed
 * For more details, see ../doc/tca-7-4.release.notes
 *
 * Revision 1.7  1993/12/14  17:32:57  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/12/01  18:02:49  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.5  1993/11/21  20:17:18  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.4  1993/08/30  21:53:10  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:14:16  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:16:57  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:16  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:16  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:16  fedor
 * Added Logging.
 *
 * 25-Jul-91 Reid Simmons at School of Computer Science, CMU
 * Removed the -b option (buffers are no longer managed).
 * Added -r option to try resending messages after module crashes.
 *
 *  8-Jul-91 Reid Simmons at School of Computer Science, CMU
 * Removed the -gc option.
 * Added -b <n> option to pre-allocate a data message buffer.
 *
 *  7-Apr-90 Christopher Fedor at School of Computer Science, CMU
 * Added -v option to simply display current server version info.
 *
 * 30-Mar-90 Christopher Fedor at School of Computer Science, CMU
 * Revised to software standards.
 *
 * $Revision: 1.34 $
 * $Date: 1996/05/09 18:30:21 $
 * $Author: reids $
 *
 *****************************************************************************/

#include "globalS.h"


/*****************************************************************************
 *
 * FUNCTION: int parseExpectedMods(argc, argv, expectedMods)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * int argc;
 * char **argv;
 * int *expectedMods;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

#ifndef VXWORKS
static void parseExpectedMods(int argc, char **argv, int *expectedMods)
{ 
  int i;
  
  /* Default to 1 module.  */
  *expectedMods = 1;

  for (i=1; i<argc; i++) {
    if (isdigit(argv[i][0])) {
      *expectedMods = atoi(argv[i]);
      return;
    }
  }
}
#endif


/*****************************************************************************
 *
 * FUNCTION: void abortCentral(s, c, sp)
 *
 * DESCRIPTION:
 * Trap fatal errors and attempt a core dump.
 *
 * INPUTS:
 * int s, c;
 * struct sigcontext *sp;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
/*
   sun os 4.0.3 added a parameter
   void abortCentral(s, c, sp, addr)
   int s, c;
   struct sigcontext *sp;
   char *addr;
   */

#if (defined(sun4) || defined(sparc)) && !defined(__svr4__)
static void abortCentral(int s, int c, struct sigcontext *sp)
#else
static void abortCentral(int s)
#endif
{
  /*  idTablePrintInfo(GET_S_GLOBAL(dispatchTable));*/
  tcaError("Central Abort : Signal %d",s);
}

/*****************************************************************************
 *
 * FUNCTION: void pipeClosedHnd(s, c, sp)
 *
 * DESCRIPTION:
 * Trap fatal errors and attempt a core dump.
 *
 * INPUTS:
 * int s, c;
 * struct sigcontext *sp;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/
/*
   sun os 4.0.3 added a parameter
   void pipeClosedHnd(s, c, sp, addr)
   int s, c;
   struct sigcontext *sp;
   char *addr;
   */

#if (defined(sun4) || defined(sparc))  && !defined(__svr4__)
static void pipeClosedHnd(int s, int c, struct sigcontext *sp)
#else
     static void pipeClosedHnd(int s)
#endif
{
#ifdef applec
#pragma unused(s)
#endif
  GET_M_GLOBAL(pipeBroken) = TRUE;
  tcaModWarning("Pipe Broke \n");
}


/******************************************************************************
 *
 * FUNCTION: char *centralFreeMalloc(amount)
 *
 * DESCRIPTION: Emergency memory free.
 *
 * INPUTS: unsigned int amount;
 *
 * OUTPUTS: void
 *
 * NOTES: 
 *
 *****************************************************************************/

static void centralFreeMemory(unsigned int amount)
{
  /* Try to free up amount of memory.  Really want to free more. */
  /* Try by deleting stuff from the pending queues. */
  /* The problem is that this can consume memory. */
  tcaModWarning("Warning MEMORY EXHAUSTED, purging queues !!!!\n");
  
  purgeResoucePending();
  
  return;
}


#ifdef applec
static 	char l[120];

int getCommandLineOptions (char **argv)
{
  int i, lastBlank = TRUE, argc = 1;
  
  argv[0] = "central";	
  fprintf(stdout, "Enter Command Line Options: "); fflush(stdout);
  gets(&l);
  for (i=0; i<strlen(l); i++) {
    if (l[i] == ' ') {
      l[i] = '\0';
      lastBlank = TRUE;
    } else if (lastBlank) {
      argv[argc++] = &(l[i]);
      lastBlank = FALSE;
    }
  }
  l[i] = '\0';
  return argc;
}
#endif

/*****************************************************************************
 *
 * FUNCTION: startWinsock()
 *
 * DESCRIPTION: We must start up the Winsock before making any socket requests
 *              Begin winsock startup.
 *
 *****************************************************************************/
#ifdef _WINSOCK_
static void startWinsock (void)
{
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;
  
  wVersionRequested = MAKEWORD( 1, 1);
  err = WSAStartup( wVersionRequested, &wsaData);
  if(err!=0){
    switch (err) {
    case WSASYSNOTREADY:
      printf("Network subsystem not ready for network communication\n");
      printf("Cannot continue\n");
      break;
    case WSAVERNOTSUPPORTED:
      printf("You do not have a version 1.1 Winsock or greater\n");
      printf("Cannot continue\n");
      break;
    case WSAEINVAL:
      printf("You winsock.dll is not 100% Winsock compatible\n");
      printf("Cannot continue\n");
      break;
    default:
      printf("Unknown Winsock start up error #%d",err);
      printf("Cannot continue\n");
    }
    return(-1);
  }
  
  if (LOBYTE(wVersionRequested) != 1 || HIBYTE(wVersionRequested) != 1) {
    printf("Your winsock.dll does not support version 1.1");
    printf("Cannot continue\n");
    WSACleanup();
    printf("Socket cleaned up.");
    return(-1);
  }
  
  /* End winsock startup */
  printf("Winsock successfully loaded\n");
  /* If we made it here, version 1.1 is supported and has been loaded */
}
#endif /* Winsock DLL loading */

/*****************************************************************************
 *
 * FUNCTION: main(argc, argv)
 *
 * DESCRIPTION:
 * start.
 *
 * INPUTS:
 * int argc;
 * char **argv;
 *
 * OUTPUTS: void
 *
 *****************************************************************************/

#ifdef VXWORKS
/* Used for cleaning stopping central */
static int centralTID = 0;

/* To be called from the shell */
void killCentral (void)
{
  if (centralTID != 0)
    kill(centralTID, SIGTERM);
}

void central(char *options)
#else
int main(int argc, char **argv)
#endif
{
  int expectedMods = 0;
  
#ifdef applec
  argc = getCommandLineOptions(argv);
#elif defined(THINK_C)
  argc = ccommand(&argv);
#endif
 
#ifdef _WINSOCK_
  startWinsock();
#endif /* Winsock DLL loading */
  
  tcaModuleInitialize();
#ifdef VXWORKS
  /* Do this only after the socket is set up (in case there is an
     old central lying around that needs killed */
  centralTID = taskIdSelf();
#endif
  globalSInit();
  
#if !defined(THINK_C) && !defined(applec) && !defined(__TURBOC__)
  (void)signal(SIGINT, abortCentral);
  (void)signal(SIGBUS, abortCentral);
  (void)signal(SIGSEGV, abortCentral);
  (void)signal(SIGPIPE, pipeClosedHnd);
  (void)signal(SIGTERM, abortCentral);
#endif /* !THINK_C && !applec */
  
#ifndef VXWORKS
  if ((argc > 1) && (STREQ(argv[1], "-v")))
    displayVersion();
  else if ((argc > 1) && (STREQ(argv[1], "-h")))
    displayOptions(argv[0]);
  else {
    parseExpectedMods(argc, argv, &expectedMods);
    parseCommandLineOptions(argc, argv);
#else
  if ((options!= NULL) && (checkOccurrence(options, "vV"))) {
    displayVersion();
  } else if ((options!= NULL) && (checkOccurrence(options, "hH"))) {
    displayOptions("central");
  } else {
    parseOpsFromStr(options, &expectedMods, FALSE);
#endif
      
    if (expectedMods < 1)
      expectedMods = 1;
      
    if (!serverInitialize(expectedMods)) {
      tcaError("ERROR: Unable to start server, Is one already running?\n");
    }
      
#ifndef VXWORKS
    /* Register a method for freeing memory in an emergency. */
    tcaRegisterFreeMemHnd(centralFreeMemory,3);
      
    if (GET_S_GLOBAL(listenToStdin))
      printPrompt();
#endif

    listenLoop();
  }
#ifdef _WINSOCK_
  WSACleanup();
  printf("Socket cleaned up.");
#endif /* Unload Winsock DLL */
#ifndef VXWORKS
  return 1;
#endif
}
