/***************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: cells
 *
 * FILE: cells.h
 *
 * ABSTRACT:
 *
 * A "cell" is an entity that can take on different values under
 * different contexts.  Each value of a cell is represented by a tms
 * node.  Cells can have any number of values.  Regular cells can have
 * many values (tms nodes) IN at once; "exclusive" cells can have only
 * one value in at a time; "constrained" cells can have many values in at
 * once, but only the "most-constrained" cell is active at any one time.
 *
 * "Rules" can be attached to a cell.  When a new value (tms node) is
 * added to the cell the rule is run, its argument being the tms node for
 * that value.  Typically, the rules are used to link up justifications
 * between the tms_nodes of other cells. Rules are defined in the file
 * "rules".
 *
 * For example, A+B can be implemented by two identical rules
 * on the cells for both A and B.  Let's say B already has the value `B =
 * 2'.  When a new value is added for A, such as `A = 1', the rule finds
 * or creates a tms node for the sum `A+B = 3' and justifies the nodes
 * with the tms nodes representing `A = 1' and `B = 2'.  There is a macro
 * for automatically creating accessor and setting functions for the
 * values of cells. 
 *
 * This file also contains mechanisms for maintaining a set of contexts.
 * The context mechanism can be implemented using an ATMS or a regular
 * TMS.  Currently it uses a modification of Williams' JTMS (in file
 * "tms"), inning and outing nodes as contexts are switched, instead of
 * keeping track of all contexts at once. 
 
 * REVISION HISTORY
 *
 * $Log: cells.h,v $
 * Revision 1.12  1996/01/30  15:03:43  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.11  1996/01/27  21:52:55  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.10  1995/12/17  20:21:06  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.9  1995/05/31  19:35:04  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.8  1995/03/30  15:42:23  rich
 * DBMALLOC works.  To use "gmake -k -w DBMALLOC=DBMALLOC install"
 * Added simple list of strings data structure that can be passed via tca
 * messages.
 * Use the string list to maintain a global variable of messages with taps.
 * Tapped messages are not sent via direct connections.
 * Implemented code to vectorize data to be sent so that it does not have
 * to be copied.  Currently, only flat, packed data structures are
 * vectored.  This can now be easily extended.
 * Changed Boolean -> BOOLEAN for consistency and to avoid conflicts with x11.
 * Fixed bug were central would try and free the "***New Module***" and
 * "*** Unkown Host***" strings when a module crashed on startup.
 * Fixed a bug reported by Jay Gowdy where the code to find the size of a
 * variable lenght array would access already freed data when called from
 * tcaFreeData.
 *
 * Revision 1.7  1995/01/18  22:39:44  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.6  1994/05/17  23:15:12  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.5  1993/12/01  18:02:43  rich
 * Fixed a problem with the port number being double converted to network
 * byte order.
 * Some general cleanup.
 *
 * Revision 1.4  1993/11/21  20:17:15  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/08/27  07:14:14  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:16:54  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:41  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:14  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:14  fedor
 * Added Logging.
 *
 * $Revision: 1.12 $
 * $Date: 1996/01/30 15:03:43 $
 * $Author: rich $
 *
 ***************************************************************************/

#ifndef INCcells
#define INCcells

typedef struct { 
  const char *name;
  LIST_PTR assumptions;
} CONTEXT_TYPE, *CONTEXT_PTR;

typedef enum {
  RegularCell=0, ExclusiveCell=1, ConstrainedCell=2
#ifdef FORCE_32BIT_ENUM
    , dummyCellValue = 0x7FFFFFFF
#endif
} CELL_VALUE_TYPE;

typedef void 
(* CELL_SAY_FN)(struct _CELL_TYPE *, const char *);

typedef BOOLEAN 
(* SAME_VALUE_FN)(const char *, const char *);

typedef struct _CELL_TYPE { 
  const char *datum;
  CELL_VALUE_TYPE type;
  LIST_PTR contextualValues;
  LIST_PTR rules;
  LIST_PTR outRules;
  LIST_PTR inRules;
  SAME_VALUE_FN sameValueFN;
  CELL_SAY_FN sayFN;
  const char *additional_field;
} CELL_TYPE, *CELL_PTR;

typedef BOOLEAN 
(* MOST_CONSTRAINED_FN)(const char *, const char *);

typedef struct {
  const char *most_constrained_value;
  MOST_CONSTRAINED_FN most_constrained_fn;
} CONSTRAINED_CELL_ADDL_TYPE, *CONSTRAINED_CELL_ADDL_PTR;

typedef struct {
  TMS_NODE_PTR node;
  int32 number;
} DEP_NODE_TYPE, *DEP_NODE_PTR; 

#define EXCLUSIVE_CELL_CURRENT_VALUE(cell) \
((cell)->additional_field)

#define CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell) \
((CONSTRAINED_CELL_ADDL_PTR) \
 (cell)->additional_field)->most_constrained_value

#define CONSTRAINED_CELL_MOST_CONSTRAINED_FN(cell) \
((CONSTRAINED_CELL_ADDL_PTR) \
 (cell)->additional_field)->most_constrained_fn

#define IS_EXCLUSIVE_CELL(cell) ((cell)->type == ExclusiveCell)

#define IS_CONSTRAINED_CELL(cell) ((cell)->type == ConstrainedCell)

CELL_PTR cellCreateConstrained(const void *datum, 
			       SAME_VALUE_FN sameValueFN, 
			       MOST_CONSTRAINED_FN moreConstrainedFN, 
			       CELL_SAY_FN sayFN);
void cellFree(CELL_PTR *cell);
void sayNodeValue(TMS_NODE_PTR node);
void tmsInNode(TMS_NODE_PTR node, int32 doResupport);
void tmsOutNode(TMS_NODE_PTR node);
TMS_NODE_PTR addValueNode(CELL_PTR cell, const void *value);
TMS_NODE_PTR getOrAddValueNode(CELL_PTR cell, const void *value);
TMS_NODE_PTR getValueNode(CELL_PTR cell, const void *value);
TMS_NODE_PTR currentValueSupporter(CELL_PTR cell);
void justify(CELL_PTR cell, const void *value, 
	     JUSTIFICATION_PTR justification);
void describeCell(CELL_PTR cell);
void printNodeDependencies (TMS_NODE_PTR node);
void printCellDependencies (CELL_PTR cell);

#endif /* INCcells */
