/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: tms
 *
 * FILE: cells.c
 *
 * ABSTRACT:
 * This file contains code to create `cells' having multiple values of
 * which only some are valid in any particular context (i.e. set of
 * assumptions). Which values are current at any time is maintained using a
 * TMS.
 *
 * A "cell" is an entity that can take on different values under
 * different contexts.  Each value of a cell is represented by a tms
 * node.  Cells can have any number of values.  Regular cells can have
 * many values (tms nodes) IN at once; "exclusive" cells can have only
 * one value in at a time; "constrained" cells can have many values in at
 * once, but only the "most-constrained" cell is active at any one time.
 *
 * "Rules" can be attached to a cell.  When a new value (tms node) is
 * added to the cell the rule is run, its argument being the tms node for
 * that value.  Typically, the rules are used to link up justifications
 * between the tms_nodes of other cells. Rules are defined in the file
 * "rules".
 *
 * For example, A+B can be implemented by two identical rules on the
 * cells for both A and B.  Let's say B already has the value `B = 2'.
 * When a new value is added for A, such as `A = 1', the rule finds or
 * creates a tms node for the sum `A+B = 3' and justifies the nodes with
 * the tms nodes representing `A = 1' and `B = 2'.
 *
 * This file also contains mechanisms for maintaining a set of contexts.
 * The context mechanism can be implemented using an ATMS or a regular TMS.
 * Currently it uses a modification of Williams' JTMS (in file "tms"), inning
 * and outing nodes as contexts are switched, instead of keeping track of all
 * contexts at once.
 *
 * EXPORTS
 *
 * cellCreate(datum, sameValueFN, sayFN) 
 * Creates a cell identified with the
 * given "datum".  The "sameValueFN" is used to determine if two values
 * of the cell are the same.  The "sayFN" is used for printing the cell
 * and its value, and is used for providing explanations.  The "sayFN" 
 * takes 2 arguments -- the cell and its value -- and prints a string that
 * says what the value of the cell means, e.g. "X is greater than Y").
 *
 * cellCreateExclusive(datum, sameValueFN, sayFN)
 * Creates an "exclusive" 
 * cell that can only have one value IN at any one time.  Exclusive 
 * cells are useful, for example, in keeping the constraint that the
 * predicate (P X) can have the value TRUE or FALSE, but not both.
 *
 * cellCreateConstrained(datum, sameValueFN, more_constrained_fn, sayFN) 
 * Creates a "constrained" cell that has an ordering on its values, so
 * that only one of IN values is the most-constrained value at any one
 * time.  The int function "more_constrained_fn" takes two "values" 
 * as arguments and returns TRUE if the first is more constrained than
 * the second. This type of cell is useful, for example, in keeping track
 * of the tightest numeric bound known on a quantity.   Note that if a
 * tms node's cell is a constrained_cell, then the node's rules run
 * only when the value of the node is the cell's most-constrained value.
 *
 * assume(cell, value)
 * Add the assumption that the "cell" has the given "value"
 * in the current context.
 *
 * unassume(cell, value)
 * Remove the assumption that the "cell" has the given
 * "value" in the current context.
 *
 * assertPremise(cell, value, informant)
 * Add the assertion that the "cell" has
 * the given "value" (in all contexts), for the "reason" specified by
 * the string "informant".
 *
 * unassertPremise(cell, value, informant)
 * Remove the assertion that the
 * "cell" has the given value.  The "informant" must be the same string
 * as the one used in the "Assert_Premise" call.
 *
 * justify(cell, value, justification)
 * Assert that the "cell" will have the given
 * "value" when (and if) all the supporters of the "justification" are 
 * believed (i.e., "IN").
 *
 * getOrAddValueNode(cell, value, justification)
 * Looks for a tms node of
 * "cell" whose datum is equal to "value", else creates one and adds it
 * to the cell's list of values.  Uses the "sameValueFN" of the cell
 * to determine equality of two values.
 *
 * getOrAddValueNode1(cell, value, justification)
 * Same as getOrAddValueNode, except if justification is provided (not NULL),
 * then the function "Justify" is called. Returns the tms node.
 *
 * getValueNode(cell, value)
 * Returns the tms_node if one exists on the cell's 
 * list whose datum is equal to "value" (the tms_node does NOT have to
 * currently be IN).  Uses the "sameValueFN" of the cell to determine
 * equality of two values.
 *
 * currentValue(cell)
 * Returns the value which is currently believed for the
 * "cell". For regular cells (not "constrained" or "exclusive"), an 
 * error occurs if more than one value is IN.	     
 *
 * currentValueSupporter(cell)
 * Returns the tms_node of the currently believed
 * (or most-constrained) value.  For regular cells (not "constrained" or
 * "exclusive"), an error occurs if more than one value is IN.
 *
 * getCurrentValues(cell)
 * Returns the list of values which are currently IN for the "cell".
 *
 * Why(cell, value)
 * Interactively prints out why the value is currently believed
 * to be IN (if "value" is NULL, it defaults to the current IN value, or
 * most-constrained value).
 * WHY accepts input to interactively print out the dependency tree. If
 * the input is not a number, then the function is exited.  If the number
 * "n" is positive, prints why the nth support is believed; if zero, pops
 * the stack; if negative, pops the stack and prints why the (-n)th
 * support is believed.
 *
 * FUNCTIONS DEALING WITH CONTEXTS
 * A "context" is a list of ASSUMED tms_nodes that are currently IN.  
 * Whenever an assumption is made, if there is a "current_context" then the 
 * assumed tms_node is added to the current_context.  Similarly, it is removed
 * from the current_context if it is unassumed.  When "switching" contexts 
 * all the nodes in the current_context OUT-ed and all the nodes in the new 
 * context are IN-ed (the actual algorithm is smarter than that, if the same 
 * node appears in both contexts, it is not changed).
 *
 * createContext(name, assumptions)
 * Create a named context with the initial assumptions.
 * DOES NOT AUTOMATICALLY MAKE IT THE CURRENT_CONTEXT.
 *
 * findContext(name)
 * find the context with the given name.  If more than one
 * has the same name, the last one created is returned.
 *
 * THE_CURRENT_CONTEXT
 * Returns the current context.
 *
 * addToContext(assumption, context)
 * Add the assumption to the specified context, and assumes the assumption
 * is IN. Usually one should just call "ASSUME", and the assumption is 
 * automatically added to the current_context.
 *
 * copyContext(new_name, context)
 * Creates a new context with the new name, and with all the 
 * assumptions of "context".
 *
 * switchContexts(new_context)
 * Make "new_context" be the current context, IN-ing and OUT-ing 
 * assumptions as necessary.
 *
 * REVISION HISTORY:
 *
 * $Log: cells.c,v $
 * Revision 1.13  1996/02/10  16:49:30  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.12  1995/12/17  20:21:04  rich
 * Have free routines set pointers to NULL.
 * Removed old makefiles.
 *
 * Revision 1.11  1995/07/10  16:16:46  rich
 * Interm save.
 *
 * Revision 1.10  1995/01/18  22:39:41  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.9  1994/04/28  16:15:19  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.8  1994/04/16  19:41:39  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1993/12/14  17:32:53  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.6  1993/11/21  20:17:12  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.5  1993/10/21  16:13:31  rich
 * Fixed compiler warnings.
 *
 * Revision 1.4  1993/08/30  21:53:04  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.3  1993/08/27  07:14:10  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.2  1993/05/26  23:16:48  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:15  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:11  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.2  1993/05/19  17:23:09  fedor
 * Added Logging.
 *
 * 15-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * The routine currentValueSupporter was missing return values
 * for some cases - added them.
 *
 *  3-Jul-91 Reid Simmons, School of Computer Science, CMU
 * Fixed cellFree to free rules before freeing the contextual values, to avoid
 * spurious rule firings.
 *
 *  6-Apr-90 Christopher Fedor, School of Computer Science, CMU
 * Changed findValueInNode to be almost a copy of listMemReturnItem.
 * This is a temporary solution to avoid using the generic listFirst and
 * listNext routines for iterating this list. This function gets
 * called by a qlattice routine that is already iterating the same list
 * with listFirst and listNext and leads to an infinite loop when the
 * iteration gets reinitialized to the begining. This will have to be
 * revisted.
 *
 *  5-Nov-89 Christopher Fedor, School of Computer Science, CMU
 * Revised to reflect software standards.
 * Removed DEF_CELL_FIELD macro which was a replacement for accessor 
 * functions created in lisp defstructs.
 *
 * Ancient   Reid Simmons, School of Computer Science, CMU
 * (c) Copyright 1988 Reid G. Simmons.  All rights reserved.
 * Adapted from LISP version used for the Quantity Lattice
 * (see Simmons, AAAI-86).   
 *
 * $Revision: 1.13 $
 * $Date: 1996/02/10 16:49:30 $
 * $Author: rich $
 * 
 *****************************************************************************/

#include "globalS.h"
#include "tcaMem.h"

/*****************************************************************************/


/*****************************************************************************
 *
 * FUNCTION: CONTEXT_PTR createContext(name, assumptions)
 * 
 * DESCRIPTION:
 * Builds a new context (set of assumptions)
 * (does not make it the current context)
 * "name" can be NULL, in which case the context will not have a name 
 * (and will shadow all other un-named contexts in function "Find_Context").
 *
 * 11/5/87 CONTEXTS changed so that they can be named.
 * Two contexts can share the same name, but the most recently created shadows
 * the older one when trying to access by name.
 *
 * A CONTEXT is a set of assumptions [tms_nodes that are assumed to be IN]
 *
 * INPUTS:
 * const char *name;
 * LIST_PTR assumptions;
 *
 * OUTPUTS: CONTEXT_PTR
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92)*/
#if 0
CONTEXT_PTR createContext(name, assumptions)
     const char *name;
     LIST_PTR assumptions;
{ 
  CONTEXT_PTR context;
  
  context = NEW(CONTEXT_TYPE);
  context->name = name;
  context->assumptions = assumptions;
  
  if (!GET_S_GLOBAL(THE_CONTEXTS)) 
    GET_S_GLOBAL(THE_CONTEXTS) = listMake1((char *)context);
  else
    listInsertItem((char *)(GET_S_GLOBAL(THE_CONTEXTS)), context->assumptions);
  
  return context;
}
#endif

/*****************************************************************************
 *
 * FUNCTION: CONTEXT_PTR findContext(name) 
 * 
 * DESCRIPTION:
 * Finds the context with the appropriate name.  
 * If more than one has the same name, the last one created is returned.
 *
 * INPUTS: const char *name;
 *
 * OUTPUTS: CONTEXT_PTR
 *
 *****************************************************************************/
#if 0
CONTEXT_PTR findContext(name) 
     const char *name;
{ 
  DoList(context, CONTEXT_PTR, GET_S_GLOBAL(THE_CONTEXTS),
	 { if (STREQ(name, context->name))
	     return context;
	 }
	 );
  return NULL;
}
#endif

/*****************************************************************************
 *
 * FUNCTION: CONTEXT_PTR copyContext(name, context)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * char  *name; 
 * CONTEXT_PTR context;
 *
 * OUTPUTS: CONTEXT_PTR
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
CONTEXT_PTR copyContext(name, context)
     char  *name; 
     CONTEXT_PTR context;
{ 
  return createContext(name, listCopy(context->assumptions));
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void addToContext(assumption, context)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 
 * TMS_NODE_PTR assumption; 
 * CONTEXT_PTR context; 
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void addToContext(assumption, context)
     TMS_NODE_PTR assumption; 
     CONTEXT_PTR context; 
{ 
  if (!listMemberItem((char *)assumption, context->assumptions)) {
    listInsertItem((char *)assumption, context->assumptions);
    if (context == THE_CURRENT_CONTEXT) 
      tmsAssumeNode(assumption);
  }
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void switchContexts(context)
 * 
 * DESCRIPTION:
 * Asserts the assumptions of context in the TMS, 
 * retracts the former context's assumptions.
 *
 * INPUTS:
 * CONTEXT_PTR context;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* NOT USED: RGS 11/11/92 */
#if 0
void switchContexts(context) 
     CONTEXT_PTR context;
{ 
  LIST_PTR list, assumptions;
  TMS_NODE_PTR assumptionToOut,assumptionToIn;
  
  if (context != GET_S_GLOBAL(THE_CURRENT_CONTEXT)) { 
    assumptions = context->assumptions;
    
    if (GET_S_GLOBAL(THE_CURRENT_CONTEXT) != NULL) {
      list = GET_S_GLOBAL(THE_CURRENT_CONTEXT)->assumptions;
      assumptionToOut = (TMS_NODE_PTR)listFirst(list);
      while (assumptionToOut) {
	if (!listMemberItem((char *)assumptionToOut, assumptions))
	  tmsUnassumeNode(assumptionToOut);
	assumptionToOut = (TMS_NODE_PTR)listNext(list);
      }
      runRules(-1);
    }
    
    assumptionToIn = (TMS_NODE_PTR)listFirst(assumptions);
    while (assumptionToIn) {
      if (!IS_ASSUMED(assumptionToIn)) 
	tmsAssumeNode(assumptionToIn);
      assumptionToIn = (TMS_NODE_PTR)listNext(assumptions);
    }
    
    runRules(-1);
    
    /* 2-Feb-90: fedor: is the previous context freed? */
    GET_S_GLOBAL(THE_CURRENT_CONTEXT) = context;
  }
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void clearContexts()
 * 
 * DESCRIPTION:
 *
 * INPUTS: none.
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* NOT USED: RGS 11/11/92 */
#if 0
void clearContexts()
{ 
  /* 2-Feb-90: fedor: looks like more memory problems */
  GET_S_GLOBAL(THE_CONTEXTS) = NULL;
  GET_S_GLOBAL(THE_CURRENT_CONTEXT) = createContext((char *)NULL,
						    (LIST_PTR)NULL);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: CELL_PTR cellCreate1(type, datum, sameValueFN, sayFN) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_VALUE_TYPE type; 
 * char *datum; 
 * SAME_VALUE_FN sameValueFn;
 * CELL_SAY_FN sayFN;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/

static CELL_PTR cellCreate1(CELL_VALUE_TYPE type, const void *datum,
			    SAME_VALUE_FN sameValueFN, CELL_SAY_FN sayFN)
{ 
  CELL_PTR cell;
  
  cell = NEW(CELL_TYPE);
  cell->type = type;
  cell->datum = datum;
  cell->sayFN = sayFN;
  cell->sameValueFN = sameValueFN;
  cell->contextualValues = listCreate();
  cell->rules = listCreate();
  cell->outRules = listCreate();
  cell->inRules = listCreate();
  cell->additional_field = NULL;
  
  return cell;
}


/*****************************************************************************
 *
 * FUNCTION: CELL_PTR cellCreate(datum, sameValueFN, sayFN) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * char *datum; 
 * SAME_VALUE_FN sameValueFN; 
 * CELL_SAY_FN sayFN;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
CELL_PTR cellCreate(datum, sameValueFN, sayFN) 
     char *datum; 
     SAME_VALUE_FN sameValueFN; 
     CELL_SAY_FN sayFN;
{ 
  return cellCreate1(RegularCell, datum, sameValueFN, sayFN);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: CELL_PTR cellCreateExclusive(datum, sameValueFN, sayFN) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * char *datum; 
 * SAME_VALUE_FN sameValueFN; 
 * CELL_SAY_FN sayFN;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
CELL_PTR cellCreateExclusive(datum, sameValueFN, sayFN) 
     char *datum; 
     SAME_VALUE_FN sameValueFN; 
     CELL_SAY_FN sayFN;
{ 
  return cellCreate1(ExclusiveCell, datum, sameValueFN, sayFN);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: 
 * CELL_PTR cellCreateConstrained(datum, sameValueFN, moreConstrainedFN, sayFN)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * void *datum; 
 * SAME_VALUE_FN sameValueFN, moreConstrainedFN;
 * CELL_SAY_FN sayFN;
 *
 * OUTPUTS: CELL_PTR
 *
 *****************************************************************************/

CELL_PTR cellCreateConstrained(const void *datum, SAME_VALUE_FN sameValueFN, 
			       MOST_CONSTRAINED_FN moreConstrainedFN, 
			       CELL_SAY_FN sayFN)
{ 
  CELL_PTR cell;
  
  cell = cellCreate1(ConstrainedCell, datum, sameValueFN, sayFN);
  cell->additional_field = 
    (char *)tcaMalloc(sizeof(CONSTRAINED_CELL_ADDL_TYPE));
  
  CONSTRAINED_CELL_MOST_CONSTRAINED_FN(cell) = moreConstrainedFN;
  CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell) = NULL;
  
  return cell;
}


/*****************************************************************************
 *
 * FUNCTION: void cellFree(cell)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 * NOTES:
 *
 *****************************************************************************/

static int iterateTmsFreeNode(const void *param, TMS_NODE_PTR node)
{
#ifdef applec
#pragma unused(param)
#endif
  tmsFreeNode(&node);
  
  return TRUE;
}

void cellFree(CELL_PTR *cell)
{
  rulesFree((*cell)->rules);
  rulesFree((*cell)->outRules);
  rulesFree((*cell)->inRules);
  (void)listIterate((LIST_ITER_FN)iterateTmsFreeNode, (void *)NULL, 
		    (*cell)->contextualValues);
  listFree(&((*cell)->contextualValues));
  if ((*cell)->type == ConstrainedCell) 
    tcaFree((char *)(*cell)->additional_field);
  tcaFree((char *)(*cell));
  *cell = NULL;
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR createContextualValue(cell, value) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char *value;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

static TMS_NODE_PTR createContextualValue(CELL_PTR cell, const void *value)
{ 
  TMS_NODE_PTR node;
  
  node = tmsCreateNode(value);
  node->cell = cell; 
  
  return node;
}


/*****************************************************************************
 *
 * FUNCTION: void cellPrint(cell) 
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

#if 0
/* No longer used */
static void cellPrint(CELL_PTR cell)
{ 
  const char *currentValue;
  const char *cellTypeChar;
  
  if (cell->type == RegularCell)
    tcaModWarning("#<CELL %s>", cell->datum);
  else {
    if (IS_EXCLUSIVE_CELL(cell)) {
      currentValue = EXCLUSIVE_CELL_CURRENT_VALUE(cell);
      cellTypeChar = "X";
    } else { 
      currentValue = CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell);
      cellTypeChar = "C";
    }
    tcaModWarning("#<%sCELL %s", cellTypeChar, cell->datum);
    if (currentValue) 
      (cell->sayFN)(cell, currentValue);
    tcaModWarning(">");
  }
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void sayNodeValue(node)
 * 
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void sayNodeValue(TMS_NODE_PTR node)
{ 
  CELL_PTR cell;
  
  cell = (CELL_PTR)node->cell;
  if (cell) 
    (cell->sayFN)(cell, node->datum);
  else 
    tcaModWarning("#%lX", (unsigned long)node);
}


/*****************************************************************************
 *
 * FUNCTION: void sayCell(cell) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void sayCell(CELL_PTR cell)
{ 
  tcaModWarning("%s", cell->datum);
}


/*****************************************************************************
 *
 * FUNCTION: void cacheValueForExclusiveCell(cell, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void cacheValueForExclusiveCell(CELL_PTR cell, TMS_NODE_PTR node)
{ 
  TMS_NODE_PTR nodeA;
  LIST_PTR contradictoryNodes, contradictoryValues;
  
  
  /* 2-Feb-90: fedor: nodeA was node looked like a mess.
     temporary lists for contradictory nodes and values? */
  
  /* 2-Feb-90: fedor: EXCLUSIVE_CELL_CURRENT needs to be removed
     defines should be similar to a function call and not be assigned 
     a value! */
  
  if (EXCLUSIVE_CELL_CURRENT_VALUE(cell)) {
    contradictoryNodes = listCreate();
    contradictoryValues = listCreate();
    
    nodeA = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (nodeA) {
      if (IS_IN(node)) {
	listInsertItem((const char *)node, contradictoryNodes);
	listInsertItem((const char *)node->datum, contradictoryValues);
      }
      nodeA = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
    
    EXCLUSIVE_CELL_CURRENT_VALUE(cell) = (char *)contradictoryValues;
    tcaModWarning("CONTRADICTORY: "); 
    sayCell(cell); 
    tcaModWarning("\n");
    
    tmsContradictory(contradictoryNodes, "one_of");
  }
  else 
    EXCLUSIVE_CELL_CURRENT_VALUE(cell) = node->datum;
}


/*****************************************************************************
 *
 * FUNCTION: void uncacheValueForExclusiveCell(cell, outNode)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * TMS_NODE_PTR outNode;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void uncacheValueForExclusiveCell(CELL_PTR cell, TMS_NODE_PTR outNode)
{ 
  const char *outValue, *cachedValue;
  LIST_PTR cachedValues;
  SAME_VALUE_FN sameValueFN;
  
  outValue = outNode->datum;
  sameValueFN = cell->sameValueFN;
  cachedValue = EXCLUSIVE_CELL_CURRENT_VALUE(cell);
  
  if (sameValueFN(outValue, cachedValue))
    EXCLUSIVE_CELL_CURRENT_VALUE(cell) = NULL;
  else { 
    cachedValues = (LIST_PTR)cachedValue;
    if (listMemReturnItem((LIST_ITER_FN) sameValueFN,
			  (char *)outValue, cachedValues)) {
      if (listLength(cachedValues) > 2) 
	perror("UNCACHE_MORE_THAN_2_VALUES");
      else {
	if (sameValueFN(outValue, listFirst(cachedValues)))
	  EXCLUSIVE_CELL_CURRENT_VALUE(cell) = (char *)listNext(cachedValues);
	else
	  EXCLUSIVE_CELL_CURRENT_VALUE(cell) = (char *)listFirst(cachedValues);
      }
    }
    else 
      perror("Something wrong with uncaching values");
  }
}


/*****************************************************************************
 *
 * FUNCTION: int cacheValueForConstrainedCell(cell, node)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * TMS_NODE_PTR node;
 *
 * OUTPUTS: int
 *
 *****************************************************************************/

static int cacheValueForConstrainedCell(CELL_PTR cell, TMS_NODE_PTR node)
{ 
  const char *currentValue, *newValue;
  
  newValue = node->datum;
  currentValue = CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell);
  if (!currentValue || 
      (CONSTRAINED_CELL_MOST_CONSTRAINED_FN(cell))(newValue, currentValue)) {
    CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell) = newValue;
    return TRUE;
  }
  else 
    return FALSE;
}


/*****************************************************************************
 *
 * FUNCTION: void uncacheValueForConstrainedCell(cell, outNode)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * TMS_NODE_PTR outNode;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static void uncacheValueForConstrainedCell(CELL_PTR cell, TMS_NODE_PTR outNode)
{ 
  MOST_CONSTRAINED_FN moreConstrainedFN;
  TMS_NODE_PTR node, constrainedNode;
  
  if ((cell->sameValueFN)(outNode->datum, 
			  CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell))) {
    constrainedNode = NULL;
    moreConstrainedFN = CONSTRAINED_CELL_MOST_CONSTRAINED_FN(cell);
    
    node = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (node) {
      if (IS_IN(node) && 
	  (!constrainedNode ||
	   (moreConstrainedFN(node->datum, constrainedNode->datum))))
	constrainedNode = node;
      node = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
    
    if (constrainedNode) {
      CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell) = constrainedNode->datum;
      runInRules(constrainedNode, cell);}
    else 
      CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell) = NULL;
  }
}


/*****************************************************************************
 *
 * FUNCTION: void tmsInNode(node, doResupport)
 * 
 * DESCRIPTION:
 * Notifies that the node is going in.
 * doResupport is TRUE if it's just going in under a different context.
 * Interface to the TMS -- called whenever node comes IN.
 *
 * INPUTS:
 * TMS_NODE_PTR node; 
 * int doResupport;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void tmsInNode(TMS_NODE_PTR node, int32 doResupport)
{ 
  int  tryRunning = TRUE;
  CELL_PTR cell;
  
  cell = (CELL_PTR)node->cell;
  if (!doResupport && cell) {
    if (cell->type == ExclusiveCell)
      cacheValueForExclusiveCell(cell, node);
    else if (cell->type == ConstrainedCell)
      tryRunning = cacheValueForConstrainedCell(cell, node);
  }
  
  if (tryRunning) 
    runInRules(node, cell);
}


/*****************************************************************************
 *
 * FUNCTION: void tmsOutNode(node) 
 * 
 * DESCRIPTION:
 * Notifies that the node is going out.
 * Interface to the TMS -- called whenever node goes OUT.
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *
 *****************************************************************************/

void tmsOutNode(TMS_NODE_PTR node)
{ 
  CELL_PTR cell;
  
  cell = (CELL_PTR)node->cell;
  if (cell) {
    if (cell->type == ExclusiveCell)
      uncacheValueForExclusiveCell(cell, node);
    else if (cell->type == ConstrainedCell)
      uncacheValueForConstrainedCell(cell, node);
    runOutRules(node, cell);
  }
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR findValueInNodes(value, contextualValues, 
 *                                         sameValueFN)
 * 
 * DESCRIPTION:
 * Finds the tms_node in the list contextualValues which has datum 
 * equal to value. Equality is determined from the cell's sameValueFN.
 *
 * INPUTS:
 * char *value; 
 * LIST_PTR contextualValues; 
 * SAME_VALUE_FN sameValueFN;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

static TMS_NODE_PTR findValueInNodes(const void *value, 
				     LIST_PTR contextualValues,
				     SAME_VALUE_FN sameValueFN)
{
  TMS_NODE_PTR node;
  LIST_ELEM_PTR tmp;
  
  if (!contextualValues)
    return NULL;
  else {
    tmp = contextualValues->first;
    while (tmp) {
      node = (TMS_NODE_PTR)tmp->item;
      if (sameValueFN(value, node->datum))
	return node;
      else
	tmp = tmp->next;
    }
    return NULL;
  }
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR addValueNode(cell, value)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char *value;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

TMS_NODE_PTR addValueNode(CELL_PTR cell, const void *value)
{ 
  TMS_NODE_PTR  contextualValue;
  
  contextualValue = createContextualValue(cell, value);
  listInsertItemFirst((char *)contextualValue, cell->contextualValues);
  fireCellRules(contextualValue, cell);
  
  return contextualValue;
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR getOrAddValueNode(cell, value, justification)
 *           TMS_NODE_PTR getOrAddValueNode1(cell, value, justification)
 *
 * DESCRIPTION:
 * If the value exists -- add the justification (if any) and return the value.
 * If the value does not exist -- create a TMS_node, push the node on to the 
 * list of values and run the rules on the tms_node.
 *
 * Returns the tms_node found or added.
 *
 * getOrAddNodeValue1: same as getOrAddNodeValue, except it takes a 
 * justification. Justifies the TMS node with the "justification".
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char *value; 
 * JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

static TMS_NODE_PTR getOrAddValueNode1(CELL_PTR cell, const void *value, 
				       JUSTIFICATION_PTR justification)
{ 
  TMS_NODE_PTR  contextualValue;
  
  contextualValue = findValueInNodes(value, cell->contextualValues,
				     cell->sameValueFN);
  if (!contextualValue)
    contextualValue = addValueNode(cell, value);
  
  if (justification)
    tmsJustifyNode(contextualValue, justification);
  
  return contextualValue;
}

TMS_NODE_PTR getOrAddValueNode(CELL_PTR cell, const void *value)
{
  return getOrAddValueNode1(cell, value, (JUSTIFICATION_PTR)NULL);
}


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR getValueNode(cell, value)
 * 
 * DESCRIPTION: Get the tms node for this value, or NULL if none exists.
 *
 * INPUTS:
 * CELL_PTR cell;
 * char *value;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

TMS_NODE_PTR getValueNode(CELL_PTR cell, const void *value)
{ 
  return findValueInNodes(value, cell->contextualValues, 
			  cell->sameValueFN);
}


/*****************************************************************************
 *
 * FUNCTION: const char *currentValue(cell) 
 * 
 * DESCRIPTION:
 * Returns the value that is currently IN.
 * For regular cells, an error occurs if more than one value is IN.
 *
 * INPUTS: CELL_PTR cell;
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

#if 0
static const char *currentValue(CELL_PTR cell)
{ 
  const char *value;
  TMS_NODE_PTR node;
  
  if (IS_EXCLUSIVE_CELL(cell)) {
    value = EXCLUSIVE_CELL_CURRENT_VALUE(cell);
  }
  else if (IS_CONSTRAINED_CELL(cell)) {
    value = CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell);
  }
  else { 
    value = NULL;
    
    node = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (node) {
      if (IS_IN(node)) {
	if (value) 
	  perror("Current_Value: cell has more than one value");
	else value = node->datum;
      }
      node = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
  }
  return value;
}
#endif


/*****************************************************************************
 *
 * FUNCTION: TMS_NODE_PTR currentValueSupporter(cell) 
 * 
 * DESCRIPTION:
 * Returns the tms_node of the currently IN (or most_constrained) value.
 * For regular cells, an error occurs if more than one value is IN.
 *
 * INPUTS: CELL_PTR cell;
 *
 * OUTPUTS: TMS_NODE_PTR
 *
 *****************************************************************************/

TMS_NODE_PTR currentValueSupporter(CELL_PTR cell)
{ 
  const char *currentValue;
  TMS_NODE_PTR node, currentNode;
  
  if (cell->type == RegularCell) {
    currentNode = NULL;
    
    node = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (node) {
      if (IS_IN(node)) {
	if (currentNode) 
	  perror("Current_Value_Supporter: cell has more than one IN node");
	else 
	  currentNode = node;
      }
      node = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
    return currentNode;
  }
  else {
    currentValue = ((IS_EXCLUSIVE_CELL(cell)) ?
		    EXCLUSIVE_CELL_CURRENT_VALUE(cell) :
		    CONSTRAINED_CELL_MOST_CONSTRAINED_VALUE(cell));
    if (currentValue) 
      return getValueNode(cell, currentValue);
    else
      return NULL;
  }
}


/*****************************************************************************
 *
 * FUNCTION: LIST_PTR getCurrentValues(cell) 
 * 
 * DESCRIPTION:
 * Returns the list of values which are currently IN for the CELL.
 *
 * INPUTS: CELL_PTR cell;
 *
 * OUTPUTS: LIST_PTR
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
LIST_PTR getCurrentValues(cell) 
     CELL_PTR cell;
{ 
  char *theValue;
  LIST_PTR theValues;
  TMS_NODE_PTR contextualValue;
  
  if (cell->type == RegularCell) {
    theValues = NULL;
    
    contextualValue = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (contextualValue) {
      if (IS_IN(contextualValue))
	listInsertItemFirst((char *)contextualValue->datum, theValues);
      contextualValue = (TMS_NODE_PTR)listNext(cell->contextualValues);
    }
  }
  else {
    theValue = currentValue(cell);
    if (theValue) 
      theValues = listMake1((char *)theValue);
  }
  
  return theValues;
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void addAssumption(assumption) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR assumption;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void addAssumption(assumption) 
     TMS_NODE_PTR assumption;
{
  if (GET_S_GLOBAL(THE_CURRENT_CONTEXT))
    addToContext(assumption, GET_S_GLOBAL(THE_CURRENT_CONTEXT));
  else 
    perror("No context is currently active");
  runRules(-1);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void removeAssumption(assumption) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: TMS_NODE_PTR assumption;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void removeAssumption(assumption) 
     TMS_NODE_PTR assumption;
{ 
  if (GET_S_GLOBAL(THE_CURRENT_CONTEXT))
    listDeleteItem((char *)assumption, 
		   GET_S_GLOBAL(THE_CURRENT_CONTEXT)->assumptions);
  
  tmsUnassumeNode(assumption);
  runRules(-1);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void assume(cell, value)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell;
 * char *value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void assume(cell, value)
     CELL_PTR cell;
     char *value;
{ 
  TMS_NODE_PTR assumption;
  
  assumption = getOrAddValueNode(cell, value);
  
  if (!(IS_ASSUMED(assumption))) 
    addAssumption(assumption);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void unassume(cell, value)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char *value;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void unassume(cell, value)
     CELL_PTR cell; 
     char *value;
{ 
  TMS_NODE_PTR assumption;
  
  assumption = getOrAddValueNode(cell, value, NULL);
  if (IS_ASSUMED(assumption)) 
    removeAssumption(assumption);
  else 
    tcaModWarning("%s is not an assumption\n", assumption->datum);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void assertPremise(cell, value, informant)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char *value; 
 * char *informant;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void assertPremise(cell, value, informant)
     CELL_PTR cell; 
     char *value; 
     char *informant;
{ 
  TMS_NODE_PTR premise;
  
  premise = getOrAddValueNode(cell, value, NULL);
  
  if (!(tmsIsPremise(premise))) {
    tmsJustifyNode(premise, tmsCreateJustification(informant, (LIST_PTR)NULL));
    runRules(-1);
  }
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void unassertPremise(cell, value, informant)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell; 
 * char  *value; 
 * char *informant;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/
/* Currently not used (RGS: 11/11/92) */
#if 0
void unassertPremise(cell, value, informant)
     CELL_PTR cell; 
     char  *value; 
     char *informant;
{ 
  TMS_NODE_PTR premise;
  
  premise = getOrAddValueNode(cell, value, NULL);
  
  if (tmsIsPremise(premise)) {
    tmsUnassertNode(premise, informant);
    runRules(-1);
  }
  else 
    tcaModWarning("%s is not a premise\n", premise->datum);
}
#endif

/*****************************************************************************
 *
 * FUNCTION: void justify(cell, value, justification)
 * 
 * DESCRIPTION:
 *
 * INPUTS:
 * CELL_PTR cell;
 * char *value; 
 * JUSTIFICATION_PTR justification;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void justify(CELL_PTR cell, const void *value, JUSTIFICATION_PTR justification)
{ 
  (void)getOrAddValueNode1(cell, value, justification);
}


/*****************************************************************************
 *
 * FUNCTION: void describeCell(cell) 
 * 
 * DESCRIPTION:
 *
 * INPUTS: CELL_PTR cell;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

void describeCell(CELL_PTR cell)
{
  TMS_NODE_PTR cv;
  
  tcaModWarning("cell %s ", cell->datum);
  if (cell->contextualValues) {
    tcaModWarning("has the value");
    if (listLength(cell->contextualValues) > 1) 
      tcaModWarning("s");
    
    cv = (TMS_NODE_PTR)listFirst(cell->contextualValues);
    while (cv) {
      tcaModWarning("\n     ");
      sayNodeValue(cv);
      tcaModWarning("(%s)", (IS_IN(cv)) ? "IN" : "OUT");
      cv = (TMS_NODE_PTR)listNext(cell->contextualValues);      
    }
  }
  else
    tcaModWarning("currently has no values\n");
}


/*********************
  NEEDED ANYWHERE???
  void value_Support (cell, value)
  CELL_PTR cell; value_ptr value;
  {
  (let ((tms_node (cv_tms_node (get_contextual_value cell value))))
  (cond ((null tms_node) :TERM_VALUE_DOES_NOT_EXIST)
  ((Is_Assumed tms_node) :ASSUMPTION)
  ((support tms_node))
  (t :PREMISE))))
  }
  *****************/

/*****************************************************************************
 *
 * FUNCTION: void printNodeDependencies(node) 
 * 
 * DESCRIPTION: Print out the (acyclic) graph of dependencies supporting 
 *              the node.
 *
 * INPUTS: TMS_NODE_PTR node;
 *
 * OUTPUTS: none.
 *
 *****************************************************************************/

static int sameNode (TMS_NODE_PTR node, DEP_NODE_PTR depNode)
{ 
  return (node == depNode->node);
}

static void printDependencies1 (TMS_NODE_PTR node, int level,
				int32 *nodeNumberPtr, LIST_PTR seenNodesList)
{
  JUSTIFICATION_PTR justification;
  LIST_PTR supportNodes;
  DEP_NODE_PTR seenNode;
  TMS_NODE_PTR supportNode;
  
  justification = node->supportJustification;
  supportNodes = justification->supporters;
  seenNode = (DEP_NODE_PTR)listMemReturnItem((LIST_ITER_FN) sameNode,
					     (char *)node, seenNodesList);
  
  tcaModWarning( "%*s", level, " ");
  tcaModWarning( "(%d) #%d ", level, 
		((seenNode) ? seenNode->number : ++(*nodeNumberPtr)));
  sayNodeValue(node);
  
  if (seenNode) {
    if (listLength(supportNodes) > 0) 
      tcaModWarning( " (support already printed)\n");
  } else {
    seenNode = NEW(DEP_NODE_TYPE);
    seenNode->node = node;
    seenNode->number = *nodeNumberPtr;
    listInsertItemFirst((char *)seenNode, seenNodesList);
    if (listLength(supportNodes) > 0) {
      tcaModWarning( " because\n");
      level++;
      supportNode = (TMS_NODE_PTR)listFirst(supportNodes);
      while (supportNode) {
	printDependencies1(supportNode, level, nodeNumberPtr, seenNodesList);
	supportNode = (TMS_NODE_PTR)listNext(supportNodes);
      }
    } else if (IS_ASSUMED(node)) {
      tcaModWarning( " (assumption).\n");
    } else {
      tcaModWarning( " (premise %s).\n", justification->informant);
    }
  }
}

void printNodeDependencies (TMS_NODE_PTR node)
{
  int nodeNumber;
  LIST_PTR seenNodes;
  
  nodeNumber = 0;
  seenNodes = listCreate();
  printDependencies1(node, 1, &nodeNumber, seenNodes);
}


void printCellDependencies (CELL_PTR cell)
{
  TMS_NODE_PTR node;
  
  node = currentValueSupporter(cell);
  if (node)
    printNodeDependencies(node);
  else
    tcaModWarning( "Cell has no currently believed value\n");
}
