/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 *
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 *
 * MODULE: behaviors
 *
 * FILE: behaviors.c
 *
 * ABSTRACT:
 * 
 * tca behavior level.
 *
 * REVISION HISTORY
 *
 * $Log: behaviors.c,v $
 * Revision 1.54  1996/08/22  16:35:42  rich
 * Check the return code on tcaQueryCentral calls.
 *
 * Revision 1.53  1996/07/25  22:24:03  rich
 * Fixed some memory leaks with handlers and removed some problems where a
 * disconnection caused a cleanup, but a variable would be assumed to still
 * be valid.
 *
 * Revision 1.52  1996/06/25  20:50:00  rich
 * Fixed memory and other problems found with purify.
 *
 * Revision 1.51  1996/05/09  18:30:16  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.50  1996/03/15  21:13:07  reids
 * Added tcaQueryNotify and plugged a memory leak -- ref was not being
 *   freed when inform/broadcast handler completed.
 *
 * Revision 1.49  1996/03/09  06:12:57  rich
 * Fixed problem where lisp could use the wrong byte order if it had to
 * query for a message format.  Also fixed some memory leaks.
 *
 * Revision 1.48  1996/03/05  05:04:11  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.47  1996/03/02  03:21:15  rich
 * Fixed memory leaks found using purify.
 *
 * Revision 1.46  1996/02/21  18:29:58  rich
 * Created single event loop.
 *
 * Revision 1.45  1996/02/12  17:41:49  rich
 * Handle direct connection disconnect/reconnect.
 *
 * Revision 1.44  1996/02/10  16:49:25  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.43  1996/02/06  19:04:05  rich
 * Changes for VXWORKS pipes.  Note: the read and write sockets descriptors
 * can be different.
 *
 * Revision 1.42  1996/01/27  21:52:49  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.41  1995/11/13  21:23:01  rich
 * Fixed tcaWaitForExecutionWithConstraints.
 *
 * Revision 1.40  1995/11/03  03:04:11  rich
 * Changed msgFind to keep if from going into an infinite loop if there is no
 * central connection.  This only happens when an exit procedure that does
 * not exit is registered.  msgFind can now return NULL, so I added some
 * checks for the return value to keep modules from seg-faulting.
 *
 * Revision 1.39  1995/10/29  18:26:23  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.38  1995/10/25  22:47:50  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.37  1995/10/17  17:36:41  reids
 * Added a "language" slot to the HND_TYPE data structure, so that the LISP
 *   TCA version will know what language to decode the data structure into
 *   (extensible for other languages, as well).
 *
 * Revision 1.36  1995/10/07  19:06:56  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.35  1995/08/05  18:13:08  rich
 * Fixed problem with writeNBuffers on partial writes.
 * Added "sharedBuffers" flag to the dataMsg structure, rather than
 * checking to see if the dataStruct pointer and the message data pointer
 * are the same.  This allows central to clear the dataStruc pointer so
 * that messages don't try to access old data structures that might have
 * changed since the  message was created.
 *
 * Revision 1.34  1995/08/05  17:16:11  reids
 * Several important bug fixes:
 *   a) Found a memory leak in the tms (when nodes are unasserted)
 *   b) Fixed a problem with direct connections that would cause TCA to crash
 *      when a command or goal message was sent from an inform or a query.
 *      As part of that fix, all command and goal messages that are sent from
 *      informs or queries are now added to the root node of the task tree.
 *
 * Revision 1.33  1995/07/24  15:39:17  reids
 * Small change to msgFind2 to take out extraneous warning.
 *
 * Revision 1.32  1995/07/19  14:25:45  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.31  1995/07/12  04:53:51  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.30  1995/07/10  16:16:42  rich
 * Interm save.
 *
 * Revision 1.29  1995/06/14  03:21:22  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.28  1995/06/05  23:58:30  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.27  1995/05/31  19:34:56  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.26  1995/04/19  14:27:43  rich
 * Fixed problems with lisp encode/decode functions.
 * Added types int32 and int16 for use where the size of the integer matters.
 *
 * Revision 1.25  1995/04/08  02:06:14  rich
 * Added waitForReplyFrom to be able to block on replies from only one
 * source.  Useful when querying for the msg info information.  Added a
 * tcaQueryCentral that only accepts input from the central server.  Fixed
 * timing problems with direct connections.
 *
 * Revision 1.24  1995/04/07  05:02:42  rich
 * Fixed GNUmakefiles to find the release directory.
 * Cleaned up libc.h file for sgi and vxworks.  Moved all system includes
 * into libc.h
 * Got direct queries to work.
 * Fixed problem in allocating/initializing generic mats.
 * The direct flag (-c) now mostly works.  Connect message has been extended to
 * indicate when direct connections are the default.
 * Problem with failures on sunOS machines.
 * Fixed problem where tcaError would not print out its message if logging had
 * not been initialized.
 * Fixed another memory problem in modVar.c.
 * Fixed problems found in by sgi cc compiler.  Many type problems.
 *
 * Revision 1.23  1995/04/04  19:41:45  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.22  1995/03/28  01:14:13  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.21  1995/03/19  19:39:19  rich
 * Implemented direct connections using tcaDirectResouce call.
 * Also made the basics.h file a module include.
 * Changed class in the interval structure to be interval_class to avoid a
 * conflict with C++.
 *
 * Revision 1.20  1995/01/25  00:00:48  rich
 * Release of tca 7.9.  Mostly speed improvements.
 * The cvs binaries may now be located in /usr/local.
 * Fixed problems with little endian translation.
 *
 * Revision 1.19  1995/01/18  22:39:35  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.18  1994/11/02  21:33:55  rich
 * Now works for linux machines (i486).
 * Got afs to work on alpha (and hopefully other vendor OS's)
 * Added generic Makefile.
 * Made libc.h and tcaMatrix.h module includes.
 * Reduced the size of libc.h by using more system includes.
 *
 * Revision 1.17  1994/10/25  17:09:34  reids
 * Changed the logging functions to accept variable number of arguments.
 *
 * Revision 1.16  1994/07/05  16:05:34  reids
 * Fixed a bug in which exception messages were incorrectly related to task
 * tree nodes if distributed responses was enabled.
 *
 * Revision 1.15  1994/05/17  23:15:06  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.14  1994/05/05  00:45:54  rich
 * Added a gmake makefile GNUmakefile so that the system can be easily
 * compiled on different machines.
 * Can now create the targets: tarfile and ftp for creating versions for
 * export.
 *
 * Fixed a number of places were tcaExitHnd was not expected to return.
 * Set the tcaSeverGlobal to 0 when the socket is closed.
 *
 * Revision 1.13  1994/04/28  16:15:15  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.12  1994/04/16  19:41:32  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.11  1993/12/14  17:32:42  rich
 * Changed getMGlobal to GET_M_GLOBAL and changed getSGlobal to
 * GET_S_GLOBAL to conform to Chris' software standards.
 *
 * Patched problem with connecting between machines with different byte
 * orders.  The real fix requires changing the way formats are stored.
 * Searching for structural similar formats does not guarantee that you
 * find the right format.
 *
 * Revision 1.10  1993/11/21  20:17:06  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.9  1993/10/21  16:13:27  rich
 * Fixed compiler warnings.
 *
 * Revision 1.8  1993/10/20  20:28:49  rich
 * Fixed a bug in findMsg2 where it was losing a self registered message
 * when it did the query to find the handler.
 *
 * Revision 1.7  1993/10/20  19:00:21  rich
 * Fixed bug with self registed messages in the lisp version.
 * Added new routine : tcaGetServerGlobal to get the server socket.
 * Fixed some bad global references for the lisp version.
 * Updated some prototypes.
 *
 * Revision 1.6  1993/10/06  18:07:16  reids
 * Fixed two bugs: tcaTplConstrain was not externed correctly and
 * -D option was not being incorporated correctly in compiling
 *
 * Revision 1.5  1993/08/30  21:52:59  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.4  1993/08/27  07:14:02  fedor
 * First Pass at V7 and V6+VXWORKS merge
 *
 * Revision 1.3  1993/06/13  23:27:58  rich
 * Made changes for lisp needed for vx works style global variables.
 * Fixed some random compiler warnings.
 * Moved test routines to test directory.
 *
 * Revision 1.2  1993/05/26  23:16:43  rich
 * Fixed up the comments at the top of the file.
 *
 * Revision 1.1.1.1  1993/05/20  05:45:52  rich
 * Importing tca version 8
 *
 * Revision 7.1  1993/05/20  00:29:07  rich
 * RTG - initial checkin of Chris Fedor's version 8 of tca
 *
 * Revision 1.3  1993/05/19  17:23:06  fedor
 * Added Logging.
 *
 * 27-Oct-92 Richard Goodwin, School of Computer Science, CMU
 * Changed printf to fprintf(stderr... for warning messages.
 *
 *  2-Jul-91 Reid Simmons, School of Computer Science, CMU
 * Hacked memory management -- closed some holes.
 *
 * 27-May-91 Christopher Fedor, School of Computer Science, CMU
 * Changed msgFind to pass back hndName to cache msgInfo from the msgInfoHnd
 * instead of from deliverDispatch to avoid timing errors.
 *
 * 30-Jan-91 Christopher Fedor, School of Computer Science, CMU
 * Added fflush(stdout) to printf for module code calls from lisp
 *
 *  3-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Removed logging and changed call from tcaError to tcaModError.
 *
 *  3-Oct-90 Christopher Fedor, School of Computer Science, CMU
 * Moved module monitor code from behaviors.c to monModule.c
 *
 *  3-Sep-90 Christopher Fedor, School of Computer Science, CMU
 * Replaced msgQuery and FindMessage with msgFind.
 * Moved msg.c comments to behaviors and removed msg.c
 * Moved msgQeury and FindMessage to behaviors.c. Updated message
 * lookup to handle HandlerRegisteredClass and multiple messages for
 * a single handler.
 *
 * 16-Aug-90 Christopher Fedor, School of Computer Science, CMU
 * moved Message_Class_Name from com.c to behaviors.c
 *
 *  7-Apr-90 Christopher Fedor, School of Computer Science, CMU
 * Revised to Software Standards.
 *
 *  1-Dec-89 Long-Ji Lin, School of Computer Science, CMU
 * Made variable "responseIssuedGlobal" global.
 *
 * 27-Nov-89 Reid Simmons, School of Computer Science, CMU
 * Added non-blocking query pairs -- tcaQuerySend and tcaQueryReceive.
 *
 * 17-aug-89 Reid Simmons, School of Computer Science, CMU
 * Moved error message from FindMessage to MsgQuery, so that MsgQuery can
 * be called to check whether a message has been registered.
 * Added exception handlers.
 *
 * 14-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Moved "CreateReference" to tasktree.c
 *
 *  2-Aug-89 Reid Simmons, School of Computer Science, CMU
 * Added checks to ensure that routines are called with the 
 * correct message class.
 *
 * 28-Jun-89 Reid Simmons, School of Computer Science, CMU
 * Added check to see if response (Reply, Success, Failure) is sent by
 * handlers.
 *
 *  9-May-89 Christopher Fedor, School of Computer Science, CMU
 * Added change to SelfRegisterMsg Calls. Probably broke modularization.
 *
 *  3-May-89 Christopher Fedor, School of Computer Science, CMU
 * Fixed Failure to send the descriptive string.
 * Changed Failure to be called at top level with NULL as a message ref.
 * Changed route_module.c to behaviors.c
 *
 *  1-May-89 Reid Simmons, School of Computer Science, CMU
 * Added code for sending temporal constraints with goal messages.  
 * Re-modularized.
 *
 * 28-Apr-89 Reid Simmons, School of Computer Science, CMU
 * Added code for sending the parent reference (dispatch) with messages.
 *
 *  7-Mar-89 Christopher Fedor, School of Computer Science, CMU
 * Moved user level routines to route_module.c Created ClassQuery, MsgQuery.
 *
 *  6-Mar-89 Christopher Fedor, School of Computer Science, CMU
 * created behaviors.
 * 
 *  1-Dec-88 Christopher Fedor, School of Computer Science, CMU
 * created message cache.
 *
 * $Revision: 1.54 $
 * $Date: 1996/08/22 16:35:42 $
 * $Author: rich $
 *
 *****************************************************************************/

#include "globalM.h"

/*****************************************************/

#if defined(LISP)
void tcaSetSizeEncodeDecode(int (* bufferSize)(int32 *, CONST_FORMAT_PTR, int),
			    long (* encode)(CONST_FORMAT_PTR, int, BUFFER_PTR),
			    long (* decode)(CONST_FORMAT_PTR, BUFFER_PTR), 
			    int (* lispReply)(char *), 
			    int (* lispExit)(void))
{
  (GET_M_GLOBAL(lispBufferSizeGlobal)) = bufferSize;
  (GET_M_GLOBAL(lispEncodeMsgGlobal)) = encode;
  (GET_M_GLOBAL(lispDecodeMsgGlobal)) = decode;
  (GET_M_GLOBAL(lispReplyMsgGlobal)) = lispReply;
  (GET_M_GLOBAL(lispExitGlobal)) = lispExit;
}
#endif /* LISP */


/******************************************************************************
 *
 * FUNCTION: MSG_PTR msgHndFind(name, hndName)
 *           MSG_PTR msgFind(name)
 *
 * DESCRIPTION: 
 * Performs a message lookup for a message of name. If the message is not
 * found the central server is queried and the information is cached.
 * If the message is still not found or the information returned from the
 * central server is not complete an error is issued. 
 *
 * INPUTS: 
 * char *name, *hndName;
 *
 * OUTPUTS: MSG_PTR
 *
 *****************************************************************************/

static void relinkMessagesAndHandlers(MSG_PTR msg, MSG_PTR oldMsg)
{
  const HND_TYPE *hnd;
  
  hnd = (const HND_TYPE *)listFirst(oldMsg->hndList);
  while (hnd) {
    listDeleteItem(oldMsg, hnd->msgList);
    listInsertItem(msg, hnd->msgList);
    listInsertItem(hnd, msg->hndList);
    hnd = listNext(oldMsg->hndList);
  }
  /* Need to free oldMsg. */
  if (oldMsg->msgData) {
    if (oldMsg->msgData->name)
      tcaFree((void *)(oldMsg->msgData->name));
    if (oldMsg->msgData->msgFormat) {
      tcaFree((void *)(oldMsg->msgData->msgFormat));
      oldMsg->msgData->msgFormat = NULL;
    }
    if (oldMsg->msgData->resFormat) {
      tcaFree((void *)(oldMsg->msgData->resFormat));
      oldMsg->msgData->resFormat = NULL;
    }
    tcaFree((void *)oldMsg->msgData);
  }
  listFree(&(oldMsg->hndList));
  listFree(&(oldMsg->tapList));
  if (oldMsg->msgFormatStr) {
    tcaFree((void *)oldMsg->msgFormatStr);
    oldMsg->msgFormatStr = NULL;
  }
  if(oldMsg->resFormatStr) {
    tcaFree((void *)oldMsg->resFormatStr);
    oldMsg->resFormatStr = NULL;
  }
  listFree(&(oldMsg->excepList));
  tcaFree((void *)(oldMsg));
}

static MSG_PTR msgHndFind(const char *name, const char *hndName)
{
  MSG_PTR msg=NULL, oldMsg=NULL;
  MSG_DATA_PTR msgData=NULL;
  MSG_ASK_TYPE msgAsk;
#if defined(LISP)
  int32 byteOrder;
  ALIGNMENT_TYPE alignment;
#endif
  
  msg = GET_MESSAGE(name);
  
  if (!msg || msg->msgData->msg_class == HandlerRegClass) {
    if (GET_C_GLOBAL(tcaServerReadGlobal) != CENTRAL_SERVER_ID) {
      oldMsg = msg;
      msgAsk.msgName = name;
      msgAsk.hndName = hndName;
      
#if defined(LISP)
      byteOrder = GET_M_GLOBAL(byteOrder);
      alignment = GET_M_GLOBAL(alignment);
#endif
      if (tcaQueryCentral(TCA_MSG_INFO_QUERY, (void *)&msgAsk,
			  (void *)&msgData) != Success) {
	return NULL;
      }
#if defined(LISP)
      GET_M_GLOBAL(byteOrder) = byteOrder;
      GET_M_GLOBAL(alignment) = alignment;
#endif
      if (!msgData) {
	tcaModError("ERROR: msgFind: Query for message %s returned NULL.\n",
		    name);
	return NULL;
      }
      
      if (msgData->msg_class != HandlerRegClass) {
	msg = msgCreate(msgData);
	if (!hndName && msgData->refId <= -1)
	  establishDirect(msg);
	if (oldMsg) relinkMessagesAndHandlers(msg, oldMsg);
	return msg;
      }
    }
    tcaModError("ERROR: msgFind: Message %s used before registered.\n", name);
    return NULL;
  } else if (msg->direct && msg->sd == NO_FD) {
    establishDirect(msg);
  }
  
  return msg;
}

MSG_PTR msgFind(const char *name)
{
  return msgHndFind(name, (const char *)NULL);
}

void tcaPreloadMessage(const char *name)
{
  msgHndFind(name, (const char *)NULL);
}

MSG_PTR msgFind2(const char *name, const char *hndName)
{
  MSG_PTR msg, oldMsg;
  MSG_DATA_PTR msgData;
  MSG_ASK_TYPE msgAsk;
  
  msg = GET_MESSAGE(name);
  
  if (!msg || msg->msgData->msg_class == HandlerRegClass) {
    oldMsg = msg;
    msgAsk.msgName = name;
    msgAsk.hndName = hndName;
    
    if (tcaQueryCentral(TCA_MSG_INFO_QUERY, (void *)&msgAsk,
			(void *)&msgData) != Success) {
      return NULL;
    }
    if (!msgData) {
      return NULL;
    }
    
    if (msgData->msg_class != HandlerRegClass) {
      msg = msgCreate(msgData);
      if (!hndName && msgData->refId <= -1)
	establishDirect(msg);
      if (oldMsg) relinkMessagesAndHandlers(msg, oldMsg);
      return msg;
    }
    return NULL;
  }
  return msg;
}


/******************************************************************************
 *
 * FUNCTION: CONST_FORMAT_PTR fmtFind(const char *name)
 *
 * DESCRIPTION:
 * Return the named formatter format.
 *
 * INPUTS: Named formatter name.
 *
 * OUTPUTS: The formatter.
 *
 *****************************************************************************/

CONST_FORMAT_PTR fmtFind(const char *name)
{
  NAMED_FORMAT_PTR format = NULL;
  format = (NAMED_FORMAT_PTR) hashTableFind(name, 
					    GET_M_GLOBAL(formatNamesTable));
  if (!format) {
    if (GET_C_GLOBAL(tcaServerReadGlobal) != CENTRAL_SERVER_ID) {
      format = (NAMED_FORMAT_PTR) tcaMalloc(sizeof(NAMED_FORMAT_TYPE));
      bzero((void *)format,sizeof(NAMED_FORMAT_TYPE));
      if (tcaQueryCentral(TCA_NAMED_FORM_QUERY, (void *)&name,
			  (void *)&(format->format)) != Success){
	tcaFree((char *)format);
	return NULL;
      }
      if (!(format->format)) {
	tcaModWarning("Unknown named format %s used.\n", name);
	format->format = NEW_FORMATTER();
	format->format->type = BadFormatFMT;
	format->format->formatter.f = NULL;
      }
    } else {
      tcaModWarning("Unknown named format %s used.\n", name);
      format->format = NEW_FORMATTER();
      format->format->type = BadFormatFMT;
      format->format->formatter.f = NULL;
    }
    hashTableInsert((void *)name, 1+strlen(name), 
		    (void *)format, GET_M_GLOBAL(formatNamesTable));
  }
  return (CONST_FORMAT_PTR) format->format;
}


/******************************************************************************
 *
 * FUNCTION: const char *messageClassName(msg_class)
 *
 * DESCRIPTION:
 * Return the name of the class (a string). Used for display purposes
 *
 * INPUTS: TCA_MSG_CLASS_TYPE msg_class;
 *
 * OUTPUTS: char *
 *
 *****************************************************************************/

const char *messageClassName(TCA_MSG_CLASS_TYPE msg_class)
{ 
  switch(msg_class) {
  case QueryClass: 
    return("Query");
  case GoalClass: 
    return("Goal");
  case CommandClass: 
    return("Command");
  case PointMonitorClass: 
    return("PMonitor");
  case DemonMonitorClass: 
  case PollingMonitorClass: 
    return("IMonitor");
  case InformClass: 
    return("Inform");
  case ExceptionClass: 
    return("Exception");
  case BroadcastClass:
    return("Broadcast");
  case MultiQueryClass:
    return("MQuery");

  case UNKNOWN:
  case HandlerRegClass:
  case ExecHndClass:
  case ReplyClass:
  case SuccessClass:
  case FailureClass:
  case FireDemonClass:
    return ("UNKNOWN CLASS");

#ifndef TEST_CASE_COVERAGE
  default: 
    return ("UNKNOWN CLASS");
#endif
  }
}


/******************************************************************************
 *
 * FUNCTION: void checkMessageClass(msg, class)
 *
 * DESCRIPTION:
 * Check to make sure that functions, such as Query, are called
 * only with messages of that class.
 *
 * INPUTS:
 * MSG_PTR msg;
 * TCA_MSG_CLASS_TYPE msg_class;
 *
 * OUTPUTS: void. Message is logged.
 *
 *****************************************************************************/

void checkMessageClass(MSG_PTR msg, TCA_MSG_CLASS_TYPE msg_class)
{ 
  const char *name, *use=NULL;
  
  if (msg->msgData->msg_class != msg_class) {
    name = messageClassName(msg_class);
    
    switch (msg->msgData->msg_class) {
    case QueryClass: 
      use = "tcaQuery"; 
      break;
    case GoalClass:
      use = "tcaExpandGoal"; 
      break;
    case CommandClass:
      use = "tcaExecuteCommand";
      break;
    case PointMonitorClass:
      use = "tcaPointMonitor";
      break;
    case InformClass: 
      use = "tcaInform"; 
      break;
    case BroadcastClass: 
      use = "tcaBroadcast"; 
      break;
    case MultiQueryClass: 
      use = "tcaMultiQuery"; 
      break;

    case UNKNOWN:
    case HandlerRegClass:
    case ExecHndClass: 
    case ExceptionClass:
    case PollingMonitorClass:
    case DemonMonitorClass:
    case ReplyClass:
    case SuccessClass:
    case FailureClass:
    case FireDemonClass:
#ifndef TEST_CASE_COVERAGE
    default:
#endif
      tcaModWarning("Internal Error:Unhandled default in checkMessageClass %d\n",
		    msg->msgData->msg_class);
      break;
    }
    
    tcaModError("ERROR: %s is not a %s class message.  Use function '%s' instead\n",
		msg->msgData->name, name, use);
  }
}
#ifdef NMP_IPC

/* Determine whether format is "[type: n]", "{int, <type: n>}", 
   or something else */
FORMAT_CLASS_TYPE ipcFormatClassType (CONST_FORMAT_PTR format)
{
  if (format->type == FixedArrayFMT) { 
    return FixedArrayFMT;
  } else if (format->type == StructFMT && format->formatter.a[0].i == 3 &&
	     format->formatter.a[1].f->type == PrimitiveFMT &&
	     format->formatter.a[1].f->formatter.i == INT_FMT &&
	     format->formatter.a[2].f->type == VarArrayFMT) {
    return VarArrayFMT;
  } else {
    return BadFormatFMT;
  }
}

char *ipcData (CONST_FORMAT_PTR formatter, char *data)
{
  char *data1;

  if (!formatter) {
    return NULL;
  } else {
    switch (ipcFormatClassType(formatter)) {
    case FixedArrayFMT: return data;
    case VarArrayFMT: 
      data1 = data;
      data = ((IPC_VAR_DATA_PTR)data1)->content;
      tcaFree(data1);
      return data; 
    default: 
      tcaModError("ipcData: Passed a general formatter\n"); return data;
      return NULL;
    }
  }
}
#endif /* NMP_IPC */

/******************************************************************************
 *
 * FUNCTION: void execHnd(directInfo, dataMsg)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * DIRECT_INFO_PTR directInfo;
 * DATA_MSG_PTR dataMsg;
 *
 * OUTPUTS: void.
 *
 *****************************************************************************/

void execHnd(DIRECT_INFO_PTR directInfo, DATA_MSG_PTR dataMsg)
{ 
  MSG_PTR msg;
  HND_PTR hnd;
  char *data=NULL, *name, **pointerToName;
  TCA_REF_PTR tcaRef;
  TCA_MSG_CLASS_TYPE msg_class;
  CLASS_FORM_PTR classForm;
  int tmpParentRef, tmpResponse, intent;
  int sd = directInfo->readSd;

  /* RTG Don't really know what intent is, but trying to use it. */
  if (dataMsg->intent == QUERY_REPLY_INTENT) {
    tcaModWarning("Warning: Unknown query reply\n");
    return;
  } else {
    intent = (directInfo->intent != 0 ? directInfo->intent : dataMsg->intent);
    hnd = (HND_PTR)idTableItem(ABS(intent), GET_C_GLOBAL(hndIdTable));
  }

  if (dataMsg->classTotal) {
    msg_class = ExecHndClass;
    classForm = GET_CLASS_FORMAT(&msg_class);
    if (!classForm) {
      tcaModError("Internal Error: execHnd, missing ExecHndClass class format.");
      return;
    }
    
    pointerToName = (char **)dataMsgDecodeClass(classForm->format, dataMsg);
    name = (char *)(*pointerToName);
    
    msg = msgHndFind(name, hnd->hndData->hndName);
    classDataFree(msg_class, (char *)pointerToName);
    hnd->msg = msg;
  }
  else
    msg = hnd->msg;
  
  if (!msg) {
    msg = msgHndFind(hnd->hndData->msgName, hnd->hndData->hndName);
    hnd->msg = msg;
    msg->direct = TRUE;
    msg->sd = directInfo->writeSd;
    if (msg->msgData->refId > -1) {
      tcaModError("Internal Error: execHnd: Sanity Check Failed!");
      return;
    }
  }
  
  tcaRef = tcaRefCreate(msg, msg->msgData->name, dataMsg->msgRef);
  
  /* If the handler is *not* in the message list, it was probably deregistered 
     after central sent it the message */
  if (listMemberItem(hnd, msg->hndList)) {
    data = (char *)decodeDataInLanguage(dataMsg, msg->msgData->msgFormat, 
					hnd->hndLanguage);
    dataMsgFree(dataMsg);
    
    tmpParentRef = GET_C_GLOBAL(parentRefGlobal);
    tmpResponse = GET_C_GLOBAL(responseIssuedGlobal);
    GET_C_GLOBAL(parentRefGlobal) = (DIRECT_CONNECTION(directInfo)
				     ? NO_REF : tcaRef->refId);
    GET_C_GLOBAL(responseIssuedGlobal) = FALSE;
    
    /* RTG Need to reset the sd, in case the msg is a query. */
    msg ->sd = directInfo->writeSd;

#ifdef NMP_IPC
    if (hnd->clientData == NO_CLIENT_DATA) {
      (*hnd->hndProc)(tcaRef, data);
    } else {
      data = ipcData(msg->msgData->msgFormat, data);
      (*((TCA_HND_DATA_FN)hnd->hndProc))(tcaRef, data, hnd->clientData);
    }
#else
    (*hnd->hndProc)(tcaRef, data);
#endif

    /* If we have disconnected, then just return.  */
    if (!hashTableFind((void *)&sd, GET_C_GLOBAL(moduleConnectionTable))) {
      return;
    }
    /* Send a "success" reponse to constraint messages -- new for version 7.
       Gets rid of race conditions that existed in old version (reids) */
    /* Only do if the connection is not direct.   */
    if (ONE_WAY_MSG(msg) && !DIRECT_CONNECTION(directInfo)
#ifdef NMP_IPC
	/* In the NMP IPC, you can "respond" to a broadcast -- need
	   to ensure that it doesn't respond twice to the same message */
	&& !tcaRef->responded
#endif
	) {
      (void)sendResponse(tcaRef, (MSG_PTR)NULL, NULL, SuccessClass, NULL,
			 GET_C_GLOBAL(tcaServerWriteGlobal));
      tcaRefFree(tcaRef);

    } else if (!GET_M_GLOBAL(enableDistributedResponses)) {

      if (!GET_C_GLOBAL(responseIssuedGlobal) && RESPONSE_EXPECTED(msg)) {
	tcaModWarning("\nWARNING: No ");
	if (TWO_WAY_MSG(msg)) {
	  tcaModWarning("Reply");
	} else {
	  tcaModWarning("Success/Failure");
	}
	tcaModWarning(" sent for message %s;\nResource will probably remain locked\n",
		      msg->msgData->name);
      }
      tcaRefFree(tcaRef);
    }
    
    GET_C_GLOBAL(parentRefGlobal) = tmpParentRef;
    GET_C_GLOBAL(responseIssuedGlobal) = tmpResponse;
  } else {
    tcaModWarning("WARNING: Message '%s' received but not processed: Handler '%s' not registered\n", 
		  msg->msgData->name, hnd->hndData->hndName);
    switch (msg->msgData->msg_class) {
    case QueryClass:
    case MultiQueryClass:
      tcaNullReply(tcaRef); break;
      
    case InformClass:
    case BroadcastClass:
      tcaSuccess(tcaRef); break;
      
    case UNKNOWN:
    case HandlerRegClass:
    case ExecHndClass:
    case GoalClass:
    case CommandClass:
    case ExceptionClass:
    case PollingMonitorClass:
    case PointMonitorClass:
    case DemonMonitorClass:
    case ReplyClass:
    case SuccessClass:
    case FailureClass:
    case FireDemonClass:
#ifndef TEST_CASE_COVERAGE
    default:
#endif
      tcaFailure(tcaRef, "Deregistered Handler", NULL); break;      
    }
    if (!GET_M_GLOBAL(enableDistributedResponses)) {
      tcaRefFree(tcaRef);
    }
  }
}

/******************************************************************************
 *
 * FUNCTION: BOOLEAN execFdHnd(fd)
 *
 * DESCRIPTION: Find the handler (if any) associated with that file descriptor,
 *              and execute the handler (possibly passing along client data).
 *
 * INPUTS: int fd
 *
 * OUTPUTS: BOOLEAN -- TRUE if a handler was executed for that fd
 *
 *****************************************************************************/

BOOLEAN execFdHnd(int fd)
{
  FD_HND_PTR fdHndData;

  fdHndData = (FD_HND_PTR)hashTableFind((void *)&fd,
					GET_M_GLOBAL(externalFdTable));
  if (fdHndData != NULL && fdHndData->hndProc != NULL) {
    (fdHndData->hndProc)(fd, fdHndData->clientData);
  }
  return (fdHndData != NULL);
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaReply(dispatch, data)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * DISPATCH_PTR dispatch;
 * void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaReply(TCA_REF_PTR ref, const void *data)
{
  TCA_RETURN_VALUE_TYPE status;
  
  GET_C_GLOBAL(responseIssuedGlobal) = TRUE;

  status = sendResponse(ref, (MSG_PTR)NULL, 
			(char *)data, ReplyClass, (char *)NULL,
			(ref->msg->direct ? ref->msg->sd :
			 GET_C_GLOBAL(tcaServerWriteGlobal)));

  if (GET_M_GLOBAL(enableDistributedResponses)) {
    tcaRefFree(ref);
  }
  
  return status;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaNullReply(ref)
 *
 * DESCRIPTION:
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaNullReply(TCA_REF_PTR ref)
{
  return tcaReply(ref, (void *)NULL);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaSuccess(ref)
 *
 * DESCRIPTION:
 *
 * INPUTS: TCA_REF_PTR ref;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaSuccess(TCA_REF_PTR ref)
{
  TCA_RETURN_VALUE_TYPE status;
  
  GET_C_GLOBAL(responseIssuedGlobal) = TRUE;
  
  status = sendResponse(ref, (MSG_PTR)NULL, (char *)NULL, 
			SuccessClass, (char *)NULL,
			GET_C_GLOBAL(tcaServerWriteGlobal));
  
  if (GET_M_GLOBAL(enableDistributedResponses)) {
    tcaRefFree(ref);
  }
  
  return status;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaFailure(ref, description, data)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * char *description;
 * void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaFailure(TCA_REF_PTR ref, const char *description, 
				 const void *data)
{
  MSG_PTR msg;
  int tplConstr, tmpParentRef=0;
  TCA_RETURN_VALUE_TYPE status;
  
  GET_C_GLOBAL(responseIssuedGlobal) = TRUE;
  
  /* If the response is to a message, but not from within the handler,
     need to force the parent reference to be correct */
  if (GET_M_GLOBAL(enableDistributedResponses) && ref) {
    tmpParentRef = GET_C_GLOBAL(parentRefGlobal);    
    GET_C_GLOBAL(parentRefGlobal) = ref->refId;
  }
  
  /* 12-May-91: fedor: this double sending of messages is silly */
  msg = msgFind2(description, (const char *)NULL);
  if (msg) {
    if (msg->msgData->msg_class != ExceptionClass) {
      tcaModError("ERROR: %s is not an exception class message for tcaFailure.\n",
		  msg->msgData->name);
      return WrongMsgClass;
    }
    
    tplConstr = NO_TPLCONSTR;
    (void)sendMessage((TCA_REF_PTR)NULL, msg, (char *)data, 
		      (char *)&tplConstr);
  }
  
  status = sendResponse(ref, (MSG_PTR)NULL, (char *)NULL,
			FailureClass, (char *)&description,
			GET_C_GLOBAL(tcaServerWriteGlobal));
  
  if (GET_M_GLOBAL(enableDistributedResponses) && ref) {
    GET_C_GLOBAL(parentRefGlobal) = tmpParentRef;
    tcaRefFree(ref);
  }
  
  return status;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaQuerySend(name, query, ref)
 *
 * DESCRIPTION:
 * Send a non-blocking query, which doesn't wait for a reply.
 * Must be paired with a corresponding tcaQueryReceive, with the same ref.
 *
 * INPUTS:
 * const char *name;
 * void *query;
 * TCA_REF_PTR *ref;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaQuerySend(const char *name, 
				   void *query, 
				   TCA_REF_PTR *ref)
{
  int32 refId;
  MSG_PTR msg;
  TCA_RETURN_VALUE_TYPE returnValue;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, QueryClass);
  
  refId = nextSendMessageRef();
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, 
			    (char *)query, 
			    (char *)NULL);
  
  *ref = tcaRefCreate(msg, name, refId);
#if defined(LISP)
  GET_M_GLOBAL(lispRefSaveGlobal) = *ref;
#endif
  
  return returnValue;
}

/*****************************************************************
 * Return to LISP the reference created for a tcaQuerySend or tcaMultiQuery.
 * There may be a cleaner way to do this, but this seems the easiest
 * and quickest to implement (reids 11/13/92)
 ****************************************************************/
#if defined(LISP)
TCA_REF_PTR tcaQuerySendRefLISPC (void)
{
  return(GET_M_GLOBAL(lispRefSaveGlobal));
}
#endif


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaQueryReceive(ref, reply)
 *
 * DESCRIPTION:
 * ref is a TCA_REF_PTR created by tcaQuerySend.
 * reply is a pointer to (already allocated) data.
 *
 * Waits for a reply to a tcaQuerySend.  The ref passed must be the same as the
 * one gotten from the tcaQuerySend function, but the send's and receive's do
 * not have to be in the same order. That is, you can send query1, send query2,
 * receive query2, receive query1.  TCA makes sure the right replies get paired
 * with the right receives.
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * void *reply;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaQueryReceive(TCA_REF_PTR ref, void *reply)
{
  TCA_RETURN_VALUE_TYPE returnValue;
  
  returnValue = waitForReply(ref, (char *)reply);
  
  tcaRefFree(ref);
  
  return returnValue;
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaQueryNotify(name, query, 
 *                                                replyHandler, clientData)
 *
 * DESCRIPTION: Send the query and invoke the replyHandler function when
 *              the query is replied to.  The handler function is invoked
 *              with the reply data and the client data.
 *
 * INPUTS:
 * const char *name; The message name
 * void *query;      The query data
 * REPLY_HANDLER_FN replyHandler; The function to invoke on the reply data.
 * void *clientData; Data passed to the handler along with the reply data.
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE queryNotifySend (MSG_PTR msg, const char *name,
				       void *query,
				       REPLY_HANDLER_FN replyHandler, 
				       HND_LANGUAGE_ENUM language,
				       void *clientData)
{
  int32 refId;
  TCA_RETURN_VALUE_TYPE returnValue;
  QUERY_NOTIFICATION_PTR queryNotif;

  refId = nextSendMessageRef();
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, (char *)query, NULL);
  if (returnValue == Success) {
    queryNotif = NEW(QUERY_NOTIFICATION_TYPE);
    queryNotif->ref = tcaRefCreate(msg, name, refId);
    queryNotif->handler = replyHandler;
    queryNotif->language = language;
    queryNotif->clientData = clientData;
    listInsertItem((char *)queryNotif, GET_C_GLOBAL(queryNotificationList));
  }
  return returnValue;
}

TCA_RETURN_VALUE_TYPE _tcaQueryNotify(const char *name, void *query, 
				      REPLY_HANDLER_FN replyHandler, 
				      HND_LANGUAGE_ENUM language,
				      void *clientData)
{
  MSG_PTR msg;
 
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, QueryClass);
  return queryNotifySend(msg, name, query, replyHandler, language, clientData);
}

TCA_RETURN_VALUE_TYPE tcaQueryNotify(const char *name, void *query, 
				     REPLY_HANDLER_FN replyHandler, 
				     void *clientData)
{
  return _tcaQueryNotify(name, query, replyHandler, C_LANGUAGE, clientData);
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaQuery(name, query, reply)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *query, *reply;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES
 * 3-May-89: fedor:
 * ReturnValue always returns Success from sendMessage - otherwise
 * sendMessage will exit. A hint of continued error handling problems.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaQueryFd(const char *name,
				 void *queryData,
				 void *replyData,
				 int fd)
{
  int32 refId;
  MSG_PTR msg;
  TCA_REF_PTR ref;
  TCA_RETURN_VALUE_TYPE returnValue;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, QueryClass);
  
  refId = nextSendMessageRef();
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, 
			    (char *)queryData, (char *)NULL);
  
  ref = tcaRefCreate(msg, name, refId);
  
  if (returnValue == Success) {
    returnValue = waitForReplyFrom(ref, (char *)replyData,
				   FALSE, WAITFOREVER, fd);
  }
  tcaRefFree(ref);
  
  return returnValue;
}

TCA_RETURN_VALUE_TYPE tcaQueryCentral(const char *name,
				      void *queryData,
				      void *replyData)
{
  if (!isValidServerConnection()) return Failure;
  return tcaQueryFd(name,queryData,replyData,
		    GET_C_GLOBAL(tcaServerReadGlobal));
}

TCA_RETURN_VALUE_TYPE tcaQuery(const char *name,
			       void *queryData,
			       void *replyData)
{
  return tcaQueryFd(name,queryData,replyData,NO_FD);
}


/******************************************************************************
 *
 * FUNCTION: int testInconsistent(tplConstraints)
 *
 * DESCRIPTION: Check if all bits of the inconsistent constraint are set.
 *
 * INPUTS: int tplConstraints;
 *
 * OUTPUTS: int
 *
 * NOTES:
 * Should this be moved to tplConstr?
 *
 *****************************************************************************/

int32 testInconsistent(int32 tplConstraints)
{
  int index;
  
  for (index=0; GET_M_GLOBAL(inconsistentConstraintsGlobal)[index] != 0;
       index++)
    if ((GET_M_GLOBAL(inconsistentConstraintsGlobal)[index] & 
	 tplConstraints) == 
	GET_M_GLOBAL(inconsistentConstraintsGlobal)[index]) {
      return TRUE;
    }
  return FALSE;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaExpandGoalWithConstraints(ref, name, 
 *                              data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref;
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExpandGoalWithConstraints(TCA_REF_PTR ref, 
						   const char *name,
						   const void *data,
						   int32 tplConstr)
{ 
  MSG_PTR msg;
  
  if (testInconsistent(tplConstr)) {
    tcaModError("ERROR: Inconsistent temporal constraints for goal %s\n", name);
    return Failure;
  }
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, GoalClass);
  return sendMessage(ref, msg, (char *)data, (char *)&tplConstr);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaExpandGoal(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExpandGoal(const char *name, const void *data)
{ 
  MSG_PTR msg;
  int tplConstr;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, GoalClass);
  
  tplConstr = SEQ_ACH;
  
  return sendMessage((TCA_REF_PTR)NULL, msg, (char *)data, (char *)&tplConstr);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaInform(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaInform(const char *name, const void *data)
{ 
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return Failure;
  checkMessageClass(msg, InformClass);
  
  return sendMessage((TCA_REF_PTR)NULL, msg, (char *)data, (char *)NULL);
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaAddConstraint(name, data)
 *
 * DESCRIPTION: THIS FUNCTION SHOULD EVENTUALLY BE REPLACED, AND "tcaInform"
 *              USED EXCLUSIVELY INSTEAD.
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaAddConstraint(const char *name, const void *data)
{ 
  return tcaInform(name, data);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaExecuteCommand(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExecuteCommand(const char *name, const void *data)
{ 
  MSG_PTR msg;
  BLOCK_COM_TYPE blockCom;
  
  msg = msgFind(name);
  if (msg == NULL)
    return MsgUndefined;
  
  checkMessageClass(msg, CommandClass);
  
  blockCom.waitFlag = 0;
  blockCom.tplConstr = SEQ_ACH;
  
  return sendMessage((TCA_REF_PTR)NULL, msg, (char *)data, (char *)&blockCom);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaExecuteCommandWithConstraints(ref, name, 
 *                                  data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExecuteCommandWithConstraints(TCA_REF_PTR ref, 
						       const char *name,
						       const void *data, 
						       int32 tplConstr)
{ 
  MSG_PTR msg;
  BLOCK_COM_TYPE blockCom;
  
  if (testInconsistent(tplConstr)) {
    tcaModError("ERROR: Inconsistent temporal constraints for command %s\n", name);
    return Failure;
  }
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, CommandClass);
  
  blockCom.waitFlag = 0;
  blockCom.tplConstr = tplConstr;
  return sendMessage(ref, msg, (char *)data, (char *)&blockCom);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaWaitForCommand(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES:
 * 14-Aug-91: fedor: should check return valuse from 
 * sendMessage and waitForReply.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaWaitForCommand(const char *name, const void *data)
{

  return tcaWaitForCommandWithConstraints((TCA_REF_PTR)NULL,name,data,SEQ_ACH);
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaWaitForCommandWithConstraints(ref, name, 
 *                                  data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES:
 * 14-Aug-91: fedor: should check return valuse from 
 * sendMessage and waitForReply.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaWaitForCommandWithConstraints(TCA_REF_PTR ref, 
						       const char *name,
						       const void *data, 
						       int32 tplConstr)
{ 
  int32 refId;
  MSG_PTR msg;
  TCA_REF_PTR tcaRef;
  BLOCK_COM_TYPE blockCom;
  TCA_MSG_CLASS_TYPE replyClass;
  
  if (testInconsistent(tplConstr)) {
    tcaModError("ERROR: Inconsistent temporal constraints for command %s\n", name);
    return Failure;
  }
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, CommandClass);
  
  blockCom.waitFlag = 1;
  blockCom.tplConstr = tplConstr;
  
  refId = nextSendMessageRef();
  if (sendMessage(ref, msg, (char *)data, (char *)&blockCom) != Success){
    return Failure;
  }
  
  tcaRef = tcaRefCreate(msg, name, refId);
  
  (void)waitForReply(tcaRef, (char *)&replyClass);
  
  tcaRefFree(tcaRef);
  
  if (replyClass == SuccessClass)
    return Success;
  else
    return Failure;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaBroadcast(name, data)
 *
 * DESCRIPTION: Broadcast a one-way ("inform"-type) message to *all* modules
 *              that registered a handler for that particular message.
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaBroadcast(const char *name, const void *data)
{ 
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, BroadcastClass);
  
  if ((GET_C_GLOBAL(broadcastMsgs) == NULL) ||
      (strListMemberItem(name, GET_C_GLOBAL(broadcastMsgs))))
    return sendMessage((TCA_REF_PTR)NULL, msg, (char *)data, (char *)NULL);
  else
    return Success;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaMultiQuery(name, query, max, refPtr)
 *
 * DESCRIPTION:
 * Send a "MultiQuery" to all handlers that can handle the message.
 * "max" is the maximum number of responses requested.
 * "refPtr" is set to an identifier to be used with a corresponding
 *  tcaMultiReceive.
 *
 * INPUTS: const char *name;
 *         void *query;
 *         int max;
 *         TCA_REF_PTR *refPtr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTE: "refPtr" is a *pointer* to a TCA_REF_PTR.
 *       Although one can set the maximum number of responses to be received,
 *       it is not guaranteed that that number will be received by
 *       tcaMultiReceive:
 *       the number of responses is MIN("max", number of registered handlers).
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaMultiQuery(const char *name, void *query, int32 max,
				    TCA_REF_PTR *refPtr)
{
  int32 refId;
  MSG_PTR msg;
  MULTI_QUERY_CLASS_TYPE multiQueryClassData;
  TCA_RETURN_VALUE_TYPE returnValue;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, MultiQueryClass);
  
  refId = nextSendMessageRef();
  multiQueryClassData.num = 0;
  multiQueryClassData.max = max;
  returnValue = sendMessage((TCA_REF_PTR)NULL, msg, (char *)query,
			    (char *)&multiQueryClassData);
  
  *refPtr = tcaRefCreate(msg, name, refId);
#if defined(LISP)
  GET_M_GLOBAL(lispRefSaveGlobal) = *refPtr;
#endif
  
  return returnValue;
}

/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaMultiReceive(ref, reply, timeout)
 *
 * DESCRIPTION: ref is a TCA_REF_PTR created by tcaMultiQuery.
 *              reply is a pointer to (already allocated) data.
 *
 * Waits for replies to a tcaMultiQuery.  The ref passed must be the same as
 *    the one gotten from the tcaMultiQuery function.
 * "timeout" is number of seconds to wait for a reply before returning.  If no
 *    message is received in that time, the return value is "TimeOut".
 *  The defined
 *    value WAITFOREVER indicates not to return until the reply is received .
 * To receive all the responses to a multi-query, loop doing a tcaMultiReceive.
 *    When no more responses will be forthcoming, the return value of 
 *    tcaMultiReceive is "NullReply".
 * For example:  
 * while (tcaMultiReceive(mqRef, WAITFOREVER, &reply) != NullReply) {
 *                 -- Process and/or save reply data here --
 *                 }
 *
 * INPUTS: TCA_REF_PTR ref;
 *         void *reply;
 *         long timeout;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *          Can have the values: 
 *          "Success" -- a reply was received
 *          "NullReply" -- indicates that no more replies will be forthcoming.
 *		          This can come about either because the "max" number
 *                         of replies were received (see tcaMultiSend), or all 
 *                         the handlers registered for that message replied.
 *          "TimeOut" -- "timeout" seconds elapsed without a message being 
 *                        received.
 *
 * NOTE: The responses are received in the order that the modules handle 
 *       the query.
 *       If "max" < number registered handlers, then the "max" fastest replies 
 *       will be reported.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaMultiReceive(TCA_REF_PTR ref, void *reply,
				      long timeout)
{
  TCA_RETURN_VALUE_TYPE returnValue;
  
  returnValue = waitForReplyFrom(ref, (char *)reply, FALSE, timeout, NO_FD);
  
  if (returnValue == NullReply) 
    tcaRefFree(ref);
  
  return returnValue;
}


/******************************************************************************
 *
 * FUNCTION: void tcaEnableDistributedResponses()
 *           void tcaDisableDistributedResponses()
 *
 * DESCRIPTION: Enable messages to be replied to (via tcaReply, tcaSuccess,
 *              tcaFailure) by a procedure other than the handler.  In other
 *              words, a handler can return without issuing a response, and
 *              (later) another procedure can asynchronously respond to the 
 *              message (using the TCA_REF_PTR that has been saved somehow by
 *              the original handler).
 *
 *              This capability is useful for modules that must interact with
 *              an asynchronous device: a command or query is passed to the 
 *              device, and the response is handled by a different call-back
 *              procedure.
 *
 *              The "disable" function turns off this capability.
 * INPUTS: none
 *
 * OUTPUTS: none
 *
 * NOTES: If distributed responses are enabled, tcaReply, tcaSuccess and 
 *        tcaFailure all free the TCA_REF_PTR.  One should not attempt to
 *        access the pointer after one of those calls have been made 
 *        (in normal situations, the reference is not freed until after 
 *        the handler returns).
 *
 *****************************************************************************/

void tcaEnableDistributedResponses(void)
{
  GET_M_GLOBAL(enableDistributedResponses) = TRUE;
}

void tcaDisableDistributedResponses(void)
{
  GET_M_GLOBAL(enableDistributedResponses) = FALSE;
}


/******************************************************************************
 *
 * FUNCTION: void tcaWillListen(int listen)
 *
 * DESCRIPTION: Sets flag so TCA does not have to quess if the program will
 * listen for messages.
 *
 * INPUTS:
 * 
 * OUTPUTS: void
 *
 *****************************************************************************/

void tcaWillListen(int listen)
{
  GET_C_GLOBAL(willListen) = listen;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaExecute(ref, name, data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExecute(const char *name,
				 const void *data)
{
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;

  switch (msg->msgData->msg_class) {
  case GoalClass:
    return tcaExpandGoal(name, data);
    break;
  case CommandClass:
    return tcaExecuteCommand(name, data);
    break;
  default:
    checkMessageClass(msg, CommandClass);
    break;
  }
  return WrongMsgClass;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaExecuteWithConstraints(ref, name, 
 *                                  data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaExecuteWithConstraints(TCA_REF_PTR ref, 
						const char *name,
						const void *data, 
						int tplConstr)
{
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;

  switch (msg->msgData->msg_class) {
  case GoalClass:
    return tcaExpandGoalWithConstraints(ref, name, data, tplConstr);
    break;
  case CommandClass:
    return tcaExecuteCommandWithConstraints(ref, name, data, tplConstr);
    break;
  default:
    checkMessageClass(msg, CommandClass);
    break;
  }
  return WrongMsgClass;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaWaitForGoalWithConstraints(ref, name, data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES:
 * 14-Aug-91: fedor: should check return valuse from 
 * sendMessage and waitForReply.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaWaitForGoalWithConstraints(TCA_REF_PTR ref, 
						    const char *name,
						    const void *data, 
						    int tplConstr)
{
  int32 refId;
  MSG_PTR msg;
  TCA_REF_PTR tcaRef;
  BLOCK_COM_TYPE blockCom;
  TCA_MSG_CLASS_TYPE replyClass;
  
  if (testInconsistent(tplConstr)) {
    tcaModError("ERROR: Inconsistent temporal constraints for command %s\n", name);
    return Failure;
  }
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;
  checkMessageClass(msg, GoalClass);
  
  blockCom.waitFlag = 1;
  blockCom.tplConstr = tplConstr;
  
  refId = nextSendMessageRef();
  (void)sendMessage(ref, msg, (char *)data, (char *)&blockCom);
  
  tcaRef = tcaRefCreate(msg, name, refId);
  
  (void)waitForReply(tcaRef, (char *)&replyClass);
  
  tcaRefFree(tcaRef);
  
  if (replyClass == SuccessClass)
    return Success;
  else
    return Failure;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE 
 * tcaWaitForExecutionWithConstraints(ref, name, data, tplConstr)
 *
 * DESCRIPTION:
 *
 * INPUTS:
 * TCA_REF_PTR ref; 
 * const char *name;
 * void *data; 
 * int tplConstr;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES:
 * 14-Aug-91: fedor: should check return valuse from 
 * sendMessage and waitForReply.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaWaitForExecutionWithConstraints(TCA_REF_PTR ref, 
							 const char *name,
							 const void *data, 
							 int tplConstr)
{ 
  MSG_PTR msg;
  
  msg = msgFind(name);
  if (msg == NULL) return MsgUndefined;

  switch (msg->msgData->msg_class) {
  case GoalClass:
    return tcaWaitForGoalWithConstraints(ref, name, data, tplConstr);
    break;
  case CommandClass:
    return tcaWaitForCommandWithConstraints(ref, name, data, tplConstr);
    break;
  default:
    checkMessageClass(msg, CommandClass);
    break;
  }
  return WrongMsgClass;
}


/******************************************************************************
 *
 * FUNCTION: TCA_RETURN_VALUE_TYPE tcaWaitForExecution(name, data)
 *
 * DESCRIPTION:
 *
 * INPUTS: const char *name;
 *         void *data;
 *
 * OUTPUTS: TCA_RETURN_VALUE_TYPE
 *
 * NOTES:
 * 14-Aug-91: fedor: should check return valuse from 
 * sendMessage and waitForReply.
 *
 *****************************************************************************/

TCA_RETURN_VALUE_TYPE tcaWaitForExecution(const char *name,
					  const void *data)
{
  
  return tcaWaitForExecutionWithConstraints((TCA_REF_PTR)NULL,
					    name,
					    data,
					    SEQ_ACH);
}
