/******************************************************************************
 *
 * PROJECT: Carnegie Mellon Planetary Rover Project
 *          Task Control Architecture 
 * 
 * (c) Copyright 1991 Christopher Fedor and Reid Simmons.  All rights reserved.
 * 
 * MODULE: tca
 *
 * FILE: tca.h
 *
 * ABSTRACT:
 * 
 *  Task Control Architecture
 *
 *  Include File
 *
 * $Source: /afs/cs.cmu.edu/project/TCA/Master/tcaV8/src/tca.h,v $ 
 * $Revision: 1.46 $
 * $Date: 1996/06/27 15:40:18 $
 * $Author: rich $
 *
 * REVISION HISTORY:
 *
 * $Log: tca.h,v $
 * Revision 1.46  1996/06/27  15:40:18  rich
 * Added tcaGetAcceptFds.
 *
 * Revision 1.45  1996/05/09  18:31:37  reids
 * Changes to keep TCA consistent with the NASA IPC package.
 * Some bug fixes (mainly to do with freeing formatters).
 *
 * Revision 1.44  1996/03/15  21:28:00  reids
 * Added tcaQueryNotify, tcaAddEventHandler, tcaRemoveEventHandler.
 *   Fixed re-registration bug; Plugged memory leaks; Fixed way task trees
 *   are killed; Added support for "enum" format type.
 *
 * Revision 1.43  1996/03/05  05:04:56  reids
 * Changes (mainly delineated by NMP_IPC conditionals) to support the
 *   New Millennium IPC.
 *
 * Revision 1.42  1996/02/21  18:30:29  rich
 * Created single event loop.
 *
 * Revision 1.41  1996/02/10  16:50:40  rich
 * Fixed header problems and a crash related to direct connections.
 *
 * Revision 1.40  1996/02/01  04:03:40  rich
 * Generalized updateVersion and added recursion.
 *
 * Revision 1.39  1996/01/30  15:05:03  rich
 * Fixed var array index problem.  Index refers to the enclosing structure.
 * Added ability to force 32 bit enums and changed some #defs to enums to
 * ease debugging.  Fixed initialization problems for central.
 *
 * Revision 1.38  1996/01/27  21:54:33  rich
 * Pre-release of 8.4.
 * Added recursive named formatters and "BAD" formats.  Also incorporated
 * Iain's windows changes.
 *
 * Revision 1.37  1996/01/05  16:31:42  rich
 * Added windows NT port.
 *
 * Revision 1.36  1995/10/29  18:27:12  rich
 * Initial creation of 8.3. Added changes made after 8.2 branch was
 * created. These mostly have to do with context switching.
 *
 * Revision 1.35.2.1  1995/10/27  01:48:15  rich
 * Removed unimplemented calls.
 *
 * Revision 1.35  1995/10/25  22:49:03  rich
 * Fixed problems with context switching.  Now the context is a separate
 * data structure accessed from the module data structure, using the
 * currentContext field.  GET_C_GLOBAL is used instead of GET_M_GLOBAL for
 * the context dependent fields.
 *
 * Revision 1.34  1995/10/17  17:36:47  reids
 * Added a "language" slot to the HND_TYPE data structure, so that the LISP
 *   TCA version will know what language to decode the data structure into
 *   (extensible for other languages, as well).
 *
 * Revision 1.33  1995/10/07  19:07:56  rich
 * Pre-alpha release of tca-8.2.
 * Added PROJECT_DIR. Added tcaWillListen.
 * Only transmit broadcast messages when there is a handler to receive them.
 * All system messages now start with "tca_".  Old messages are also supported.
 *
 * Revision 1.32  1995/08/14  21:32:05  rich
 * Got rid of the "sharedBuffers" flag on the dataMessages.  It was not the
 * right solution, and possibly caused a memory leak.
 * Limit pending for TCA_TAPPED_MSG_VAR to 1.
 *
 * Revision 1.31  1995/08/08  05:43:32  rich
 * Removed junk from tca.h file.
 *
 * Revision 1.30  1995/07/26  20:42:41  rich
 * Recognize dump when not compiled with DBMALLOC, remove proc from prototypes.
 *
 * Revision 1.29  1995/07/24  15:56:59  reids
 * Somehow tca.h got all screwed up (confused with tca.lisp)!
 *
 * Revision 1.27  1995/07/19  14:26:50  rich
 * Added display and dump to the central interface.
 * Fixed problem with direct querries not returning to the correct module.
 * Added Argv versions of provides and requires.
 *
 * Revision 1.26  1995/07/12  04:55:34  rich
 * Release of 8.0.
 * Fixed problems with sending between machines of different endien.
 *
 * Revision 1.25  1995/07/10  16:18:53  rich
 * Interm save.
 *
 * Revision 1.24  1995/06/14  03:22:35  rich
 * Added DBMALLOC_DIR.
 * More support for DOS.  Fixed some problems with direct connections.
 *
 * Revision 1.23  1995/06/05  23:59:12  rich
 * Improve support of detecting broken pipes.  Add support for OSF 2.
 * Add return types to the global variable routines.
 *
 * Revision 1.22  1995/05/31  19:37:00  rich
 * Fixed problem with reply data being freed early from replys.
 * Initial work on getting the PC version to work.
 *
 * Revision 1.21  1995/04/21  03:53:35  rich
 * Added central commands to kill the task tree and close a module.
 * Added tcaGetContext and tcaSetContext to support connections to multiple
 * central servers.  tcaConnectModules can be called multiple times.
 * Fixed a bug in the resource limit pending.
 * Created seperate routines to print help and option messages.
 *
 * Revision 1.20  1995/04/04  19:43:08  rich
 * Added sgi support.
 * Split low level com routines out to be used in devUtils.
 * Improved some error messages.
 * Added central switch to default to direct connections.  Does not work yet.
 * Fixed the vectorization code.
 *
 * Revision 1.19  1995/03/28  01:15:07  rich
 * - Added ability to log data with direct connections.  Also fixed some
 * problems with global variables. It now uses broadcasts for watching variables.
 * - Added preliminary memory recovery routines to handle out of memory
 * conditions.  It currently purges items from resource queues.  Needs to
 * be tested.
 * - If the CENTRALHOST environment variable is not set, try the current
 * host.
 * - Fixed a problem with central registered messages that caused the parsed
 * formatters to be lost.
 * - Added const declarations where needed to the prototypes in tca.h.
 * - tcaGetConnections: Get the fd_set.  Needed for direct connections.
 * - Added tcaExecute and tcaExecuteWithConstraints.  Can "execute" a goal
 *   or command.
 * - tcaPreloadMessage: Preload the definition of a message from the
 *   central server.
 *
 * Revision 1.18  1995/03/19  19:39:47  rich
 * Implemented direct connections using tcaDirectResouce call.
 * Also made the basics.h file a module include.
 * Changed class in the interval structure to be interval_class to avoid a
 * conflict with C++.
 *
 * Revision 1.17  1995/03/18  15:11:06  rich
 * Fixed updateVersion script so it can be run from any directory.
 *
 * Revision 1.16  1995/02/06  14:46:13  reids
 * Removed the "Global" suffix from tcaRootNode, tcaServer and tcaDefaultTime
 *
 * Revision 1.15  1995/01/18  22:43:14  rich
 * TCA 7.9: Speed improvements.
 * Use unix sockets for communication on the same machine.
 * Eliminate copying.
 * Optimize loop for arrays, especially simple, primitive arrays.
 * Optimize the buffer size.
 *
 * Revision 1.14  1994/05/25  17:32:41  reids
 * Added utilities to limit the number of pending messages
 *
 * Revision 1.13  1994/05/25  04:58:06  rich
 * Defined macros for registering simple messages and handlers at once.
 * Added function to ignore logging for all messages associated with a
 * global variable.
 * Moved module global variable routines to a new file so they are not
 * included in the .sa library file.  Gets better code sharing and lets you
 * debug these routines.
 * Added code to force the module variables to be re-initialized after the
 * server goes down.
 * tcaClose now will not crash if the server is down and frees some module
 * memory.
 * The command line flag "-u" turns off the simple user interface.
 * Added routines to free hash tables and id tables.
 *
 * Revision 1.12  1994/05/17  23:18:00  rich
 * Added global variables and associated routines.
 * Added some error checking.  The central connection is now set to -1
 * rather than zero to prevent tca messages from being send to stdout.
 * Now compiles on the sgi machines.  Still need to have the endian and
 * alignment figured out automatically.
 *
 * Revision 1.11  1994/05/11  01:57:36  rich
 * Now set an invalid tcaServerGlobal (a socket fd) to -1 rather than 0
 * which is stdout.
 * Added checks to make sure tcaServerGlobal is a valid socket before
 * sending messages or waiting for messages.
 *
 * Revision 1.10  1994/04/28  16:17:39  reids
 * Changes in TCA Version 7.6:
 *  1) New functions: tcaIgnoreLogging and tcaResumeLogging
 *  2) Code for MacIntosh (MPW) version of TCA
 *
 * Revision 1.9  1994/04/25  16:12:07  reids
 * Fixed external declaration of tcaByPassException, which was mis-spelled
 *
 * Revision 1.8  1994/04/16  19:43:25  rich
 * First release of TCA for the DEC alpha.
 * Changes were needed because longs are 64 bits.
 * Fixed alignment assumption in the data message format.
 * Fixed the way offsets are calculated for variable length arrays.  This
 * was a problem even without 64 bit longs and pointers.
 *
 * Added the commit date to the version information printed out with the -v
 * option.
 *
 * Now uses standard defines for byte order
 * (BYTE_ORDER = BIG_ENDIAN, LITTLE_ENDIAN or PDP_ENDIAN)
 *
 * Defined alignment types: ALIGN_INT ALINE_LONGEST and ALIGN_WORD.
 *
 * *** WARNING ***
 * sending longs between alphas and non-alpha machines will probably not work.
 * *** WARNING ***
 *
 * Revision 1.7  1993/11/21  20:19:49  rich
 * Added shared library for sun4c_411 sunos machines.
 * Added install to the makefile.
 * Fixed problems with global variables.
 *
 * Revision 1.3  1993/10/20  18:56:11  rich
 * Got rid of VOID_FN an INT_FN.
 *
 * Revision 1.6  1993/10/06  18:07:24  reids
 * Fixed two bugs: tcaTplConstrain was not externed correctly and
 * -D option was not being incorporated correctly in compiling
 *
 * Revision 1.5  1993/08/30  21:54:48  fedor
 * V7+V6+VXWORKS Everything compiles but there are initialization problems.
 *
 * Revision 1.6  1993/07/08  05:37:45  rich
 * Added function prototypes
 *
 * Revision 1.5  1993/07/06  20:30:05  reids
 * Accidentally deleted prototypes for tcaEnableDistributedResponses and
 * Disable...
 *
 * Revision 1.4  1993/07/05  18:11:30  reids
 * Added tcaDeregisterHandler function
 *
 * Revision 1.2  1993/05/27  22:15:24  rich
 * Added automatic logging.
 *
 * 13-Oct-92  Reid Simmons, School of Computer Science, CMU
 * Changed ConstraintClass to InformClass (more informative name).
 * Added BroadcastClass and MultiQueryClass messages.
 * Added WhenSuccess and WhenFailure message tap conditions
 *
 ****************************************************************************/

/* This flag ensures that multiple occurrences of the include file 
   don't cause problems*/
#ifndef INCtca
#define INCtca

#include <sys/types.h>

/* Support for HandleMessage */
#define WAITFOREVER ((long)-1) /* must be less than 0 */

#ifdef Success
#undef Success
#endif

/* Some parameters need to be of a fixed number of bits. */

#ifndef __TURBOC__
typedef unsigned short u_int16;
typedef short int16;
typedef unsigned int u_int32;
typedef int int32;
#else
typedef unsigned int u_int16;
#define int16 short
typedef unsigned long u_int32;
#define int32 long
#endif

/* Communciation Return Values */
typedef enum {
  Success=0, TimeOut=1, Failure=2, WrongVersion=3, MsgUndefined=4,
  WrongMsgClass=5, NullReply=6
#ifdef FORCE_32BIT_ENUM
    , dummyTcaReturnValue = 0x7FFFFFFF
#endif
} TCA_RETURN_VALUE_TYPE;

/* Low Level Communication Status Values */
typedef enum {
  StatEOF=0, StatError=1, StatOK=2, StatSendEOF=3, StatSendError=4,
  StatRecvEOF=5, StatRecvError=6
#ifdef FORCE_32BIT_ENUM
    , dummyTcaReturnStatus = 0x7FFFFFFF
#endif
} TCA_RETURN_STATUS_TYPE;

/* Class Type */
#ifndef __TURBOC__
typedef enum {
  UNKNOWN=0, HandlerRegClass=1, ExecHndClass=2, QueryClass=3,
  GoalClass=4, CommandClass=5, InformClass=6, ExceptionClass=7,
  PollingMonitorClass=8, PointMonitorClass=9,
  DemonMonitorClass=10, ReplyClass=11, SuccessClass=12,
  FailureClass=13, FireDemonClass=14, BroadcastClass=15,
  MultiQueryClass=16
#ifdef FORCE_32BIT_ENUM
    , dummyTcaMsgClass = 0x7FFFFFFF
#endif
} TCA_MSG_CLASS_TYPE;
#else
typedef int32 TCA_MSG_CLASS_TYPE;
#define UNKNOWN              0
#define HandlerRegClass      1
#define ExecHndClass         2
#define QueryClass           3
#define GoalClass            4
#define CommandClass         5
#define InformClass          6
#define ExceptionClass       7
#define PollingMonitorClass  8
#define PointMonitorClass    9
#define DemonMonitorClass   10
#define ReplyClass          11
#define SuccessClass        12
#define FailureClass        13
#define FireDemonClass      14
#define BroadcastClass      15
#define MultiQueryClass     16
#endif

/* For upward compatibility */
#define ConstraintClass InformClass

/* Reference Status Return Values */
typedef 
enum { UnknownRefStatus=0, InactiveRef=1, PendingRef=2, ActiveRef=3,
	 HandledRef=4, PlannedRef=5, AchievedRef=6, KilledRef=7 
     } TCA_REF_STATUS_TYPE;

typedef struct _TCA_REF *TCA_REF_PTR;

typedef struct _TCA_CONTEXT *TCA_CONTEXT_PTR;

typedef void (* TCA_HND_FN)(TCA_REF_PTR, char *);

typedef void (* TCA_HND_DATA_FN)(TCA_REF_PTR, char *, void *clientData);

typedef struct { 
  int maxFire;     /* maximum times to fire action message;
		      -1 indicates infinite (default is -1) */
  int fireEvery;  /* fire action message every n times condition is satisfied.
		     (default is 1) */
  int duration;    /* maximum time, in seconds, that the monitor can run;
		      -1 indicates infinite time (default is -1) */
  int period;      /* time between polls, in seconds */
  int initialWait; /* if TRUE, don't poll immediately (default is FALSE) */
} INTERVAL_MONITOR_OPTIONS_TYPE, *INTERVAL_MONITOR_OPTIONS_PTR;

/*
 * For representing temporal constraints within modules (central uses
 * a different representation).
 */

#ifndef __TURBOC__
typedef enum {
  NoTime=0, StartPoint=1, EndPoint=2
#ifdef FORCE_32BIT_ENUM
    , dummyTcaPointClass = 0x7FFFFFFF
#endif
} TCA_POINT_CLASS_TYPE;
#else
typedef int32 TCA_POINT_CLASS_TYPE;
#define NoTime 0
#define StartPoint 1
#define EndPoint 2
#endif

#ifndef __TURBOC__
typedef enum {
  NoInterval=0, HandlingInterval=1, AchievementInterval=2, 
  PlanningInterval=3
#ifdef FORCE_32BIT_ENUM
    , dummyTcaIntervalClass = 0x7FFFFFFF
#endif
} TCA_INTERVAL_CLASS_TYPE;
#else
typedef int32 TCA_INTERVAL_CLASS_TYPE;
#define  NoInterval          0
#define  HandlingInterval    1
#define  AchievementInterval 2
#define  PlanningInterval    3
#endif

typedef enum {
  UnknownCondition=0, WhenSent=1, BeforeHandling=2,
  WhileHandling=3, AfterHandled=4, AfterReplied=5,
  AfterSuccess=6, AfterFailure=7, BeforeAchieving=8,
  WhileAchieving=9, AfterAchieved=10, WhenAchieved=11,
  BeforePlanning=12, WhilePlanning=13, AfterPlanned=14,
  WhenPlanned=15,	WhenSuccess=16, WhenFailure=17, WhenKilled=18
#ifdef FORCE_32BIT_ENUM
    , dummyTapCondition = 0x7FFFFFFF
#endif
} TAP_CONDITION_TYPE;

typedef struct {
  TCA_INTERVAL_CLASS_TYPE interval_class;
  int msgRef;
} TCA_INTERVAL_TYPE, *TCA_INTERVAL_PTR;

#define TCA_INTERVAL_FORMAT "{int,int}"

typedef struct { 
  TCA_POINT_CLASS_TYPE point_class;
  TCA_INTERVAL_TYPE interval;
} TCA_TIME_POINT_TYPE, *TCA_TIME_POINT_PTR;

#define TCA_TIME_POINT_FORMAT "{int,TCA_INTERVAL}"

typedef struct _TIMELIST { 
  TCA_TIME_POINT_TYPE timePoint;
  struct _TIMELIST *Next;
} TCA_TIME_LIST_TYPE, *TCA_TIME_LIST_PTR;

#define TCA_TIME_LIST_FORMAT "{TCA_TIME_POINT,*!}"

typedef struct { 
  TCA_TIME_POINT_TYPE timePoint1;
  const char *relationship;
  TCA_TIME_POINT_TYPE timePoint2;
} TCA_TPL_CONSTRAINT_TYPE, *TCA_TPL_CONSTRAINT_PTR;

#define TCA_TPL_CONSTRAINT_FORMAT "{TCA_TIME_POINT,*char,TCA_TIME_POINT}"

typedef struct {
  TCA_REF_PTR ERef, CRef;
  int numOfRetries;		/* number of times of CNode being retried */
} EXC_INFO_TYPE, * EXC_INFO_PTR;

/* The language in which the message handler is written */
typedef enum {
  UNKNOWN_LANGUAGE=0, C_LANGUAGE=1, LISP_LANGUAGE=2 
#ifdef FORCE_32BIT_ENUM
    , dummyHndLanguageEnum = 0x7FFFFFFF
#endif
} HND_LANGUAGE_ENUM;

/***************************************************************************/

typedef enum {
  NO_TPLCONSTR = 0,
  
  /* End(Achievement(Last_Node)) <= Start(Achievement(Node)) */
  SEQ_ACH = 01,

  /* End(Planning(Last_Node)) <= Start(Planning(Node)) */
  SEQ_PLANNING = 02,
  
  /* End(Planning(Node)) <= Start(Achievement(Node)) */
  PLAN_FIRST = 04,

  /* Start(Planning(Node)) = Start(Achievement(Node)) */
  DELAY_PLANNING = 010 
#ifdef FORCE_32BIT_ENUM
    , dummyTplconstrEnum = 0x7FFFFFFF
#endif
} TPLCONSTR_ENUM;

#define tcaRegisterGoalMessage(name, format) \
tcaRegisterMessage(name, GoalClass, format, (const char *)NULL)

#define tcaRegisterCommandMessage(name, format) \
tcaRegisterMessage(name, CommandClass, format, (const char *)NULL)

#define tcaRegisterInformMessage(name, format) \
tcaRegisterMessage(name, InformClass, format, (const char *)NULL)

/* Kept for upward compatibility -- use "tcaRegisterInformMessage" */
#define tcaRegisterConstraintMessage(name, format) \
tcaRegisterInformMessage(name, format)

#define tcaRegisterQueryMessage(name, msgFormat, resFormat) \
tcaRegisterMessage(name, QueryClass, msgFormat, resFormat)

#define tcaRegisterExceptionMessage(name, format) \
tcaRegisterMessage(name, ExceptionClass, format, (const char *)NULL)

#define tcaRegisterBroadcastMessage(name, format) \
tcaRegisterMessage(name, BroadcastClass, format, (const char *)NULL)

#define tcaRegisterMultiQueryMessage(name, msgFormat, resFormat) \
tcaRegisterMessage(name, MultiQueryClass, msgFormat, resFormat)

#define VALID_TIME(timePoint) ((timePoint).point_class != NoTime)

#define TPL_ORDER(interval1, interval2) \
(tcaTplConstrain(tcaEndOf(interval1), "<=", tcaStartOf(interval2)))

#define tcaRaiseException(a,b,c)  tcaFailure(a,b,c)

/***************************************************************************/

/*  EXTERN_FUNCTION( rtn name, (arg types) );
 *	Macro to make external function declarations portable between 
 *      the major C dialects: C, ANSI C, and C++.
 */

#undef EXTERN_FUNCTION

#if defined(__cplusplus) /* C++ */
#define EXTERN_FUNCTION( rtn, args ) extern "C" { rtn args; }
#elif defined(__STDC__) || defined(__GNUC__) /* ANSI C */
#define EXTERN_FUNCTION( rtn, args ) extern rtn args
#elif defined(__TURBOC__) /* ANSI C */
#define EXTERN_FUNCTION( rtn, args ) extern rtn args
#else /* "Regular" C */
#define EXTERN_FUNCTION( rtn, args ) extern rtn()
#endif

/*********************
 * Library Functions *
 *********************/

/* 
 * Functions for starting TCA and setting parameters 
 */

EXTERN_FUNCTION( const char *tcaServerMachine,
		(void));

EXTERN_FUNCTION( void tcaConnectModule,
		(const char *moduleName, const char *serverHost));

EXTERN_FUNCTION( void tcaModuleProvides,
		(const char *resourceName, ...));

EXTERN_FUNCTION( void tcaModuleProvidesArgv,
		(const char **resourceNames));

EXTERN_FUNCTION( void tcaModuleRequires,
		(const char *resourceName, ...));

EXTERN_FUNCTION( void tcaModuleRequiresArgv,
		(const char **resourceNames));

EXTERN_FUNCTION( char **tcaQueryAvailable,
		(void));

EXTERN_FUNCTION( void tcaEnableDistributedResponses, 
		(void));

EXTERN_FUNCTION( void tcaDisableDistributedResponses,
		(void));

EXTERN_FUNCTION( void tcaRegisterFreeMemHnd,
		(void (*func)(unsigned int), int retry));

#if defined(DBMALLOC)
EXTERN_FUNCTION( void tcaRegisterMallocHnd,
		(void *(*func)(const char*,int, SIZETYPE), int retry));
#elif defined(__sgi)
EXTERN_FUNCTION( void tcaRegisterMallocHnd,
		(void *(*func)(unsigned int), int retry));
#elif defined(__TURBOC__)
     void tcaRegisterMallocHnd(void *(*func)(unsigned long), long retry);
#else
EXTERN_FUNCTION( void tcaRegisterMallocHnd,
		(void *(*func)(unsigned long), int retry));
#endif

EXTERN_FUNCTION( void tcaWillListen,
		(int listen));

EXTERN_FUNCTION( void tcaWaitUntilReady,
		(void));

/* 
 * Functions for registering messages, handlers...
 */

EXTERN_FUNCTION( void tcaRegisterNamedFormatter,
		(const char *formName, const char *formString));

EXTERN_FUNCTION( void tcaRegisterLengthFormatter,
		(const char *formName, int32 length));

EXTERN_FUNCTION( void tcaRegisterMessage,
		(const char *name, TCA_MSG_CLASS_TYPE msg_class, 
		 const char *msgFormat, const char *resFormat));

EXTERN_FUNCTION( void tcaPreloadMessage,
		(const char *name));

EXTERN_FUNCTION( void _tcaRegisterHandlerL,
		(const char *msgName, const char *hndName, 
		 TCA_HND_FN hndProc, HND_LANGUAGE_ENUM hndLanguage));

#define tcaRegisterHandler(msgName, hndName, hndProc) \
_tcaRegisterHandlerL(msgName, hndName, (TCA_HND_FN) hndProc, C_LANGUAGE)
     
EXTERN_FUNCTION( void tcaDeregisterHandler,
		(const char *msgName, const char *hndName));

/* 
 * Functions for getting information about registered messages.
 */

EXTERN_FUNCTION( int tcaMessageRegistered,
		(const char *msgName));

EXTERN_FUNCTION( int tcaMessageHandlerRegistered,
		(const char *msgName));

EXTERN_FUNCTION( char **tcaGetRegisteredMsgs,
		(void));

typedef struct {
  char *name;
  char *msgFormat, *resFormat;
  TCA_MSG_CLASS_TYPE msg_class;
  char *resourceName;
  int numberOfHandlers;
} MSG_INFO_TYPE, *MSG_INFO_PTR;

/* Call this function with a pointer to a valid MSG_INFO_TYPE structure with 
 * the name field filled and it will fill in the rest of the fields.
 */

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaGetMsgInfo,
		(MSG_INFO_PTR info));

/* 
 * define a set of macros that make it easy to register simple 
 * messages and handlers 
 */

#define tcaRegisterGoal(name, format, hndProc) \
{tcaRegisterGoalMessage(name, format); \
   tcaRegisterHandler(name, name, hndProc);}

#define tcaRegisterCommand(name, format, hndProc) \
{tcaRegisterCommandMessage(name, format); \
   tcaRegisterHandler(name, name, hndProc);}

#define tcaRegisterInform(name, format, hndProc) \
{tcaRegisterInformMessage(name, format); \
   tcaRegisterHandler(name, name, hndProc);}

#define tcaRegisterQuery(name, format, reply, hndProc) \
{tcaRegisterQueryMessage(name, format, reply); \
   tcaRegisterHandler(name, name, hndProc);}

#define tcaRegisterException(name, format, hndProc) \
{tcaRegisterExceptionMessage(name, format); \
   tcaRegisterHandler(name, name, hndProc);}

EXTERN_FUNCTION( void tcaRegisterExitProc,
		(void (*)(void)));

EXTERN_FUNCTION( void tcaLimitPendingMessages,
		(const char *msgName, const char *resName, int32 limit));

EXTERN_FUNCTION( void tcaIgnoreLogging, (char *msgName));

EXTERN_FUNCTION( void tcaResumeLogging, (char *msgName));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaCleanUpAfterAchieved,
		(const char *msgName));

/* 
 * Functions for sending/replying to messages 
 */

/* 
 * "tcaInform" and "tcaAddConstraint" are equivalent -- the former name
 * is just more informative of what the function does
 */
EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaInform,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaAddConstraint,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaQuery,
		(const char *name, void *queryData, void *replyData));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaReply,
		(TCA_REF_PTR ref, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaNullReply,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaQuerySend,
		(const char *name, void *query, TCA_REF_PTR *ref));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaQueryReceive,
		(TCA_REF_PTR ref, void *reply));

typedef void (*REPLY_HANDLER_FN)(void *replyData, void *clientData);

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaQueryNotify,
		(const char *name, void *query, REPLY_HANDLER_FN replyHandler,
		 void *clientData));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExpandGoal,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExpandGoalWithConstraints,
		(TCA_REF_PTR ref, const char *name,
		 const void *data, int32 tplConstraints));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExecuteCommand,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExecuteCommandWithConstraints,
		(TCA_REF_PTR ref, const char *name,
		 const void *data, int32 tplConstraints));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaWaitForCommand,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaWaitForCommandWithConstraints,
		(TCA_REF_PTR ref, const char *name,
		 const void *data, int32 tplConstraints));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExecute,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaExecuteWithConstraints,
		(TCA_REF_PTR ref, const char *name,
		 const void *data, int tplConstraints));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaWaitForGoalWithConstraints,
		(TCA_REF_PTR ref, 
		 const char *name,
		 const void *data, 
		 int tplConstr));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaWaitForExecutionWithConstraints,
		(TCA_REF_PTR ref, 
		 const char *name,
		 const void *data, 
		 int tplConstr));
		
EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaWaitForExecution,
		(const char *name, const void *data));

#define tcaExecuteCommandAsync(name, data) \
tcaExecuteCommandWithConstraints(NULL,name,data,NO_TPLCONSTR)

#define tcaExecuteAsync(name, data) \
tcaExecuteWithConstraints(NULL,name,data,NO_TPLCONSTR)

#define tcaMaybeExecute(name, data) \
{ if (tcaMessageHandlerRegistered(name)) tcaExecute(name,data); }

#define tcaMaybeExecuteWithConstraints(ref, name, data, tplConstraints) \
{ if (tcaMessageHandlerRegistered(name))\
    tcaExecuteWithConstraints(ref,name,data,tplConstraints); }

#define tcaMaybeExecuteAsync(name, data) \
{ if (tcaMessageHandlerRegistered(name)) tcaExecuteAsync(name,data); }

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaSuccess,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaFailure,
		(TCA_REF_PTR ref, const char *description, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaBroadcast,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaMultiQuery,
		(const char *name, void *query, int32 max, TCA_REF_PTR *refPtr));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaMultiReceive,
		(TCA_REF_PTR ref, void *reply, long timeout));

/* 
 * Functions for handling messages 
 */

EXTERN_FUNCTION( void tcaModuleListen,
		(void));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaHandleMessage,
		(long seconds));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaHandleFdInput,
		(int fd));

typedef void (* TCA_FD_HND_FN)(int, void *);

EXTERN_FUNCTION( void tcaAddEventHandler,
		(int fd, TCA_FD_HND_FN hnd, void *clientData));

EXTERN_FUNCTION( void tcaRemoveEventHandler,
		(int fd));

/* 
 * Functions for handling message data.
 */

EXTERN_FUNCTION( void tcaFreeData,
		(const char *msgName, void *data));

EXTERN_FUNCTION( void *tcaAllocateReply,
		(const char *msgName));

EXTERN_FUNCTION( void tcaFreeReply,
		(const char *msgName, void *replyData));

/* 
 * Functions for handling temporal relations.
 */

EXTERN_FUNCTION( TCA_TIME_POINT_TYPE tcaStartOf,
		(TCA_INTERVAL_TYPE interval));

EXTERN_FUNCTION( TCA_TIME_POINT_TYPE tcaEndOf,
		(TCA_INTERVAL_TYPE interval));

EXTERN_FUNCTION( TCA_INTERVAL_TYPE tcaHandlingOf,
		(TCA_REF_PTR msgRef));

EXTERN_FUNCTION( TCA_INTERVAL_TYPE tcaAchievementOf,
		(TCA_REF_PTR msgRef));

EXTERN_FUNCTION( TCA_INTERVAL_TYPE tcaPlanningOf,
		(TCA_REF_PTR msgRef));

EXTERN_FUNCTION( void tcaTplConstrain,
		(TCA_TIME_POINT_TYPE timePoint1, 
		 const char *relationship,
		 TCA_TIME_POINT_TYPE timePoint2));

/* 
 * Functions for handling message taps.
 */

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaTapMessage,
		(TAP_CONDITION_TYPE condition,
		 const char *tappedMsg, const char *listenMsg));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaTapReference,
		(TAP_CONDITION_TYPE condition,
		 TCA_REF_PTR tappedRef,
		 const char *listenMsg));

EXTERN_FUNCTION( TCA_REF_PTR tcaFindTappedReference,
		(TCA_REF_PTR listeningRef));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaRemoveTap,
		(TAP_CONDITION_TYPE condition,
		 const char *tappedMsg, const char *listenMsg));

/* 
 * Functions for handling resources.
 */

EXTERN_FUNCTION( void tcaRegisterResource,
		(const char *resName, int32 capacity));

EXTERN_FUNCTION( void tcaDirectResource,
		(const char *resName));

EXTERN_FUNCTION( void tcaAddHndToResource,
		(const char *hndName, const char *resName));

EXTERN_FUNCTION( void tcaLimitPendingResource,
		(const char *resName, int32 limit));

EXTERN_FUNCTION( TCA_REF_PTR tcaReserveResource,
		(const char *resName));

EXTERN_FUNCTION( TCA_REF_PTR tcaReserveModResource,
		(const char *modName, const char *resName));

EXTERN_FUNCTION( void tcaCancelReservation,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( TCA_REF_PTR tcaLockResource,
		(const char *resName));

EXTERN_FUNCTION( TCA_REF_PTR tcaLockModResource,
		(const char *modName, const char *resName));

EXTERN_FUNCTION( void tcaUnlockResource,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaAddHndToResource,
		(const char *hndName, const char *resName));

/* 
 * Functions for handling variables.
 */

/* Experimental version of global variables */
EXTERN_FUNCTION( void tcaRegisterVar,
		(const char*, const char*));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE _tcaSetVar,
		(const char*, const void*));

#define tcaSetVar(msgName, value) \
_tcaSetVar(msgName, (const void*) value)

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE _tcaGetVar,
		(const char*, void*));

#define tcaGetVar(msgName, value) \
_tcaGetVar(msgName, (void*) value)

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE _tcaGetSetVar,
		(const char*, void*, void*));

#define tcaGetSetVar(msgName, setValue, getValue) \
_tcaGetSetVar(msgName, (void*) setValue, (void*) getValue)

EXTERN_FUNCTION( void _tcaWatchVar,
		(const char *name, const char *format, TCA_HND_FN watchFn));

#define tcaWatchVar(varName, format, handler) \
_tcaWatchVar(varName, format, (TCA_HND_FN) handler)

EXTERN_FUNCTION( void tcaUnwatchVar,
		(const char *name));

EXTERN_FUNCTION( void tcaIgnoreVarLogging, (const char *varName));

EXTERN_FUNCTION( void tcaFreeVar, (const char *varName, void *varData));

/* 
 * Functions for handling the task tree.
 */

EXTERN_FUNCTION( TCA_REF_PTR tcaRootNode,
		(void));

/* For upward compatibility */
#define tcaRootNodeGlobal tcaRootNode

EXTERN_FUNCTION( int tcaIsSameReference,
		(TCA_REF_PTR ref1, TCA_REF_PTR ref2));

EXTERN_FUNCTION( int tcaReferenceId,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( const char *tcaReferenceName,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void *tcaReferenceData,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaReferenceRelease,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( int tcaReferenceStatus,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaRefFree,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( TCA_REF_PTR tcaAddChildReference,
		(TCA_REF_PTR parent, const char *name));

EXTERN_FUNCTION( TCA_REF_PTR tcaCreateReference,
		(const char *name));

EXTERN_FUNCTION( TCA_REF_PTR tcaFindParentReference,
		(TCA_REF_PTR child));

EXTERN_FUNCTION( TCA_REF_PTR tcaFindFirstChild,
		(TCA_REF_PTR parent));

EXTERN_FUNCTION( TCA_REF_PTR tcaFindLastChild,
		(TCA_REF_PTR parent));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindNextChild,
		(TCA_REF_PTR child));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindPreviousChild,
		(TCA_REF_PTR child));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindFailedReference,
		(TCA_REF_PTR exception));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindTopLevelReference,
		(TCA_REF_PTR child));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindAnteReferenceByName,
		(TCA_REF_PTR child, const char *antecedentName));
EXTERN_FUNCTION( TCA_REF_PTR tcaFindChildByName,
		(TCA_REF_PTR parent, const char *childName));

EXTERN_FUNCTION( void tcaKillTaskTree,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaKillSubTaskTree,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaDisplayTaskTree,
		(TCA_REF_PTR ref));

/* 
 * Functions for handling monitors.
 */

EXTERN_FUNCTION( void tcaRegisterPointMonitor,
		(const char *monitor, const char *condition, const char *act));

EXTERN_FUNCTION( void tcaRegisterDemonMonitor,
		(const char *monitor, const char *setUp, 
		 const char *act, const char *cancel));

EXTERN_FUNCTION( void tcaRegisterPollingMonitor,
		(const char *monitor, const char *condition, const char *act));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaPointMonitor,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaPointMonitorWithConstraints,
		(TCA_REF_PTR ref, const char *name,
		 const void *data, int32 tplConstraint));

EXTERN_FUNCTION( INTERVAL_MONITOR_OPTIONS_PTR tcaCreateMonitorOptions,
		(void));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaIntervalMonitor,
		(const char *name, const void *data));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaIntervalMonitorWithConstraints,
		(TCA_REF_PTR ref, const char *name, const void *data,
		 TCA_TIME_POINT_TYPE start,
		 TCA_TIME_POINT_TYPE expire,
		 INTERVAL_MONITOR_OPTIONS_PTR ops));

EXTERN_FUNCTION( TCA_RETURN_VALUE_TYPE tcaFireDemon,
		(const char *name, int32 idValue, const void *data));

/* 
 * Functions for getting information about TCA.
 */

/* Get the version of TCA. */
EXTERN_FUNCTION(int32 tcaGetVersionMajor, (void));

EXTERN_FUNCTION(int32 tcaGetVersionMinor, (void));

EXTERN_FUNCTION( TCA_TIME_POINT_TYPE tcaDefaultTime,
		(void));

/* For upward compatibility */
#define tcaDefaultTimeGlobal tcaDefaultTime

EXTERN_FUNCTION( int tcaGetServer,
		(void));

/* For upward compatibility */
#define tcaGetServerGlobal tcaGetServer

EXTERN_FUNCTION( fd_set* tcaGetConnections,
		(void));

EXTERN_FUNCTION( fd_set* tcaGetAcceptFds,
		(void));

EXTERN_FUNCTION( int tcaGetMaxConnection,
		(void));

/*
 * Functions for handling exceptions.
 */

EXTERN_FUNCTION( void tcaRetry,
		(TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaByPassException,
		( TCA_REF_PTR ref));

EXTERN_FUNCTION( void tcaGetExceptionInfo,
		(TCA_REF_PTR ref, EXC_INFO_PTR info));

EXTERN_FUNCTION( void tcaAddExceptionHandler,
		(TCA_REF_PTR ref, const char *hndName));

EXTERN_FUNCTION( void tcaAddExceptionHandlerToMessage,
		(const char *msgName, const char *hndName));

/* 
 * Functions for handling connections to multiple central servers. 
 */

EXTERN_FUNCTION(void tcaSetContext, (TCA_CONTEXT_PTR context));

EXTERN_FUNCTION(TCA_CONTEXT_PTR tcaGetContext, (void));

/* 
 * Functions for shutting down.
 */

EXTERN_FUNCTION( void tcaClose,
		(void));

/* 
 * Don't export the definition of EXTERNAL_FUNCTION, 
 * it may conflict with some open windows code. 
 */
#undef EXTERN_FUNCTION

#if defined(VXWORKS)
void central(char *options);
#endif

#endif /* INCtca */
