/*
  triwg.cxx
  
  Mark Sibenac
  96-6-7
  Field Robotics Center
  Lunar Rover Terrestrial Prototype
  TRIWG control of Ambler Amps.

  Using VADM10 with 64 Pin connector module and interfaced to a DB-15

  Left Command  - AO0, DB1, c9, P17, White
  Right Command - A02, DB13, c10, P19, Brown
  Command Ground- OGND, DB12, a7, P14, Green
  Enable        - R0b, DB7, c2, P3, Black
  Reset         - R1b, DB8, a2, P4, Blue
  Digital Ground- R0c/R1c, DB6, c3/a3, P5/P6, Red
  
  call initTriwg() to initialize things to a known state.
  */

#include <vxWorks.h>
#include <shellLib.h>
#include <stdio.h>
#include <stdlib.h>
#include "triwg.h"
#include "vadm/vadm.h"
#include "joy.h"

int nLeft_command, nRight_command;
int nArray_max;
int nArray_pos;

#define enableAmps() vadm.setRelays(1)
#define disableAmps() vadm.setRelays(0)

void resetAmps (Vadm vadm)
{
  volatile unsigned int n;

  vadm.setRelays(0x02); // close reset relay
  for (n=0; n<65000; n++);
  vadm.setRelays(0x00); // open reset relay
}
  
void clearArray(double *pArray)
{
  int n;

  for (n=0; n < nArray_max; n++)
    pArray[n] = 0.0;
}

void insertArray(double *pArray, double dVal)
{
  if (dVal < 0.0)
    {
      if (dVal >= -2.0)
	dVal = 0.0;
      else
	dVal = (dVal + 2.0) * 1.33333;
    }
  else
    {
      if (dVal <= 2.0)
	dVal = 0.0;
      else
	dVal = (dVal - 2.0) * 1.33333;
    }

  pArray[nArray_pos] = dVal;
}

void getAvg(double *pArray, double &pVal)
{
  int n;
  double dSum;

  dSum = 0.0;
  for (n=0; n<nArray_max; n++)
    dSum += pArray[n];

  pVal = (dSum/nArray_max);
}

// assumes connected to /tyCo/1
void goJoy(void)
{
  Vadm vadm;
  joystruct joy;
  double dLeft, dRight;
  double dLeft_array[30], dRight_array[30];
  int nLeft_trigger, nLast;

  vadm.init(DIFFERENTIAL, IN_10_10, TWOS_COMP);
  disableAmps();
  nLeft_command = 0; nRight_command = 0;
  vadm.setDAC(0,0);
  vadm.setDAC(2,0);

  fd = 0;
  nLeft_trigger = 0; nLast = 0;
  nArray_max = 30;
  nArray_pos = 0;
  clearArray(dLeft_array);
  clearArray(dRight_array);
  joy.left_trigger = 0; joy.right_trigger = 0;
  strcpy(pPort, "/tyCo/1");

  printf("To stop, press ctrl-c or left trigger only ten times in a row.\n");
  printf("To go, press both triggers.\n");
  printf("Okay, play with it...\n\n");

  while(nLeft_trigger < 10)
    {
      read_joystick(&joy);
      if (joy.left_trigger && !joy.right_trigger)
	{
	  if (!nLast)
	    {
	      nLeft_trigger++;
	      nLast = 1;
	      resetAmps(vadm);
	    }
	}
      else 
	{
	  nLast = 0;
	}
      
      if (joy.left_trigger && joy.right_trigger)
	{
	  nLeft_trigger = 0;

	  if ((joy.left_y != 0.0) && ( joy.right_y != 0.0))
	    {	    
	      dLeft = (joy.left_y-50.0)/3.75;
	      dRight = (joy.right_y-50.0)/3.75;
	    }

	  if (++nArray_pos >= nArray_max)
	    nArray_pos = 0;
	  insertArray(dLeft_array, dLeft);
	  insertArray(dRight_array, dRight);
	  
	  getAvg(dLeft_array, dLeft);
	  getAvg(dRight_array, dRight);

	  nLeft_command = (int)(dLeft * 204.8);
	  nRight_command = (int)(dRight * 204.8);
	  
	  if (nLeft_command >= 2047) nLeft_command = 2047;
	  if (nLeft_command <= -2048) nLeft_command = -2048;
	  if (nRight_command >= 2047) nRight_command = 2047;
	  if (nRight_command <= -2048) nRight_command = -2048;

	  vadm.setDAC(0,nLeft_command);
	  vadm.setDAC(2,nRight_command);
	  if ((nLeft_command == 0) && (nRight_command == 0))
	    disableAmps();
	  else
	    {
	      enableAmps();
	    }
	}
      else
	{
	  if (++nArray_pos >= nArray_max)
	    nArray_pos = 0;
	  insertArray(dLeft_array, 0.0);
	  insertArray(dRight_array, 0.0);
	  
	  disableAmps();
	  vadm.setDAC(0,0);
	  vadm.setDAC(2,0);
	}
    }

  disableAmps();
  vadm.setDAC(0,0);
  vadm.setDAC(2,0);

  close_joystick();

  shellInit(0,1);
}

extern "C" void testInit(void)
{

  while(1)
    {
      printf("It worked!\n");
    }
}

