/*
   test_drive.cxx

   Mark Sibenac
   97-1-29
   Atacama Desert Trek
   Field Robotics Center

   This tests the encoder interface.

*/

#include "test_drive.h"
#include "scopeprobe.h"
#include "taskLib.h"
#include "sysLib.h"
#include "components/pid/pid.h"
#include "components/amps/amps.h"
#include "components/traject/traject.h"
#include "modules/drive/drivesys.h"
#include "vxLib.h"
#include "usrLib.h"
#include "semLib.h"
#include "logLib.h"

#define AUX_CLK_RATE_MIN 30
#define AUX_CLK_RATE_MAX 3000

#define STRLEN 80
static char s[STRLEN]; // general purpose string

int g_freq=200; // global frequency
Cdrivesys *drivesys; // declare the pointer here but should be in robot.cxx
Cdrive *drive[6]; // just used for this test program

// Get string ending with '\n'. returns numchars not including '\n' or null,
//  0 for empty string, -1 for overrun.
// len is the length of the string to be read in including '\n' and null
static int GetString(char *str, int len)
{
  for (int n=0; n<len; n++)
    {
      str[n] = getchar();
      if (str[n] == '\n')
	{ 
	  str[n] = 0;
	  return n;
	}
    }
  str[n-1] = 0;
  return -1;
}

int drive_servoISR (int)
{
  semGive (drivesys->GetSemServo());
  return 0;
}

void drive_startISR (int freq)
{
  if ((freq < AUX_CLK_RATE_MIN) || (freq > AUX_CLK_RATE_MAX))
    {
      printf("drive_startISR() illegal freq=%d. %d<freq<%d\n", freq, 
	     AUX_CLK_RATE_MIN, AUX_CLK_RATE_MAX);
      return;
    }
  
  taskPrioritySet(taskNameToId("tShell"), 50);
  
  if (sysAuxClkConnect ((FUNCPTR)drive_servoISR, 0))
    {
      printf("AuxClock could not connect\n");
      return;
    }

  if (!sysAuxClkRateSet (freq))
    Dprintf(("AuxClk Connected at freq=%d\n", freq));

  sysAuxClkEnable  ();
}

void drive_stopISR ()
{
  sysClkDisable ();
}

int drive_Init        (int drive_freq, int pid_freq, int traj_freq)
{
  g_freq = pid_freq;

  if ((drive_freq == 0) || (pid_freq == 0) || (traj_freq == 0))
    {
      printf("put three arguments on this command line, dumbass!\n");
      return -1;
    }

  drivesys = new Cdrivesys;

  printf("Initializing drivesys\n");
  drivesys->Init (); // create amp objects and run servo process in background
  drivesys->SetUpdateFreq(pid_freq, traj_freq, drive_freq);

  drive[MOTOR_DRIVE_FL] = drivesys->pFL;
  drive[MOTOR_DRIVE_FR] = drivesys->pFR;
  drive[MOTOR_DRIVE_BL] = drivesys->pBL;
  drive[MOTOR_DRIVE_BR] = drivesys->pBR;
  drive[MOTOR_STEER_L]  = drivesys->pLeft;
  drive[MOTOR_STEER_R]  = drivesys->pRight;

  printf("Spawning Scope at 50 Hz\n");
  sp((FUNCPTR)drive_Scope, MOTOR_DRIVE_FL, 50, 3, 4, 5, 6, 7, 8, 9); // default

  printf("Starting ISR at %d Hz\n", pid_freq);
  drive_startISR (pid_freq);

  return 0;
}

int drive_Kill ()
{
  drive_stopISR();
  return 0;
}

int drive_SetFreq        (int drive_freq, int pid_freq, int traj_freq)
{
  g_freq = pid_freq;

  if ((drive_freq == 0) || (pid_freq == 0) || (traj_freq == 0))
    {
      printf("put three arguments on this command line, dumbass!\n");
      return -1;
    }

  drivesys->SetUpdateFreq(pid_freq, traj_freq, drive_freq);
  return 0;
}

int drive_ChangeState (int drive_num, motor_state_t value)
{
  return (drivesys->GetDrivePtr(drive_num)->ChangeState(value));
}

int drive_GetState    (int drive_num)
{
  register motor_state_t state=drivesys->GetDrivePtr(drive_num)->GetState();

  printf("State = %d\n", state);
  return state;
}

int drive_ChangeMode  (DRIVING_MODE_t mode)
{
  return (drivesys->ChangeMode(mode));
}

DRIVING_MODE_t drive_GetMode ()
{
  register DRIVING_MODE_t mode = drivesys->GetMode();
  
  switch(mode)
    {
      case AUTONOMOUS: printf("Autonomous mode\n"); break;
      case MANUAL: printf("Manual mode\n"); break;
    }

  return mode;
}


int drive_GetPID      (int drive_num)
{
  register Cpid *pid = &drivesys->GetDrivePtr(drive_num)->m_pid;

  printf("drive%d Values:\n", drive_num);
  printf("\tKp = %f\n", pid->GetKp());
  printf("\tKi = %f\n", pid->GetKi());
  printf("\tKd = %f\n", pid->GetKd());
  printf("\tIlimit = %f\n", pid->GetILimit());
  printf("\tMaxPosErr = %f\n", pid->GetMaxPosErr());
  printf("\tCommand = %f\n", pid->GetCommand());
  printf("\tDesPos = %f\n", pid->GetDesPos());
  printf("\tCurPos = %f\n", pid->GetCurPos());
  return 0;
}

int drive_GetAmp      (int drive_num)
{
  register Camps *Amp = drivesys->GetDrivePtr(drive_num)->m_amp;

  printf("drive%d Amp Status:\n", drive_num);
  printf("\tPosition = %ld\n", Amp->GetPosition());
  printf("\tAngle = %f\n", Amp->GetAngle());
  printf("\tFault = %d\n", Amp->GetFault());
  printf("\tTemp = %d\n", Amp->GetTemp());
  printf("\tCurrentMon = %f\n", Amp->GetCurrentMon());
  printf("\tCommand = %f\n", Amp->GetCommand());
  printf("\tEnable = %d\n", Amp->GetEnable());
  printf("\tMaxCommand = %f\n", Amp->GetMaxCommand());
  return 0;
}

int drive_SetPID      (int drive_num)
{
  register Cpid *pid = &drivesys->GetDrivePtr(drive_num)->m_pid;
  double fNum;

  printf("SetPID parameters for drive%d:\n", drive_num);
  printf("Kp [%f] = ", pid->GetKp());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pid->SetKp(fNum);
  printf("Ki [%f] = ", pid->GetKi());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pid->SetKi(fNum);
  printf("Kd [%f] = ", pid->GetKd());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pid->SetKd(fNum);
  printf("ILimit [%f] = ", pid->GetILimit());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pid->SetILimit(fNum);
  printf("MaxPosErr [%f] = ", pid->GetMaxPosErr());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pid->SetMaxPosErr(fNum);
  return 0;
}

int drive_SetDes      (int drive_num, double des)
{
  return (drivesys->GetDrivePtr(drive_num)->m_pid.SetDesPos(des));
}

int drive_SetAmp      (int drive_num)
{
  register Camps *Amp = drivesys->GetDrivePtr(drive_num)->m_amp;
  double fNum;

  printf("SetAmp parameters for amp%d:\n", drive_num);
  printf("Command [%f] = ", Amp->GetCommand());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) Amp->SetCommand(fNum);
  printf("MaxCommand [%f] = ", Amp->GetMaxCommand());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) Amp->SetMaxCommand(fNum);
  return 0;
}

int drive_Disable     (int drive_num)
{
  return (drivesys->GetDrivePtr(drive_num)->m_amp->Disable());
}

int drive_Enable      (int drive_num)
{
  return (drivesys->GetDrivePtr(drive_num)->m_amp->Enable());
}

char *ScopeName(char *name, int drive_num)
{
  static char str[20];

  sprintf(str, "drive%d.%s", drive_num, name);
  return str;
}

int drive_Scope(int drive_num, int freq)
{
  double fCommand=0.0, fDesPos=0.0, fCurrent=0.0, fAngle=0.0, 
         fVelocity=0.0, fPosError=0.0, fCurPos=0.0;
  int    nTargetPos=0, nSetPoint=0;
  int    nVelTrajPos=0, nVelTraj=0;

  if ((freq < 0) || (freq > sysClkRateGet()))
    {
      printf("freq = %d must be between 0 and %d\n", freq, sysClkRateGet());
      return -1;
    }
  
  ScopeInstallSignal(ScopeName("Command", drive_num), "volt", &fCommand, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("CurPos", drive_num), "tick", &fCurPos, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("DesPos", drive_num), "tick", &fDesPos, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("PosError", drive_num), "tick", &fPosError, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("Current", drive_num), "amp", &fCurrent, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("Angle", drive_num), "rad", &fAngle, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("TargetPos", drive_num), "tick", &nTargetPos, 
		     "int", 0);
  ScopeInstallSignal(ScopeName("SetPoint", drive_num), "tick", &nSetPoint, 
		     "int", 0);
  ScopeInstallSignal(ScopeName("Velocity", drive_num), "tick/s", &fVelocity, 
		     "double", 0);
  ScopeInstallSignal(ScopeName("VelTrajPos", drive_num), "tick", &nVelTrajPos, 
		     "int", 0);
  ScopeInstallSignal(ScopeName("VelTraj", drive_num), "tick/s", &nVelTraj, 
		     "int", 0);

  ScopeChangeSampleRate(double(freq), 0);

  while(1)
    {
      fCommand = drivesys->GetDrivePtr(drive_num)->m_pid.GetCommand();
      fCurPos  =double(drivesys->GetDrivePtr(drive_num)->m_amp->GetPosition());
      fDesPos  = drivesys->GetDrivePtr(drive_num)->m_pid.GetDesPos();
      fPosError= drivesys->GetDrivePtr(drive_num)->m_pid.GetPosError();
      fCurrent = drivesys->GetDrivePtr(drive_num)->m_amp->GetCurrentMon();
      fAngle   = drivesys->GetDrivePtr(drive_num)->m_amp->GetAngle();
      nTargetPos= drivesys->GetDrivePtr(drive_num)->m_trajectory.GetTarget();
      nSetPoint=  drivesys->GetDrivePtr(drive_num)->m_trajectory.GetNewPos();
      fVelocity=  drivesys->GetDrivePtr(drive_num)->m_trajectory.GetCurrentVel();
      nVelTrajPos = drivesys->GetDrivePtr(drive_num)->m_veltraject.GetTarget();
      nVelTraj = drivesys->GetDrivePtr(drive_num)->m_veltraject.GetCurrentVel();
      ScopeCollectSignals(0);
      taskDelay(sysClkRateGet()/freq);
    }

  return 0;
}

int drive_SetTarget (int drive_num)
{
  register Ctraject *traject = &drivesys->GetDrivePtr(drive_num)->m_trajectory;
  long int nNum;

  printf("SetTarget parameters for amp%d:\n", drive_num);
  printf("Target Position [%ld ticks] = ", traject->GetTarget());
  GetString(s, STRLEN);
  if (sscanf(s, "%ld", &nNum)) traject->SetNewTarget(nNum);
  drivesys->GetDrivePtr(drive_num)->SetControlMode(POSITION_MODE);
  return 0;
}

int drive_SetTargetAngle (int drive_num)
{
  register Cdrive *pDrive = drivesys->GetDrivePtr(drive_num);
  double fNum;

  pDrive->SetControlMode(POSITION_MODE);
  printf("SetTarget parameters for amp%d:\n", drive_num);
  printf("Target Angle [%f rads] = ", pDrive->GetTargetAngle());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pDrive->SetTargetAngle(fNum);
  return 0;
}

int drive_SetTargetVel (int drive_num)
{
  register Cdrive *pDrive = drivesys->GetDrivePtr(drive_num);
  double fNum;

  pDrive->SetControlMode(VELOCITY_MODE);
  printf("SetTarget parameters for amp%d:\n", drive_num);
  printf("Target Vel [%f rads/s] = ", pDrive->GetTargetVel());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fNum)) pDrive->SetTargetVel(fNum);
  return 0;
}

int drive_SetTraject (int drive_num)
{
  register Ctraject *traject = &drivesys->GetDrivePtr(drive_num)->m_trajectory;
  double fAccel, fMaxVel;

  printf("SetTarget parameters for amp%d:\n", drive_num);
  printf("Accel  [%f] = ", traject->GetAccel());
  GetString(s, STRLEN);
  sscanf(s, "%lf", &fAccel);
  printf("MaxVel [%f] = ", traject->GetMaxVel());
  GetString(s, STRLEN);
  if (sscanf(s, "%lf", &fMaxVel)) traject->SetParams(fMaxVel, fAccel);
  return 0;
}

int drive_SetVelTraject (int drive_num)
{
  register Ctrajectvel *traject = &drivesys->GetDrivePtr(drive_num)->m_veltraject;
  int nAccel, nMaxVel;

  printf("SetTarget parameters for amp%d:\n", drive_num);
  printf("Accel  [%d] = ", traject->GetAccel());
  GetString(s, STRLEN);
  sscanf(s, "%d", &nAccel);
  printf("MaxVel [%d] = ", traject->GetMaxVel());
  GetString(s, STRLEN);
  if (sscanf(s, "%d", &nMaxVel)) traject->SetParams(nMaxVel, nAccel);
  return 0;
}

int drive_GetTraject (int drive_num)
{
  register Ctraject *traject = &drivesys->GetDrivePtr(drive_num)->m_trajectory;

  printf("GetTraject parameters for amp%d:\n", drive_num);
  printf("\tTrajectory Position = %ld\n", traject->GetNewPos());
  printf("\tMax Velocity = %f\n", traject->GetMaxVel());
  printf("\tAcceleration = %f\n", traject->GetAccel());
  printf("\tFrequency = %d\n", traject->GetFreq());
  printf("\tTime = %d\n", traject->GetTime());
  printf("\tDirection = %d = %s\n", traject->GetDirection(), (traject->GetDirection() == 1) ? "Positve, Clockwise" : "Negative, Counter-Clockwise");
  printf("\tCurrent Velocity = %f\n", traject->GetCurrentVel());
  printf("\tTarget Position = %ld\n", traject->GetTarget());
  printf("\tStatus = %d\n", traject->GetStatus());
  printf("\tCase = %d\n", traject->GetCase());
  return 0;
}

int drive_Reset (int drive_num)
{
  Cdrive *pDrive = drivesys->GetDrivePtr(drive_num);

  drive_ChangeState (drive_num, INHIBITED);
  pDrive->m_amp->Reset();
  pDrive->m_trajectory.Reset();
  pDrive->m_veltraject.Reset();
  pDrive->SetTargetAngle(0.0);
  return 0;
}
