/*
   point.cxx

   Mark Sibenac
   97-2-4
   Atacama Desert Trek
   Field Robotics Center

   This module handles the two pointing system motors (azimuth, pitch).

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, SERVO_ERROR}

   The pointing servos opoerate in angular mode.
*/

#include "point.h"
#include "point_param.h"
#include "taskLib.h"
#include "math.h"

// Angular 
Cpoint::Cpoint (MOTOR_t axis_number, Camps *amp_in) : m_pid(amp_in, 1), m_trajectory(amp_in, 1) /* 1 means angular */
{
  Dprintf(("Cpoint::Cpoint() creating a new point object %d\n", axis_number));

  if ((axis_number != MOTOR_POINT_AZIMUTH) &&
      (axis_number != MOTOR_POINT_ELEVATION))
    {
      Dprintf(("Cpoint::Cpoint() illegal axis_number=%d\n", axis_number));
      return;
    }

  if (!amp_in)
    {
      Dprintf(("Cpoint::Cpoint() amp=%x is not initialized!!!\n", 
	       (unsigned int)amp_in));
      return;
    }

  m_axis_number = axis_number;
  m_amp = amp_in;

  if (axis_number == MOTOR_POINT_AZIMUTH) // azimuth
    { 
      m_trajectory.SetParams(AZ_MAX_MOTOR_TICKS_SEC, AZ_DEFAULT_MOTOR_ACCEL);
      m_fAccel = AZ_DEFAULT_ACCEL;
    }
  else // elevation
    {
      m_trajectory.SetParams(EL_MAX_MOTOR_TICKS_SEC, EL_DEFAULT_MOTOR_ACCEL);
      m_fAccel = EL_DEFAULT_ACCEL;
    }

  m_traject_time_preset = 1; // update every servo update by default
  m_traject_timer = m_traject_time_preset; // start timer at preset value
  m_traject_current_des_pos = m_trajectory.GetNewPos();
  m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());

  m_fDesVel = 0.0;
  m_fCurrentVel = 0.0;
}

Cpoint::~Cpoint ()
{
  m_state = INHIBITED;
  m_amp->Disable();
  m_amp->SetCommand(0.0);
}

// Init
//  This initializes all of the point member variables to a known state
//   motor_state = INHIBITED
//   
int Cpoint::Init (double Kp, double Ki, double Kd, double I_limit,
		  double MaxPosErr, double MaxAmp, double Accel)
{
  Dprintf(("Cpoint::Init()"));

  m_state = INHIBITED;
  m_amp->Reset();
  m_amp->SetMaxCommand(MaxAmp);
  m_trajectory.SetAccel(Accel);
  m_pid.Init(Kp, Ki, Kd, I_limit, MaxPosErr); // Some close values
  m_amp->SelectEncoderMode (4); // encoder count synchronous with all edges

  return 0;
}

// UpdateServos
//  This runs the PID loop and trajectory calculation. 
int Cpoint::UpdateServo ()
{
  register long target;

  m_amp->UpdateEncoder();
  m_amp->UpdateCurrentMon();
  m_amp->UpdateFaults();
  
  if (--m_traject_timer <= 0)
    {
      m_traject_timer = m_traject_time_preset; // reset timer
      m_traject_current_des_pos = m_trajectory.CalcNewPos();
      m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());
//      m_pid.SetDesPos (m_traject_current_des_pos);
    }

  target = m_trajectory.GetTarget();

  m_pid.SetDesPos (target);
  m_pid.ServoLaw  (double(target-m_lLastTarget));
  
  m_lLastTarget = target;

  if (m_state != SERVOING) return 1;
  
  m_amp->SetCommand(m_pid.GetCommand());
  return 0;
}

int Cpoint::ChangeState (motor_state_t state)
{
  switch (state)
    {
      case INHIBITED:
           m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
      case SERVOING:
	   m_pid.Reset();
	   m_amp->SetCommand(0.0);
	   m_amp->Enable();
	   break;
      case OVERCURRENT:
      case OVERTEMP:
      case SERVO_ERROR:
	   m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
    }

  m_state = state;
  return 0;
}

int Cpoint::SetTarget (double fTargetAngle)
{
  m_fTargetAngle = fTargetAngle;
  m_trajectory.SetNewTarget(long(fTargetAngle / PI_2 * double(m_amp->GetEncoderTicksPerRev())));

  return 0;
}

/*
   Sets the PID and Trajectory Generator's update frequency variables.
   Makes sure amp is in the INHIBITED state before the change is made.
*/
int Cpoint::SetUpdateFreq (int pid_freq, int trajectory_freq, int point_freq)
{
  motor_state_t state = m_state;
  
  ChangeState (INHIBITED); // change to an inhibited state to avoid problems
  m_pid.SetUpdateFreq (pid_freq);
  m_pid.SetVelUpdateRate(point_freq);
  m_trajectory.SetFreq (trajectory_freq);
  m_traject_time_preset = pid_freq/trajectory_freq;
  m_traject_timer = m_traject_time_preset;
  
  ChangeState (state); // go back to previous state
  return 0;
}


