/*
   drivesys.cxx

   Mark Sibenac
   97-2-18
   Atacama Desert Trek
   Field Robotics Center

   This module is responsible for handling the four drive motors.

*/

#include "drivesys.h"
#include "drive_param.h"
#include "components/amps/amps.h"
#include "taskLib.h"

int DrivePIDFun (int pDrivesys, int, int, int, int, int, int, int, int, int)
{
  return (((Cdrivesys *)(pDrivesys))->UpdateServos());
}

Cdrivesys::Cdrivesys()
{
  Dprintf(("Cdrivesys::Cdrivesys() being constructed.\n"));

  pFL = pFR = pBL = pBR = pLeft = pRight = 0;

  m_drive_time_preset = 1; // update every servo update by default
  m_drive_timer = m_drive_time_preset; // start timer at preset value
  m_Mode = MANUAL; // start in Manual driving mode

  /* create the amp objects for the two axes */
  WHEEL_FL_AMP = new Camps(1, WHEEL_TICKS_REV); // 1 means angular
  WHEEL_FR_AMP = new Camps(1, WHEEL_TICKS_REV); // 1 means angular
  WHEEL_BL_AMP = new Camps(1, WHEEL_TICKS_REV); // 1 means angular
  WHEEL_BR_AMP = new Camps(1, WHEEL_TICKS_REV); // 1 means angular
  STEER_LEFT_AMP = new Camps(0, STEER_TICKS_REV); // 0 means not angular
  STEER_RIGHT_AMP = new Camps(0, STEER_TICKS_REV); // 0 means not angular

  /* Initialize each amp, 0 means no temperature signal */
  WHEEL_FL_AMP->Init(&WHEEL_FL_ENCODER, &WHEEL_FL_CURRENT, &WHEEL_FL_COMMAND,
		     &WHEEL_FL_FAULT, &WHEEL_FL_TEMP, &WHEEL_FL_ENABLE);
  WHEEL_FR_AMP->Init(&WHEEL_FR_ENCODER, &WHEEL_FR_CURRENT, &WHEEL_FR_COMMAND,
		     &WHEEL_FR_FAULT, &WHEEL_FR_TEMP, &WHEEL_FR_ENABLE);
  WHEEL_BL_AMP->Init(&WHEEL_BL_ENCODER, &WHEEL_BL_CURRENT, &WHEEL_BL_COMMAND,
		     &WHEEL_BL_FAULT, &WHEEL_BL_TEMP, &WHEEL_BL_ENABLE);
  WHEEL_BR_AMP->Init(&WHEEL_BR_ENCODER, &WHEEL_BR_CURRENT, &WHEEL_BR_COMMAND,
		     &WHEEL_BR_FAULT, &WHEEL_BR_TEMP, &WHEEL_BR_ENABLE);
  STEER_LEFT_AMP->Init(&STEER_LEFT_ENCODER, &STEER_LEFT_CURRENT,
		       &STEER_LEFT_COMMAND, &STEER_LEFT_FAULT,
		       &STEER_LEFT_TEMP, &STEER_LEFT_ENABLE);
  STEER_RIGHT_AMP->Init(&STEER_RIGHT_ENCODER, &STEER_RIGHT_CURRENT,
		       &STEER_RIGHT_COMMAND, &STEER_RIGHT_FAULT,
		       &STEER_RIGHT_TEMP, &STEER_RIGHT_ENABLE);

  /* create the drive objects for the six axes */
  pFL    = new Cdrive (MOTOR_DRIVE_FL, WHEEL_FL_AMP);
  pFR    = new Cdrive (MOTOR_DRIVE_FR, WHEEL_FR_AMP);
  pBL    = new Cdrive (MOTOR_DRIVE_BL, WHEEL_BL_AMP);
  pBR    = new Cdrive (MOTOR_DRIVE_BR, WHEEL_BR_AMP);
  pLeft  = new Cdrive (MOTOR_STEER_L, STEER_LEFT_AMP);
  pRight = new Cdrive (MOTOR_STEER_R, STEER_RIGHT_AMP);
}

Cdrivesys::~Cdrivesys()
{
  char pString[20];
  int tid;
  
  semFlush (m_semServo);
  semDelete (m_semServo);

  sprintf(pString, "tDSPID");
  if ((tid = taskNameToId(pString)) != -1) taskDelete (tid);

  delete pFL;
  delete pFR;
  delete pBL;
  delete pBR;
  delete pLeft;
  delete pRight;
}

int Cdrivesys::Init ()
{
  char pString[20];
  int tid;
  
  Dprintf(("Cdrivesys::Init()\n"));

  if (!(m_semServo = semBCreate (SEM_Q_FIFO, SEM_EMPTY)))
    {
      Dprintf(("Cdrivesys::Cdrivesys() semBCreate() was not able to allocate "
	       "memory.\n"));
      return -1;
    }

  sprintf(pString, "tDSPID");
  if ((tid = taskNameToId(pString)) != -1) taskDelete (tid);
  taskSpawn (pString, PRIORITY_DSPID, VX_SUPERVISOR_MODE | VX_DEALLOC_STACK | 
	     VX_FP_TASK | VX_STDIO, 20000, (FUNCPTR)DrivePIDFun, 
	     (int)this, 0, 0, 0, 0, 0, 0, 0, 0, 0);

  pFL->Init (WHEEL_DEFAULT_KP, WHEEL_DEFAULT_KI, WHEEL_DEFAULT_KD,
	     WHEEL_DEFAULT_ILIM, WHEEL_DEFAULT_MAXPOSERR,
	     WHEEL_DEFAULT_MAXCMD, WHEEL_DEFAULT_MOTOR_ACCEL);
  pFR->Init (WHEEL_DEFAULT_KP, WHEEL_DEFAULT_KI, WHEEL_DEFAULT_KD,
	     WHEEL_DEFAULT_ILIM, WHEEL_DEFAULT_MAXPOSERR,
	     WHEEL_DEFAULT_MAXCMD, WHEEL_DEFAULT_MOTOR_ACCEL);
  pBL->Init (WHEEL_DEFAULT_KP, WHEEL_DEFAULT_KI, WHEEL_DEFAULT_KD,
	     WHEEL_DEFAULT_ILIM, WHEEL_DEFAULT_MAXPOSERR,
	     WHEEL_DEFAULT_MAXCMD, WHEEL_DEFAULT_MOTOR_ACCEL);
  pBR->Init (WHEEL_DEFAULT_KP, WHEEL_DEFAULT_KI, WHEEL_DEFAULT_KD,
	     WHEEL_DEFAULT_ILIM, WHEEL_DEFAULT_MAXPOSERR,
	     WHEEL_DEFAULT_MAXCMD, WHEEL_DEFAULT_MOTOR_ACCEL);
  pLeft->Init (STEER_DEFAULT_KP, STEER_DEFAULT_KI, STEER_DEFAULT_KD,
	     STEER_DEFAULT_ILIM, STEER_DEFAULT_MAXPOSERR,
	     STEER_DEFAULT_MAXCMD, STEER_DEFAULT_MOTOR_ACCEL);
  pRight->Init (STEER_DEFAULT_KP, STEER_DEFAULT_KI, STEER_DEFAULT_KD,
	     STEER_DEFAULT_ILIM, STEER_DEFAULT_MAXPOSERR,
	     STEER_DEFAULT_MAXCMD, STEER_DEFAULT_MOTOR_ACCEL);

  return 0;
}

/*
   This routine runs at the PID frequency.
*/
int Cdrivesys::UpdateServos ()
{
  while (1)
    {
      semTake (m_semServo, WAIT_FOREVER);

      if (--m_drive_timer <= 0) // time to update drive formulas
	{
	  long pos[6];

	  m_drive_timer = m_drive_time_preset; // reset timer

	  // do steering transform and compute new wheel positions in pos[]
	  // SteerFormulae(pos);

	  switch (m_Mode)
	    {
	      case AUTONOMOUS : 
	        pFL->SetTargetTick   (pos[MOTOR_DRIVE_FL]);
		pFR->SetTargetTick   (pos[MOTOR_DRIVE_FR]);
		pBL->SetTargetTick   (pos[MOTOR_DRIVE_BL]);
		pBR->SetTargetTick   (pos[MOTOR_DRIVE_BR]);
		pLeft->SetTargetTick (pos[MOTOR_STEER_L]);
		pRight->SetTargetTick(pos[MOTOR_STEER_R]);
		break;
	      case MANUAL :
		ManualModeUpdate();
	      default : /* nothing */ break;
	    }
	}

      pFL->UpdateServo();
      pFR->UpdateServo();
      pBL->UpdateServo();
      pBR->UpdateServo();
      pLeft->UpdateServo();
      pRight->UpdateServo();
    }
}

/*
   pid_freq   = servo PID updates
   traj_freq  = servo trajectory updates
   drive_freq = New Angle calculations

   This relationship must be followed:
     (pid_freq >= traj_freq) && (pid_freq >= drive_freq)

   pid_freq is the main clock source, traj_freq and drive_freq are derived
     from it. So even divisible numbers should be used.
     EG:    pid_freq  traj_freq  drive_freq
            200       50         25
	    100       25         10
	    100       50         50  (eliminates traj_gen)
*/
int Cdrivesys::SetUpdateFreq (int pid_freq, int traj_freq, int drive_freq)
{
  if (pid_freq < traj_freq)
    {
      Dprintf(("Cdrivesys::SetUpdateFreq() illegal pid_freq=%d and traj_freq="
	       "%d\n", pid_freq, traj_freq));
      return -1;
    }

  if (pid_freq < drive_freq)
    {
      Dprintf(("Cdrivesys::SetUpdateFreq() illegal pid_freq=%d and drive_freq="
	       "%d\n", pid_freq, drive_freq));
      return -1;
    }

  if (drive_freq <= 0)
    {
      Dprintf(("Cdrivesys::SetUpdateFreq() illegal drive_freq=%d\n", 
	       drive_freq));
      return -1;
    }

  m_drive_time_preset = pid_freq / drive_freq;
  m_drive_timer = m_drive_time_preset;
  m_drive_freq = drive_freq;

  pFL->SetUpdateFreq   (pid_freq, traj_freq, drive_freq);
  pFR->SetUpdateFreq   (pid_freq, traj_freq, drive_freq);
  pBL->SetUpdateFreq   (pid_freq, traj_freq, drive_freq);
  pBR->SetUpdateFreq   (pid_freq, traj_freq, drive_freq);
  pLeft->SetUpdateFreq (pid_freq, traj_freq, drive_freq);
  pRight->SetUpdateFreq(pid_freq, traj_freq, drive_freq);

  return 0;
}

int Cdrivesys::ChangeMode (DRIVING_MODE_t mode)
{
  switch (mode)
    {
      case AUTONOMOUS : Dprintf(("Going to AUTONOMOUS MODE.\n")); break;
      case MANUAL : Dprintf(("Going to MANUAL MODE.\n")); break;
    }

  return (m_Mode = mode);
}

Cdrive *Cdrivesys::GetDrivePtr (int drive_num)
{
  switch (drive_num) {
    case MOTOR_DRIVE_FL: return pFL; break;
    case MOTOR_DRIVE_FR: return pFR; break;
    case MOTOR_DRIVE_BL: return pBL; break;
    case MOTOR_DRIVE_BR: return pBR; break;
    case MOTOR_STEER_L: return pLeft; break;
    case MOTOR_STEER_R: return pRight; break;
    default: Dprintf(("Cdrivesys::GetDrivePtr() illegal drive_num=%d\n",
		      drive_num));
  }
  return 0;
}

void Cdrivesys::ManualModeUpdate ()
{
  Cdrive *pDrive;

  for (MOTOR_t m=MOTOR_DRIVE_FL; m <= MOTOR_STEER_R; m++)
    {
      pDrive = GetDrivePtr(m);
      if (pDrive->GetControlMode() == VELOCITY_MODE)
	{
	  /* nothing... */
	} else { // POSITION MODE
	  /* nothing... */
	}
    }
}

