/*
   drive.cxx

   Mark Sibenac
   97-1-28
   Atacama Desert Trek
   Field Robotics Center

   This module controls the coordinated motion and velocity servo loops on the
   six motors. It handles the PID loops and the trajectory generator
   algorithms.

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, SERVO_ERROR}

   The Wheel servo operate in angular mode, but the steering servos 
   operate in non-angular mode.
*/

#include "drive.h"
#include "taskLib.h"
#include "drive_param.h"

#define FABS(x) (((x) < 0.0) ? (-(x)) : (x))

// angular control = 1, otherwise 0
#define ANGULAR(x) (((axis_number) < MOTOR_STEER_L) ? (1) : (0))

// axis_number = [0,5]
//               0 - MOTOR_DRIVE_FL = Wheel Front Right
//               1 - MOTOR_DRIVE_FR = Wheel Front Left
//               2 - MOTOR_DRIVE_BL = Wheel Back Right
//               3 - MOTOR_DRIVE_BR = Wheel Back Left
//               4 - MOTOR_STEER_L = Left Side Steering 
//               5 - MOTOR_STEER_R = Right Side Steering 
Cdrive::Cdrive (MOTOR_t axis_number, Camps *amp_in) : m_pid(amp_in, 0), m_trajectory(amp_in, 0), m_veltraject(amp_in)
{
  Dprintf(("Cdrive::Cdrive() creating a new drive object %d\n", axis_number));

  if ((axis_number < MOTOR_DRIVE_FL) || (axis_number > MOTOR_STEER_R))
    {
      Dprintf(("Cdrive::Cdrive() illegal axis_number=%d\n", axis_number));
      return;
    }

  if (!amp_in)
    {
      Dprintf(("Cdrive::Cdrive() amp=%x is not initialized!!!\n", 
	       (unsigned int)amp_in));
      return;
    }

  m_axis_number = axis_number;
  m_amp = amp_in;
  
  if (axis_number <= MOTOR_DRIVE_BR) // Wheel servos
    {
      m_trajectory.SetParams(WHEEL_MAX_MOTOR_TICKS_SEC,
			     WHEEL_DEFAULT_MOTOR_ACCEL);
      m_veltraject.SetParams(WHEEL_MAX_MOTOR_TICKS_SEC,
			  WHEEL_DEFAULT_MOTOR_ACCEL);
      m_fAccel = WHEEL_DEFAULT_ACCEL;
    } else {
      m_trajectory.SetParams(STEER_MAX_MOTOR_TICKS_SEC,
			     STEER_DEFAULT_MOTOR_ACCEL);
      m_veltraject.SetParams(STEER_MAX_MOTOR_TICKS_SEC,
			  STEER_DEFAULT_MOTOR_ACCEL);
      m_fAccel = STEER_DEFAULT_ACCEL;
    }

  m_traject_time_preset = 1; // update every servo update by default
  m_traject_timer = m_traject_time_preset; // start timer at preset value
  m_traject_current_des_pos = m_trajectory.GetNewPos();
  m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());

  m_fDesVel = 0.0;
  m_nDesVelTicks = 0;
  m_fCurrentVel = 0.0;

  m_lTargetTick = 0L;
  m_fTargetAngle = 0.0;

  m_ControlMode = POSITION_MODE;
}

Cdrive::~Cdrive ()
{
  m_state = INHIBITED;
  m_amp->Disable();
  m_amp->SetCommand(0.0);
}

// Init
//  This initializes all of the drive member variables to a known state
//   motor_state = INHIBITED
//   
int Cdrive::Init (double Kp, double Ki, double Kd, double I_limit,
		  double MaxPosErr, double MaxAmp, double Accel)
{
  Dprintf(("Cdrive::Init()\n"));

  m_state = INHIBITED;
  m_amp->Reset();
  m_amp->SetMaxCommand(MaxAmp);
  m_trajectory.SetAccel(Accel);
  m_veltraject.SetAccel(int(Accel));
  m_pid.Init(Kp, Ki, Kd, I_limit, MaxPosErr);
  return 0;
}

// UpdateServo
//  This routine is called from a mother process that runs at the servo update
//  rate. It updates the trajectory generator if the timer hits 0. It always
//  updates the PID controller.
int Cdrive::UpdateServo ()
{
  m_amp->UpdateEncoder();
  m_amp->UpdateCurrentMon();
  m_amp->UpdateFaults();

  if (--m_traject_timer <= 0)
    {
      m_traject_timer = m_traject_time_preset; // reset timer
      if (m_ControlMode == VELOCITY_MODE)
	{
	  m_traject_current_des_pos = m_veltraject.CalcNewPos();
	  m_traject_current_des_vel = int(m_veltraject.GetCurrentVel());
	} else { // POSITION MODE	  
	  m_traject_current_des_pos = m_trajectory.CalcNewPos();
	  m_traject_current_des_vel = int(m_trajectory.GetCurrentVel());
	}
      m_pid.SetDesPos (m_traject_current_des_pos);
    }

  m_pid.ServoLaw(double(m_traject_current_des_vel));

  if (m_state != SERVOING) return 1;

  m_amp->SetCommand(m_pid.GetCommand());
  return 0;
}

int Cdrive::ChangeState (motor_state_t state)
{
  switch (state)
    {
      case INHIBITED:
           m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   m_veltraject.Reset();
	   m_trajectory.Reset();
	   break;
      case SERVOING:
	   m_pid.Reset();	
	   m_trajectory.Reset();
	   m_veltraject.Reset();
	   m_amp->SetCommand(0.0);
	   m_amp->Enable();
	   break;
      case OVERCURRENT:
      case OVERTEMP:
      case SERVO_ERROR:
	   m_amp->Disable();
	   m_amp->SetCommand(0.0);
	   break;
    }

  m_state = state;
  return 0;
}

/*
   Set's the velocity of the wheel in radians per second.
*/
int Cdrive::SetTargetVel (double rads_sec)
{
  if (rads_sec < -WHEEL_MAX_VEL)
    { rads_sec = -WHEEL_MAX_VEL; }
  else if (rads_sec > WHEEL_MAX_VEL)
    { rads_sec =  WHEEL_MAX_VEL; }

  m_fDesVel = rads_sec;
  m_nDesVelTicks = int(double(rads_sec/PI_2*double(m_amp->GetEncoderTicksPerRev())));
  m_veltraject.SetNewVel (m_nDesVelTicks);

  return 0;
}

/*
   rads = [-pi, pi]
   Goes to the nearest angle on the wheel output.
*/
int Cdrive::SetTargetAngle (double fTargetAngle)
{
  m_fTargetAngle = fTargetAngle;
  SetTargetTick(long(fTargetAngle / PI_2 * double(m_amp->GetEncoderTicksPerRev())));
  return 0;
}

/*
   This function is called to do a trapezoidal move to a target motor position.
   Does not automatically change the control type. Both position and velocity
   control functions should use this as an interface to the trajectory gen.
*/
int Cdrive::SetTargetTick (long motor_position)
{
  m_lTargetTick = motor_position;
  if (m_ControlMode == VELOCITY_MODE)
    { return (m_veltraject.SetNewTarget (motor_position)); }
  else // POSITION_MODE
    { return (m_trajectory.SetNewTarget (motor_position)); }
}

/*
   Sets the PID and Trajectory Generator's update frequency variables.
   Makes sure amp is in the INHIBITED state before the change is made.
*/
int Cdrive::SetUpdateFreq (int pid_freq, int trajectory_freq, int drive_freq)
{
  motor_state_t state = m_state;
  
  ChangeState (INHIBITED); // change to an inhibited state to avoid problems
  m_pid.SetUpdateFreq (pid_freq);
  m_pid.SetVelUpdateRate(trajectory_freq);
  m_trajectory.SetFreq (trajectory_freq);
  m_veltraject.SetFreq (trajectory_freq);
  m_traject_time_preset = pid_freq/trajectory_freq;
  m_traject_timer = m_traject_time_preset;
  
  ChangeState (state); // go back to previous state
  return 0;
}

int Cdrive::SetControlMode (CONTROL_t mode)
{
  m_ControlMode = mode;
  return 0;
}

