/*
   vmio28.cxx

   VME I/O Class File
   VMIO-28 Piggy Back for a VMIO-10 card (opto-isolated encoder input)

   97-1-8
   Mark Sibenac
   Field Robotics Center
   Lunar Rover Terrestrial Prototype

   nVmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
   pBoardVmeAddress (VMIO10) = I/O address on VMEbus (not short space)

   Port A 0,1,2 for mode 1..7 for counter 1 and 2
   Port A 3,4,5 for mode 1..7 for counter 3 and 4
   Port A 6,7   for chip select counter 1..4
   Port C 0,1   for read and write enable
   Port C 2,3   for addressing LSB, LSB+1, MSB of 24Bit counter
   Port C   3   for enabling software reset with Port C 0,1 active

   Port B
*/

#include "common/nomad_global.h"
#include "vmio28.h"

/*
   vmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
*/
Cvmio28::Cvmio28 (const char *pBoardVmeAddress_in, int nVmioBay_in) : 
                 Cvmio10(pBoardVmeAddress_in, nVmioBay_in)
{
  if (!pBoardVmeAddress_in)
    {
      Dprintf(("Cvmio28::Cvmio28 constructor called without BoardVmeAddress\n"));
      return;
    }

  cCs = 0; // default to counter 1
  cMode01 = cMode23 = 1; // default to single count synchron to UA1

  Dprintf(("Cvmio28::Cvmio28 Creating a new vmio28 object\n"));

  InitPorts (0x00, 0x00, // PORTA = no inversion, all output
	     0x00, 0xff, // PORTB = no inversion, all input
	     0x00, 0x00);// PORTC = no inversion, all output

  SelectMode (0, 1);
  SelectMode (2, 1);
  CounterReset();
}

/*
   enc_num = {0..3}
   Mode 1..7 from counter 1 and 2 are selected with port A0,1,2
   cMode = {1..7}

   This function changes the mode for two encoders!!! 0 and 1 go together, and
   2 and 3 go together. It is only necessary to call this once for each pair
   of encoder registers.

   Mode 1: single count synchron with Ua1
   Mode 2: single count synchron with Ua2
   Mode 3: double count synchron with Ua1
   Mode 4: double count synchron with Ua2
   Mode 5: quadruple count synchron with all edges
   Mode 6: pulse width measurement
   Mode 7: frequency measurement
*/
int Cvmio28::SelectMode (int enc_num, Byte cMode)
{
  if ((enc_num < 0) || (enc_num > 3))
    {
      Dprintf(("Cvmio28::SelectMode() got illegal enc_num=%d\n", enc_num));
      return (-1);
    }

  if ((cMode < 1) || (cMode > 7))
    {
      Dprintf(("Cvmio28::SelectMode() got illegal cMode=%d\n", cMode));
      return (-1);
    }

  if ((enc_num == 0) || (enc_num == 1))
    {
      cMode01 = cMode;
      *pPortA = cCs | cMode23 | cMode01;
    }

  if ((enc_num == 2) || (enc_num == 3))
    {
      cMode23 = cMode << 3;
      *pPortA = cCs | cMode23 | cMode01;
    }

  return (0);
}

/*
   Resets counter registers and counters to 0
*/
int Cvmio28::CounterReset ()
{
  // A reset is generated whenever WE, RD, and port C3 are active

  semTake (*m_pMutex, WAIT_FOREVER);

  *pPortC = 0x0f; // %0000_11_11 _LSB_
  *pPortC = 0x04; // %0000_01_00 _PC3_WE+RD
  *pPortC = 0x0d; // %0000_11_01 _LSB_WE
  *pPortC = 0x0f; // %0000_11_11 _LSB_

  semGive (*m_pMutex);

  // To avoid a spurious count after a reset (+/- 1 error), it
  // is necessary to set all counters to zero.
  PresetEncoder (0, 0);
  PresetEncoder (1, 0);
  PresetEncoder (2, 0);
  PresetEncoder (3, 0);
  
  return (0);
}

/*
   Presets counter nCounter to a value of nCount.

   nCounter = {0..3}
   nCount = {-2^23..2^23-1} == {0xffff.ffff.ffff - 0x0fff.ffff.ffff} must be
            sign extended as an int
*/
int Cvmio28::PresetEncoder (int nCounter, int nCount)
{
  volatile int i;

  Dprintf(("Cvmio28::PresetEncoder(%d, %d) called\n", nCounter, nCount));

  if ((nCounter < 0) || (nCounter > 3))
    {
      Dprintf(("Cvmio28::PresetEncoder() illegal argument nCounter=%d\n",
	       nCounter));
      return (-1);
    }

  if ((nCount < 0) || (nCount > 16777215))
    {
      Dprintf(("Cvmio28::PresetEncoder() illegal argument nCount=%d\n",
	       nCount));
      return (-1);
    }
    
  SetPortDirection ('b', 0x00); // Set Port B as output

  // The two MSBits of CS get counter number minus 1
  cCs = nCounter << 6;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pPortA = cCs | cMode23 | cMode01;

  *pPortC = 0x0f; // %0000_11_11 _LSB_
  *pPortC = 0x0d; // %0000_11_01 _LSB_WE
  *pPortB = (nCount & 0xff);
  *pPortC = 0x0f; // %0000_11_11 _LSB_
for (i=0; i<50; i++);
  *pPortC = 0x0b; // %0000_10_11 _LSB+1_
  *pPortC = 0x09; // %0000_10_01 _LSB+1_WE
  *pPortB = (nCount & 0xff00) >> 8;
  *pPortC = 0x0b; // %0000_10_11 _LSB+1_
for (i=0; i<50; i++);
  *pPortC = 0x07; // %0000_01_11 _MSB_
  *pPortC = 0x05; // %0000_01_01 _MSB_WE
  *pPortB = (nCount & 0xff0000) >> 16;
  *pPortC = 0x07; // %0000_01_11 _MSB_

  semGive (*m_pMutex);

  SetPortDirection ('b', 0xff); // Set Port B as input

  return 0;
}

/*
   nCounter = {0..3}
   returns = value from {-2^23..2^23-1} current value of the counter; will be
             sign extended for an int
*/
int Cvmio28::GetEncoder (int nCounter)
{
  int nCount;
  unsigned int nLSB, nLMSB, nMSB;

  if ((nCounter < 0) || (nCounter > 3))
    {
      Dprintf(("Cvmio28::GetEncoder() illegal argument nCounter=%d\n",
	       nCounter));
      return (-1);
    }
  
  cCs = nCounter << 6;
 
  semTake (*m_pMutex, WAIT_FOREVER);

  *pPortC = 0x0f; // %0000_11_11 _LSB_
  *pPortC = 0x0e; // %0000_11_10 _LSB_RD
  *pPortA = cCs | cMode23 | cMode01;
  nLSB    = (*pPortB & 0xff);

  *pPortC = 0x0b; // %0000_10_11 _LSB+1_
  *pPortC = 0x0a; // %0000_10_10 _LSB+1_RD
  nLMSB   = (*pPortB & 0xff);

  *pPortC = 0x07; // %0000_01_11 _MSB_
  *pPortC = 0x06; // %0000_01_10 _MSB_RD
  nMSB    = (*pPortB & 0xff);

  *pPortC = 0x07; // %0000_01_11 _MSB_
                  // Read (RD) disabled

  semGive (*m_pMutex);

  nCount = ((nMSB << 16) | (nLMSB << 8) | nLSB);

  return nCount;
}
