/*
   vmio10.cxx

   96-12-19
   Mark Sibenac
   Field Robotics Center
   Atacama Desert Trek

   Driver Class for the o+r VMIO10 vme card


   nVmioBay := 0 -> P5,P6,P9,P10; 01-16; chip 1, IC7
               1 -> P3,P4,P7,P8; 17-32, chip 0, IC6
   cPortX_invert := 0->non-invert, 1->invert (each bit)
   cPortX_dir := 0->output, 1->input (each bit)
   pBoardBaseAddress = short I/O address on VMEbus
*/

#include "common/nomad_global.h"
#include "vmio10.h"
#include <vxLib.h>

vmio10_mutex_t *vmio10_mutexes=0;
int mutexes_installed=0; // initially false (set to 0 in do_sib_global_ctors())


/*
   Constructor for vmio10_mutex_s
*/
vmio10_mutex_s::vmio10_mutex_s ()
{
  Dprintf(("vmio10_mutex_s() constructor being called; vmio10_mutexes=%x\n",
	   INT32(this)));
  mutexes_installed = 1; // Set the flag to true

  for (int n=0; n < MAX_VMIO10_BOARDS; n++)
    {
      sem_mutex[n]     = 0;
      pBoardAddress[n] = 0;
    }
}

vmio10_mutex_s::~vmio10_mutex_s ()
{
  for (int n=0; n < MAX_VMIO10_BOARDS; n++)
    {
      if (sem_mutex[n] && pBoardAddress[n])
	{
	  semFlush (sem_mutex[n]);
	  semDelete(sem_mutex[n]);
	}
    }

  Dprintf(("CVmio10 Mutex Semaphores are gone!\n"));
}

/*
   This looks up a semaphore for the Address given and returns a pointer to it.
   If there is no semaphore for that address, one is created and its pointer
   is returned. cNewBoard is written to with a 0-not new or 1-new board.
*/
SEM_ID *vmio10_mutex_s::GetPointer (char *pAddress, char *cNewBoard)
{
  for (int n=0; n < MAX_VMIO10_BOARDS; n++)
    {
      if (!sem_mutex[n]) break; // break outta here since we found an uninitialized slot
      if (pAddress == pBoardAddress[n])
	{
	  *cNewBoard = 0;
	  return &sem_mutex[n];
	}
    }

  // Create a new semaphore at the current location
  sem_mutex[n] = semMCreate(SEM_Q_FIFO);
  Dprintf(("vmio10_mutex_s::GetPointer() creating a new semaphore at 0x%x\n",
	   sem_mutex[n]));
  *cNewBoard = 1;
  pBoardAddress[n] = pAddress;
  return &sem_mutex[n];
}

/*
   Constructor

   nVmioBay := 0 -> P5,P6,P9,P10; 01-16; chip 1, IC7
              1 -> P3,P4,P7,P8; 17-32, chip 0, IC6
   pBoardVmeAddress = address on card (bus address)
*/
Cvmio10::Cvmio10 (const char *pBoardVmeAddress, int nVmioBay_in)
{
  STATUS s;
  char cNewBoard;

  if (!mutexes_installed) { vmio10_mutexes = new vmio10_mutex_t; }

  if (!pBoardVmeAddress)
    {
      Dprintf(("Cvmio10:Cvmio10 constructor was called without a "
	       "BoardVmeAddress.\n"));
      pBoardBaseAddress = 0;  // set it to zero to signify missing board
      return;
    }

  Dprintf(("Cvmio10::Cvmio10 Constructor being called\n"));

  s = sysBusToLocalAdrs (VME_AM_USR_SHORT_IO, pBoardVmeAddress,
			 &pBoardBaseAddress);

  if (s == OK)
    {
      Byte cTestR;

      s = vxMemProbe(ByteRegister(pBoardBaseAddress + PORT_0A_OFFSET), VX_READ,
		     1, &cTestR);
    }

  if (s == OK)
    {
      Dprintf(("VMIO10 board found at 0x%08x\n", UINT32(pBoardBaseAddress)));
    } else {
      Dprintf(("VMIO10 board not found at 0x%08x\n",UINT32(pBoardVmeAddress)));
      pBoardBaseAddress = 0; // set it to zero to signify missing board
      return;
    }

  if (nVmioBay_in == 0)
    {
      pCntl  = ByteRegister(pBoardBaseAddress + CNTL1_OFFSET);
      pPortA = ByteRegister(pBoardBaseAddress + PORT_1A_OFFSET);
      pPortB = ByteRegister(pBoardBaseAddress + PORT_1B_OFFSET);
      pPortC = ByteRegister(pBoardBaseAddress + PORT_1C_OFFSET);
    }
  else if (nVmioBay_in == 1)
    {
      pCntl  = ByteRegister(pBoardBaseAddress + CNTL0_OFFSET);
      pPortA = ByteRegister(pBoardBaseAddress + PORT_0A_OFFSET);
      pPortB = ByteRegister(pBoardBaseAddress + PORT_0B_OFFSET);
      pPortC = ByteRegister(pBoardBaseAddress + PORT_0C_OFFSET);
    }
  else
    {
      Dprintf(("VmioBay = %d is an illegal argument for Cvmio10 constructor\n",
	       nVmioBay_in));
      return ;
    }

  nVmioBay = nVmioBay_in;

  m_pMutex = vmio10_mutexes->GetPointer(pBoardBaseAddress, &cNewBoard);

  if (cNewBoard)
    {
      ByteRegister pCntlTemp = pCntl;

      Dprintf(("Cvmio10::Cvmio10() Resetting both CIO chips\n"));
      pCntl = ByteRegister(pBoardBaseAddress + CNTL1_OFFSET);
      ResetCIO(); // reset both CIO chip IC07
      pCntl = ByteRegister(pBoardBaseAddress + CNTL0_OFFSET);
      ResetCIO(); // reset both CIO chip IC06
      pCntl = pCntlTemp;
    }

  Dprintf(("CVmio10::CVmio10() m_pMutex = %x\n", *m_pMutex));
}

Cvmio10::~Cvmio10 ()
{
  if (!pBoardBaseAddress) return;

}

int Cvmio10::ResetCIO ()
{
  char cHelp;

  if (!pBoardBaseAddress) return -1;

  Dprintf(("Cvmio10::ResetCIO nVmioBay = %d\n", nVmioBay));

  semTake (*m_pMutex, WAIT_FOREVER);

  cHelp = *pCntl;
  *pCntl = 0;
  cHelp = *pCntl;
  *pCntl = 0;
  *pCntl = 1;
  *pCntl = 0;

  semGive (*m_pMutex);

  return 0;
}

/*
  cPortX_invert := 0->non-invert, 1->invert (each bit)
  cPortX_dir := 0->output, 1->input (each bit)
*/
int Cvmio10::InitPorts (Byte cPortA_invert, Byte cPortA_dir,
		        Byte cPortB_invert, Byte cPortB_dir,
			Byte cPortC_invert, Byte cPortC_dir)
{
  if (!pBoardBaseAddress) return -1;

  ResetCIO ();

  Dprintf(("Cvmio10::InitPorts Initializing vmioBay %d\n", nVmioBay));

  InitPortA (cPortA_invert, cPortA_dir);
  InitPortB (cPortB_invert, cPortB_dir);
  InitPortC (cPortC_invert, cPortC_dir);
  EnableCIOChip (); // necessary for I/O to exist to piggies
  return (0);
}

/*
  cPortX_invert := 0->non-invert, 1->invert (each bit)
  cPortX_dir := 0->output, 1->input (each bit)
*/
int Cvmio10::InitPortA (Byte cPortA_invert, Byte cPortA_dir)
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = PORT_A_OFFSET;
  *pCntl = cPortA_invert;
  *pCntl = PORT_A_OFFSET + 1;
  *pCntl = cPortA_dir;
  *pCntl = PORT_A_OFFSET + 2;
  *pCntl = 0;
  semGive (*m_pMutex);

  return 0;
}

/*
  cPortX_invert := 0->non-invert, 1->invert (each bit)
  cPortX_dir := 0->output, 1->input (each bit)
*/
int Cvmio10::InitPortB (Byte cPortB_invert, Byte cPortB_dir)
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = PORT_B_OFFSET;
  *pCntl = cPortB_invert;
  *pCntl = PORT_B_OFFSET + 1;
  *pCntl = cPortB_dir;
  *pCntl = PORT_B_OFFSET + 2;
  *pCntl = 0;
  semGive (*m_pMutex);

  return 0;
}

/*
  cPortX_invert := 0->non-invert, 1->invert (each bit)
  cPortX_dir := 0->output, 1->input (each bit)
*/
int Cvmio10::InitPortC (Byte cPortC_invert, Byte cPortC_dir)
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = PORT_C_OFFSET;
  *pCntl = (cPortC_invert & 0x0f);
  *pCntl = PORT_C_OFFSET + 1;
  *pCntl = (cPortC_dir & 0x0f);
  *pCntl = PORT_C_OFFSET + 2;
  *pCntl = 0;
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::EnableCIOChip ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = 0x94; // enable Port A, B, and C
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::DisableCIOChip ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select mast control register
  *pCntl = 0x94; // disable Port A, B, and C
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::EnablePortA ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl | PORT_A_ENABLE; // enable Port A
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::EnablePortB ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl | PORT_B_ENABLE; // enable Port B
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::EnablePortC ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl | PORT_C_ENABLE; // enable Port C
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::DisablePortA ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl & ~Byte(PORT_A_ENABLE); // disable Port A
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::DisablePortB ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl & ~Byte(PORT_B_ENABLE); // disable Port B
  semGive (*m_pMutex);

  return 0;
}

int Cvmio10::DisablePortC ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  *pCntl = 0x01; // select master control register
  *pCntl = *pCntl & ~Byte(PORT_C_ENABLE); // disable Port C
  semGive (*m_pMutex);

  return 0;
}

/*
   cPortLetter = {'a' | 'b' | 'c' | 'A' | 'B' | 'C'}
   cPortDir = 0->output, 1->input (each bit)
   returns 0 for no error
*/
int Cvmio10::SetPortDirection (char cPortLetter, Byte cPortDir)
{
  register int s=0;

  if (!pBoardBaseAddress) return -1;

  semTake (*m_pMutex, WAIT_FOREVER);

  switch (cPortLetter)
    {
      case 'a':
      case 'A': 
            *pCntl = PORT_A_OFFSET + 1;
	    *pCntl = cPortDir;
	    break;
      case 'b':
      case 'B':
	    *pCntl = PORT_B_OFFSET + 1;
	    *pCntl = cPortDir;
	    break;
      case 'c':
      case 'C':
	    *pCntl = PORT_C_OFFSET + 1;
	    *pCntl = (cPortDir & 0x0f);
	    break;
      default:
	    Dprintf(("Cvmio10::SetDirection() illegal parameter: cPortLetter"
		     "='%c'\n", cPortLetter));
	    s = -1;
    }

  semGive (*m_pMutex);

  return s;
}

