/*
   pid.cxx

   Mark Sibenac
   97-1-14
   Atacama Desert Trek
   Field Robotics Center

   This is a general PID controller.

*/

#include "pid.h"

#define FABS(x) (((x)<0.0) ? (-(x)) : (x))
#define SIGN(x) (((x)<0.0) ? (-1.0) : (1.0))

// PID Class Constructor
Cpid::Cpid (Camps *amp_in, int angular)
{
  if (!(m_amp = amp_in))
    { 
      Dprintf(("Cpid::Cpid() illegal amp_in=%x", UINT32(amp_in)));
      return;
    }
      
  Dprintf(("Cpid::Cpid() creating a new PID controller for amp=%x\n", 
	   UINT32(amp_in)));

  nEncoderTicksPerRev = m_amp->GetEncoderTicksPerRev();
  nAngular = angular;

  fCurPos = 0.0;
  fLastPos = 0.0;
  fSumErr = 0.0;
  fDesPos = 0.0;
  fCommand = 0.0;

  nServoUpdateRate = 1; // good base number
  nVelUpdateRate = 1;   // good base number
  nVelUpdateCount = 0;
  nVelUpdateCountMask = 0;
  fVelocity = 0.0;
  fDeltaPos = 0.0;

  Init (0.0, 0.0, 0.0, 0.0, 0.0);
}

/*
   This standardizes all gains no matter what the update frequency is.
*/
int Cpid::RecalcGains (int old_freq)
{
  double fOldFreq = double(old_freq);

  SetKp(fKp * fOldFreq);
  SetKi(fKi * fOldFreq);
  SetKd(fKd * fOldFreq);
  return 0;
}

/*
   This should be called first to set up the coefficients.
*/
int Cpid::Init (double Kp, double Ki, double Kd, double I_limit, 
		double MaxPosErr)
{
  SetKp(Kp);
  SetKi(Ki);
  SetKd(Kd);
  SetILimit(I_limit);
  SetMaxPosErr(MaxPosErr);

  return 0;
}

int Cpid::Reset ()
{
  fCommand = 0.0;
  fSumErr  = 0.0;
  fLastPos = fCurPos;
  return 0;
}

int Cpid::SetVelUpdateRate (int vel_update_rate_hz)
{
  if (vel_update_rate_hz == 0)
    {
      Dprintf(("Cpid::SetVelUpdateRate() illegal vel_update_rate_hz=%d\n",
	       vel_update_rate_hz));
      return -1;
    }
  
  nVelUpdateRate = vel_update_rate_hz;
  nVelUpdateCountMask = nServoUpdateRate / vel_update_rate_hz;
  nVelUpdateCount = 0;
  return 0;
}

int Cpid::SetUpdateFreq (int servo_update_rate_hz)
{
  int old_freq = nServoUpdateRate;

  if (servo_update_rate_hz == 0)
    {
      Dprintf(("Cpid::SetUpdateFreq() illegal servo_update_rate_hz=%d\n",
	       servo_update_rate_hz));
      return -1;
    }

  nServoUpdateRate = servo_update_rate_hz;
  RecalcGains(old_freq); // recalculates the gains based on the new frequency
  return 0;
}

/*
   This is called at the servoing frequency.
   It returns the new calculated command.
   fDesiredVel = delta in ticks per second
*/
int Cpid::ServoLaw(double fDesiredVel)
{
  register double error;
  register double vel;
  register double vel_delta;
  
  fDesiredVel /= double(nVelUpdateRate); // normalize value to this frequency
  fCurPos = GetPosition();

  error = fDesPos - fCurPos;
  if (nAngular && (FABS(error) > double(nEncoderTicksPerRev/2)))
    { error = -(SIGN(error))*(double(nEncoderTicksPerRev - FABS(error))); }

  if (++nVelUpdateCount >= nVelUpdateCountMask)
    {
      nVelUpdateCount = 0;
      vel = (fCurPos - fDeltaPos);
      if (nAngular && (FABS(vel) > double(nEncoderTicksPerRev/2)))
	{ vel = -(SIGN(vel)) * (double(nEncoderTicksPerRev - FABS(vel))); }
      
      vel_delta = fDesiredVel - vel;
      fDeltaPos = fCurPos;
      fVelocity = vel;
    }

  fSumErr += error;
  if (error > fMaxPosErr)
    { error = fMaxPosErr; }
  else if (error < -fMaxPosErr)
    { error = -fMaxPosErr; }

  fPosError = error;

  if (fSumErr > fI_limit)
    { fSumErr = fI_limit; }
  else if (fSumErr < -fI_limit)
    { fSumErr = -fI_limit; }

/*  if (FABS(error) < double(10.0))
    {
      fCommand = double(0.0);
    } else {
      fCommand = fKp*fPosError + fKi*fSumErr + fKd*vel_delta;
    }
*/
/*  if (FABS(error) < double(10.0))
    { fSumErr = double(0.0); }
*/
  fCommand = fKp*fPosError + fKi*fSumErr + fKd*vel_delta;
  
  fLastPos = fCurPos;
  return (0);
}

