/*
   amps.cxx
   
   Mark Sibenac
   97-1-22
   Atacama Desert Trek
   Field Robotics Center

   This is the implementation of the amps class.

   The integer positions in the amps class are 24 bit signed numbers. The
   positions are fixed to 24 bit unsigned numbers when interfacing to the
   encoder interfaces. So, a 0 position is really -2^23 in the amp class.
*/

#include "amps.h"
#include "taskLib.h"
#include "math.h"

#define SIGN(x) (((x)<0L) ? (-1.0) : (1.0))

Camps *amps[8]; // pointer to array of amp objects

Camps::Camps (int angular, int encoder_ticks_per_rev)
{
  ASSERT(((angular == 0) || (angular == 1)));
  ASSERT((encoder_ticks_per_rev >=0));

  Dprintf(("Camps::Camps() creating a new amps object.\n"));

  nAngular = angular;
  nEncoderTicksPerRev = encoder_ticks_per_rev;
  
  fCurrentMon = 0.0;
  nFault = 0;
  nTemp = -1;
  nEnable = 0;
  fCommand = 0.0;
  nEncoderPos = 0;
  fAngle = 0.0;
  fMaxCommandAmps = MAX_COMMAND_AMPS;
}

// Set temp_in=0 if motor does not have a temperature sensor
int Camps::Init (Cencoder *enc_in, Cadc *cur_in, Cdac *com_in, Cdi *fault_in, 
		 Cdi *temp_in, Cdo *enable_in)
{
  if (enc_in) encoder = enc_in;
    else Dprintf(("Camps::Init() illegal enc_in\n"));
  if (cur_in) current_adc = cur_in;
    else Dprintf(("Camps::Init() illegal cur_in\n"));
  if (com_in) command_dac = com_in;
    else Dprintf(("Camps::Init() illegal com_in\n"));
  if (fault_in) fault_di = fault_in;
    else Dprintf(("Camps::Init() illegal fault_in\n"));
  if (temp_in) temp_di = temp_in;
    else {
      Dprintf(("Camps::Init() illegal temp_in, not using temp function\n"));
      nTemp = -1; // set to does not apply state
    }
  if (enable_in) enable_do = enable_in;
    else Dprintf(("Camps::Init() illegal enable_in\n"));

  PresetPosition(0);
  return 0;
}


 // Real-Time Scheduler calls these functions periodically
int Camps::UpdateEncoder()
{ 
  register double dummy;

  nEncoderPos = encoder->GetEncoder();

  if (nAngular)
    {
      fAngle = modf (double(nEncoderPos) / double(nEncoderTicksPerRev), &dummy)
	       * PI_2;
      if (fAngle > PI)
	{ fAngle -= PI_2; }
      else if (fAngle < NEG_PI)
	{ fAngle += PI_2; }
    }

  return 0;
}

int Camps::UpdateCurrentMon()
{ 
  fCurrentMon = current_adc->GetADC();
  return 0;
}

int Camps::UpdateFaults()
{ 
  nFault = fault_di->GetDI();
  if (nTemp != -1)
    { nTemp  = temp_di->GetDI(); }
  return 0;
}

// User interface functions

 // set position to 0, disable
int Camps::Reset()
{ 
  Disable();
  SetCommand(0.0);
  PresetPosition(nEncoderPos = 0L);
  return 0;
}

double Camps::GetRPM()
{ 
  return 0.0;
}

// position is a 64-bit signed number
long Camps::GetPosition()
{ 
  return nEncoderPos;
}

// angle in radians of the motor shaft
double Camps::GetAngle()
{
  return fAngle;
}


// This is used for 24-bit signed positions
int Camps::PresetPosition(long position)
{ 
  encoder->PresetEncoder(position);
  nEncoderPos = position;
  return 0;
}

// This is used for angle positions of the motor shaft
int Camps::PresetAngle(double angle)
{
  fAngle = fmod(angle, PI_2);
  if (fAngle > PI)
    { fAngle += NEG_PI_2; }

  return (PresetPosition(long(fAngle / PI_2 * double(nEncoderTicksPerRev))));
}
 // 0-no fault, 1-fault
int Camps::GetFault()
{ 
  return nFault;
}

 // 0-ok 1-over temperature -1-does not apply
int Camps::GetTemp()
{ 
  return nTemp;
}

double Camps::GetCurrentMon()
{ 
  return fCurrentMon;
}

double Camps::GetCommand()
{ 
  return fCommand;
}

 // 0-disabled 1-enabled
int Camps::GetEnable()
{ 
  return nEnable;
}

int Camps::Enable()
{ 
  ReleaseBrake();
  SetCommand(0.0);
  enable_do->SetDO(1);
  nEnable = 1;

  return 0;
}

int Camps::Disable()
{ 
  enable_do->SetDO(0);
  nEnable = 0;
  SetCommand(0.0);
  EngageBrake();

  return 0;
}

int Camps::SetCommand(double command)
{ 
  if (command < -fMaxCommandAmps)
    command = -fMaxCommandAmps;
  if (command > fMaxCommandAmps)
    command = fMaxCommandAmps;

  command_dac->SetDAC(fCommand = command);
  return 0;
}

int Camps::ReleaseBrake()
{
  return 0;
}

int Camps::EngageBrake()
{
  return 0;
}

int Camps::SetMaxCommand (double command)
{
  if (command < 0.0)
    {
      Dprintf(("Camps::SetMaxCommand() illegal command = %f\n", command));
      return -1;
    }

  fMaxCommandAmps = command;
  return 0;
}

int Camps::SelectEncoderMode (int mode)
{
  return (encoder->SelectMode (mode));
}
