/*
   trajectvel.h

   Mark Sibenac
   97-1-30
   Atacama Desert Trek
   Field Robotics Center

   This is a VELocity  TRAJECTory controller.

   All positions are 24 bit signed numbers that the amps work with directly.
   Accel is totally dependant on the pid update rate and what command means.

   The accel and max velocity variables are scaled according to frequency. So
   set them no differently with a freq of 60Hz or a freq of 200Hz. Accel and 
   max velocity are in units of ticks/s and ticks/s/s respectively.
*/

#ifndef __TRAJECTVEL_H__
#define __TRAJECTVEL_H__

#include "common/nomad_global.h"
#include "components/amps/amps.h"
#include "semLib.h"

class Ctrajectvel {
 private:

  int  m_nTargetVel;  // Taget Velocity (ticks/slice)
  int  m_nCommandVel; // Velocity in time t (ticks/slice 
  int  m_nLastVel;    // Velocity in time t-1 (ticks/slice)

  long m_lCommandPos; // Commanded position in time t (ticks)

  int  m_nDirection;  // 1:Positive, -1:Negative

  int  m_nMaxVel;     // Maximum velocity (ticks/slice)
  int  m_nAccel;      // Accelration (ticks/slice/slice) 
  int  m_nFreq;       // Frequency that this updates (Hz)

  SEM_ID  m_pMutex;   // Semaphore to mutually exclude
  Camps  *m_amp;      // pointer to object that contains info on positions

  long     GetCurPos        (); // uses m_amp to get current positon feedback
                                // returns it too
 protected:

 public:

  Ctrajectvel  (Camps *amp_in);
 ~Ctrajectvel ();

  long    CalcNewPos        (); // returns m_lCommandPos; called at servo freq
  int     SetNewVel         (int velocity); // desired velocity in ticks
  int     SetParams         (int max_vel, int accel, int freq=-1);
  int     SetAccel          (int accel);
  int     SetFreq           (int freq);
  int     Reset             (); // Called when starting servoing a motor

  int     GetMaxVel         () {return m_nMaxVel*m_nFreq;}
  int     GetAccel          () {return m_nAccel*m_nFreq*m_nFreq;}
  int     GetFreq           () {return m_nFreq;}
  int     GetDirection      () {return m_nDirection;}
  int     GetCurrentVel     () {return m_nCommandVel*m_nFreq;} // ticks/sec
  int     GetCurrentVelSlice() {return m_nCommandVel;}         // ticks/slice
  long    GetTarget         () {return m_lCommandPos;}
};

#endif // __TRAJECTVEL_H__
