/*
   vmio14.cxx

   VME I/O Class File
   VMIO-14 Piggy Back for a VMIO-10 card (opto-isolated encoder input)

   97-1-17
   Mark Sibenac
   Field Robotics Center
   Lunar Rover Terrestrial Prototype

   nVmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
   nPort_in = (if nVmioBay==0 then  0:P6,P10;01-08, 1:P5,P9;09-16)
              (if nVmioBay==1 then  0:P4,P8;17-24,  1:P3,P7;25-32)
   pBoardVmeAddress (VMIO10) = I/O address on VMEbus (not short space)

*/

#include "common/nomad_global.h"
#include "vmio14.h"

/*
   vmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
   nPort_in = (if nVmioBay==0 then  0:P6,P10;01-08, 1:P5,P9;09-16)
              (if nVmioBay==1 then  0:P4,P8;17-24,  1:P3,P7;25-32)
*/
Cvmio14::Cvmio14 (const char *pBoardVmeAddress_in, int nVmioBay_in, int 
		  nPort_in) : Cvmio10(pBoardVmeAddress_in, nVmioBay_in)
{
  if (!pBoardVmeAddress_in)
    {
      Dprintf(("Cvmio14::Cvmio14 constructor called without BoardVmeAddress\n"));
      return;
    }

  if ((nPort_in < 0) || (nPort_in > 1))
    {
      Dprintf(("Cvmio14::Cvmio14() illegal nPort_in=%d\n", nPort_in));
      return;
    }

  Dprintf(("Cvmio14::Cvmio14 Creating a new vmio14 object in VmioBay=%d, "
	   "VmioPort=%d\n", nVmioBay_in, nPort_in));

  m_nVmioPort = nPort_in;

  if (nPort_in == 1)
    {
      InitPortA (0x00, 0xff); // PORTA = no inversion, all input
      InitPortC (0x00, 0x0f); // PORTC = no inversion, all input
      m_pPortAB = pPortA;
      m_pPortC = pPortC; 
    } else {
      InitPortB (0x00, 0xff); // PORTB = no inversion, all input
      InitPortC (0x00, 0x0f); // PORTC = no inversion, all input
      m_pPortAB = pPortB;
      m_pPortC = pPortC;
    }

  EnableCIOChip();
}

Cvmio14::~Cvmio14 ()
{
  DisableIO ();
}

/*
   channel = [0-7]
   returns: 1-value is high
            0-value is low
*/
int Cvmio14::GetDI (int channel)
{
  register char value;

  if (!pBoardBaseAddress) return -1;

  if ((channel < 0) || (channel > 11))
      {
	Dprintf(("Cvmio14::GetDI() illegal channel=%d\n", channel));
	return -1;
      }
  
  if (channel < 8)
    { value = (*m_pPortAB) & (1 << channel); }
  else
    { value = (*m_pPortC) & (1 << channel); }

  return (value ? 1 : 0);}

/*
   channel = [0-7]
   value   = 1-high
             0-low
*/
int Cvmio14::SetDO (int channel, int value)
{
  ByteRegister pPort;

  if (!pBoardBaseAddress) return -1;

  if ((channel < 0) || (channel > 11))
      {
	Dprintf(("Cvmio14::SetDO() illegal channel=%d\n", channel));
	return -1;
      }
  
  if (channel < 8)
    { pPort = m_pPortAB; }
  else
    { pPort = m_pPortC; }

    if (value)
      { 
	semTake (*m_pMutex, WAIT_FOREVER);
      *pPort = *pPort | (1 << channel); 
      semGive (*m_pMutex);
    } else {
      semTake (*m_pMutex, WAIT_FOREVER);
      *pPort = *pPort & ~Byte(1 << channel);
      semGive (*m_pMutex);
    }

  return 0;
}

int Cvmio14::EnableIO ()
{
  if (!pBoardBaseAddress) return -1;

  if (m_nVmioPort == 1)
    { EnablePortA(); }
  else
    { EnablePortB(); }

  return 0;
}

int Cvmio14::DisableIO ()
{
  if (!pBoardBaseAddress) return -1;

  if (m_nVmioPort == 1)
    { DisablePortA(); }
  else
    { DisablePortB(); }

  return 0;
}

