/* 
   vadm.h

   Mark Sibenac
   Lunar Rover Terrestrial Prototype
   Field Robotics Center
   96-5-15 original
   97-1-6  modified

   This is the vadm interface class.
*/

#include "common/nomad_global.h"
#include "vadm.h"
#include <vxLib.h>

// Constructor
Cvadm::Cvadm (const char *pBoardVmeAddress, int nSingleDiff_in)
{
  STATUS s;

  if (!pBoardVmeAddress)
    {
      Dprintf(("Cvadm::Cvadm constructor called without BoardVmeAddress\n"));
      pBoardBaseAddress = 0;
      return;
    }

  Dprintf(("Cvadm::Cvadm Creating a new vadm object\n"));

  nSingleDiff = nSingleDiff_in;

  s = sysBusToLocalAdrs(VME_AM_USR_SHORT_IO, pBoardVmeAddress,
			&pBoardBaseAddress);

  pDO      = WordRegister(pBoardBaseAddress + VADM_DO_OFFSET);
  pDORB    = WordRegister(pBoardBaseAddress + VADM_DORB_OFFSET);
  pADSTART = WordRegister(pBoardBaseAddress + VADM_ADSTART_OFFSET);
  pADC     = WordRegister(pBoardBaseAddress + VADM_ADC_OFFSET);
  pADMUX   = WordRegister(pBoardBaseAddress + VADM_ADMUX_OFFSET);
  pDACA    = WordRegister(pBoardBaseAddress + VADM_DACA_OFFSET);
  pDI      = WordRegister(pBoardBaseAddress + VADM_DI_OFFSET);
  pDACB    = WordRegister(pBoardBaseAddress + VADM_DACB_OFFSET);

  if (s == OK)
    {
      int nTestR;

      s = vxMemProbe((char *)pDORB, VX_READ, 2, (char *)&nTestR);
      if (s == OK)
	{
	  Dprintf(("VADM board found at 0x%08x\n", UINT32(pBoardBaseAddress)));
	}
    }
  
  if (s == ERROR)
    {
      printf("VADM board not found at 0x%08x\n", UINT32(pBoardBaseAddress));
      pBoardBaseAddress = 0;
      return;
    }

  if (*pDI & 0x0001) // check for correct ADC chip
    {
      printf("Cvadm::init Error... Do not have the correct ADS7808/9 chip\n");
      pBoardBaseAddress = 0;
      return;
    }

  *pDO = *pDORB | 0x0004; // select single-shot mode (slow mode) for ADC
  cRelay_map = 0; // Set all relays open on startup

  m_Mutex = semMCreate (SEM_Q_FIFO);
}

Cvadm::~Cvadm ()
{
  int n;

  for (n=0; n<8; n++)
    { SetDAC (n, 0.0); }

  for (n=0; n<3; n++)
    { SetRelay (n, 0); }

  DisableIO ();
  
  semFlush (m_Mutex); 
  semDelete(m_Mutex); 
 
}

int Cvadm::Init()
{
  if (!pBoardBaseAddress) return -1;

  return 0;
}

// SetDACI()
//   nChannel - int from 0-7
//   nValue   - 12 bit signed value from -2048 --> 2047
//   returns - value written or 0xffff if error
int Cvadm::SetDACI(int nChannel, int nValue)
{
  if (!pBoardBaseAddress) return -1;

  if ((nChannel < 0) || (nChannel > 7))
    {
      printf ("Cvadm::SetDACI error. nChannel = %d\n", nChannel);
      return 0xffff;
    }

  if ((nValue < -2048) || (nValue > 2047))
    {
      printf ("Cvadm::SetDACI error. nValue = %d\n", nValue);
      return 0xffff;
    }

  *pDACA = ((nChannel+1) << 12) | (nValue + 2048);

  return nValue;
}

// SetDAC()
//   nChannel - int from 0-7
//   fValue   - 12 bit double-precision floating point from -10.0 -> 10.0
//   returns - value written or 0xffff if error
int Cvadm::SetDAC(int nChannel, double fValue)
{
  if (!pBoardBaseAddress) return -1;

  if ((fValue < -10.0) || (fValue > 10.0))
    {
      printf("Cvadm::SetDACF error. nValue = %f\n", fValue);
      return 0xffff;
    }
  
  if (fValue > 9.995) fValue = 9.995;

  return (SetDACI(nChannel, Word(fValue/VOLT_PER_BIT)));
}

// GetADCI()
//   nChannel - int from 0-15 (single-end), 0-7 (differential)
//   returns - int value from -2048..2047 or 0xffff for error
int Cvadm::GetADCI(int nChannel)
{
  unsigned command;
  register int nValue;

  if (!pBoardBaseAddress) return -1;

  command = 0x0000; // Choose twos-complement read mode over single-ended

  if (nSingleDiff == DIFFERENTIAL)
    {
      if ((nChannel < 0) || (nChannel > 7))
	{
	  printf ("Cvadm::GetADCI error. nChannel = %d\n", nChannel);
	  return 0xffff;
	}

      command |= ((nChannel + 8) << 12);
    }
  else // SINGLE_ENDED
    {
      if ((nChannel < 0) || (nChannel > 15))
	{
	  printf ("Cvadm::GetADCI error. nChannel = %d\n", nChannel);
	  return 0xffff;
	}

      command |= (nChannel << 12);
    }

  semTake (m_Mutex, WAIT_FOREVER);

  // next few lines are repeated to wait 10 usec
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX
  *pADMUX = command; // write to ADMUX

  *pADSTART = 1; // start a conversion

  nValue =  *pADC & 0x0fff;

  semGive (m_Mutex);

  nValue = (nValue & 0x0800) ? ((nValue & 0x07ff) - 0x0800) : (nValue);

  return nValue;
}

// GetADC()
//   nChannel - int from 0-15 (single-end), 0-7 (differential)
//   returns - float value from -10..10 or 0xffff for error
double Cvadm::GetADC(int nChannel)
{
  if (!pBoardBaseAddress) return -1;

  return (GetADCI(nChannel) * VOLT_PER_BIT);
}

// SetDO
//   bitmap  - 8 bit number specifying the actual output states 1-on, 0-off
//   returns - 0-no error, 0xffff-error
int Cvadm::SetDO (unsigned char bitmap)
{
  if (!pBoardBaseAddress) return -1;

  *pDO = (((unsigned) bitmap) << 8) | 0x000c; // RS=1, AM=1, EC=0, ED=0

  return 0;
}

int Cvadm::SetDO (int nChan, int nValue)
{
  if (!pBoardBaseAddress) return -1;

  if (nValue)
    { return (SetDOhigh (1 << nChan)); }
  else
    { return (SetDOlow (1 << nChan)); }
}

// SetDOhigh
//   bitmask - 8 bit number specifying which bits should be high leaving the
//             others what they are 1-high 0-oldstate
//   returns - 0-no error, 0xffff-error
int Cvadm::SetDOhigh (unsigned char bitmask)
{
  if (!pBoardBaseAddress) return -1;

  semTake (m_Mutex, WAIT_FOREVER);

  *pDO = (((unsigned) bitmask) << 8) | (*pDORB & 0xff00) | 0x000c; // RS=1, AM=1, EC=0, ED=0

  semGive (m_Mutex);

  return 0;
}

// SetDOlow
//   bitmask - 8 bit number specifying which bits should be low leaving the
//             others what they are 1-low 0-oldstate
//   returns - 0-no error, 0xffff-error
int Cvadm::SetDOlow (unsigned char bitmask)
{
  if (!pBoardBaseAddress) return -1;

  semTake (m_Mutex, WAIT_FOREVER);

  *pDO = ((((unsigned) (~bitmask)) << 8) & (*pDORB & 0xff00)) | 0x000c; // RS=1, AM=1, EC=0, ED=0

  semGive (m_Mutex);

  return 0;
}

// GetDI
//   returns - 8 bit value of input register (not inverted_ or 0xffff for error
int Cvadm::GetDI (void)
{
  Word nDI;

  if (!pBoardBaseAddress) return -1;

  nDI = *pDI;

  if (!(nDI & 0x0080)) // thermal shutdown or other error when ER is low
    {
      printf("Cvadm::GetDI Error... problem with output port\n");
      return 0xffff;
    }

  return ((~nDI >> 8) & 0xff);
}

// Function that interface calls with a channel number between 0 and 7
int Cvadm::GetDI (int nChannel)
{
  if (!pBoardBaseAddress) return -1;

  return (GetDI() & (int(1) << nChannel));
}

// EnableDigital
int Cvadm::EnableIO ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (m_Mutex, WAIT_FOREVER);

  *pDO = *pDORB | 0x0008; // RS=1

  semGive (m_Mutex);

  return 0;
}

// DisableDigital
int Cvadm::DisableIO ()
{
  if (!pBoardBaseAddress) return -1;

  semTake (m_Mutex, WAIT_FOREVER);

  *pDO = *pDORB & ~0x0008; // RS=0

  semGive (m_Mutex);

  return 0;
}

// SetRelays
//   bitmap - 4 bit value specifying what relays should be energized 1-on,0-off
//   returns- 0 on success, 0xffff for error
int Cvadm::SetRelays (unsigned char bitmap)
{
  if (!pBoardBaseAddress) return -1;

  if (bitmap & 0xf0)
    {
      printf("Cvadm::SetRelays Error... bitmap=0x%x is wrong\n", bitmap);
      return 0xffff;
    }

  cRelay_map = bitmap;
  *pDO = (((unsigned) bitmap) << 4) | 0x0c; // RS=1. AM=1. EC=0, ED=0

  return 0;
}

// SetRelay Interface Function
//   0 < nChan < 3
//   nState:   0=opened, 1=closed
int Cvadm::SetRelay  (int nChan, int nState)
{
  if (!pBoardBaseAddress) return -1;

  if (nState)
    { return (SetRelays(cRelay_map | (1 << nChan))); }
  else
    { return (SetRelays((cRelay_map & ~(1 << nChan)) & 0x0f)); }
}
