/* 
   vadc.h

   Mark Sibenac
   Lunar Rover Terrestrial Prototype
   Field Robotics Center
   97-1-9

   This is the vadc device class.

*/

#include "common/nomad_global.h"
#include "vadc.h"
#include <vxLib.h>

// Constructor
//  nVadcBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
Cvadc::Cvadc(const char *pBoardVmeAddress, int nVadcBay_in, int nSingleDiff_in)
{
  STATUS s;

  if (!pBoardVmeAddress)
    {
      Dprintf(("Cvadc::Cvadc constructor called without BoardVmeAddress\n"));
      pBoardBaseAddress = 0;
      return;
    }

  Dprintf(("Cvadc::Cvadc Creating a new vadc object\n"));

  if ((nSingleDiff_in != SINGLE_ENDED) && (nSingleDiff_in != DIFFERENTIAL))
    {
      Dprintf(("Cvadc::Cvadc() nSingleDiff_in=%d is illegal\n", nSingleDiff_in));
      pBoardBaseAddress = 0;
      return;
    }

  nSingleDiff = nSingleDiff_in;

  if ((nVadcBay_in != 0) && (nVadcBay_in != 1))
    {
      Dprintf(("Cvadc::Cvadc() nVadcBay_in=%d is illegal\n", nVadcBay_in));
      pBoardBaseAddress = 0;
      return;
    }

  nVadcBay = nVadcBay_in;

  s = sysBusToLocalAdrs(VME_AM_USR_SHORT_IO, pBoardVmeAddress,
			&pBoardBaseAddress);

  pSelectReg   = WordRegister(pBoardBaseAddress + VADC_SELECT_OFFSET);
  pDataReg     = WordRegister(pBoardBaseAddress + VADC_DATA_OFFSET);
  pRes12BitReg = WordRegister(pBoardBaseAddress + VADC_RES12BIT_OFFSET);

  if (s == OK)
    {
      int nTestR;

      s = vxMemProbe((char *)pDataReg, VX_READ, 2, (char *)&nTestR);
      if (s == OK)
	{
	  Dprintf(("VADC board found at 0x%08x\n", UINT32(pBoardBaseAddress)));
	}
    }
  
  if (s == ERROR)
    {
      printf("VADC board not found at 0x%08x\n", UINT32(pBoardBaseAddress));
      pBoardBaseAddress = 0;
      return;
    }

  m_Mutex = semMCreate (SEM_Q_FIFO);
}

Cvadc::~Cvadc ()
{
  semFlush (m_Mutex);
  semDelete(m_Mutex);
}

int Cvadc::Init()
{
  if (!pBoardBaseAddress) return -1;

  return 0;
}

// GetADCI()
//   nChannel - int from 0-15 (single-end), 0-7 (differential)
//   returns - int value from -2048..2047 or 0xffff for error
int Cvadc::GetADCI(int nChannel)
{
  Word nConversion;
  volatile int z;

  if (!pBoardBaseAddress) return -1;

  if (nSingleDiff == DIFFERENTIAL)
    {
      if ((nChannel < 0) || (nChannel > 7))
	{
	  printf ("Cvadc::GetADCI error. nChannel = %d\n", nChannel);
	  return 0xffff;
	}
    }
  else // SINGLE_ENDED
    {
      if ((nChannel < 0) || (nChannel > 15))
	{
	  printf ("Cvadc::GetADCI error. nChannel = %d\n", nChannel);
	  return 0xffff;
	}
    }

  semTake (m_Mutex, WAIT_FOREVER);

  *pSelectReg = ((nChannel & 0x0f) | ((nVadcBay & 0x01) << 4));

  for (z=0; z<260; z++); // wait 13us max assuming 40Mhz and 2 CPI

  *pRes12BitReg = 0xfff; // start 12 bit conversion

  for (z=0; z<80; z++); // wait 3us for voltage acq assuming 40Mhz and 2 CPI

  while (*pDataReg & VADC_COMPLETE_MASK); // poll STS until end of conversion

  nConversion = (*pDataReg & 0xfff) - 0x0800; // assuming 12 bit signed convers

  semGive (m_Mutex);

  return nConversion;
}

// GetADC()
//   nChannel - int from 0-15 (single-end), 0-7 (differential)
//   returns - double value from -10..10 or 0xffff for error
double Cvadc::GetADC(int nChannel)
{
  if (!pBoardBaseAddress) return 0xffff;

  return (GetADCI(nChannel) * VOLT_PER_BIT);
}


