/*
   trajectvel.cxx

   Mark Sibenac
   97-1-30
   Atacama Desert Trek
   Field Robotics Center

   This is a general TRAJECTory generator that is used with a closed loop
   controller (Cpid).

   The accel and max velocity variables are scaled according to frequency. So
   set them no differently with a freq of 60Hz or a freq of 200Hz. Accel and 
   max velocity are in units of ticks/s and ticks/s/s respectively.
*/

#include "trajectvel.h"
#include "stdlib.h"

#define MAX(x,y) (((x)>(y)) ? (x) : (y))
#define MIN(x,y) (((x)<(y)) ? (x) : (y))

// TRAJECTVEL Class Constructor
Ctrajectvel::Ctrajectvel (Camps *amp_in)
{
  if (!amp_in)
    {
      Dprintf(("Ctraject::Ctraject() amp_in not initialized!!\n"));
      return;
    }

  Dprintf(("Ctraject::Ctraject() initialize new velocity generator \n"));

  m_amp = amp_in;

  m_nDirection = 1;
  m_nTargetVel = m_nLastVel = m_nCommandVel = 0;
  m_lCommandPos = m_amp->GetPosition();
  m_nMaxVel = m_nAccel = 1;
  m_nFreq = 1;

  m_pMutex = semMCreate(SEM_Q_FIFO | SEM_DELETE_SAFE);
  Dprintf(("Ctrajectvel::Ctrajectvel() m_pMutex = %x\n", UINT32(m_pMutex)));
}

Ctrajectvel::~Ctrajectvel ()
{
  semDelete (m_pMutex);
  Dprintf(("Ctrajectvel::~Ctrajectvel() Mutex Semaphore is gone!\n"));
}

// Desired velocity in ticks
int Ctrajectvel::SetNewVel (int velocity)
{
  semTake(m_pMutex, WAIT_FOREVER);
  m_nTargetVel = velocity / m_nFreq; // ticks
  semGive(m_pMutex);
  return 0;
}
  
//   This is called at the servoing frequency.
//   It returns the new position that the controller needs to servo to.
long Ctrajectvel::CalcNewPos ()
{
  semTake(m_pMutex, WAIT_FOREVER);

  if ((abs(m_nTargetVel - m_nLastVel) >= m_nAccel) ||
      abs(m_nTargetVel) > m_nAccel)
    {
      if (m_nTargetVel < m_nLastVel)
	{
	  m_nDirection = -1;
	  m_nCommandVel -= m_nAccel;
	} else { // m_nTargetVel > m_nLastVel
	  m_nDirection = 1;
	  m_nCommandVel += m_nAccel;
	}

      if (m_nTargetVel < 0) // negative
	{ m_nCommandVel = MAX(-m_nMaxVel, m_nCommandVel); }
      else // positive
	{ m_nCommandVel = MIN(m_nMaxVel, m_nCommandVel); }

      m_lCommandPos += long(m_nCommandVel); 
      m_nLastVel = m_nCommandVel;
   }

  semGive (m_pMutex);

  return m_lCommandPos;
}

int Ctrajectvel::SetParams (int max_vel, int accel, int freq)
{
  Dprintf(("Ctrajectvel::SetParams max_vel=%d, accel=%d, freq=%d\n", max_vel,
	   accel, freq));

  if (freq == 0)
    {
      Dprintf(("Ctrajectvel::SetParams() illegeal freq = %d\n", freq));
      return -1;
    }

  if (accel == 0)
    {
      Dprintf(("Ctrajectvel::SetParams() illegeal accel = %d\n", accel));
      return -1;
    }

  semTake(m_pMutex, WAIT_FOREVER);
  if ((freq != -1) && (freq != 0))
    { m_nFreq   = freq; }
  m_nMaxVel = max_vel / m_nFreq;
  m_nAccel  = accel / m_nFreq / m_nFreq; 
  semGive(m_pMutex);
  return 0;
}

int Ctrajectvel::SetFreq (int freq)
{
  if (freq == 0)
    {
      Dprintf(("Ctrajectvel::SetFreq() illegal freq=%d", freq));
      return -1;
    }
  
  return (SetParams(m_nMaxVel, m_nAccel, freq));
}

//  Sets the Acceleration by calling SetParams with the current MaxVel and freq
int Ctrajectvel::SetAccel (int accel)
{
  return (SetParams (m_nMaxVel, accel));
}

// Called when starting up a motor into the SERVOING state so things are 
//  stopped at startup
int Ctrajectvel::Reset()
{
  semTake(m_pMutex, WAIT_FOREVER);

  m_nLastVel = m_nCommandVel = 0;
  m_lCommandPos = GetCurPos();
  
  SetNewVel(0);
  
  semGive(m_pMutex);
  return 0;
}

// Retrieves the current position of the motor (encoder) from the m_amp object
long Ctrajectvel::GetCurPos ()
{
  return (m_amp->GetPosition());
}

