/*
 * 
 * gps.h
 *
 * Mike Krebs
 * Atacama Desert Trek
 * 96-2-27
 *
 */

#ifndef __GPS_H__
#define __GPS_H__


#include "common/nomad_global.h"


/* commands from/to receiver */
#define ACK 0x06
#define NAK 0x15
#define ENQ 0x05
#define STX 0x02
#define ETX 0x03
 
/* Message types (for the type field of the GpsMessage structure */
#define MESSAGE_OK 0
#define MESSAGE_ERROR 1
#define MESSAGE_GETSERIAL 0x06
#define MESSAGE_RSERIAL 0x07
#define MESSAGE_GETSTAT1 0x08
#define MESSAGE_RECSTAT1 0x09
#define MESSAGE_GENOUT 0x40
#define MESSAGE_GETRAW 0x56
#define MESSAGE_RAWDATA 0x57
#define MESSAGE_APPFILE 0x64
#define MESSAGE_KEYSIM 0x81
#define MESSAGE_SCRDUMP 0x82


#define MAX_SVS 32      /* maximum number of satellites */
#define MAX_CONFIGS 10  /* maximum number of messages to keep track of */

typedef struct GpsMessage {  
  unsigned char type;
  unsigned char status;
  unsigned char length;
  unsigned char *pData;
} GpsMessage, *GpsMessagePtr;

typedef struct {
  unsigned char cPrn;
  unsigned char cSVflag1;
  unsigned char cSVflag2;
} t_sSV;

typedef struct {
  unsigned char c_type;
  unsigned char c_freq;
} config_t;

class Cgps {
 private:
  int m_fd;                  /* file descriptor of device */
  unsigned char m_cPort;     /* port number of GPS receiver */
  int m_nFreq;               /* frequency of status update */
  
  /* LAT, LONG, HEIGHT (type=2) */
  double m_dLatitude;
  double m_dLongitude;
  double m_dHeight;
  
  /* ECEF POSITION (type=3) */
  double m_dX;
  double m_dY;
  double m_dZ;

  /* ECEF DELTA (type=6) */
  double m_dDeltaX;
  double m_dDeltaY;
  double m_dDeltaZ;
  
  /* PDOP INFO (type=9) */
  float m_fPdop;
/*  float m_fHdop;*/
/*  float m_fVdop;*/
/*  float m_fTdop;*/

  /* POSITION SIGMA INFO (type=12) */
  float m_fPosRMS;
  float m_fSigmaE;
  float m_fSigmaN;
/*  float m_fCovarEN;*/
  float m_fSigmaU;
/*  float m_fSemiMajor;*/
/*  float m_fSemiMinor;*/
/*  float m_fOrientation;*/
/*  float m_fUnitVar;*/
/*  short m_sEpochs;*/

  /* SV BRIEF INFO (type=13) */
  unsigned char m_cSVtotal;
  t_sSV m_pSVS[MAX_SVS];

  config_t m_pConfigs[MAX_CONFIGS];

  unsigned char m_TxID;
  
  
  /* see if the receiver is ready for commands
   * returns: zero - receiver ok
   *          non-zero - receiver busy */
  int checkOperational(void);
  
  /* read a response from the serial line - wait for one if there isn't
   * already one there.
   * returns: pointer to the message */
  GpsMessagePtr getResponse();

  /* send a command on the serial line
   * takes: pointer to a valid message structure */
  void sendCommand(GpsMessagePtr);

  void disableMessages();
  void enableMessages();

  void procGENOUT(GpsMessagePtr);
  void procSCRDUMP(GpsMessagePtr);
  double ptod(unsigned char *);
  float ptof(unsigned char *);
  short ptos(unsigned char *);
  void clearConfigs();
  void addConfig(unsigned char cType, unsigned char cFreq);
  void delConfig(unsigned char cType);


 public:
  Cgps (); // constructor
 ~Cgps (); // destructor

  int init (char *pSerialDeviceName, int nPort, int nFreq);
  void flushSerial(void);

  int handleResponse(void);
  /* task that updates registers */
  void updateState ();

  /* sets the message type to send back
   * takes: cType -- message type (page C-5):
   *          0 == All messages off
   *          2 == LAT, LONG, HEIGHT
   *          3 == ECEF POSITION
   *          6 == ECEF DELTA
   *          9 == PDOP INFO
   *          12 == POSITION SIGMA INFO
   *          13 == SV BRIEF INFO
   *        cFreq -- frequency of message (page C-4):
   *          0 -- off
   *          2 -- 5 Hz
   *          4 -- 2 seconds
   *          8 -- 60 seconds
   *          255 -- once only, now */
  void setMessage (unsigned char cType, unsigned char cFreq);
  
  void showPanel(void);
  void panelMode(void);
  void keySequence(char *seq);

  double getLat() { return m_dLatitude; }
  double getLong() { return m_dLongitude; }
  double getHeight() { return m_dHeight; }
  
  double getX() { return m_dX; }
  double getY() { return m_dY; }
  double getZ() { return m_dY; }
  
  double getDX() { return m_dDeltaX; }
  double getDY() { return m_dDeltaY; }
  double getDZ() { return m_dDeltaZ; }

  float getPdop() { return m_fPdop; }

  float getRMS() { return m_fPosRMS; }
  float getSigmaE() { return m_fSigmaE; }
  float getSigmaN() { return m_fSigmaN; }
  float getSigmaU() { return m_fSigmaU; }

  unsigned char getSVS() { return m_cSVtotal; }
  t_sSV getSV(int svnum) { return m_pSVS[svnum]; }

};

#endif // __GPS_H__
