/*
   amps.h
   
   Mark Sibenac
   97-1-22
   Atacama Desert Trek
   Field Robotics Center

   This is the header file for the amps component.

   When creating amp objects, call the constructor with two parameters:
     angular : 0 - not angular, linear, don't mod encoder
               1 - angular, mod encoder and put values in terms of -pi to pi
     encoder_ticks_per_rev : 24 bit unsigned number of the number of encoder
               ticks per revolution of the motor (eg. 1024)
*/

#ifndef __AMPS_H__
#define __AMPS_H__

#include "common/nomad_global.h"
#include "interfaces/encoder/encoder.h"
#include "interfaces/adc/adc.h"
#include "interfaces/dac/dac.h"
#include "interfaces/di/di.h"
#include "interfaces/do/do.h"

#define  MAX_COMMAND_AMPS (0.5)

class Camps {
 private:
  Cencoder *encoder;
  Cadc     *current_adc;
  Cdac     *command_dac;
  Cdi      *fault_di;
  Cdi      *temp_di;
  Cdo      *enable_do;
  Cdo      *brake_do; // still need to implement this!!!!!!!

  int     nEncoderTicksPerRev; // 32 bit signed number
  int     nAngular; // 0-linear 1-angular

  double  fAngle; // between -pi and pi
  double  fCurrentMon; // current in amps
  int     nFault; // 0-no fault 1-fault
  int     nTemp;  // 0-ok 1-overheat -1-does not apply
  int     nEnable; // 0-disabled 1-enabled
  double  fCommand; // voltage between -10 and 10
  long    nEncoderPos; // 64 bit signed number
  double  fMaxCommandAmps;  // Max command current in amps
  
 public:

  Camps (int angular, int encoder_ticks_per_rev);
  int Init  (Cencoder *enc_in, Cadc *cur_in, Cdac *com_in, Cdi *fault_in, 
	     Cdi *temp_in, Cdo *enable_in);

 // Real-Time Scheduler calls these functions periodically
  int    UpdateEncoder();
  int    UpdateCurrentMon();
  int    UpdateFaults();
  char  *Name () {return "Camps";}

 // User interface functions
  int    Reset(); // set position to 0, disable
  int    Enable();
  int    Disable();
  int    ReleaseBrake();
  int    EngageBrake();

  int    SetCommand(double command); // this is a voltage for the amps
  int    SetMaxCommand (double command); // sets MaxCommandAmps
  int    SelectEncoderMode (int mode);
  int    PresetPosition(long position); // give this a 64-bit signed number
  int    PresetAngle(double position); // give this an angle in rads

  double GetRPM();
  long   GetPosition(); // returns a 64 bit signed number
  double GetAngle();   // returns an angle in rads between -pi and pi
  int    GetEncoderTicksPerRev () {return nEncoderTicksPerRev;}
  int    GetFault(); // 0-no fault, 1-fault
  int    GetTemp(); // 0-ok 1-over temperature
  double GetCurrentMon(); // current in amps
  double GetCommand(); // voltage/current going to the amps
  int    GetEnable(); // 0-disabled 1-enabled
  double GetMaxCommand () {return fMaxCommandAmps;}

};

extern Camps *amps[8]; // pointer to array of amp objects

#endif //  __AMPS_H__
