/*
   drive.h

   Mark Sibenac
   97-1-28
   Atacama Desert Trek
   Field Robotics Center

   This module controls the coordinated motion and velocity servo loops on the
   six motors. It handles the PID loops and the trajectory generator
   algorithms. It assumes that the PID loop is run at a higher frequency than
   the trajectory generator. 

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, ERROR}

*/

#ifndef __DRIVE_H__
#define __DRIVE_H__

#include "common/nomad_global.h"
#include "components/amps/amps.h"
#include "components/pid/pid.h"
#include "components/traject/traject.h"
#include "components/traject/trajectvel.h"


class Cdrive {
 private:
  motor_state_t m_state; // motor state
  int           m_axis_number; // [MOTOR_DRIVE_FL, MOTOR_STEER_RIGHT]
  double        m_fTargetAngle; // in radians [-PI,PI]
  long          m_lTargetTick; // in ticks

  int           m_traject_time_preset;  // trajectory update sleep length
  int           m_traject_timer; // countdown timer for trajectory update
  long          m_traject_current_des_pos; // since last trajectory update
  int           m_traject_current_des_vel; // since last trajectory update

  double        m_fDesVel; // Desired Velocity in rad/s
  int           m_nDesVelTicks; // Desired Velocity in ticks/s
  double        m_fCurrentVel; // Current Velocity at this point in time rad/s
  double        m_fAccel;  // Acceleration in rad/s/s

  CONTROL_t m_ControlMode; // of motor [VELOCITY_MODE, POSITION_MODE]

 public:
  Cdrive (MOTOR_t axis_number, Camps *amp);
 ~Cdrive ();

  int           Init          (double Kp, double Ki, double Kd, double I_limit,
			       double MaxPosErr, double MaxAmp, double Accel);
  int           SetUpdateFreq (int pid_freq, int trajectory_freq, int drive_freq); //Freq in Hz
  int           UpdateServo   (); // Called at the servo update rate for PID
  int           ChangeState   (motor_state_t state);

  int           SetTargetVel  (double rads_sec); // given as angular velocity
  int           SetTargetTick (long motor_position); // for steering
  int           SetTargetAngle(double fTargetAngle); // given as an angle [-pi, pi]
  int           SetControlMode(CONTROL_t mode); // [VELOCITY, POSITION]

  motor_state_t GetState      () {return m_state;}
  CONTROL_t     GetControlMode() {return m_ControlMode;}
  long          GetTargetTick () {return m_lTargetTick;} // ticks
  double        GetTargetAngle() {return m_fTargetAngle;} // rads
  double        GetTargetVel()   {return m_fDesVel;} // rads/s
  int           GetTargetVelTicks(){return m_nDesVelTicks;}; // ticks/s

  Camps        *m_amp; // angular and 1024 ticks/rev
  Cpid          m_pid; // PID controller
  Ctraject      m_trajectory; // Trapezoidal trajectory generator
  Ctrajectvel   m_veltraject; // Velocity trajectory generator
};

#endif // __DRIVE_H__
