/* 
   vmio10.h

   VME I/O Library Header File
   VMIO-10 Board

   96-12-18
   Mark Sibenac
   Field Robotics Center
   Lunar Rover Terrestrial Prototype

   If an object is created for this class and the actual board does not exist
   in the VME cage or there is another problem that makes the constructor fail,
   nothing will execute even if the functions are called.

   A VME bus address of 0x000000 is not allowed for this class. This internally
   signifies that the board does not exist.
   
   vmioBay := 0 -> P5,P6,P9,P10; 01-16; chip 1, IC7
              1 -> P3,P4,P7,P8; 17-32, chip 0, IC6
   cPortX_invert := 0->non-invert, 1->invert (each bit)
   cPortX_dir := 0->output, 1->input (each bit)
   boardBaseAddress = short I/O address on VMEbus
*/

#ifndef __VMIO10_H__
#define __VMIO10_H__

#include "common/nomad_global.h"
#include "devices/devices.h"
#include "semLib.h"

#define MAX_VMIO10_BOARDS (4)

#define CNTL0_OFFSET   0x31
#define PORT_0A_OFFSET 0x21
#define PORT_0B_OFFSET 0x11
#define PORT_0C_OFFSET 0x01

#define CNTL1_OFFSET   0x39
#define PORT_1A_OFFSET 0x29
#define PORT_1B_OFFSET 0x19
#define PORT_1C_OFFSET 0x09

#define PORT_A_OFFSET  0x22
#define PORT_B_OFFSET  0x2a
#define PORT_C_OFFSET  0x05

#define PORT_A_ENABLE  0x04
#define PORT_B_ENABLE  0x80
#define PORT_C_ENABLE  0x10

typedef struct vmio10_mutex_s {
  vmio10_mutex_s(); // constructor for this struct
 ~vmio10_mutex_s(); // destructor for this struct

  // returns the pointer to a semaphore for a board address
  SEM_ID *GetPointer (char *pAddress, char *cNewBoard);

  SEM_ID sem_mutex[MAX_VMIO10_BOARDS]; // semaphores for each VMIO10 board
  char  *pBoardAddress[MAX_VMIO10_BOARDS]; // addresses for each VMIO10 board
} vmio10_mutex_t;

class Cvmio10 : public Cdevices {
 private:

  int nVmioBay;  /* 0 -> P5,P6,P9,P10; 01-16; chip 1, IC7
                    1 -> P3,P4,P7,P8; 17-32, chip 0, IC6 */

 protected:

  ByteRegister pCntl;   // value base on nVmioBay
  ByteRegister pPortA;  // value base on nVmioBay
  ByteRegister pPortB;  // value base on nVmioBay
  ByteRegister pPortC;  // value base on nVmioBay
  SEM_ID    *m_pMutex;  // pointer to semaphore in sem_mutex array

 public:

  /* Constructor; pBoardVmeAddress is VME address (not local) */
  Cvmio10 (const char *pBoardVmeAddress, int nVmioBay_in); 
 ~Cvmio10 ();

  int ResetCIO ();
  int InitPorts (Byte cPortA_invert, Byte cPortA_dir,
	         Byte cPortB_invert, Byte cPortB_dir,
	         Byte cPortC_invert, Byte cPortC_dir);

  int InitPortA (Byte cPortA_invert, Byte cPortA_dir);
  int InitPortB (Byte cPortB_invert, Byte cPortB_dir);
  int InitPortC (Byte cPortC_invert, Byte cPortC_dir);

  int EnableCIOChip (); // Enables  Ports A, B, and C
  int DisableCIOChip(); // Disables Ports A, B, and C

  int EnablePortA (); // Enables Port A on CIO Chip
  int EnablePortB (); // Enables Port B on CIO Chip
  int EnablePortC (); // Enables Port C on CIO Chip

  int DisablePortA (); // Disables Port A on CIO Chip
  int DisablePortB (); // Disables Port B on CIO Chip
  int DisablePortC (); // Disables Port C on CIO Chip

  int SetPortDirection (char cPortLetter, Byte cPortDir);
};

#endif /* __VMIO10_H__ */

