/*
   traject.h

   Mark Sibenac
   97-1-30
   Atacama Desert Trek
   Field Robotics Center

   This is a general TRAJECT controller.

   All positions are 24 bit signed numbers that the amps work with directly.
   Accel is totally dependant on the pid update rate and what command means.

   The accel and max velocity variables are scaled according to frequency. So
   set them no differently with a freq of 60Hz or a freq of 200Hz. Accel and 
   max velocity are in units of ticks/s and ticks/s/s respectively.
*/

#ifndef __TRAJECT_H__
#define __TRAJECT_H__

#include "common/nomad_global.h"
#include "components/amps/amps.h"
#include "semLib.h"

class Ctraject {
 private:

 protected:

  // Parameters that user sets
  double  m_fTargetPos;  // Target position that user wants to servo to
  double  m_fMaxVel;     // Max velocity this generator will ramp
  double  m_fAccel;      // Acceleration at which this gen will ramp
  int     m_nFreq;       // Frequency at which this runs
  int     m_nEncoderTicksPerRev; // Number of encoder ticks per output rev
  int     m_nAngular;    // 1-mod everytime with m_nEncoderTicksPerRev 0-do not

  // Parameters that user can get
  double  m_fTime;       // Time ticker
  double  m_fDirection;  // Direction of move; 1:higher numbers
                         //                   -1:lower numbers
  double  m_fSetPoint;   // Trajectory servo point (like a bunny at a dog race)
  double  m_fCurrentVel; // Current velocity at this Time
  Traj_Stat_t m_nStatus; // Status of trajectory
  int     m_nCase;       // Case1-trapezoid, Case2-triangular

  // Parameters internal to trajectory generator
  double  m_fStartPos;   // Starting position at Time=0
  double  m_fInitVel;    // Velocity at Time=0
  double  m_fSMaxVel;    // Sign*MaxVel
  double  m_fSAccel;     // Sign*Accel
  double  m_fLastVel;    // Velocity at Time-1
  double  m_fLastPos;    // Position at Time-1
  double  m_fCase1Zone1Bound; // Max time for zone 1, case 1
  double  m_fCase1Zone2Bound; // Max time for zone 2, case 1
  double  m_fCase1Zone3Bound; // Max time for zone 3, case 1
  double  m_fCase2Zone1Bound; // Max time for zone 1, case 2
  double  m_fCase2Zone2Bound; // Max time for zone 2, case 2
  
  SEM_ID  m_pMutex; // Semaphore to prevent coll between CalcNewPos and SetNewPos
  Camps  *m_amp;         // pointer to object that contains info on positions

  long     GetCurPos         (); // uses m_amp to get current positon feedback
                                // returns it too
  virtual int CalcNewParams (); // calculates new trajectory

 public:

  Ctraject  (Camps *amp_in, int angular);
  virtual ~Ctraject ();

  virtual int CalcNewPos    (); // returns m_fSetPoint; called at servo freq
  int     SetNewTarget      (long target_pos); // calculates new params
  // SetParams returns -1 if it cannot take action at this time, 0 otherwise
  int     SetParams         (double max_vel, double accel, int freq=-1);
  int     SetAccel          (double accel);
  int     SetFreq           (int freq);
  int     Reset             (); // Called when starting servoing a motor

  long    GetNewPos         () {return long(m_fSetPoint);}
  double  GetMaxVel         () {return m_fMaxVel*m_nFreq;}
  double  GetAccel          () {return m_fAccel*m_nFreq*m_nFreq;}
  int     GetFreq           () {return m_nFreq;}
  int     GetTime           () {return int(m_fTime);}
  int     GetDirection      () {return int(m_fDirection);}
  double  GetCurrentVel     () {return m_fCurrentVel*m_nFreq;} // ticks/sec
  int     GetCurrentVelSlice() {return int(m_fCurrentVel);}    // ticks/slice
  long    GetTarget         () {return long(m_fTargetPos);}
  int     GetStatus         () {return int(m_nStatus);}
  int     GetCase           () {return m_nCase;}
};

#endif // __TRAJECT_H__
