/*
   drive.h

   Mark Sibenac
   97-1-28
   Atacama Desert Trek
   Field Robotics Center

   This module controls the coordinated motion and velocity servo loops on the
   four motors. It handles the PID loops and the trajectory generator
   algorithms. It assumes that the PID loop is run at a higher frequency than
   the trajectory generator. 

   Each motor can be in one of the following states: {INHIBITED, SERVOING,
   OVERCURRENT, OVERTEMP, ERROR}

*/

#ifndef __DRIVE_H__
#define __DRIVE_H__

#include "common/nomad_global.h"
#include "components/amps/amps.h"
#include "components/pid/pid.h"
#include "components/traject/traject.h"

/* MAX_DRIVE_VEL = 2700(motor rpm) / 100(harmonic) / 48(pinion) * 22(spurr) 
                   * 2PI(rads/rev) / 60(secs/min)
   Assuming wheel diameter is 76cm, 2700rpm, Max Speed of robot is 49.24cm/s
   DRIVE_ACCEL= (50cm/s)/(4s) = 12.5cm/s^2 = 685 rpm/s = 0.328776 rad/s^2
   MAX_MOTOR_TICKS_SEC = 2700rpm / 60s/min * 1024 tick/rev = 46080 tick/s
   MOTOR_ACCEL = 685 rpm/s / 60 s/min * 1024 tick/rev = 11691 ticks/s^2
*/
const int    MAX_MOTOR_TICKS_SEC  = 46080; // ticks/s
const int    DEFAULT_MOTOR_ACCEL  = 11691; // ticks/s^2
const double MAX_DRIVE_VEL        = 1.29590696961; // rad/s
const double DEFAULT_DRIVE_ACCEL  = 0.328776397843;// rad/s^2

class Cdrive {
 private:
  motor_state_t m_state; // motor state
  int           m_axis_number;

  int           m_traject_time_preset;  // trajectory update sleep length
  int           m_traject_timer; // countdown timer for trajectory update
  int           m_traject_current_des_pos; // since last trajectory update
  int           m_traject_current_des_vel; // since last trajectory update

  double        m_fDesVel; // Desired Velocity in rad/s
  double        m_fCurrentVel; // Current Velocity at this point in time rad/s
  double        m_fAccel;  // Acceleration in rad/s/s

 public:
  Cdrive (int axis_number, Camps *amp);
  ~Cdrive ();

  int           Init          (double Kp, double Ki, double Kd, double I_limit,
			       double MaxPosErr, double MaxAmp, double Accel);
  int           UpdateServo   (); // Called at the servo update rate
  int           ChangeState   (motor_state_t state);
  motor_state_t GetState      () {return m_state;}
  int           SetVelocity   (double rads_sec);
  int           SetUpdateFreq (int pid_freq, int trajectory_freq); //Freq in Hz

  Camps        *m_amp; // angular and 1024 ticks/rev
  Cpid          m_pid; // PID controller
  Ctraject      m_trajectory; // Trapezoidal trajectory generator
};

extern Cdrive *drive; // array of drive objects

#endif // __DRIVE_H__
