/*
   vmio22.cxx

   VME I/O Class File
   VMIO-22 Piggy Back for a VMIO-10 card (opto-isolated A->D)

   96-12-19
   Mark Sibenac
   Field Robotics Center
   Lunar Rover Terrestrial Prototype

   This device assumes you are in -10V to 10V mode and fills the sensors 
   matrix with twos complement numbers from -10V to 10V.

   nSingleDiff = (0-Single-ended, 1-Differential)
   nVmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
   pBoardVmeAddress (VMIO10) = I/O address on VMEbus (not short space)
*/

#include "common/nomad_global.h"
#include "vmio22.h"

/*
   vmioBay = (0->P5,P6,P9,P10;01-16,   1->P3,P4,P7,P8;17-32)
*/
Cvmio22::Cvmio22(const char *pBoardVmeAddress_in, int nVmioBay_in, 
		 int nSingleDiff_in) : Cvmio10(pBoardVmeAddress_in, 
					       nVmioBay_in)
{
  if (!pBoardVmeAddress_in)
    {
      Dprintf(("Cvmio22::Cvmio22 constructor called without BoardVmeAddress\n"));
      return;
    }

  nSingleDiff = nSingleDiff_in;

  Dprintf(("Cvmio22::Cvmio22 Creating a new Cvmio22 object\n"));

  InitPorts (0x00, 0x80, // PORTA = no inversion, all output except bit7,STS
	     0x00, 0xff, // PORTB = no inversion, all input
	     0x00, 0x0f);// PORTC = no inversion, all input
}

int Cvmio22::GetADCI (int nChannel)
{
  register int nData;
  static int nLastChannel=-1;

  semTake (*m_pMutex, WAIT_FOREVER);

  /* clear MA0-MA3=PA0-PA3, 
     A0 = PA5 = 0 (12 bit conv), 
     R/C=PA4=1 MA0-MA3 = channel */
  *pPortA = Byte(0x10) | (Byte(nChannel) & Byte(0x0f)); 

  if ((nSingleDiff == SINGLE_ENDED) && ((nChannel < 0) || (nChannel > 15)))
    {
      Dprintf(("nChannel = %d is illegal in Cvmio22::GetADCI "
	       "Single_ended\n", nChannel));
      return ERROR_ADC_I;
    }
  else if ((nSingleDiff == DIFFERENTIAL) && ((nChannel < 0) || (nChannel > 7)))
    {
      Dprintf(("nChannel = %d is illegal in Cvmio22::GetADCI "
	       "Differential\n", nChannel));
      return ERROR_ADC_I;
    }

  if (nLastChannel != nChannel)
    {
      for (volatile int nCounter=0; nCounter<50; nCounter++);
      nLastChannel = nChannel;
    }

  // R/C = 0 to convert (falling edge causes conversion)
  *pPortA = (Byte(nChannel) & Byte(0x0f)); 

  while ((*pPortA & Byte(0x80)) == Byte(0x80)); // spin until STS goes low

  // read and adjust for +/- measurement
  nData = (((int(*pPortB) & 0xff) << 4) | (int(*pPortC) & 0x0f)) - 0x800; 

  // R/C = 1 to read
  *pPortA = Byte(0x10) | (Byte(nChannel) & Byte(0x0f));
  
  semGive (*m_pMutex);

  return nData;
}

double Cvmio22::GetADCF (int nChannel)
{
  register int nData;

  if ((nData = GetADCI(nChannel)) != ERROR_ADC_I)
    {
      return (double(nData) * VOLT_PER_BIT);
    }

  return ERROR_ADC_I;
}
