/*
   traject.cxx

   Mark Sibenac
   97-1-30
   Atacama Desert Trek
   Field Robotics Center

   This is a general TRAJECTory generator that is used with a closed loop
   controller (Cpid).

   The accel and max velocity variables are scaled according to frequency. So
   set them no differently with a freq of 60Hz or a freq of 200Hz. Accel and 
   max velocity are in units of ticks/s and ticks/s/s respectively.
*/

#include "traject.h"
#include "math.h"

#define ABS(x)   (((x)<0)   ? (-(x))   : ((x)))
#define DABS(x)  (((x)<0.0) ? (-(x))   : ((x)))
#define SIGN(x)  (((x)<0)   ? (-1)     : (1))
#define DSIGN(x) (((x)<0.0) ? (-1.0)   : (1.0))
 
// TRAJECT Class Constructor
Ctraject::Ctraject (Camps *amp_in, int angular)
{
  if (!amp_in)
    {
      Dprintf(("Ctraject::Ctraject() amp_in not initialized!!\n"));
      return;
    }

  Dprintf(("Ctraject::Ctraject() initialize new generator for amp=%x\n", 
	   UINT32(amp_in)));

  m_amp = amp_in;
  m_nEncoderTicksPerRev = m_amp->GetEncoderTicksPerRev();
  m_nAngular = angular;

  m_fDirection = 1.0; // just initialize it to something
  m_fSMaxVel = m_fMaxVel = 1.0;  // initialize it to something
  m_fSAccel   = m_fAccel  = 1.0;  // just initialize it to something
  m_nFreq   = 1;    // just initialize it to something

  m_nCase   = 1;    // Whatever
  m_fTime   = 0.0;    // Time starts at 0
  m_fLastVel = m_fInitVel = m_fCurrentVel = 0.0;  // should be 0
  m_nStatus = IDLE; // Start in the idle state so no new positions are calc'ed
  
  
  m_fLastPos = m_fStartPos = GetCurPos();   // setup m_fStartPos;
  m_fTargetPos = m_fSetPoint = m_fStartPos; // set those params to current pos

  m_pMutex = semMCreate(SEM_Q_FIFO | SEM_DELETE_SAFE);
  Dprintf(("Ctraject::Ctraject() m_pMutex = %x\n", UINT32(m_pMutex)));
}

Ctraject::~Ctraject ()
{
  semFlush (m_pMutex);
  semDelete (m_pMutex);
  Dprintf(("Ctraject::~Ctraject() Mutex Semaphore is gone!\n"));
}

// Retrieves the current position of the motor (encoder) from the m_amp object
int Ctraject::GetCurPos ()
{
  return (m_amp->GetPosition());
}

// Calculates new transfer points and resets the setpoint.
//  Since this function is virtual, any daughter class can change this.
//  This is set up to do a trapezoidal move.
int Ctraject::CalcNewParams ()
{
  register double delta;

  m_fTime      = 0.0; // reset time to 0
//  m_fStartPos  = GetCurPos();
  m_fStartPos  = m_fLastPos;
  m_fLastPos   = m_fSetPoint = m_fStartPos;
  m_fInitVel   = m_fLastVel;
  delta        = m_fTargetPos - m_fStartPos;
  m_fDirection = DSIGN(delta);
  if (m_nAngular && (DABS(delta)>double(m_nEncoderTicksPerRev/2)))
    {
      m_fDirection = -m_fDirection;
      delta = m_fDirection*(m_nEncoderTicksPerRev - DABS(delta));
    }
  m_fSAccel    = m_fDirection * m_fAccel;

  if ((delta*2.0*m_fSAccel+m_fInitVel*m_fInitVel)/2.0 >= m_fMaxVel*m_fMaxVel)
    { 
      m_nCase = 1; 
      m_fSMaxVel   = m_fDirection * m_fMaxVel;
      m_nStatus = CASE1_ZONE1; // need this to run even though it may be false
      m_fCase1Zone1Bound = (m_fSMaxVel - m_fInitVel)/m_fSAccel;
      m_fCase1Zone2Bound = m_fCase1Zone1Bound + (delta-(2*m_fSMaxVel*m_fSMaxVel-m_fInitVel*m_fInitVel)/2.0/m_fSAccel)/m_fSMaxVel;
      m_fCase1Zone3Bound = m_fCase1Zone2Bound + m_fSMaxVel/m_fSAccel;
    } else {
      m_nCase = 2;
      m_fSMaxVel = m_fDirection * sqrt((delta*2.0*m_fSAccel+m_fInitVel*m_fInitVel)/2.0);
      m_nStatus = CASE2_ZONE1; // need this to run even though it may be false
      m_fCase2Zone1Bound = ((m_fSMaxVel - m_fInitVel)/m_fSAccel);
      m_fCase2Zone2Bound = m_fCase2Zone1Bound + m_fSMaxVel/m_fSAccel;
    }
  return 0;
}

//  This function is called when a move needs to be done by the user.
//  It updates the parametrs for the trajectory generator.
//  It can be called when Status is anything
//  target_pos is a 64 bit signed number
int Ctraject::SetNewTarget (long target_pos)
{
  semTake(m_pMutex, WAIT_FOREVER);
  m_fTargetPos = target_pos;
  CalcNewParams();
  semGive(m_pMutex);
  return 0;
}

//  max_vel and accel get stored into the appropriate variables
//  may only be called when Status = IDLE
//  if Status != IDLE, it returns -1 and doesn't do anything  
int Ctraject::SetParams (double max_vel, double accel, int freq)
{
  Dprintf(("Ctraject::SetParams max_vel=%f, accel=%f, freq=%d\n", max_vel,
	   accel, freq));

  if (freq == 0)
    {
      Dprintf(("Ctraject::SetParams() illegeal freq = %d\n", freq));
      return -1;
    }

  if (accel == 0.0)
    {
      Dprintf(("Ctraject::SetParams() illegeal accel = %f\n", accel));
      return -1;
    }

  semTake(m_pMutex, WAIT_FOREVER);
  if ((freq != -1) && (freq != 0))
    { m_nFreq   = freq; }
  m_fSMaxVel = m_fMaxVel = max_vel / double(m_nFreq);
  m_fSAccel  = m_fAccel  = accel   / double(m_nFreq) / double(m_nFreq); 
  CalcNewParams();
  semGive(m_pMutex);
  return 0;
}

int Ctraject::SetFreq (int freq)
{
  if (freq == 0)
    {
      Dprintf(("Ctraject::SetFreq() illegal freq=%d", freq));
      return -1;
    }
  
  return (SetParams(m_fMaxVel, m_fAccel, freq));
}


//  Sets the Acceleration by calling SetParams with the current MaxVel and freq
int Ctraject::SetAccel (double accel)
{
  return (SetParams (m_fMaxVel, accel));
}

//   This is called at the servoing frequency.
//   It returns the new position that the controller needs to servo to.
//   Since this function is virtual, any daughter class can change this.
//   This is set up to do a trapezoidal move.
int Ctraject::CalcNewPos ()
{
  semTake(m_pMutex, WAIT_FOREVER);

  m_fLastPos = m_fSetPoint;
  m_fLastVel = m_fCurrentVel;

  if (m_nStatus == IDLE) 
    {  
      semGive(m_pMutex);
      return int(m_fSetPoint);
    }

  if (m_nCase == 1) // Case 1
    {
      if      (m_fTime < m_fCase1Zone1Bound) 
	       m_nStatus = CASE1_ZONE1;
      else if (m_fTime < m_fCase1Zone2Bound)
	       m_nStatus = CASE1_ZONE2;
      else if (m_fTime < m_fCase1Zone3Bound)
	       m_nStatus = CASE1_ZONE3;
      else     m_nStatus = CASE1_ZONE4;
    } else { // Case 2
      if      (m_fTime < m_fCase2Zone1Bound) 
	       m_nStatus = CASE2_ZONE1;
      else if (m_fTime < m_fCase2Zone2Bound)
	       m_nStatus = CASE2_ZONE2;
      else     m_nStatus = CASE2_ZONE3;
    }

  switch (m_nStatus)
    {
      case CASE1_ZONE1: 
          m_fSetPoint = m_fLastPos + m_fLastVel + m_fSAccel/2.0;
	  m_fCurrentVel = m_fLastVel + m_fSAccel; 
	  break;
      case CASE1_ZONE2: 
	  m_fSetPoint = m_fLastPos + m_fSMaxVel;
	  m_fCurrentVel = m_fSMaxVel; 
	  break;
      case CASE1_ZONE3: 
	  m_fSetPoint = m_fLastPos + m_fLastVel - m_fSAccel/2.0;
	  m_fCurrentVel = m_fLastVel - m_fSAccel; 
	  break;
      case CASE1_ZONE4: 
	  m_fSetPoint = m_fTargetPos;
	  m_fCurrentVel = 0.0; 
	  m_nStatus = IDLE; 
	  break;

      case CASE2_ZONE1: 
	  m_fSetPoint = m_fLastPos + m_fLastVel + m_fSAccel/2.0;
	  m_fCurrentVel = m_fLastVel + m_fSAccel; 
	  break;
      case CASE2_ZONE2:  
	  m_fSetPoint = m_fLastPos + m_fLastVel - m_fSAccel/2.0;
	  m_fCurrentVel = m_fLastVel - m_fSAccel; 
	  break;
      case CASE2_ZONE3: 
	  m_fSetPoint = m_fTargetPos;
	  m_fCurrentVel = 0.0; 
	  m_nStatus = IDLE; 
	  break;
      case IDLE:
	  break;
    }

  if (m_nAngular)
    {
      if (m_fSetPoint > double(m_nEncoderTicksPerRev/2))
	{
	  m_fSetPoint = m_fSetPoint - double(m_nEncoderTicksPerRev);
	}
      if (m_fSetPoint < double(-m_nEncoderTicksPerRev/2))
	{
	  m_fSetPoint = m_fSetPoint + double(m_nEncoderTicksPerRev);
	}
    }

  m_fTime++;

  semGive(m_pMutex);
  return int(m_fSetPoint);
}

// Called when starting up a motor into the SERVOING state so things are 
//  stopped at startup
int Ctraject::Reset()
{
  semTake(m_pMutex, WAIT_FOREVER);

  m_fLastVel = 0.0;
  m_fLastPos = GetCurPos();
  
  SetNewTarget(int(m_fLastPos));
  
  semGive(m_pMutex);
  return 0;
}
