/*
   pid.h

   Mark Sibenac
   97-1-14
   Atacama Desert Trek
   Field Robotics Center

   This is a general PID controller based on positions and velocities. Every
   update cycle, it calls amp->GetPosition() for the current position. The
   velocity is sampled at an interval assigned by calling SetVelUpdateRate().
   The PID gains are normalized so the different servo update rates don't
   affect the commands. The gains are defined as the amount of command gain
   per second.

*/

#ifndef __PID_H__
#define __PID_H__

#include "common/nomad_global.h"
#include "components/amps/amps.h"

class Cpid {
 private:

  // Parameters
  double fKp;
  double fKi;
  double fKd;
  double fI_limit;
  double fMaxPosErr;

  int    nServoUpdateRate; // Update rate in Hz of the servo routine
  int    nVelUpdateRate; // Update rate in Hz of velocity data
  int    nVelUpdateCount; // counter that increments every cycle
  int    nVelUpdateCountMask; //if (nUpdateCount == this) it does new diff calc
  double fVelocity; // velocity at delta reading
  double fDeltaPos; // position at delta reading

  int    nEncoderTicksPerRev; // Encoder ticks per output revolution
  int    nAngular; // 1-yes mod everytime with nEncoderTicksPerRev, 0-no

  // Servo Variables
  double fPosError;
  double fCurPos;
  double fLastPos;
  double fSumErr;
  double fDesPos;
  double fCommand;

  Camps  *m_amp; // pointer to the amp that is using this PID controller
  double GetPosition () {return m_amp->GetPosition();}
  int    RecalcGains (int old_freq); // called after a frequency change occurs

 protected:

 public:

  Cpid (Camps *amp_in, int angular);

  int    Init (double Kp, double Ki, double Kd, double I_limit, 
	       double MaxPosErr);
  int    Reset ();
  int    ServoLaw(double fDesiredVel); // called at the update frequency
  int    SetDesPos (double DesPos) {fDesPos = DesPos; return 0;}

  double SetKp (double Kp) {fKp = Kp / double(nServoUpdateRate);}
  double SetKi (double Ki) {fKi = Ki / double(nServoUpdateRate);}
  double SetKd (double Kd) {fKd = Kd / double(nServoUpdateRate);}
  double SetILimit (double Ilimit) {fI_limit = Ilimit;}
  double SetMaxPosErr (double MaxPosErr) {fMaxPosErr = MaxPosErr;}
  int    SetVelUpdateRate (int vel_update_rate_hz);
  int    SetUpdateFreq    (int servo_update_rate_hz);

  double GetKp ()          {return fKp * double(nServoUpdateRate);}
  double GetKi ()          {return fKi * double(nServoUpdateRate);}
  double GetKd ()          {return fKd * double(nServoUpdateRate);}
  double GetILimit ()      {return fI_limit;}
  double GetMaxPosErr ()   {return fMaxPosErr;}
  double GetCommand ()     {return fCommand;}
  double GetDesPos ()      {return fDesPos;}
  double GetCurPos ()      {return fCurPos;}
  double GetPosError()     {return fPosError;}
};

#endif // __PID_H__
