/*
 * rt_data.h
 *
 * Mark Sibenac
 * 97-2-24
 * Field Robotics Center
 * Atacama Desert Trek
 *
 * This file defines the data structures in shared memory on the real-time
 * computer box. Periodically, processes will update this information and
 * write a time stamp to denote the last update. 
 *
 * The functions in rt_interface.h are used to access these data structures
 * for reading. 
 *
 */

#ifndef __RT_DATA_H__
#define __RT_DATA_H__

typedef struct {
  boolean rts_status;     /* 1=up, 0-down */
  boolean nav_status;     /* 1=up, 0-down */
  boolean panos_status;   /* 1=up, 0-down */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} computing_status_t, *computing_status_ptr;

typedef struct {
  float   current_fb[6];  /* -10.0 Amps->+10.0 Amps current monitor from amp */
  float   motor_rpm[6];   /* motor rpm derived from encoder ticks/s */
  float   wheel_speed[4]; /* wheel speed derived from motor_rpm */
  float   command[6];     /* -10.0 Amps->+10.0 Amps commanded current to amp */
  long    abs_position[6];/* 64 bit absolute position of encoder(motor) */
  long    target_position[6]; /* absolute target position of encoder(motor) */
  float   target_vel[4];  /* target velocity of motor (N/A for steering) */

  boolean amp_fault[6];   /* fault status from amp */
  boolean temp_fault[6];  /* over temperature fault from add-on board */
  boolean enable[6];      /* enable status for amp */

  boolean brake_enable[2];/* 0=not engaged, 1=engaged */
  float   steering_pot[2];/* meters */
  float   bogie_angle[2]; /* -pi -> +pi rads */

  float   robot_speed;     /* speed in m/s */
  float   robot_radius;    /* radius in m */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} locomotion_data_t, *locomotion_data_ptr;

typedef struct {
  float   current[2];     /* -10.0 Amps->+10.0 Amps current monitor from amp */
  float   motor_rpm[2];   /* motor rpm derived from encoder ticks/s */
  float   command[2];     /* -10.0 Amps->+10.0 Amps commanded current to amp */
  long    abs_position[2];/* 64 bit absolute position of encoder */
  long    target_position[2]; /* absolute target position of encoder(motor) */

  boolean fault[2];       /* fault status from amp */
  boolean temp_fault[2];  /* over temperature fault from add-on board */
  boolean enable[2];      /* enable status for this amp */

  float   azimuth;        /* target azimuth angle in rads [-pi,pi] */
  float   elevation;      /* target elevation angle in rads [-pi,pi] */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} aps_data_t, *aps_data_ptr;

typedef struct {
  int     fuel_level;     /* 0 -> 100 (percentage) */
  float   bus_voltage;    /* 0.0->200.0 Volts (read from UPS) */
  int     gen_rpm;        /* generator rpm */
  boolean gen_status;     /* 0=down, 1=up */

  boolean ups_staus;      /* 0=down, 1=up */
  float   ups_volt;       /* ups voltage */
  float   ups_remain;     /* juice remaining */

  boolean nav_power;      /* 0=not powered, 1=powered */
  boolean pan_power;      /* 0=not powered, 1=powered */
  boolean comm_power;     /* 0=not powered, 1=powered */
  boolean science_power;  /* 0=not powered, 1=powered */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} power_status_t, *power_status_ptr;

typedef struct {
  int     temp;           /* temperature degrees C */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} weather_data_t *weather_data_ptr;

typedef struct {
  float   radius;         /*radius in meters of an arc (positive is to right)*/
  float   speed;          /* speed in m/s to travel in this arc */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} nav_command_t, *nav_command_ptr;

typedef struct {
  boolean passable;       /* 1=passable terrain, 0=not passable */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} laser_sight_t, *laser_sight_ptr;

typedef struct {
  int     x_pos;          /* joystick x position -120 -> 120 */
  int     y_pos;          /* joystick y position -120 -> 120 */
  boolean enable_but;     /* 0=not pressed, 1=pressed */
  boolean button_2;       /* 0=not pressed, 1=pressed */
  boolean sw1;            /* 0, 1 */
  boolean sw2;            /* 0, 1 */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} pendant_command_t, *pendant_command_ptr;

typedef struct {
  long    msecOfGPSWeek;  
  short   GPSWeek;
  uchar   numSVSTracked;
  uchar   numSVSVisible;
  uchar   positionFlags1;
  uchar   positionFlags2;
  uchar   initializationNumber;
  time_t  lastDiffCorr;
  double  userDatumNorthOrX;
  double  userDatumEastOrY;
  double  userDatumUporZ;
  uchar   velocity_flags;
  float   horizVelocity;
  float   heading;
  float   vertVelocity;
  float   positionRMSError;
  float   sigmaEast;
  float   sigmaNorth;
  float   eastNorthCovar;
  float   sigmaUp;
  float   semiMajorAxis;
  float   semiMinorAxis;
  float   sigmaOrientation;
  float   unitVariance;
  short   numEpochs;

  time_t  timestamp;
  SEM_ID  sem_mutex;
} gps_derived_t, *gps_derived_ptr;

typedef struct {
  double  receiveTime;
  double  clockOffset;
  uchar   numSVSVisible;
  uchar   visibleSvsPRN[10];
  uchar   svFlags1[10];
  uchar   svFlags2[10];
  char    svsElevation[10];
  short   svsAzimuth[10];
  char    svsL1SNR[10];
  char    svsL2SNR[10];
  double  L1PseudoRange[10];
  double  L1ContPhase[10];
  float   L1Doppler[10];
  double  L2ContPhase[10];
  float   L2L1PseudoRange[10];
  char    issueOfDataEphem[10];
  char    L1Slip[10];
  char    L2Slip[10];
  double  wgs84Latitude;
  double  wgs84Longitude;
  double  wgs84Altitude;
  double  clockOffset;
  double  positionDOP;
  double  latitudeRate;
  double  longitudeRate;
  double  alititudeRate;
  ulong   msecOfGPSWeek;
  char    positionFlags;
  uchar   numSVSTracked;
  uchar   svsChannelNumber[10];
  uchar   trackedSVS_PRN[10];

  time_t  timestamp;
  SEM_ID  sem_mutex;
} gps_raw_t, *gps_raw_ptr;

typedef struct {
  float   mag_azimuth;           /* -pi -> +pi rads */ 
  float   pitch;                 /* -pi -> +pi rads */ 
  float   roll;                  /* -pi -> +pi rads */ 

  time_t  timestamp;
  SEM_ID  sem_mutex;
} kvh_data_t, *kvh_data_ptr;

typedef struct {
  float   x_rate;                /* rate in rads/s */
  float   y_rate;                /* rate in rads/s */
  float   z_rate;                /* rate in rads/s */

  time_t  timestamp;
  SEM_ID  sem_mutex;
} imu_raw_t, *imu_raw_ptr;

/* /~ Use this as a blank structure ~/

typedef struct {
  
  time_t  timestamp;
  SEM_ID  sem_mutex;
} _t, *_ptr;

*/

#endif /* __RT_DATA_H__ */
